/*
 *  Copyright (C) 2008-2012, Parallels, Inc. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <sys/ioctl.h>
#include <limits.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <dirent.h>

#include "ploop.h"
#include "common.h"

static void usage_summary(void)
{
	fprintf(stderr, "Usage: ploop init -s SIZE [-f FORMAT] NEW_DELTA\n"
			"       ploop mount [-rP] [-f raw] [-d DEVICE] BASE_DELTA [ ... TOP_DELTA]\n"
			"       ploop umount -d DEVICE\n"
			"       ploop { delete | rm } -d DEVICE -l LEVEL\n"
			"       ploop merge -d DEVICE [-l LEVEL[..TOP_LEVEL]]\n"
			"       ploop fsck [-fcr] DELTA\n"
			"       ploop getdev\n"
			"       ploop resize -s SIZE BASE_DELTA\n"
			"       ploop snapshot [-F] -d DEVICE NEW_DELTA\n"
			"       ploop snapshot DiskDescriptor.xml\n"
			"       ploop snapshot-delete -u <uuid> DiskDescriptor.xml\n"
			"       ploop snapshot-merge [-u <uuid>] DiskDescriptor.xml\n"
			"       ploop snapshot-switch -u <uuid> DiskDescriptor.xml\n"
			"Also:  ploop { stat | start | stop | clear } ...\n"
	       );
}

static void usage_init(void)
{
	fprintf(stderr, "Usage: ploop init -s SIZE [-f FORMAT] [-t FSTYPE] [-b BLOCKSIZE] DELTA\n"
			"       SIZE := NUMBER[kmg]\n"
			"       FORMAT := { raw | ploop1 }\n"
			"       DELTA := path to new image file\n"
			"       BLOCKSIZE := block size in sectors\n"
			"       FSTYPE := make file system\n");
}

static int plooptool_init(int argc, char **argv)
{
	int i, f, ret;
	off_t size_sec = 0;
	struct ploop_create_param param = {};

	param.mode = PLOOP_EXPANDED_MODE;
	while ((i = getopt(argc, argv, "s:b:f:t:")) != EOF) {
		switch (i) {
		case 's':
			if (parse_size(optarg, &size_sec)) {
				usage_init();
				return SYSEXIT_PARAM;
			}
			break;
		case 'b': {
			  char * endptr;

			  param.blocksize = strtoul(optarg, &endptr, 0);
			  if (*endptr != '\0') {
				  usage_init();
				  return SYSEXIT_PARAM;
			  }
			  break;
		}
		case 'f':
			f = parse_format_opt(optarg);
			if (f < 0) {
				usage_init();
				return SYSEXIT_PARAM;
			}
			param.mode = f;
			break;
		case 't':
			if (!strcmp(optarg, "ext4") || !strcmp(optarg, "ext3")) {
				param.fstype = strdup(optarg);
			} else {
				fprintf(stderr, "Incorrect file system type specified: %s\n",
						optarg);
				return SYSEXIT_PARAM;
			}
			break;
		default:
			usage_init();
			return SYSEXIT_PARAM;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1 || size_sec == 0) {
		usage_init();
		return SYSEXIT_PARAM;
	}
	param.size = (__u64) size_sec;
	param.image = argv[0];
	ret = ploop_create_image(&param);
	if (ret)
		return ret;

	return 0;
}

static void usage_mount(void)
{
	fprintf(stderr, "Usage: ploop mount [-r] [-f FORMAT] [-b BLOCKSIZE] [-d DEVICE]\n"
			"             [-m MOUNT_POINT] [-t FSTYPE] [-o MOUNT_OPTS]\n"
			"             BASE_DELTA [ ... TOP_DELTA ]\n"
			"       ploop mount [-r] [-m MOUNT_POINT] [-u UUID] DiskDescriptor.xml\n"
			"       FORMAT := { raw | ploop1 }\n"
			"       BLOCKSIZE := block size (for raw image format)\n"
			"       DEVICE := ploop device, e.g. /dev/ploop0\n"
			"       MOUNT_POINT := directory to mount in-image filesystem to\n"
			"       FSTYPE := in-image filesystem type (ext4 by default)\n"
			"       MOUNT_OPTS := additional mount options, comma-separated\n"
			"       *DELTA := path to image file\n"
			"       -r     - mount images read-only\n"
		);
}

static int plooptool_mount(int argc, char **argv)
{
	int i, f, ret = 0;
	int raw = 0;
	int base = 0;
	struct ploop_mount_param mountopts = {};
	const char *component_name = NULL;

	while ((i = getopt(argc, argv, "rf:d:m:t:u:o:b:c:")) != EOF) {
		switch (i) {
		case 'd':
			strncpy(mountopts.device, optarg, sizeof(mountopts.device)-1);
			break;
		case 'r':
			mountopts.ro = 1;
			break;
		case 'f':
			f = parse_format_opt(optarg);
			if (f < 0) {
				usage_mount();
				return SYSEXIT_PARAM;
			}
			raw = (f == PLOOP_RAW_MODE);
			break;
		case 'm':
			mountopts.target = strdup(optarg);
			break;
		case 't':
			mountopts.fstype = strdup(optarg);
			break;
		case 'u':
			if (!strcmp(optarg, "base"))
				base = 1;
			else
				mountopts.guid = strdup(optarg);
			break;
		case 'o':
			mountopts.mount_data = strdup(optarg);
			break;
		case 'b': {
			  char * endptr;

			  mountopts.blocksize = strtoul(optarg, &endptr, 0);
			  if (*endptr != '\0') {
				  usage_mount();
				  return SYSEXIT_PARAM;
			  }
			  break;
		case 'c':
			component_name = optarg;
			break;
		}
		default:
			usage_mount();
			return SYSEXIT_PARAM;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc < 1) {
		usage_mount();
		return SYSEXIT_PARAM;
	}

	if (argc == 1 && is_xml_fname(argv[0]))
	{
		struct ploop_disk_images_data *di;
		ret = read_dd(&di, argv[0]);
		if (ret)
			return ret;

		if (component_name != NULL)
			ploop_set_component_name(di, component_name);
		if (base) {
			mountopts.guid = ploop_get_base_delta_uuid(di);
			if (mountopts.guid == NULL) {
				ret = SYSEXIT_DISKDESCR;
				fprintf(stderr, "Unable to find base delta uuid\n");
				goto err;
			}
		}
		ret = ploop_mount_image(di, &mountopts);
err:
		ploop_free_diskdescriptor(di);
	}
	else
		ret = ploop_mount(NULL, argv, &mountopts, raw);

	return ret;
}

static void usage_start(void)
{
	fprintf(stderr, "Usage: ploop start -d DEVICE\n"
			"       DEVICE := ploop device, e.g. /dev/ploop0\n"
			);
}

static int plooptool_start(int argc, char **argv)
{
	int i;
	int lfd;
	struct {
		char * device;
	} startopts = { };

	while ((i = getopt(argc, argv, "d:")) != EOF) {
		switch (i) {
		case 'd':
			startopts.device = optarg;
			break;
		default:
			usage_start();
			return SYSEXIT_PARAM;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc || !startopts.device) {
		usage_start();
		return SYSEXIT_PARAM;
	}

	lfd = open(startopts.device, O_RDONLY);
	if (lfd < 0) {
		perror("Can't open device");
		return SYSEXIT_DEVICE;
	}

	if (ioctl(lfd, PLOOP_IOC_START, 0) < 0) {
		perror("PLOOP_IOC_START");
		return SYSEXIT_DEVIOC;
	}

	if (ioctl(lfd, BLKRRPART, 0) < 0) {
		perror("BLKRRPART");
	}

	return 0;
}

static void usage_stop(void)
{
	fprintf(stderr, "Usage: ploop stop -d DEVICE\n"
			"       DEVICE := ploop device, e.g. /dev/ploop0\n");
}

static int plooptool_stop(int argc, char **argv)
{
	int i;
	int lfd;
	struct {
		char * device;
	} stopopts = { };

	while ((i = getopt(argc, argv, "d:")) != EOF) {
		switch (i) {
		case 'd':
			stopopts.device = optarg;
			break;
		default:
			usage_stop();
			return SYSEXIT_PARAM;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc || !stopopts.device) {
		usage_stop();
		return SYSEXIT_PARAM;
	}

	lfd = open(stopopts.device, O_RDONLY);
	if (lfd < 0) {
		perror("Can't open device");
		return SYSEXIT_DEVICE;
	}

	if (ioctl(lfd, PLOOP_IOC_STOP, 0) < 0) {
		perror("PLOOP_IOC_STOP");
		return SYSEXIT_DEVIOC;
	}

	return 0;
}

static void usage_clear(void)
{
	fprintf(stderr, "Usage: ploop clear -d DEVICE\n"
			"       DEVICE := ploop device, e.g. /dev/ploop0\n");
}

static int plooptool_clear(int argc, char **argv)
{
	int i;
	int lfd;
	struct {
		char * device;
	} stopopts = { };

	while ((i = getopt(argc, argv, "d:")) != EOF) {
		switch (i) {
		case 'd':
			stopopts.device = optarg;
			break;
		default:
			usage_clear();
			return SYSEXIT_PARAM;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc || !stopopts.device) {
		usage_clear();
		return SYSEXIT_PARAM;
	}

	lfd = open(stopopts.device, O_RDONLY);
	if (lfd < 0) {
		perror("Can't open device");
		return SYSEXIT_DEVICE;
	}

	if (ioctl(lfd, PLOOP_IOC_CLEAR, 0) < 0) {
		perror("PLOOP_IOC_CLEAR");
		return SYSEXIT_DEVIOC;
	}

	return 0;
}

static void usage_umount(void)
{
	fprintf(stderr, "Usage: ploop umount -d DEVICE\n"
			"       ploop umount -m DIR\n"
			"       ploop umount DiskDescriptor.xml\n"
			"       ploop umount DELTA\n"
			"       DEVICE := ploop device, e.g. /dev/ploop0\n"
			"       DIR := mount point\n"
			"       DELTA := path to (mounted) image file\n");
}

static int plooptool_umount(int argc, char **argv)
{
	int i, ret;
	char *mnt = NULL;
	char device[PATH_MAX];
	struct {
		char * device;
	} umountopts = { };
	const char *component_name = NULL;

	while ((i = getopt(argc, argv, "d:m:c:")) != EOF) {
		switch (i) {
		case 'd':
			umountopts.device = optarg;
			break;
		case 'm':
			mnt = optarg;
			break;
		case 'c':
			component_name = optarg;
			break;
		default:
			usage_umount();
			return SYSEXIT_PARAM;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1 && !umountopts.device && !mnt) {
		usage_umount();
		return SYSEXIT_PARAM;
	}

	if (umountopts.device != NULL) {
		ret = ploop_umount(umountopts.device, NULL);
	}else if (mnt != NULL) {
		if (ploop_get_dev_by_mnt(mnt, device, sizeof(device))) {
			fprintf(stderr, "Unable to find ploop device by %s\n",
					mnt);
			return SYSEXIT_PARAM;
		}
		ret = ploop_umount(device, NULL);
	} else if (is_xml_fname(argv[0])) {
		struct ploop_disk_images_data *di;
		ret = read_dd(&di, argv[0]);
		if (ret)
			return ret;

		if (component_name != NULL)
			ploop_set_component_name(di, component_name);

		ret = ploop_umount_image(di);

		ploop_free_diskdescriptor(di);
	} else {
		if (ploop_find_dev(component_name, argv[0], device, sizeof(device)) != 0) {
			fprintf(stderr, "Image %s is not mounted\n", argv[0]);
			return SYSEXIT_PARAM;
		}
		ret = ploop_umount(device, NULL);
	}

	return ret;
}

static void usage_rm(void)
{
	fprintf(stderr, "Usage: ploop { delete | rm } -d DEVICE -l LEVEL\n"
			"       LEVEL := NUMBER, distance from base delta\n"
			"       DEVICE := ploop device, e.g. /dev/ploop0\n");
}

static int plooptool_rm(int argc, char **argv)
{
	int i;
	int lfd;
	__u32 level;

	struct {
		int level;
		char * device;
	} rmopts = { .level = -1, };

	while ((i = getopt(argc, argv, "d:l:")) != EOF) {
		switch (i) {
		case 'd':
			rmopts.device = optarg;
			break;
		case 'l':
			rmopts.level = atoi(optarg);
			break;
		default:
			usage_rm();
			return SYSEXIT_PARAM;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc || !rmopts.device || rmopts.level < 0 || rmopts.level > 127) {
		usage_rm();
		return SYSEXIT_PARAM;
	}

	lfd = open(rmopts.device, O_RDONLY);
	if (lfd < 0) {
		perror("open dev");
		return SYSEXIT_DEVICE;
	}

	level = rmopts.level;
	if (ioctl(lfd, PLOOP_IOC_DEL_DELTA, &level) < 0) {
		perror("PLOOP_IOC_DEL_DELTA");
		return SYSEXIT_DEVIOC;
	}
	return 0;
}

static void usage_snapshot(void)
{
	fprintf(stderr, "Usage: ploop snapshot [-u <uuid>] DiskDescriptor.xml\n"
			"       ploop snapshot [-F] -d DEVICE DELTA\n"
			"       DEVICE := ploop device, e.g. /dev/ploop0\n"
			"       DELTA := path to new image file\n"
			"       -F     - synchronize file system before taking snapshot\n"
		);
}

static int plooptool_snapshot(int argc, char **argv)
{
	int i, ret;
	char *device = NULL;
	int syncfs = 0;
	struct ploop_snapshot_param param = {};

	while ((i = getopt(argc, argv, "Fd:u:")) != EOF) {
		switch (i) {
		case 'd':
			device = optarg;
			break;
		case 'F':
			syncfs = 1;
			break;
		case 'u':
			param.guid = optarg;
			break;
		default:
			usage_snapshot();
			return SYSEXIT_PARAM;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1) {
		usage_snapshot();
		return SYSEXIT_PARAM;
	}

	if (is_xml_fname(argv[0])) {
		struct ploop_disk_images_data *di;
		ret = read_dd(&di, argv[0]);
		if (ret)
			return ret;

		ret = ploop_create_snapshot(di, &param);

		ploop_free_diskdescriptor(di);
	} else {
		__u32 blocksize = 0;
		if (!device) {
			usage_snapshot();
			return SYSEXIT_PARAM;
		}
		if (ploop_get_attr(device, "block_size", (int*) &blocksize))
			return SYSEXIT_SYSFS;
		ret = create_snapshot(device, argv[0], blocksize, syncfs);
	}

	return ret;
}

static void usage_snapshot_switch(void)
{
	fprintf(stderr, "Usage: ploop snapshot-switch -u <uuid> DiskDescriptor.xml\n"
			"       -u <uuid>     snapshot uuid\n");
}

static int plooptool_snapshot_switch(int argc, char **argv)
{
	int i, ret;
	char *uuid = NULL;
	int flags = 0;
	struct ploop_disk_images_data *di = NULL;

	while ((i = getopt(argc, argv, "u:D")) != EOF) {
		switch (i) {
		case 'u':
			uuid = optarg;
			break;
		case 'D':
			/* for test purposes */
			flags = PLOOP_SNAP_SKIP_TOPDELTA_DESTROY;
			break;
		default:
			usage_snapshot_switch();
			return SYSEXIT_PARAM;
		}
	}

	argc -= optind;
	argv += optind;

	if ((argc != 1 && !is_xml_fname(argv[0])) || uuid == NULL) {
		usage_snapshot_switch();
		return SYSEXIT_PARAM;
	}

	ret = read_dd(&di, argv[0]);
	if (ret)
		return ret;

	ret = ploop_switch_snapshot(di, uuid, flags);

	ploop_free_diskdescriptor(di);

	return ret;
}

static void usage_snapshot_delete(void)
{
	fprintf(stderr, "Usage: ploop snapshot-delete -u <uuid> DiskDescriptor.xml\n"
			"       -u <uuid>     snapshot uuid\n");
}

static int plooptool_snapshot_delete(int argc, char **argv)
{
	int i, ret;
	char *uuid = NULL;
	struct ploop_disk_images_data *di = NULL;

	while ((i = getopt(argc, argv, "u:")) != EOF) {
		switch (i) {
		case 'u':
			uuid = optarg;
			break;
		default:
			usage_snapshot_delete();
			return SYSEXIT_PARAM;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1 || !is_xml_fname(argv[0]) || uuid == NULL) {
		usage_snapshot_delete();
		return SYSEXIT_PARAM;
	}

	ret = read_dd(&di, argv[0]);
	if (ret)
		return ret;

	ret = ploop_delete_snapshot(di, uuid);

	ploop_free_diskdescriptor(di);

	return ret;
}

static void usage_snapshot_merge(void)
{
	fprintf(stderr, "Usage: ploop snapshot-merge [-u <uuid> | -A] DiskDescriptor.xml\n"
			"       -u <uuid>     snapshot uuid (top delta if not specified)\n");
}

static int plooptool_snapshot_merge(int argc, char ** argv)
{
	int i, ret;
	struct ploop_merge_param param = {};

	while ((i = getopt(argc, argv, "u:A")) != EOF) {
		switch (i) {
		case 'u':
			param.guid = optarg;
			break;
		case 'A':
			param.merge_all = 1;
			break;
		default:
			usage_snapshot_merge();
			return SYSEXIT_PARAM;
		}
	}

	argc -= optind;
	argv += optind;

	if (param.guid != NULL && param.merge_all != 0) {
		fprintf(stderr, "Options -u and -A can't be used together\n");
		usage_snapshot_merge();
		return SYSEXIT_PARAM;
	}

	if (argc == 1 && is_xml_fname(argv[0])) {
		struct ploop_disk_images_data *di;
		ret = read_dd(&di, argv[0]);
		if (ret)
			return ret;

		ret = ploop_merge_snapshot(di, &param);

		ploop_free_diskdescriptor(di);
	} else {
		usage_snapshot_merge();
		return SYSEXIT_PARAM;
	}

	return ret;
}


static void usage_getdevice(void)
{
	fprintf(stderr, "Usage: ploop getdev\n"
			"(ask /dev/ploop0 about first unused minor number)\n"
		);
}

static int plooptool_getdevice(int argc, char **argv)
{
	int minor, fd;

	if (argc != 1) {
		usage_getdevice();
		return SYSEXIT_PARAM;
	}
	fd = ploop_getdevice(&minor);
	if (fd < 0)
		return 1;
	close(fd);
	printf("Next unused minor: %d\n", minor);

	return 0;
}

static void usage_resize(void)
{
	fprintf(stderr, "Usage: ploop resize -s NEW_SIZE DiskDescriptor.xml\n"
			"       NEW_SIZE := NUMBER[kmg]\n");
}

static int plooptool_resize(int argc, char **argv)
{
	int i, ret;
	off_t new_size = 0; /* in sectors */
	int max_balloon_size = 0; /* make balloon file of max possible size */
	struct ploop_resize_param param = {};
	struct ploop_disk_images_data *di;

	while ((i = getopt(argc, argv, "s:b")) != EOF) {
		switch (i) {
		case 's':
			if (parse_size(optarg, &new_size)) {
				usage_resize();
				return SYSEXIT_PARAM;
			}
			param.size = new_size;
			break;
		case 'b':
			max_balloon_size = 1;
			break;
		default:
			usage_resize();
			return SYSEXIT_PARAM;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1 ||
			(new_size == 0 && !max_balloon_size) ||
			!is_xml_fname(argv[0]))
	{
		usage_resize();
		return SYSEXIT_PARAM;
	}

	ret = read_dd(&di, argv[0]);
	if (ret)
		return ret;

	ret = ploop_resize_image(di, &param);

	ploop_free_diskdescriptor(di);

	return ret;
}

static void usage_convert(void)
{
	fprintf(stderr, "Usage: ploop convert -f FORMAT DiskDescriptor.xml\n"
			"       FORMAT := { raw | preallocated }\n"
			);
}

static int plooptool_convert(int argc, char **argv)
{
	int i, ret;
	struct ploop_disk_images_data *di;
	int mode = -1;

	while ((i = getopt(argc, argv, "f:")) != EOF) {
		switch (i) {
		case 'f':
			mode = parse_format_opt(optarg);
			break;
		default:
			usage_convert();
			return SYSEXIT_PARAM;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc == 0 || mode == -1) {
		usage_convert();
		return SYSEXIT_PARAM;
	}

	ret = read_dd(&di, argv[0]);
	if (ret)
		return ret;

	ret = ploop_convert_image(di, mode, 0);

	ploop_free_diskdescriptor(di);

	return ret;
}

static void usage_info(void)
{
	fprintf(stderr, "Usage: ploop info DiskDescriptor.xml\n");
}

static void print_info(struct ploop_info *info)
{
	printf("%11s %14s %14s\n",
			"resource", "Size", "Used");
	printf("%11s %14llu %14llu\n",
			"1k-blocks",
			(info->fs_blocks * info->fs_bsize) >> 10,
			((info->fs_blocks - info->fs_bfree) * info->fs_bsize) >> 10);
	printf("%11s %14llu %14llu\n",
			"inodes",
			info->fs_inodes,
			(info->fs_inodes - info->fs_ifree));
}

static int plooptool_info(int argc, char **argv)
{
	int ret;
	struct ploop_info info = {};

	/* skip the command itself (i.e. "info") */
	argc -= 1;
	argv += 1;

	if (argc != 1) {
		usage_info();
		return SYSEXIT_PARAM;
	}

	ret = ploop_get_info_by_descr(argv[0], &info);
	if (ret == 0)
		print_info(&info);

	return ret;
}

static int plooptool_list(int argc, char **argv)
{
	char fname[PATH_MAX];
	char image[PATH_MAX];
	DIR *dp;
	struct stat st;
	struct dirent *de;
	char cookie[PLOOP_COOKIE_SIZE];

	snprintf(fname, sizeof(fname) - 1, "/sys/block/");
	dp = opendir(fname);
	if (dp == NULL) {
		fprintf(stderr, "Can't opendir %s: %m", fname);
		return 1;
	}
	while ((de = readdir(dp)) != NULL) {
		if (strncmp("ploop", de->d_name, 5))
			continue;

		snprintf(fname, sizeof(fname), "/sys/block/%s/pdelta/0/image",
				de->d_name);
		if (stat(fname, &st))
			continue;
		if (read_line(fname, image, sizeof(image)))
			continue;
		snprintf(fname, sizeof(fname), "/sys/block/%s/pstate/cookie",
				de->d_name);
		if (stat(fname, &st) == 0) {
			if (read_line(fname, cookie, sizeof(cookie)))
				continue;
		}

		printf("%-12s %s %s\n", de->d_name, image, cookie);
	}
	closedir(dp);

	return 0;
}


static void cancel_callback(int sig)
{
	ploop_cancel_operation();
}

static void init_signals(void)
{
	struct sigaction act = {};

	sigemptyset(&act.sa_mask);
	act.sa_handler = cancel_callback;
	sigaction(SIGTERM, &act, NULL);
	sigaction(SIGINT, &act, NULL);
	sigaction(SIGHUP, &act, NULL);
}

int main(int argc, char **argv)
{
	char * cmd;

	if (argc < 2) {
		usage_summary();
		return SYSEXIT_PARAM;
	}

	cmd = argv[1];
	argc--;
	argv++;

	ploop_set_verbose_level(3);
	init_signals();

	if (strcmp(cmd, "init") == 0)
		return plooptool_init(argc, argv);
	if (strcmp(cmd, "start") == 0)
		return plooptool_start(argc, argv);
	if (strcmp(cmd, "stop") == 0)
		return plooptool_stop(argc, argv);
	if (strcmp(cmd, "clear") == 0)
		return plooptool_clear(argc, argv);
	if (strcmp(cmd, "mount") == 0)
		return plooptool_mount(argc, argv);
	if (strcmp(cmd, "umount") == 0)
		return plooptool_umount(argc, argv);
	if (strcmp(cmd, "delete") == 0 || strcmp(cmd, "rm") == 0)
		return plooptool_rm(argc, argv);
	if (strcmp(cmd, "snapshot") == 0)
		return plooptool_snapshot(argc, argv);
	if (strcmp(cmd, "snapshot-switch") == 0)
		return plooptool_snapshot_switch(argc, argv);
	if (strcmp(cmd, "snapshot-delete") == 0)
		return plooptool_snapshot_delete(argc, argv);
	if (strcmp(cmd, "snapshot-merge") == 0)
		return plooptool_snapshot_merge(argc, argv);
	if (strcmp(cmd, "getdev") == 0)
		return plooptool_getdevice(argc, argv);
	if (strcmp(cmd, "resize") == 0)
		return plooptool_resize(argc, argv);
	if (strcmp(cmd, "convert") == 0)
		return plooptool_convert(argc, argv);
	if (strcmp(cmd, "info") == 0)
		return plooptool_info(argc, argv);
	if (strcmp(cmd, "list") == 0)
		return plooptool_list(argc, argv);

	if (cmd[0] != '-') {
		char ** nargs;

		nargs = calloc(argc+1, sizeof(char*));
		nargs[0] = malloc(sizeof("ploop-") + strlen(cmd));
		sprintf(nargs[0], "ploop-%s", cmd);
		memcpy(nargs + 1, argv + 1, (argc - 1)*sizeof(char*));
		nargs[argc] = NULL;

		execvp(nargs[0], nargs);
	}

	usage_summary();
	return SYSEXIT_PARAM;
}
