# Copyright 1999-2011 Gentoo Foundation
# Distributed under the terms of the GNU General Public License v2
# $Header: /var/cvsroot/gentoo-x86/eclass/multilib.eclass,v 1.90 2011/03/18 22:03:25 vapier Exp $

# @ECLASS: multilib.eclass
# @MAINTAINER:
# amd64@gentoo.org
# toolchain@gentoo.org
# @BLURB: This eclass is for all functions pertaining to handling multilib configurations.
# @DESCRIPTION:
# This eclass is for all functions pertaining to handling multilib configurations.

___ECLASS_RECUR_MULTILIB="yes"
[[ -z ${___ECLASS_RECUR_TOOLCHAIN_FUNCS} ]] && inherit toolchain-funcs

# Defaults:
export MULTILIB_ABIS=${MULTILIB_ABIS:-"default"}
export DEFAULT_ABI=${DEFAULT_ABI:-"default"}
export CFLAGS_default
export LDFLAGS_default
export CHOST_default=${CHOST_default:-${CHOST}}
export CTARGET_default=${CTARGET_default:-${CTARGET:-${CHOST_default}}}
export LIBDIR_default=${CONF_LIBDIR:-"lib"}
export KERNEL_ABI=${KERNEL_ABI:-${DEFAULT_ABI}}

# @FUNCTION: has_multilib_profile
# @DESCRIPTION:
# Return true if the current profile is a multilib profile and lists more than
# one abi in ${MULTILIB_ABIS}.  When has_multilib_profile returns true, that
# profile should enable the 'multilib' use flag. This is so you can DEPEND on
# a package only for multilib or not multilib.
has_multilib_profile() {
	[ -n "${MULTILIB_ABIS}" -a "${MULTILIB_ABIS}" != "${MULTILIB_ABIS/ /}" ]
}

# @FUNCTION: get_libdir
# @RETURN: the libdir for the selected ABI
# @DESCRIPTION:
# This function simply returns the desired lib directory. With portage
# 2.0.51, we now have support for installing libraries to lib32/lib64
# to accomidate the needs of multilib systems. It's no longer a good idea
# to assume all libraries will end up in lib. Replace any (sane) instances
# where lib is named directly with $(get_libdir) if possible.
#
# Jeremy Huddleston <eradicator@gentoo.org> (23 Dec 2004):
#   Added support for ${ABI} and ${DEFAULT_ABI}.  If they're both not set,
#   fall back on old behavior.  Any profile that has these set should also
#   depend on a newer version of portage (not yet released) which uses these
#   over CONF_LIBDIR in econf, dolib, etc...
get_libdir() {
	local CONF_LIBDIR
	if [ -n  "${CONF_LIBDIR_OVERRIDE}" ] ; then
		# if there is an override, we want to use that... always.
		echo ${CONF_LIBDIR_OVERRIDE}
	else
		get_abi_LIBDIR
	fi
}

# @FUNCTION: get_multilibdir
# @RETURN: Returns the multilibdir
get_multilibdir() {
	if has_multilib_profile; then
		eerror "get_multilibdir called, but it shouldn't be needed with the new multilib approach.  Please file a bug at http://bugs.gentoo.org and assign it to eradicator@gentoo.org"
		exit 1
	fi
	echo ${CONF_MULTILIBDIR:=lib32}
}

# @FUNCTION: get_libdir_override
# @DESCRIPTION:
# Sometimes you need to override the value returned by get_libdir. A good
# example of this is xorg-x11, where lib32 isnt a supported configuration,
# and where lib64 -must- be used on amd64 (for applications that need lib
# to be 32bit, such as adobe acrobat). Note that this override also bypasses
# portage version sanity checking.
# get_libdir_override expects one argument, the result get_libdir should
# return:
#
#   get_libdir_override lib64
get_libdir_override() {
	if has_multilib_profile; then
		eerror "get_libdir_override called, but it shouldn't be needed with the new multilib approach.  Please file a bug at http://bugs.gentoo.org and assign it to eradicator@gentoo.org"
		exit 1
	fi
	CONF_LIBDIR="$1"
	CONF_LIBDIR_OVERRIDE="$1"
	LIBDIR_default="$1"
}

# @FUNCTION: get_abi_var
# @USAGE: <VAR> [ABI]
# @RETURN: returns the value of ${<VAR>_<ABI>} which should be set in make.defaults
# @DESCRIPTION:
# ex:
# CFLAGS=$(get_abi_var CFLAGS sparc32) # CFLAGS=-m32
#
# Note that the prefered method is to set CC="$(tc-getCC) $(get_abi_CFLAGS)"
# This will hopefully be added to portage soon...
#
# If <ABI> is not specified, ${ABI} is used.
# If <ABI> is not specified and ${ABI} is not defined, ${DEFAULT_ABI} is used.
# If <ABI> is not specified and ${ABI} and ${DEFAULT_ABI} are not defined, we return an empty string.
get_abi_var() {
	local flag=$1
	local abi
	if [ $# -gt 1 ]; then
		abi=${2}
	elif [ -n "${ABI}" ]; then
		abi=${ABI}
	elif [ -n "${DEFAULT_ABI}" ]; then
		abi=${DEFAULT_ABI}
	else
		abi="default"
	fi

	local var="${flag}_${abi}"
	echo ${!var}
}

# @FUNCTION: get_abi_CFLAGS
# @USAGE: [ABI]
# @DESCRIPTION:
# Alias for 'get_abi_var CFLAGS'
get_abi_CFLAGS() { get_abi_var CFLAGS "$@"; }

# @FUNCTION: get_abi_ASFLAGS
# @USAGE: [ABI]
# @DESCRIPTION:
# Alias for 'get_abi_var ASFLAGS'
get_abi_ASFLAGS() { get_abi_var ASFLAGS "$@"; }

# @FUNCTION: get_abi_LDFLAGS
# @USAGE: [ABI]
# @DESCRIPTION:
# Alias for 'get_abi_var LDFLAGS'
get_abi_LDFLAGS() { get_abi_var LDFLAGS "$@"; }

# @FUNCTION: get_abi_CHOST
# @USAGE: [ABI]
# @DESCRIPTION:
# Alias for 'get_abi_var CHOST'
get_abi_CHOST() { get_abi_var CHOST "$@"; }

# @FUNCTION: get_abi_CTARGET
# @USAGE: [ABI]
# @DESCRIPTION:
# Alias for 'get_abi_var CTARGET'
get_abi_CTARGET() { get_abi_var CTARGET "$@"; }

# @FUNCTION: get_abi_FAKE_TARGETS
# @USAGE: [ABI]
# @DESCRIPTION:
# Alias for 'get_abi_var FAKE_TARGETS'
get_abi_FAKE_TARGETS() { get_abi_var FAKE_TARGETS "$@"; }

# @FUNCTION: get_abi_LIBDIR
# @USAGE: [ABI]
# @DESCRIPTION:
# Alias for 'get_abi_var LIBDIR'
get_abi_LIBDIR() { get_abi_var LIBDIR "$@"; }

# @FUNCTION: get_install_abis
# @DESCRIPTION:
# Return a list of the ABIs we want to install for with
# the last one in the list being the default.
get_install_abis() {
	local order=""

	if [[ -z ${MULTILIB_ABIS} ]] ; then
		echo "default"
		return 0
	fi

	if [[ ${EMULTILIB_PKG} == "true" ]] ; then
		for x in ${MULTILIB_ABIS} ; do
			if [[ ${x} != "${DEFAULT_ABI}" ]] ; then
				hasq ${x} ${ABI_DENY} || order="${order} ${x}"
			fi
		done
		hasq ${DEFAULT_ABI} ${ABI_DENY} || order="${order} ${DEFAULT_ABI}"

		if [[ -n ${ABI_ALLOW} ]] ; then
			local ordera=""
			for x in ${order} ; do
				if hasq ${x} ${ABI_ALLOW} ; then
					ordera="${ordera} ${x}"
				fi
			done
			order=${ordera}
		fi
	else
		order=${DEFAULT_ABI}
	fi

	if [[ -z ${order} ]] ; then
		die "The ABI list is empty.  Are you using a proper multilib profile?  Perhaps your USE flags or MULTILIB_ABIS are too restrictive for this package."
	fi

	echo ${order}
	return 0
}

# @FUNCTION: get_all_abis
# @DESCRIPTION: 
# Return a list of the ABIs supported by this profile.
# the last one in the list being the default.
get_all_abis() {
	local order=""

	if [[ -z ${MULTILIB_ABIS} ]] ; then
		echo "default"
		return 0
	fi

	for x in ${MULTILIB_ABIS}; do
		if [[ ${x} != ${DEFAULT_ABI} ]] ; then
			order="${order:+${order} }${x}"
		fi
	done
	order="${order:+${order} }${DEFAULT_ABI}"

	echo ${order}
	return 0
}

# @FUNCTION: get_all_libdirs
# @DESCRIPTION:
# Returns a list of all the libdirs used by this profile.  This includes
# those that might not be touched by the current ebuild and always includes
# "lib".
get_all_libdirs() {
	local libdirs
	local abi
	local dir

	for abi in ${MULTILIB_ABIS}; do
		libdirs+=" $(get_abi_LIBDIR ${abi})"
	done
	[[ " ${libdirs} " != *" lib "* ]] && libdirs+=" lib"

	echo "${libdirs}"
}

# @FUNCTION: is_final_abi
# @DESCRIPTION:
# Return true if ${ABI} is the last ABI on our list (or if we're not
# using the new multilib configuration.  This can be used to determine
# if we're in the last (or only) run through src_{unpack,compile,install}
is_final_abi() {
	has_multilib_profile || return 0
	set -- $(get_install_abis)
	local LAST_ABI=$#
	[[ ${!LAST_ABI} == ${ABI} ]]
}

# @FUNCTION: number_abis
# @DESCRIPTION:
# echo the number of ABIs we will be installing for
number_abis() {
	set -- `get_install_abis`
	echo $#
}

# @FUNCTION: get_libname
# @USAGE: [version]
# @DESCRIPTION:
# Returns libname with proper suffix {.so,.dylib,.dll,etc} and optionally
# supplied version for the current platform identified by CHOST.
#
# Example:
#     get_libname ${PV}
#     Returns: .so.${PV} (ELF) || .${PV}.dylib (MACH) || ...
get_libname() {
	local libname
	local ver=$1
	case ${CHOST} in
		*-cygwin|mingw*|*-mingw*) libname="dll";;
		*-darwin*)                libname="dylib";;
		*-mint*)                  libname="irrelevant";;
		hppa*-hpux*)              libname="sl";;
		*)                        libname="so";;
	esac

	if [[ -z $* ]] ; then
		echo ".${libname}"
	else
		for ver in "$@" ; do
			case ${CHOST} in
				*-darwin*) echo ".${ver}.${libname}";;
				*-mint*)   echo ".${libname}";;
				*)         echo ".${libname}.${ver}";;
			esac
		done
	fi
}

# @FUNCTION: get_modname
# @USAGE:
# @DESCRIPTION:
# Returns modulename with proper suffix {.so,.bundle,etc} for the current
# platform identified by CHOST.
#
# Example:
#     libfoo$(get_modname)
#     Returns: libfoo.so (ELF) || libfoo.bundle (MACH) || ...
get_modname() {
	local modname
	local ver=$1
	case ${CHOST} in
		*-darwin*)                modname="bundle";;
		*)                        modname="so";;
	esac

	echo ".${modname}"
}

# This is for the toolchain to setup profile variables when pulling in
# a crosscompiler (and thus they aren't set in the profile)
multilib_env() {
	local CTARGET=${1:-${CTARGET}}

	case ${CTARGET} in
		x86_64*)
			export CFLAGS_x86=${CFLAGS_x86--m32}
			export CHOST_x86=${CTARGET/x86_64/i686}
			export CTARGET_x86=${CHOST_x86}
			if [[ ${SYMLINK_LIB} == "yes" ]] ; then
				export LIBDIR_x86="lib32"
			else
				export LIBDIR_x86="lib"
			fi

			export CFLAGS_amd64=${CFLAGS_amd64--m64}
			export CHOST_amd64=${CTARGET}
			export CTARGET_amd64=${CHOST_amd64}
			export LIBDIR_amd64="lib64"

			export CFLAGS_x32=${CFLAGS_x32--mx32}
			export CHOST_x32=${CTARGET}
			export CTARGET_x32=${CHOST_x32}
			export LIBDIR_x32="libx32"

			: ${MULTILIB_ABIS=amd64 x86}
			: ${DEFAULT_ABI=amd64}
		;;
		mips64*)
			export CFLAGS_o32=${CFLAGS_o32--mabi=32}
			export CHOST_o32=${CTARGET/mips64/mips}
			export CTARGET_o32=${CHOST_o32}
			export LIBDIR_o32="lib"

			export CFLAGS_n32=${CFLAGS_n32--mabi=n32}
			export CHOST_n32=${CTARGET}
			export CTARGET_n32=${CHOST_n32}
			export LIBDIR_n32="lib32"

			export CFLAGS_n64=${CFLAGS_n64--mabi=64}
			export CHOST_n64=${CTARGET}
			export CTARGET_n64=${CHOST_n64}
			export LIBDIR_n64="lib64"

			: ${MULTILIB_ABIS=n64 n32 o32}
			: ${DEFAULT_ABI=n32}
		;;
		powerpc64*)
			export CFLAGS_ppc=${CFLAGS_ppc--m32}
			export CHOST_ppc=${CTARGET/powerpc64/powerpc}
			export CTARGET_ppc=${CHOST_ppc}
			export LIBDIR_ppc="lib"

			export CFLAGS_ppc64=${CFLAGS_ppc64--m64}
			export CHOST_ppc64=${CTARGET}
			export CTARGET_ppc64=${CHOST_ppc64}
			export LIBDIR_ppc64="lib64"

			: ${MULTILIB_ABIS=ppc64 ppc}
			: ${DEFAULT_ABI=ppc64}
		;;
		s390x*)
			export CFLAGS_s390=${CFLAGS_s390--m31} # the 31 is not a typo
			export CHOST_s390=${CTARGET/s390x/s390}
			export CTARGET_s390=${CHOST_s390}
			export LIBDIR_s390="lib"

			export CFLAGS_s390x=${CFLAGS_s390x--m64}
			export CHOST_s390x=${CTARGET}
			export CTARGET_s390x=${CHOST_s390x}
			export LIBDIR_s390x="lib64"

			: ${MULTILIB_ABIS=s390x s390}
			: ${DEFAULT_ABI=s390x}
		;;
		sparc*)
			export CFLAGS_sparc32=${CFLAGS_sparc32}
			export CHOST_sparc32=${CTARGET/sparc64/sparc}
			export CTARGET_sparc32=${CHOST_sparc32}
			export LIBDIR_sparc32="lib"

			export CFLAGS_sparc64=${CFLAGS_sparc64--m64}
			export CHOST_sparc64=${CTARGET}
			export CTARGET_sparc64=${CHOST_sparc64}
			export LIBDIR_sparc64="lib64"

			: ${MULTILIB_ABIS=sparc64 sparc32}
			: ${DEFAULT_ABI=sparc64}
		;;
		*)
			: ${MULTILIB_ABIS=default}
			: ${DEFAULT_ABI=default}
		;;
	esac

	export MULTILIB_ABIS DEFAULT_ABI
}

# @FUNCTION: multilib_toolchain_setup
# @DESCRIPTION:
# Hide multilib details here for packages which are forced to be compiled for a
# specific ABI when run on another ABI (like x86-specific packages on amd64)
multilib_toolchain_setup() {
	local v vv

	export ABI=$1

	# First restore any saved state we have laying around.
	if [[ ${__DEFAULT_ABI_SAVED} == "true" ]] ; then
		for v in CHOST CBUILD AS CC CXX LD ; do
			vv="__abi_saved_${v}"
			export ${v}="${!vv}"
			unset ${vv}
		done
		unset __DEFAULT_ABI_SAVED
	fi

	# We want to avoid the behind-the-back magic of gcc-config as it
	# screws up ccache and distcc.  See #196243 for more info.
	if [[ ${ABI} != ${DEFAULT_ABI} ]] ; then
		# Back that multilib-ass up so we can restore it later
		for v in CHOST CBUILD AS CC CXX LD ; do
			export __abi_saved_${v}="${!v}"
		done
		export __DEFAULT_ABI_SAVED="true"

		# Set the CHOST native first so that we pick up the native
		# toolchain and not a cross-compiler by accident #202811.
		export CHOST=$(get_abi_CHOST ${DEFAULT_ABI})
		export AS="$(tc-getAS) $(get_abi_ASFLAGS)"
		export CC="$(tc-getCC) $(get_abi_CFLAGS)"
		export CXX="$(tc-getCXX) $(get_abi_CFLAGS)"
		export LD="$(tc-getLD) $(get_abi_LDFLAGS)"
		export CHOST=$(get_abi_CHOST $1)
		export CBUILD=$(get_abi_CHOST $1)
	fi
}
