 /*
 |----------------------------------------------------------------
 |             Copyright (c) 2002, venus.org.
 +----------------------------------------------------------------  
 |
 |      IP relay source file
 |
 |      ralao@venus.org (2002)
 |                                              	
 +----------------------------------------------------------------*/


/*---------------------------------------------------
 |     Include files below this line      
 +---------------------------------------------------*/

#ifdef _WIN32

#include <Winsock2.h>

#else /* _WIN32 */

#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#endif /* _WIN32 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>



/*---------------------------------------------------
 |     Constant Definitions below this line      
 +---------------------------------------------------*/
#define IR_ERROR_NONE           0                 /* normal exit code */
#define IR_ERROR_BAD_PARAMETERS 1                 /* bad command line parameters error code */
#define IR_ERROR_INIT_FAILED    2                 /* init failed error code */
#define IR_READ_BUFFER_SIZE     1500              /* read buffer size in bytes */
#define IR_DEFAULT_TIMEOUT      86400             /* deault timeout in seconds (1 day) */

#ifndef _WIN32

#define BOOL           int
#define FALSE          0
#define TRUE           (!FALSE)
#define SOCKET         int
#define INVALID_SOCKET (-1)
#define closesocket(s) (close(s))

#endif /* !_WIN32 */


/*---------------------------------------------------
 |     Type Definitions below this line      
 +---------------------------------------------------*/

/* tcp/ip relay node */
typedef struct _IR_tcp_client_node {
    struct _IR_tcp_client_node *next;
    long   client_ip;
    int    client_port;
    SOCKET client_socket;
	SOCKET target_socket;
    time_t last_used_time;
} IR_tcp_client_node;


/* udp/ip relay node */
typedef struct _IR_udp_client_node {
    struct _IR_udp_client_node *next;
    long   client_ip;
    int    client_port;
    SOCKET target_socket;
    time_t last_used_time;
} IR_udp_client_node;


/* packet library context type */
typedef struct {
	int    listen_port;
	SOCKET tcp_client_listen_socket;
	SOCKET udp_client_listen_socket;
    long   target_ip;
    int    target_port;
    time_t timeout;
    IR_tcp_client_node *tcp_client_list;
    IR_udp_client_node *udp_client_list;
} IR_context;


/*---------------------------------------------------
 |     Global vars Definition below this line      
 +---------------------------------------------------*/

/* global context */
static IR_context global_c = { 0, -1, -1, 0, 0, IR_DEFAULT_TIMEOUT, NULL, NULL };

/*---------------------------------------------------
 |     External functions Definition below this line      
 +---------------------------------------------------*/


/* just converts an ascii string to an IP address */
static long ascii_to_ip( char *str )
{
    int  i, j;
    long ip;
    char ip_field[4];
    
    /* gets each ip field, converts it into an int and appends it to the ip */
    for ( ip = 0, i = 0; i < 4; i++ ) {
        /* gets current field */
        for ( j = 0; j < 3 && *str && *str != '.'; ip_field[j++] = *(str++) );
        ip_field[j] = '\0';

        /* skips '.' if needed */
        if ( *str == '.' ) str++;
        
        /* updates ip address */
        ip = ( ip << 8 ) + atoi( ip_field );
    }
    
    /* returns */
    return ip;    
}


/* inits socket library */
static int init_socket_lib( void )
{
#ifdef _WIN32

    WORD    wVersionRequested;
    WSADATA wsaData;
 
    wVersionRequested = MAKEWORD( 2, 2 );
    if ( WSAStartup( wVersionRequested, &wsaData ) != 0 ) {
        fprintf( stderr, "ip_relay: ERROR - could not init socket library!\n");	
        exit( IR_ERROR_INIT_FAILED );
    }

#endif /* _WIN32 */

    return 0;
}


/* closes socket library */
static int close_socket_lib( void ) 
{
#ifdef _WIN32

    WSACleanup( );

#endif /* _WIN32 */

    return 0;    
}



/* parses command line
 * returns 1 on success, 0 on failure.
 */
static int parse_command_line( int argc, char **argv )
{
    long           ip;
    struct hostent *hostentry;

    /* exits here if OS is not posix compliant! */
    if ( argc == 0 ) return 0; 

    /* parses command line */
	if ( argc < 4 ) {
		fprintf( stderr, "usage:\n" );
		fprintf( stderr, "\tip_relay listen_port target_ip target_port [timeout]\n\n" );
        fprintf( stderr, "\tlisten_port: \n\t\tlocal port to listen on for udp/tcp connections.\n" );
        fprintf( stderr, "\ttarget_ip: \n\t\tip address to redirect connections to\n" );
        fprintf( stderr, "\ttarget_port: \n\t\tport number to redirect connections to\n" );
        fprintf( stderr, "\ttimeout: \n\t\toptional udp/tcp connection timeout in seconds, default = %d\n",
                 IR_DEFAULT_TIMEOUT );
        close_socket_lib( );
        exit( IR_ERROR_BAD_PARAMETERS );
	}
	global_c.listen_port = atoi( argv[1] );
    global_c.target_port = atoi( argv[3] );	
    hostentry            = gethostbyname( argv[2] );
    if ( hostentry ) {        
        memcpy( &ip, hostentry->h_addr, hostentry->h_length );
        global_c.target_ip = ntohl( ip );
    }
    else {
        global_c.target_ip = ascii_to_ip( argv[2] );
    }
    if ( argc > 4 ) {
        global_c.timeout = atoi( argv[4] );	
    }
    
	/* returns */
    return 0;
}




/* adds a tcp client node to the tcp_client_list */
static int add_tcp_client( long client_ip, int client_port, SOCKET tcp_client_socket, 
                           SOCKET tcp_target_socket )
{
    IR_tcp_client_node *node;

    node = malloc( sizeof( IR_tcp_client_node ) );
    if ( !node ) {
        fprintf( stderr, "ip_relay: Warning - could not allocate memory for tcp node!\n");	
        return -1;
    }
    node->client_ip          = client_ip;
    node->client_port        = client_port;
    node->client_socket      = tcp_client_socket;
    node->target_socket      = tcp_target_socket;
    node->last_used_time     = time( NULL );
    node->next               = global_c.tcp_client_list;
    global_c.tcp_client_list = node;
    
    return 0;
}


/* adds a udp client node the udp client list */
static int add_udp_client( long client_ip, int client_port, SOCKET udp_target_socket )
{
    IR_udp_client_node *node;

    node = malloc( sizeof( IR_udp_client_node ) );
    if ( !node ) {
        fprintf( stderr, "ip_relay: Warning - could not allocate memory for udp node!\n");	
        return -1;
    }
    node->client_ip          = client_ip;
    node->client_port        = client_port;
    node->target_socket      = udp_target_socket;
    node->last_used_time     = time( NULL );
    node->next               = global_c.udp_client_list;
    global_c.udp_client_list = node;
    
    return 0;
}



/* lookups a client from a udp_client_list */
static IR_udp_client_node *lookup_udp_client( long client_ip, int client_port )
{
    IR_udp_client_node *node;

    for ( node = global_c.udp_client_list; node; node = node->next ) {
        if ( node->client_ip == client_ip && node->client_port == client_port ) {
            return node;
        }
    }

    return NULL;
}



/* just creates the listen sockets */
static int create_listen_sockets( void ) 
{
	BOOL   reuse_address = TRUE;  /* reuse address option */ 
    struct sockaddr_in sock_addr;

    /* creates server sockets */
	global_c.tcp_client_listen_socket = socket(AF_INET, SOCK_STREAM, 0); 
	global_c.udp_client_listen_socket = socket(AF_INET, SOCK_DGRAM, 0); 
	if ( global_c.tcp_client_listen_socket == INVALID_SOCKET || 
         global_c.udp_client_listen_socket == INVALID_SOCKET ) 
    {
		fprintf( stderr, "ip_relay: ERROR - could not create listen sockets!\n") ;	
		exit( IR_ERROR_INIT_FAILED );
	}

	/* binds server sockets (and starts listening on tcp socket ) */
	memset( &sock_addr, 0, sizeof(sock_addr) );
	sock_addr.sin_family      = AF_INET;
	sock_addr.sin_addr.s_addr = htonl( INADDR_ANY );
	sock_addr.sin_port        = htons((unsigned short)global_c.listen_port);
	setsockopt( global_c.tcp_client_listen_socket, SOL_SOCKET, SO_REUSEADDR, (char *)&reuse_address, sizeof(BOOL) );
	if ( bind( global_c.tcp_client_listen_socket, (struct sockaddr *)&sock_addr, sizeof(sock_addr) ) < 0 ) {
		fprintf( stderr, "ip_relay: Warning - could not create bind tcp listen socket!\n");	
	}
	if ( bind( global_c.udp_client_listen_socket, (struct sockaddr *)&sock_addr, sizeof(sock_addr) ) < 0 ) {
		fprintf( stderr, "ip_relay: Warning - could not create bind udp listen socket!\n");	
	}
	if ( listen( global_c.tcp_client_listen_socket, 5 ) < 0 ) {
		fprintf( stderr, "ip_relay: Warning - could not create listen on tcp socket!\n");	
	}

    /* returns */
    return 0;
}


/* just creates tcp target socket */
static SOCKET create_tcp_target_socket( void ) 
{
    SOCKET             target_socket;
    int                port = global_c.target_port;
    long               ip   = global_c.target_ip;
	struct sockaddr_in sock_addr;

   /* creates server sockets */
	target_socket = socket(AF_INET, SOCK_STREAM, 0); 
	if ( target_socket == INVALID_SOCKET ) {
		fprintf( stderr, "ip_relay: Warning - could not create tcp target socket!\n") ;	
		return INVALID_SOCKET;
	}

    /* connects target sockets */
    memset(&sock_addr, 0, sizeof(sock_addr));
	sock_addr.sin_family      = AF_INET;
	sock_addr.sin_addr.s_addr = htonl( global_c.target_ip );
	sock_addr.sin_port        = htons( (unsigned short)global_c.target_port );
    if ( connect( target_socket, (struct sockaddr *)&sock_addr, sizeof(sock_addr) ) < 0 ) {
        fprintf( stderr, "ip_relay: Warning - could not connect tcp target socket to %d.%d.%d.%d:%d!\n",
                 (ip>>24)&0xff, (ip>>16)&0xff, (ip>>8)&0xff, ip&0xff, port );
        closesocket( target_socket );
        return INVALID_SOCKET;
    }

    /* returns */
    return target_socket;
}


/* just creates udp target socket */
static SOCKET create_udp_target_socket( void ) 
{
    SOCKET             target_socket;
    int                port = global_c.target_port;
    long               ip   = global_c.target_ip;
	struct sockaddr_in sock_addr;

   /* creates server sockets */
	target_socket = socket(AF_INET, SOCK_DGRAM, 0); 
	if ( target_socket == INVALID_SOCKET ) {
		fprintf( stderr, "ip_relay: ERROR - could not create target udp socket!\n") ;	
		return INVALID_SOCKET;
	}

    /* connects target sockets */
    memset(&sock_addr, 0, sizeof(sock_addr));
	sock_addr.sin_family      = AF_INET;
	sock_addr.sin_addr.s_addr = htonl( global_c.target_ip );
	sock_addr.sin_port        = htons( (unsigned short)global_c.target_port );
    if ( connect( target_socket, (struct sockaddr *)&sock_addr, sizeof(sock_addr) ) < 0 ) {
        fprintf( stderr, "ip_relay: Warning - could not connect udp target socket to %d.%d.%d.%d:%d!\n",
                 (ip>>24)&0xff, (ip>>16)&0xff, (ip>>8)&0xff, ip&0xff, port );
        closesocket( target_socket );
        return INVALID_SOCKET;
    }

    /* returns */
    return target_socket;
}



/* just clean up unused client nodes */
static int cleanup_unused_clients( void ) 
{
	int                  port;
	long                 ip;
    IR_tcp_client_node   *tcp_client, *prev_tcp_client;
    IR_udp_client_node   *udp_client, *prev_udp_client;

    /* cleanup tcp clients */
    for ( prev_tcp_client = NULL, tcp_client = global_c.tcp_client_list; 
          tcp_client; /* inc in loop */ ) 
    {
        if ( tcp_client->last_used_time + global_c.timeout < time( NULL ) ) {
            ip   = tcp_client->client_ip;
            port = tcp_client->client_port;
            fprintf( stderr, "ip_relay: INFO - tcp client connection %d.%d.%d.%d:%d timeout! cleaning up...\n",
                     (ip>>24)&0xff, (ip>>16)&0xff, (ip>>8)&0xff, ip&0xff, port );   
            if ( prev_tcp_client ) prev_tcp_client->next = tcp_client->next;
            else global_c.tcp_client_list = tcp_client->next;
            closesocket( tcp_client->client_socket );
            closesocket( tcp_client->target_socket );
            free( tcp_client );
            tcp_client = ( prev_tcp_client ) ? prev_tcp_client->next : global_c.tcp_client_list;
        }
        else {
            prev_tcp_client = tcp_client; 
            tcp_client      = tcp_client->next;
        }
    }

    /* cleanup udp clients */
    for ( prev_udp_client = NULL, udp_client = global_c.udp_client_list; 
          udp_client; /* inc in loop */ ) 
    {
        if ( udp_client->last_used_time + global_c.timeout < time( NULL ) ) {
            ip   = udp_client->client_ip;
            port = udp_client->client_port;
            fprintf( stderr, "ip_relay: INFO - udp client connection %d.%d.%d.%d:%d timeout! cleaning up...\n",
                     (ip>>24)&0xff, (ip>>16)&0xff, (ip>>8)&0xff, ip&0xff, port ) ;   
            if ( prev_udp_client ) prev_udp_client->next = udp_client->next;
            else global_c.udp_client_list = udp_client->next;
            closesocket( udp_client->target_socket );
            free( udp_client );
            udp_client = ( prev_udp_client ) ? prev_udp_client->next : global_c.udp_client_list;
        }
        else {
            prev_udp_client = udp_client;
            udp_client      = udp_client->next;
        }
    }

    return 0;
}



/* just goes through all tcp clients and forwards packet as needed */
static int forward_tcp_packets( fd_set *mask )
{
	int                  port;
	long                 ip;
    long                 read_size;
    IR_tcp_client_node   *tcp_client, *prev_tcp_client;
    static unsigned char read_buffer[IR_READ_BUFFER_SIZE];

    for ( prev_tcp_client = NULL, tcp_client = global_c.tcp_client_list; 
          tcp_client; /* inc in loop */ ) 
    {
        /* sends packet from client to target */
        if ( FD_ISSET( tcp_client->client_socket, mask ) ) {
            ip   = tcp_client->client_ip;
            port = tcp_client->client_port;
            read_size = recv( tcp_client->client_socket, read_buffer, IR_READ_BUFFER_SIZE, 0 );            
            if ( read_size < 0 ) {
                fprintf( stderr, "ip_relay: Warning - tcp_recv client %d.%d.%d.%d:%d failed! ignoring...\n",
                         (ip>>24)&0xff, (ip>>16)&0xff, (ip>>8)&0xff, ip&0xff, port );
            }
            if ( read_size == 0 ) {
	            fprintf( stderr, "ip_relay: INFO - connection to tcp client %d.%d.%d.%d:%d closed! cleaning up...\n",
                         (ip>>24)&0xff, (ip>>16)&0xff, (ip>>8)&0xff, ip&0xff, port ) ;
                if ( prev_tcp_client ) prev_tcp_client->next = tcp_client->next;
                else global_c.tcp_client_list = tcp_client->next;
                closesocket ( tcp_client->client_socket );
                closesocket ( tcp_client->target_socket );
                free( tcp_client );
                tcp_client = ( prev_tcp_client ) ? prev_tcp_client->next : global_c.tcp_client_list;
                continue;
            }
            if ( send( tcp_client->target_socket, read_buffer, read_size, 0 ) < 0 ) 
            {
                fprintf( stderr, "ip_relay: Warning - tcp_send target failed! ignoring...\n") ;	
            }
            tcp_client->last_used_time = time( NULL );
        }

        /* sends packet from target to client */
        if ( FD_ISSET( tcp_client->target_socket, mask ) ) {
            ip   = tcp_client->client_ip;
            port = tcp_client->client_port;
            read_size = recv( tcp_client->target_socket, read_buffer, IR_READ_BUFFER_SIZE, 0 );
            if ( read_size < 0 ) {
                fprintf( stderr, "ip_relay: Warning - tcp_recv target failed! ignoring...\n") ;	
            }
            if ( read_size == 0 ) {
	            fprintf( stderr, "ip_relay: INFO - connection to tcp target closed! cleaning up %d.%d.%d.%d:%d...\n",
                         (ip>>24)&0xff, (ip>>16)&0xff, (ip>>8)&0xff, ip&0xff, port );
                if ( prev_tcp_client ) prev_tcp_client->next = tcp_client->next;
                else global_c.tcp_client_list = tcp_client->next;
                closesocket ( tcp_client->client_socket );
                closesocket ( tcp_client->target_socket );
                free( tcp_client );
                tcp_client = ( prev_tcp_client ) ? prev_tcp_client->next : global_c.tcp_client_list;
                continue;
            }
            if ( send( tcp_client->client_socket, read_buffer, read_size, 0 ) < 0 ) 
            {
                fprintf( stderr, "ip_relay: Warning - tcp_send client %d.%d.%d.%d:%d failed! ignoring...\n",
                         (ip>>24)&0xff, (ip>>16)&0xff, (ip>>8)&0xff, ip&0xff, port );	
            }
            tcp_client->last_used_time = time( NULL );
        }

        /* moves to next client */
        prev_tcp_client = tcp_client; 
        tcp_client      = tcp_client->next;
    }

    return 0;
}



/* just goes through all udp clients and forwards packet as needed */
static int forward_udp_packets( fd_set *mask )
{
	int                  sa_length;
    SOCKET               target_socket;
	int                  port;
	long                 ip;
    long                 read_size;
    IR_udp_client_node   *udp_client;
	struct sockaddr_in   sock_addr;
    static unsigned char read_buffer[IR_READ_BUFFER_SIZE];

    /* reads packet from udp listener */
	if ( FD_ISSET( global_c.udp_client_listen_socket, mask ) ) {
        sa_length  = sizeof( struct sockaddr_in );
        memset( &sock_addr, 0, sa_length );
        read_size  = recvfrom( global_c.udp_client_listen_socket, read_buffer, IR_READ_BUFFER_SIZE, 0, 
                               (struct sockaddr *)&sock_addr, &sa_length );
        if ( read_size < 0 ) {
            fprintf( stderr, "ip_relay: Warning - udp_recvfrom failed! ignoring...\n") ;	
        }
        else {
            ip         = ntohl( sock_addr.sin_addr.s_addr );
            port       = ntohs( sock_addr.sin_port );
            udp_client = lookup_udp_client( ip, port );
            if ( !udp_client ) {
                target_socket = create_udp_target_socket( );
                if ( target_socket == INVALID_SOCKET ) {
                    fprintf( stderr, "ip_relay: INFO - rejecting new udp connection from %d.%d.%d.%d:%d! (failure)\n",
			                (ip>>24)&0xff, (ip>>16)&0xff, (ip>>8)&0xff, ip&0xff, port ) ;
                }
                else {
                    add_udp_client( ip, port, target_socket );
                    fprintf( stderr, "ip_relay: INFO - accepted new udp connection from %d.%d.%d.%d:%d.\n",
			                 (ip>>24)&0xff, (ip>>16)&0xff, (ip>>8)&0xff, ip&0xff, port );
                    if ( send( target_socket, read_buffer, read_size, 0 ) < 0 ) {
                        fprintf( stderr, "ip_relay: Warning - udp_send target failed! ignoring...\n") ;	
                    }
                }
            }
            else {
                if ( send( udp_client->target_socket, read_buffer, read_size, 0 ) < 0 ) {
                    fprintf( stderr, "ip_relay: Warning - udp_send target failed! ignoring...\n");
                    udp_client->last_used_time = time( NULL );
                }
            }
        }
    }

    /* goes through all udp target sockets and forward packets as needed */
    for ( udp_client = global_c.udp_client_list; udp_client; udp_client = udp_client->next ) {
		if ( FD_ISSET( udp_client->target_socket, mask ) ) {
            ip   = udp_client->client_ip;
            port = udp_client->client_port;
            read_size = recv( udp_client->target_socket, read_buffer, IR_READ_BUFFER_SIZE, 0 );
            if ( read_size < 0 ) {
                fprintf( stderr, "ip_relay: Warning - udp_recv from target failed! ignoring...\n" );
                continue;
            }
            sa_length  = sizeof( struct sockaddr_in );
            memset( &sock_addr, 0, sa_length );
            sock_addr.sin_family      = AF_INET;
            sock_addr.sin_addr.s_addr = htonl( ip );
            sock_addr.sin_port        = htons( (unsigned short)port );
            if ( sendto( global_c.udp_client_listen_socket, read_buffer, read_size, 0, 
                (struct sockaddr *)&sock_addr, sizeof(struct sockaddr_in) ) < 0 ) 
            {
                fprintf( stderr, "ip_relay: Warning - udp_sendto %d.%d.%d.%d:%d failed! ignoring...\n",
                         (ip>>24)&0xff, (ip>>16)&0xff, (ip>>8)&0xff, ip&0xff, port );	
                continue;   
            }
            udp_client->last_used_time = time( NULL );
        }
    }

    return 0;
}



/* 
 * this is the main function of the IR generic server 
 */
int main( int argc, char **argv )
{ 
	int                  n_fd_ready;
	SOCKET               client_socket;
    SOCKET               target_socket;
	int                  sa_length;
	int                  port;
	long                 ip;
    time_t               oldest_last_used_time;
    IR_tcp_client_node   *tcp_client, *tmp_tcp_client;
    IR_udp_client_node   *udp_client, *tmp_udp_client;
	struct sockaddr_in   sock_addr;
    struct timeval       timeval;
    fd_set               mask;

    /* inits socket library (Windows only) */
    init_socket_lib( );

    /* parses command line */
    parse_command_line( argc, argv );

    /* creates listen sockets */
    create_listen_sockets( );

    /*  prints running trace */
    ip   = global_c.target_ip;
    port = global_c.target_port;
	printf( "ip_relay: running on localhost:%d <-> %d.%d.%d.%d:%d\n",
			global_c.listen_port, (ip>>24)&0xff, (ip>>16)&0xff, (ip>>8)&0xff, ip&0xff, port ) ;	

	/* do forever */
	for ( ; /* ever */ ; ) {
        /* resets oldest_last_used_time */ 
        oldest_last_used_time = time( NULL );
        
        /* sets select mask */
        FD_ZERO( &mask );
        FD_SET( global_c.tcp_client_listen_socket, &mask );
        FD_SET( global_c.udp_client_listen_socket, &mask );
        for ( tcp_client = global_c.tcp_client_list; tcp_client; tcp_client = tcp_client->next ) {
            FD_SET( tcp_client->client_socket, &mask );
            FD_SET( tcp_client->target_socket, &mask );
            if ( tcp_client->last_used_time < oldest_last_used_time ) {
                oldest_last_used_time = tcp_client->last_used_time;
            }
        }
        for ( udp_client = global_c.udp_client_list; udp_client; udp_client = udp_client->next ) {
            FD_SET( udp_client->target_socket, &mask );
            if ( udp_client->last_used_time < oldest_last_used_time ) {
                oldest_last_used_time = udp_client->last_used_time;
            }
        }

        /* computes timeout */
        timeval.tv_sec  = oldest_last_used_time + global_c.timeout - time( NULL );
        if ( timeval.tv_sec < 0 ) timeval.tv_sec = 0;
        timeval.tv_usec = 0;

        /* sleeps on select */
        n_fd_ready = select( FD_SETSIZE, &mask, NULL, NULL, &timeval );

        /* ignores select errors */
        if ( n_fd_ready < 0 ) {
            fprintf( stderr, "ip_relay: Warning - select failed! ignoring...\n") ;	
            continue;
        }

        /* catches select timeout -- gets read of unused clients */
        if ( n_fd_ready == 0  ) {
            cleanup_unused_clients( );
            continue;
        }

		/* accepts TCP connections */
		if ( FD_ISSET( global_c.tcp_client_listen_socket, &mask ) ) {
            sa_length  = sizeof( struct sockaddr_in );
            memset( &sock_addr, 0, sa_length );
            client_socket = accept( global_c.tcp_client_listen_socket, (struct sockaddr *)&sock_addr, &sa_length );
            if ( client_socket == INVALID_SOCKET ) {
                fprintf( stderr, "ip_relay: Warning - tcp_accept failed! ignoring...\n") ;	
                continue;    
            }
            ip            = ntohl( sock_addr.sin_addr.s_addr );
            port          = ntohs( sock_addr.sin_port );            
            target_socket = create_tcp_target_socket( );
            if ( target_socket == INVALID_SOCKET ) {
	            fprintf( stderr, "ip_relay: INFO - rejecting new tcp connection from %d.%d.%d.%d:%d! (failure)\n",
			             (ip>>24)&0xff, (ip>>16)&0xff, (ip>>8)&0xff, ip&0xff, port ) ;
                closesocket ( client_socket );
                continue;
            }
            add_tcp_client( ip, port, client_socket, target_socket );
	        fprintf( stderr, "ip_relay: INFO - accepted new tcp connection from %d.%d.%d.%d:%d!\n",
			         (ip>>24)&0xff, (ip>>16)&0xff, (ip>>8)&0xff, ip&0xff, port );
        }

		/* forwards TCP packets  */
        forward_tcp_packets( &mask );

		/* forwards UDP packets to target */
        forward_udp_packets( &mask );
	}

    /* socket cleanup */    
    for ( tcp_client = global_c.tcp_client_list; tcp_client; /* inc in loop */ ) {
        tmp_tcp_client = tcp_client;
        tcp_client     = tcp_client->next;
        closesocket ( tmp_tcp_client->client_socket );
        closesocket ( tmp_tcp_client->target_socket );
        free( tmp_tcp_client );
    }
    for ( udp_client = global_c.udp_client_list; udp_client; /* inc in loop */ ) {
        tmp_udp_client = udp_client;
        udp_client     = udp_client->next;
        closesocket ( tmp_udp_client->target_socket );
        free( tmp_udp_client );
    }    
    closesocket( global_c.udp_client_listen_socket );
    closesocket( global_c.tcp_client_listen_socket ) ;
    close_socket_lib( );

    /* returns */
    return 0;
}

