/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: OOXMLSaxHandler.cxx,v $
 *
 *  $Revision: 1.22 $
 *
 *  last change: $Author: hbrinkm $ $Date: 2007/06/27 09:16:33 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "OOXMLSaxHandler.hxx"
#include "TokenMap.hxx"

#include <iostream>

#define DEBUG_CONTEXT_STACK
#define DEBUG_ATTRIBUTE
#define DEBUG_ELEMENT

namespace ooxml {

using namespace ::std;
using namespace ::com::sun::star;

OOXMLSaxHandler::OOXMLSaxHandler
(Stream & rStream, OOXMLDocument * pDocument)
: mrStream(rStream), mpDocument(pDocument)  
{
}

OOXMLSaxHandler::~OOXMLSaxHandler()
{
}

void OOXMLSaxHandler::startDocument()
    throw (xml::sax::SAXException, uno::RuntimeException)
{
    OOXMLContext::Pointer_t pContext(new OOXMLContext(mrStream));
    pContext->setDocument(mpDocument);
    pContext->setXNoteId(msXNoteId);

    mContextStack.push_back(pContext);
}

void OOXMLSaxHandler::endDocument()
    throw (xml::sax::SAXException, uno::RuntimeException)
{
    OOXMLContext::Pointer_t pTopContext = mContextStack.back();

    if (pTopContext.get() != NULL)
    {
        pTopContext->endCharacterGroup();
        pTopContext->endParagraphGroup();
    }

    mContextStack.pop_back();    
}

void OOXMLSaxHandler::handleNamespaceMappings
(OOXMLContext::Pointer_t pContext,
 const uno::Reference<xml::sax::XAttributeList> & attriblist)
{
    sal_Int16 nAttrCount = attriblist->getLength();
    for (sal_Int16 i = 0; i < nAttrCount; i++)
    {
        rtl::OUString sStrName(attriblist->getNameByIndex(i));
        
        rtl::OUString sPrefix;
        bool bIsNamespaceMapping = false;
        if (sStrName.compareToAscii("xmlns:", 6) == 0)
        {
            sPrefix = sStrName.copy(6);
            bIsNamespaceMapping = true;
        }
        else if (sStrName.compareToAscii("xmlns:", 6) == 0)
            bIsNamespaceMapping = true;

        if (bIsNamespaceMapping)
        {
            rtl::OUString sUrl = attriblist->getValueByIndex(i);
            
            pContext->addNamespaceMapping(sPrefix, sUrl);
        }
    }
}

void OOXMLSaxHandler::handleAttributes
(OOXMLContext::Pointer_t pContext,
 const uno::Reference<xml::sax::XAttributeList> & attriblist)
{
    sal_Int16 nAttrCount = attriblist->getLength();
    for (sal_Int16 i = 0; i < nAttrCount; i++)
    {
        rtl::OUString sStrName(attriblist->getNameByIndex(i));
        
        if (sStrName.compareToAscii("xmlns:", 6) != 0)
        {
            TokenEnum_t nTokenAttr =
                pContext->getToken(TokenMap::ATTRIBUTE, sStrName);
            pContext->attribute(nTokenAttr, attriblist->getValueByIndex(i));
        }
    }
}

void OOXMLSaxHandler::startElement
(const rtl::OUString & sElement,
 const uno::Reference<xml::sax::XAttributeList> & attriblist)
    throw (xml::sax::SAXException, uno::RuntimeException)
{
    string sInfo = "start element: ";
    sInfo += rtl::OUStringToOString(sElement, RTL_TEXTENCODING_ASCII_US).
        getStr();

    mrStream.info(sInfo);

#ifdef DEBUG_ELEMENT
    clog << sInfo << endl;
#endif

    OOXMLContext::Pointer_t pTopContext = mContextStack.back();
    OOXMLContext::Pointer_t pContext;
    TokenEnum_t nToken = OOXML_TOKENS_END;

    if (pTopContext.get() != NULL)
    {
        handleNamespaceMappings(pTopContext, attriblist);
        nToken = pTopContext->getToken(TokenMap::ELEMENT, sElement);
        pContext = OOXMLContext::Pointer_t(pTopContext->element(nToken));
    }
#ifdef DEBUG_ELEMENT
    else
        clog << "failed!" << endl;
#endif

    pushContext(pContext);
    if (pContext.get() != NULL)
    {
        pContext->setParentResource(pTopContext->getResource());
        handleAttributes(pContext, attriblist);
        pContext->resolvePropertySetAttrs();
        pContext->startAction();
        pContext->setToken(nToken);
    }
}

void OOXMLSaxHandler::endElement(const rtl::OUString & sElement)
    throw (xml::sax::SAXException, uno::RuntimeException)
{
    OOXMLContext::Pointer_t pContext = popContext();

    OOXMLContext::Pointer_t pTopContext = mContextStack.back();

    if (pTopContext.get() != NULL && pContext.get() != NULL)
    {
        TokenEnum_t nToken = pTopContext->getToken(TokenMap::ELEMENT, sElement);
        pContext->processToken(nToken);
        pContext->endAction();
        pTopContext->endElement(nToken, pContext);
    }

    string sInfo = "end element: ";
    sInfo += rtl::OUStringToOString(sElement, RTL_TEXTENCODING_ASCII_US).
        getStr();

#ifdef DEBUG_ELEMENT
    clog << sInfo << endl;
#endif

    mrStream.info(sInfo);
}

void OOXMLSaxHandler::characters(const rtl::OUString & sText)
    throw (xml::sax::SAXException, uno::RuntimeException)
{
#ifdef DEBUG_ELEMENT
    clog << "--> characters: " << OUStringToOString(sText, RTL_TEXTENCODING_ASCII_US).getStr() << endl;
#endif

    OOXMLContext::Pointer_t pContext = mContextStack.back();
    if (pContext.get() != NULL)
    {
        pContext->characters(sText);
    }

#ifdef DEBUG_ELEMENT
    clog << "<-- characters" << endl;
#endif
}

void OOXMLSaxHandler::ignorableWhitespace(const rtl::OUString & /*str*/)
    throw (xml::sax::SAXException, uno::RuntimeException)
{
}

void  OOXMLSaxHandler::processingInstruction
(const rtl::OUString& /*str*/,
 const rtl::OUString& /*str2*/)
    throw (xml::sax::SAXException, uno::RuntimeException)
{
}

void  OOXMLSaxHandler::setDocumentLocator
(const uno::Reference<xml::sax::XLocator>& /*doclocator*/)
    throw (xml::sax::SAXException, uno::RuntimeException)
{
}

void OOXMLSaxHandler::pushContext(OOXMLContext::Pointer_t pContext)
{
#ifdef DEBUG_CONTEXT_STACK
    if (pContext.get() != NULL)
        clog << "+-->" << pContext->toString() << endl;
    else
        clog << "+-->(null)" << endl;
#endif

    mContextStack.push_back(pContext);
}

OOXMLContext::Pointer_t OOXMLSaxHandler::popContext()
{
    OOXMLContext::Pointer_t pContext = mContextStack.back();
    mContextStack.pop_back();

#ifdef DEBUG_CONTEXT_STACK
    if (pContext.get() != NULL)
        clog << "+<--" << pContext->toString() << endl;
    else
        clog << "+<--(null)" << endl;
#endif

    return pContext;
}

void OOXMLSaxHandler::dumpContextStack() const
{
    OOXMLContextStack_t::const_iterator aIt = mContextStack.begin();
    OOXMLContextStack_t::const_iterator aEndIt = mContextStack.end();

    while (aIt != aEndIt)
    {
        clog << (*aIt)->toString() << endl;
        aIt++;
    }
}

void OOXMLSaxHandler::setXNoteId(const rtl::OUString & rId)
{
    msXNoteId = rId;
}

}
