/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: userpaintoverlay.cxx,v $
 *
 *  $Revision: 1.10.4.1 $
 *
 *  last change: $Author: obo $ $Date: 2007/05/08 10:05:11 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_slideshow.hxx"

#include <canvas/debug.hxx>
#include <osl/mutex.hxx>

#include <comphelper/anytostring.hxx>
#include <cppuhelper/exc_hlp.hxx>

#include <com/sun/star/awt/MouseButton.hpp>

#include <basegfx/polygon/b2dpolygon.hxx>
#include <cppcanvas/basegfxfactory.hxx>

#include "userpaintoverlay.hxx"
#include "mouseeventhandler.hxx"
#include "vieweventhandler.hxx"

#include <boost/utility.hpp> // for boost::noncopyable


using namespace ::com::sun::star;

namespace slideshow
{
    namespace internal
    {
        class ScreenUpdateActivity : public Activity, 
                                     private boost::noncopyable
        {
            bool mbIsActive;
        public:
            ScreenUpdateActivity() : mbIsActive(true) {}
            
            // Activity:
            virtual bool perform() {
                mbIsActive = false;
                return false;
            }
            virtual double calcTimeLag() const { return 0.0; }
            virtual bool isActive() const { return mbIsActive; }
            virtual bool needsScreenUpdate() const { return true; }
            virtual void dequeued() {}
            virtual void end() { mbIsActive = false; }
            // Disposable:
            virtual void dispose() { mbIsActive = false; }
        };
    
        class PaintOverlayHandler : public MouseEventHandler,
                                    public ViewEventHandler
        {
        public:
            PaintOverlayHandler( const RGBColor&         rStrokeColor,
                                 double                  nStrokeWidth,
                                 ActivitiesQueue&        rActivitiesQueue,
                                 const UnoViewContainer& rViews ) :
                mrActivitiesQueue( rActivitiesQueue ),
                maViews(),
                mpScreenUpdateActivity( new ScreenUpdateActivity ),
                maStrokeColor( rStrokeColor ),
                mnStrokeWidth( nStrokeWidth ),
                maLastPoint(),
                maLastMouseDownPos(),
                mbIsLastPointValid( false ),
                mbIsLastMouseDownPosValid( false )
            {
                std::for_each( rViews.begin(),
                               rViews.end(),
                               boost::bind( &PaintOverlayHandler::viewAdded,
                                            this,
                                            _1 ));
            }

            virtual void dispose()
            {
                ::osl::MutexGuard aGuard( maMutex );

                maViews.clear();
            }

            // ViewEventHandler methods
            virtual void viewAdded( const UnoViewSharedPtr& rView )
            {
                ::osl::MutexGuard aGuard( maMutex );
                maViews.push_back( rView );
            }

            virtual void viewRemoved( const UnoViewSharedPtr& rView )
            {
                ::osl::MutexGuard aGuard( maMutex );

                maViews.erase( std::remove( maViews.begin(),
                                            maViews.end(),
                                            rView ),
                               maViews.end() );
            }

            virtual void viewChanged( const UnoViewSharedPtr& /*rView*/ )
            {
                // TODO(F2): for persistent drawings, need to store
                // polygon and repaint here.
            }
            
            // MouseEventHandler methods
            virtual bool handleMousePressed( const awt::MouseEvent& e )
            {
                if (e.Buttons != awt::MouseButton::LEFT)
                    return false;
                
                ::osl::MutexGuard aGuard( maMutex );

                maLastMouseDownPos.setX( e.X );
                maLastMouseDownPos.setY( e.Y );
                mbIsLastMouseDownPosValid = true;

                // eat mouse click (though we don't process it
                // _directly_, it enables the drag mode
                return true;
            }

            virtual bool handleMouseReleased( const awt::MouseEvent& e )
            {
                if (e.Buttons != awt::MouseButton::LEFT)
                    return false;
                
                ::osl::MutexGuard aGuard( maMutex );

                // check, whether up- and down press are on exactly
                // the same pixel. If that's the case, ignore the
                // click, and pass on the event to low-prio
                // handlers. This effectively permits effect
                // advancements via clicks also when user paint is
                // enabled.
                if( mbIsLastMouseDownPosValid &&
                    ::basegfx::B2DPoint( e.X,
                                         e.Y ) == maLastMouseDownPos )
                {
                    mbIsLastMouseDownPosValid = false;
                    return false;
                }

                // invalidate, next downpress will have to start a new
                // polygon.
                mbIsLastPointValid = false;

                // eat mouse click (though we don't process it
                // _directly_, it enables the drag mode
                return true;
            }

            virtual bool handleMouseEntered( const awt::MouseEvent& e )
            {
                ::osl::MutexGuard aGuard( maMutex );

                mbIsLastPointValid = true;
                maLastPoint.setX( e.X );
                maLastPoint.setY( e.Y );

                return true;
            }

            virtual bool handleMouseExited( const awt::MouseEvent& )
            {
                ::osl::MutexGuard aGuard( maMutex );

                mbIsLastPointValid = false;
                mbIsLastMouseDownPosValid = false;

                return true;
            }

            virtual bool handleMouseDragged( const awt::MouseEvent& e )
            {
                ::osl::MutexGuard aGuard( maMutex );

                if( !mbIsLastPointValid )
                {
                    mbIsLastPointValid = true;
                    maLastPoint.setX( e.X );
                    maLastPoint.setY( e.Y );
                }
                else
                {
                    ::basegfx::B2DPolygon aPoly;
                    aPoly.append( maLastPoint );

                    maLastPoint.setX( e.X );
                    maLastPoint.setY( e.Y );

                    aPoly.append( maLastPoint );

                    // paint to all views
                    for( UnoViewVector::iterator aIter=maViews.begin(), aEnd=maViews.end();
                         aIter!=aEnd;
                         ++aIter )
                    {
                        ::cppcanvas::PolyPolygonSharedPtr pPolyPoly( 
                            ::cppcanvas::BaseGfxFactory::getInstance().createPolyPolygon( (*aIter)->getCanvas(), 
                                                                                          aPoly ) );
                        
                        if( pPolyPoly )
                        {
                            pPolyPoly->setRGBALineColor( maStrokeColor.getIntegerColor() );
                            pPolyPoly->draw();
                        }
                    }

                    // update screen, force update (layer manager is
                    // bypassed in this implementation)
                    mrActivitiesQueue.addActivity( mpScreenUpdateActivity );
                }

                // mouse events captured
                return true;
            }

            virtual bool handleMouseMoved( const awt::MouseEvent& /*e*/ )
            {
                // not used here
                return false; // did not handle the event
            }

        private:
            ::osl::Mutex            maMutex;
            ActivitiesQueue&        mrActivitiesQueue;
            UnoViewVector           maViews;
            ActivitySharedPtr       mpScreenUpdateActivity;
			RGBColor                maStrokeColor;
            double                  mnStrokeWidth;
            ::basegfx::B2DPoint     maLastPoint;
            ::basegfx::B2DPoint     maLastMouseDownPos;
            bool                    mbIsLastPointValid;
            bool                    mbIsLastMouseDownPosValid;
        };

        UserPaintOverlaySharedPtr UserPaintOverlay::create( const RGBColor&         rStrokeColor,
                                                            double                  nStrokeWidth,
                                                            const SlideShowContext& rContext )
        {
            UserPaintOverlaySharedPtr pRet( new UserPaintOverlay( rStrokeColor,
                                                                  nStrokeWidth,
                                                                  rContext ));

            return pRet;
        }

        UserPaintOverlay::UserPaintOverlay( const RGBColor&         rStrokeColor,
                                            double                  nStrokeWidth,
                                            const SlideShowContext& rContext ) : 
            mpHandler( new PaintOverlayHandler( rStrokeColor, 
                                                nStrokeWidth,
                                                rContext.mrActivitiesQueue,
                                                rContext.mrViewContainer )),
            mrMultiplexer( rContext.mrEventMultiplexer )
        {
            mrMultiplexer.addClickHandler( mpHandler, 3.0 );
            mrMultiplexer.addMouseMoveHandler( mpHandler, 3.0 );
            mrMultiplexer.addViewHandler( mpHandler );
        }
        
        UserPaintOverlay::~UserPaintOverlay()
        {
            try
            {
                mrMultiplexer.removeMouseMoveHandler( mpHandler );
                mrMultiplexer.removeClickHandler( mpHandler );
                mpHandler->dispose();
            }
            catch (uno::Exception &) {
                OSL_ENSURE( false, rtl::OUStringToOString(
                                comphelper::anyToString(
                                    cppu::getCaughtException() ),
                                RTL_TEXTENCODING_UTF8 ).getStr() );
            }
        }
    }
}
