/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: layermanager.cxx,v $
 *
 *  $Revision: 1.9 $
 *
 *  last change: $Author: kz $ $Date: 2006/12/13 15:17:06 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_slideshow.hxx"

// must be first
#include <canvas/debug.hxx>
#include <layermanager.hxx>

#include <rtl/logfile.hxx>

#include <basegfx/range/b1drange.hxx>

#include <boost/optional.hpp>
#include <boost/utility.hpp>
#include <boost/bind.hpp>
#include <boost/mem_fn.hpp>

#include <algorithm>
#include <iterator>

using namespace ::com::sun::star;

namespace slideshow
{
    namespace internal
    {
        LayerManager::LayerManager( const ::basegfx::B2DRectangle& rPageBounds ) :
            maViews(),
            maLayers(),
            maXShapeHash( 101 ),
            maUpdateShapes(),
            maUpdateAreas(),
            maPageBounds( rPageBounds )
        {
            // TODO(F3): Currently, LayerManager is a dummy (containing a single Layer). 
            // Implement it fully.

            // create background layer
            maLayers.push_back( LayerSharedPtr(new Layer()) );
        }

        ::basegfx::B2DRectangle LayerManager::getPageBounds() const
        {
            return maPageBounds;
        }

        void LayerManager::addView( const ViewSharedPtr& rView )
        {
            const ViewEntryVector::iterator aEnd( maViews.end() );

            // already added (searching for one of the maViews 
            // entries having a mpView member equal to rView)?
            if( ::std::find_if( maViews.begin(), 
                                aEnd, 
                                ::boost::bind<bool>( 
                                    ::std::equal_to< ViewSharedPtr >(),
                                    ::boost::bind( &ViewEntry::getView,
                                                   _1 ),
                                    ::boost::cref( rView ) ) ) != aEnd )
            {
                return; // yes, nothing to do
            }

            ViewEntry aNewEntry;
            aNewEntry.mpView = rView;

            // create ViewLayers for all registered Layers
            const ::std::size_t nNumLayers( maLayers.size() );
            for( ::std::size_t i=0; i<nNumLayers; ++i )
            {
                aNewEntry.maLayerVector.push_back( rView->createViewLayer() );

                ENSURE_AND_THROW( maLayers[i],
                                  "LayerManager::addView(): Invalid layer encountered" );

                maLayers[i]->addViewLayer( aNewEntry.maLayerVector.back() );
            }

            maViews.push_back( aNewEntry );
        }

        bool LayerManager::removeView( const ViewSharedPtr& rView )
        {
            const ViewEntryVector::iterator aEnd( maViews.end() );

            OSL_ENSURE( ::std::count_if( maViews.begin(), 
                                         aEnd, 
                                         ::boost::bind<bool>( 
                                             ::std::equal_to< ViewSharedPtr >(),
                                             ::boost::bind( &ViewEntry::getView,
                                                            _1 ),
                                             ::boost::cref( rView ) ) ) < 2,
                        "LayerManager::removeView(): Duplicate View entries!" );

            ViewEntryVector::iterator aIter;

            // search in local View list
            if( (aIter=::std::find_if( maViews.begin(), 
                                       aEnd, 
                                       ::boost::bind<bool>( 
                                           ::std::equal_to< ViewSharedPtr >(),
                                           ::boost::bind( &ViewEntry::getView,
                                                          _1 ),
                                           ::boost::cref( rView ) ) )) == aEnd )
            {
                return false; // View was not added
            }

            // now, aIter points to the appropriate ViewEntry - 
            // remove ViewLayers from all added Layers
            const ::std::size_t nNumLayers( maLayers.size() );
            ENSURE_AND_THROW( nNumLayers==aIter->maLayerVector.size(),
                              "LayerManager::removeView(): Mismatching view numbers" );
            for( ::std::size_t i=0; i<nNumLayers; ++i )
            {
                ENSURE_AND_THROW( maLayers[i],
                                  "LayerManager::removeView(): Invalid layer encountered" );

                maLayers[i]->removeViewLayer( aIter->maLayerVector[i] );
            }

            // remove from vector
            maViews.erase( aIter );

            return true;
        }

        bool LayerManager::viewChanged( const ViewSharedPtr& rView )
        {
            const ViewEntryVector::iterator aEnd( maViews.end() );
            ViewEntryVector::iterator aIter;

            // search in local View list
            if( (aIter=::std::find_if( maViews.begin(), 
                                       aEnd, 
                                       ::boost::bind<bool>( 
                                           ::std::equal_to< ViewSharedPtr >(),
                                           ::boost::bind( &ViewEntry::getView,
                                                          _1 ),
                                           ::boost::cref( rView ) ) )) == aEnd )
            {
                return false; // View was not added
            }

            // now, aIter points to the appropriate ViewEntry - 
            // forward change notification to all added Layers
            const ::std::size_t nNumLayers( maLayers.size() );
            ENSURE_AND_THROW( nNumLayers==aIter->maLayerVector.size(),
                              "LayerManager::viewChanged(): Mismatching view numbers" );

            for( ::std::size_t i=0; i<nNumLayers; ++i )
            {
                ENSURE_AND_THROW( maLayers[i],
                                  "LayerManager::viewChanged(): Invalid layer encountered" );

                maLayers[i]->viewLayerChanged( aIter->maLayerVector[i] );
            }

            return true;
        }

        void LayerManager::addShape( const ShapeSharedPtr& rShape )
        {
            ENSURE_AND_THROW( !maLayers.empty(), "LayerManager::addShape(): no layers" );

            if( !rShape )
                return; // no valid shape, no adding

            // add shape to XShape hash map
            if( !maXShapeHash.insert( 
                    XShapeHash::value_type( rShape->getXShape(),
                                            rShape) ).second )
            {
                // entry already present, bail out
                return; 
            }

            // add shape to background layer
            maLayers.front()->addShape( rShape );
            
            // shape needs update, if visible.
            if( !maViews.empty() && rShape->isVisible() )
                notifyShapeUpdate( rShape );
        }
        
        bool LayerManager::implRemoveShape( const ShapeSharedPtr& rShape )
        {
            ENSURE_AND_THROW( !maLayers.empty(), "LayerManager::removeShape(): no layers" );
            ENSURE_AND_THROW( rShape, "LayerManager::removeShape(): invalid Shape" );

            // store area early, once the shape is removed from
            // the layers, it no longer has any view references
            ::basegfx::B2DRectangle aShapeBounds;
            
            // guard with non-empty views (otherwise, shape has
            // nothing to pixel-justify on)
            if( !maViews.empty() )
                aShapeBounds = rShape->getUpdateArea();

            // remove shape from background layer
            if( !maLayers.front()->removeShape( rShape ) )
            {
                ENSURE_AND_RETURN( false,
                                   "LayerManager::removeShape(): Cannot remove shape from background layer" );
            }

            // area needs update. store in update set. Note that we
            // don't need to store the shape here, the dirty area
            // cannot change until next update() call (shape is
            // removed, after all)
            updateShapeArea( rShape,
                             aShapeBounds );

            // remove shape from maUpdateShapes, if inserted there -
            // no point in updating a removed shape
            maUpdateShapes.erase( rShape );

            return true;
        }
            
        bool LayerManager::removeShape( const ShapeSharedPtr& rShape )
        {
            if( !implRemoveShape( rShape ) )
                return false;

            // remove shape from XShape hash map
            if( maXShapeHash.erase( rShape->getXShape() ) == 0 )
                return false; // TODO(E2): return above leaves shape
                              // and layer untouched, this return has
                              // already removed the shape.

            return true;
        }
            
        ShapeSharedPtr LayerManager::lookupShape( const uno::Reference< drawing::XShape >& xShape ) const
        {
            ENSURE_AND_THROW( xShape.is(), "LayerManager::lookupShape(): invalid Shape" );

            XShapeHash::const_iterator aIter;
            if( (aIter=maXShapeHash.find( xShape )) == maXShapeHash.end() )
                return ShapeSharedPtr(); // not found

            // found, return data part of entry pair.
            return aIter->second;
        }

        AttributableShapeSharedPtr LayerManager::getSubsetShape( const AttributableShapeSharedPtr& 	rOrigShape,
                                                                 const DocTreeNode&					rTreeNode )
        {
            AttributableShapeSharedPtr pSubset;

            // shape already added?
            if( rOrigShape->createSubset( pSubset,
                                          rTreeNode ) )
            {
                ENSURE_AND_THROW( pSubset, "LayerManager::getSubsetShape(): failed to create subset" );
                ENSURE_AND_THROW( !maLayers.empty(), "LayerManager::getSubsetShape(): no layers" );

                // don't add to shape hash, we're dupes to the original
                // XShape anyway

                // TODO(F2): Since we don't store subset shapes, layers added after 
                // getSubsetShape() will _not_ contain previously generated subset 
                // shapes!

                // nope, also add to front layer
                maLayers.front()->addShape( pSubset );

                // empty views? no need to update anything. As soon as
                // a view is added, everything's painted anyway
                if( !maViews.empty() )
                {
                    // update subset shape, it's just added and not yet painted
                    if( pSubset->isVisible() )
                        notifyShapeUpdate( pSubset );

                    // update original shape, it now shows less content
                    // (the subset is removed from its displayed output)
                    if( rOrigShape->isVisible() )
                        notifyShapeUpdate( rOrigShape );
                }
            }

            return pSubset;
        }

        void LayerManager::revokeSubset( const AttributableShapeSharedPtr& rOrigShape,
                                         const AttributableShapeSharedPtr& rSubsetShape )
        {
            if( rOrigShape->revokeSubset( rSubsetShape ) )
            {
                if( !implRemoveShape( rSubsetShape ) )
                    ENSURE_AND_THROW(false,
                                     "LayerManager::revokeSubset(): subset shape removal failed");

                // only update for non-empty views. Empty views don't
                // need no update, and as soon as a view is added,
                // everything is painted from scratch anyway.
                if( !maViews.empty() )
                {
                    // update original shape, it now shows more content
                    // (the subset is added back to its displayed output)
                    if( rOrigShape->isVisible() )
                        notifyShapeUpdate( rOrigShape );
                }
            }
        }

        void LayerManager::enterAnimationMode( const AnimatableShapeSharedPtr& rShape )
        {
            ENSURE_AND_THROW( !maLayers.empty(), "LayerManager::enterAnimationMode(): no layers" );
            ENSURE_AND_THROW( rShape, "LayerManager::enterAnimationMode(): invalid Shape" );

            const bool bPrevAnimState( rShape->isBackgroundDetached() );

            // enter animation on given shape
            maLayers.front()->enterAnimationMode( rShape );

            // area needs update (shape is removed from normal slide,
            // and now rendered as an autonomous sprite). store in
            // update set
            if( !maViews.empty() &&
                bPrevAnimState != rShape->isBackgroundDetached() &&
                rShape->isVisible() )
            {
                addUpdateArea( rShape->getUpdateArea() );
            }
        }

        void LayerManager::leaveAnimationMode( const AnimatableShapeSharedPtr& rShape )
        {
            ENSURE_AND_THROW( !maLayers.empty(), "LayerManager::leaveAnimationMode(): no layers" );
            ENSURE_AND_THROW( rShape, "LayerManager::leaveAnimationMode(): invalid Shape" );

            const bool bPrevAnimState( rShape->isBackgroundDetached() );

            // leave animation on given shape
            maLayers.front()->leaveAnimationMode( rShape );

            // shape needs update. Don't just store in update set,
            // shape might change until final update() call happens
            if( !maViews.empty() && 
                bPrevAnimState != rShape->isBackgroundDetached() &&
                rShape->isVisible() )
            {
                notifyShapeUpdate( rShape );
            }
        }

        void LayerManager::notifyShapeUpdate( const ShapeSharedPtr& rShape )
        {
            ENSURE_AND_THROW( rShape,
                              "LayerManager::notifyAnimationUpdate(): NULL shape" );

            // store entry in update set (it's a unique container,
            // i.e. multiple requests for the same shape will still
            // result in only one entry)
            maUpdateShapes.insert( rShape );
        }

        bool LayerManager::isUpdatePending() const
        {
            return 
                !maUpdateShapes.empty() || 
                !maUpdateAreas.isEmpty();
        }

        void LayerManager::clearPendingUpdates()
        {
            // update request processed, clear lists
            maUpdateShapes.clear();
            maUpdateAreas.reset();
        }

        bool LayerManager::update()
        {
            RTL_LOGFILE_CONTEXT( aLog, "::presentation::internal::LayerManager::update()" );
            
            ENSURE_AND_THROW( !maLayers.empty(), "LayerManager::update(): no layers" );

            bool bRet( true );

            if( !maUpdateShapes.empty() )
            {
                // send update() calls to every shape in the maUpdateShapes set,
                // which is _animated_ (i.e. a sprite). If there are unanimated
                // sprites, we need a proper layer update
                const ShapeSet::const_iterator 	aEnd=maUpdateShapes.end();
                ShapeSet::const_iterator 		aCurrShape=maUpdateShapes.begin();
                while( aCurrShape != aEnd ) 
                {
                    try
                    {
                        if( (*aCurrShape)->isBackgroundDetached() )
                        {
                            // can update shape directly, without affecting layer 
                            // content (shape is currently displayed in a sprite)
                            if( !(*aCurrShape)->update() )
                                bRet = false; // delay error exit
                        }
                        else
                        {
                            // cannot update shape directly, it's not
                            // animated and update() calls will prolly
                            // overwrite other page content.
                            addUpdateArea( (*aCurrShape)->getUpdateArea() );
                        }
                    }
                    catch( uno::Exception& )
                    {
                        // shape could not be updated - catch and
                        // delay error, we want all other shapes
                        // still updated.
                        bRet = false;
                    }

                    ++aCurrShape;
                }
            }

            RTL_LOGFILE_CONTEXT_TRACE( aLog, "::presentation::internal::LayerManager: Sprites updated" );

            if( !maUpdateAreas.isEmpty() )
            {
                // perform proper layer update. That means, setup proper
                // clipping, and render each shape that intersects with 
                // the calculated update area
                ::basegfx::B2DPolyPolygon aClip( maUpdateAreas.getPolyPolygon() );

                // actually, if there happen to be shapes with zero
                // update area in the maUpdateAreas vector, the
                // resulting clip polygon will be empty.
                if( aClip.count() )
                {
                    // set clip
                    // ::boost::bind does not seem to work here for gcc...
                    // ...either becomes ambiguous, or didn't find matching call.
                    // Reason is perhaps the fact that ViewEntry::getView
                    // is const, whereas View::setClip() is not.
                    ViewEntryVector::const_iterator aEnd ( maViews.end()   );
                    ViewEntryVector::const_iterator aCurr( maViews.begin() );
                    while( aCurr != aEnd )
                    {
                        aCurr->getView()->setClip( aClip );
                        ++aCurr;
                    }

                    // clear update area
                    ::std::for_each( maViews.begin(), 
                                     maViews.end(), 
                                     ::boost::bind( 
                                         &View::clear,
                                         ::boost::bind( &ViewEntry::getView,
                                                        _1 ) ) );

                    // render shapes on all layers.
                    if( !maLayers.front()->update( maUpdateAreas ) )
                        bRet = false;

                    RTL_LOGFILE_CONTEXT_TRACE( aLog, "::presentation::internal::LayerManager: All shapes updated" );

                    // clear clip
                    // ::boost::bind does not seem to work here for gcc...
                    // ...either becomes ambiguous, or didn't find matching call.
                    // Reason is perhaps the fact that ViewEntry::getView
                    // is const, whereas View::setClip() is not.
                    ::basegfx::B2DPolyPolygon aEmpty;
                    aEnd  = maViews.end();
                    aCurr = maViews.begin();
                    while( aCurr != aEnd )
                    {
                        aCurr->getView()->setClip( aEmpty );
                        ++aCurr;
                    }
                }
            }
            
            // all updates processed now.
            clearPendingUpdates();

            return bRet;
        }

        bool LayerManager::render()
        {
            ENSURE_AND_THROW( !maLayers.empty(), "LayerManager::render(): no layers" );

            bool bRet( true );

            // clear view area
            ::std::for_each( maViews.begin(), 
                             maViews.end(), 
                             ::boost::bind( 
                                 &View::clear,
                                 ::boost::bind( &ViewEntry::getView,
                                                _1 ) ) );

            if( !maLayers.front()->render() )
                bRet = false;

            // send update() calls to all animated shapes in the
            // update set (the unanimated ones are already painted via
            // Layer::render, unconditionally).
            if( !maUpdateShapes.empty() )
            {
                // send update() calls to every shape in the maUpdateShapes set,
                // which is _animated_ (i.e. a sprite).
                const ShapeSet::const_iterator	aEnd=maUpdateShapes.end();
                ShapeSet::const_iterator		aCurrShape=maUpdateShapes.begin();
                while( aCurrShape != aEnd )
                {
                    try
                    {
                        if( (*aCurrShape)->isBackgroundDetached() )
                        {
                            // can update shape directly, without affecting layer 
                            // content (shape is currently displayed in a sprite)
                            if( !(*aCurrShape)->update() )
                                bRet = false; // delay error exit
                        }
                    }
                    catch( uno::Exception& )
                    {
                        // shape could not be updated - catch and
                        // delay error, we want all other shapes
                        // still updated.
                        bRet = false;
                    }

                    ++aCurrShape;
                }
            }

            // all updates processed now.
            clearPendingUpdates();

            return bRet;
        }

        void LayerManager::addUpdateArea( const ::basegfx::B2DRectangle& rArea ) const
        {
            // keep update area clean of empty bounds - it's a waste
            // of space, and currently, the clipper seems to dislike
            // it.
            if( !rArea.isEmpty() )
                maUpdateAreas.addRange( rArea );
        }

        void LayerManager::updateShapeArea( const ShapeSharedPtr& 			rShape,
                                            const ::basegfx::B2DRectangle&	rShapeBounds ) const
        {
            // Enter shape area to the update area, but only if shape
            // is visible and not in sprite mode (otherwise, updating
            // the area doesn't do actual harm, but costs time)
            if( !maViews.empty() && 
                rShape->isVisible() &&
                !rShape->isBackgroundDetached() )
            {
                addUpdateArea( rShapeBounds );
            }
        }
    }
}
