/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: Query.java,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 16:59:08 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

package com.sun.xmlsearch.xml.qe;

import com.sun.xmlsearch.util.Location;
import com.sun.xmlsearch.xml.*;

import java.io.PrintStream;

class Query {
    protected static double MissingTermPenalty = 10.0;
  
    protected final XmlIndex      _env;
    protected final ContextTables _ctx;
  
    protected final HitStore  _store;
  
    protected double     _currentStandard;
    protected final int  _nColumns;
    protected double[]  _missingPenalty;
    protected double[]  _upperboundTemplate;
    protected double[]  _penalties;
  
    private final int _nHitsRequested;
    private double    _missingTermsPenalty;
    protected boolean  _vote = false;
  
    private boolean[] _ignoredElements;
  
    // for use with Start/Stop
    private RoleFiller _roleFillerList;
  
    public Query(XmlIndex env, int nColumns, int nHits, double[] missingPenalties) {
	_env = env;
	// for the EmptyQuery case (awaits arch improvement pass)
	_ctx = env != null ? _env.getContextInfo() : null;
	_nColumns = nColumns;
	_nHitsRequested = nHits;
	_missingPenalty = new double[nColumns];
	_upperboundTemplate = new double[nColumns];
	_penalties = missingPenalties;
	_currentStandard = (nColumns - 1)*MissingTermPenalty +
	    MissingTermPenalty - 0.0001;
	_store = new HitStore(_currentStandard, nHits, nColumns);
	for (int i = 0; i < _nColumns; i++)
	    _missingPenalty[i] = missingPenalties != null
		? missingPenalties[i]
		: MissingTermPenalty;
	makePenaltiesTable();
	//  _roleFillerList = RoleFiller.STOP;
    }

    // overloaded in subclasses
    public ConceptData makeConceptData(int query, int col, int concept, double penalty) {
	return new ConceptData(concept, col, penalty, query, _nColumns, _ctx);
    }

    public void addControlConceptData(Search search, int queryNo) {
	// overloaded in subclasses
    }

    public boolean zoned() {
	return false;
    }

    public final void saveRoleFillers(RoleFiller rfList) {
	_roleFillerList = rfList;
    }
  
    public final RoleFiller getRoleFillers() {
	return _roleFillerList;
    }

    public final void missingTerms(int nMissingTerms) {
	_missingTermsPenalty = MissingTermPenalty * nMissingTerms;
    }
  
    public final double lookupPenalty(int pattern) {
	return _penalties[pattern];
    }
  
    public final double getOutOufOrderPenalty() {
	return 0.25;
    }
  
    public final double getGapPenalty() {
	return 0.005;
    }
  
    public final int getNColumns() {
	return _nColumns;
    }
  
    public final void setIgnoredElements(String[] ignoredElements) {
	if (_ctx != null) {
	    _ignoredElements = _ctx.getIgnoredElementsSet(ignoredElements);
	}
    }

    public final QueryHit maybeCreateQueryHit(double penalty,
					      int doc, int begin, int end, int parentContext) {
	// hits are located using only terms actually present in text
	// if B is not present, the query A B C reduces to A C and penalties
	// are computed as if B did not occur in query
	// to meaningfully merge results from different servers, some of which
	// may have B, penalty has to be normalized to the common computing scheme
	return _store.goodEnough(penalty += _missingTermsPenalty, begin, end)
	    && (_ignoredElements == null
		|| _ctx.notIgnored(parentContext, _ignoredElements))
	    ? _store.createQueryHit(penalty, doc, begin, end)
	    : null;
    }
  
    public final void resetForNextDocument() {
	_currentStandard = _store.getCurrentStandard();
	// "everything's missing"
	for (int i = 0; i < _nColumns; i++) {
	    _upperboundTemplate[i] = _missingPenalty[i];
	}
	_vote = false;
    }

    public final boolean vote() {
	double sum = 0.0;
	for (int i = 0; i < _nColumns; i++) {
	    sum += _upperboundTemplate[i];
	}
	return _vote = (sum <= _currentStandard);
    }

    public final void updateEstimate(int role, double penalty) {
	if (penalty < _upperboundTemplate[role]) {
	    _upperboundTemplate[role] = penalty;
	}
    }
  
    public final void printHits(PrintStream out, int n) throws Exception {
	if (n > 0) {
	    QueryHit qh = _store.firstBestQueryHit();
	    for ( ; qh != null; qh = --n > 0
		      ? _store.nextBestQueryHit()
		      : null)
		out.println(_env.hitToString(qh));
	    out.println("** hit **");
	}
    }

    public final String getHitsInXml(int n) throws Exception {
	final StringBuffer result = new StringBuffer(1024*16);
	result.append("<QueryResults>");
	if (n > 0) {
	    QueryHit qh = _store.firstBestQueryHit();
	    for ( ; qh != null; qh = --n > 0
		      ? _store.nextBestQueryHit()
		      : null)
		_env.hitToXml(qh, result);
	}
	result.append("</QueryResults>");
	System.out.println("|getHitsInXml|=" + result.length());
	return result.toString();
    }

    public final QueryHitData[] getHits(final int n) throws Exception {
	if (n > 0) {
	    QueryHit qh = _store.firstBestQueryHit();
	    if (qh != null) {
		final QueryHitData[] result = new QueryHitData[n];
		int index = 0;
		do {
		    result[index++] = _env.hitToData(qh);
		    qh = index < n ? _store.nextBestQueryHit() : null;
		}
		while (qh != null);
		return result;
	    }
	}
	return null;
    }

    private final void makePenaltiesTable() {
	final int nPatterns = 1 << _nColumns;
	_penalties = new double[nPatterns];
	for (int i = 0; i < nPatterns; i++) {
	    _penalties[i] = computePenalty(i);
	}
    }
    
    private final double computePenalty(int n) {
	double penalty = 0.0;
	for (int i = 0; i < _nColumns; i++) {
	    if ((n & 1 << i) == 0) {
		penalty += _missingPenalty[i];
	    }
	}
	return penalty;
    }
}
