<?php

namespace {

	/**
	 * <p>Represents a connection to a set of memcached servers.</p>
	 * @link https://php.net/manual/en/class.memcached.php
	 * @since PECL memcached >= 0.1.0
	 */
	class Memcached {

		/**
		 * @var integer Enables or disables payload compression. When enabled, item values longer than a certain threshold (currently 2000 bytes) will be compressed during storage and decompressed during retrieval transparently. The threshold can be configured using the memcached.compression_threshold ini setting.  <p>Type: <code>bool</code>, default: <b><code>true</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_COMPRESSION = -1001;

		/**
		 * @var integer Specifies the compression algorithm to use, if compression is enabled. The valid algorithms are <b><code>Memcached::COMPRESSION_FASTLZ</code></b>, <b><code>Memcached::COMPRESSION_ZLIB</code></b> and <b><code>Memcached::COMPRESSION_ZSTD</code></b>.  <p>Type: <code>int</code>, default: <b><code>Memcached::COMPRESSION_FASTLZ</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_COMPRESSION_TYPE = -1004;

		/**
		 * @var integer The FASTLZ compression algorithm.
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int COMPRESSION_FASTLZ = 2;

		/**
		 * @var integer The ZLIB compression algorithm.
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int COMPRESSION_ZLIB = 1;

		/**
		 * @var integer The ZSTD compression algorithm.   Available as of Memcached 3.3.0.
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int COMPRESSION_ZSTD = 3;

		/**
		 * @var integer Specifies the compression level to use, if compression is enabled.  <ul> <li><b><code>Memcached::COMPRESSION_FASTLZ</code></b> does not support compression levels.</li> <li><b><code>Memcached::COMPRESSION_ZSTD</code></b>: minimum level is <code>-22</code> and maximum level is <code>22</code>.</li> <li><b><code>Memcached::COMPRESSION_ZLIB</code></b>: minimum level is <code>0</code> and maximum level is <code>9</code>.</li> </ul> <p>Type: <code>int</code>, default: <code>3</code>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_COMPRESSION_LEVEL = -1007;

		/**
		 * @var integer Specifies the serializer to use for serializing non-scalar values. The valid serializers are <b><code>Memcached::SERIALIZER_PHP</code></b>, <b><code>Memcached::SERIALIZER_IGBINARY</code></b>, <b><code>Memcached::SERIALIZER_JSON</code></b>, <b><code>Memcached::SERIALIZER_JSON_ARRAY</code></b> and <b><code>Memcached::SERIALIZER_MSGPACK</code></b>.  <p>Type: <code>int</code>, default: <b><code>Memcached::SERIALIZER_IGBINARY</code></b> if available, then <b><code>Memcached::SERIALIZER_MSGPACK</code></b> if available, then <b><code>Memcached::SERIALIZER_PHP</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_SERIALIZER = -1003;

		/**
		 * @var integer <p>The default PHP serializer.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int SERIALIZER_PHP = 1;

		/**
		 * @var integer The igbinary serializer. Instead of textual representation it stores PHP data structures in a compact binary form, resulting in space and time gains.   This serializer is supported only if memcached is configured with <b>--enable-memcached-igbinary</b> option and the igbinary extension is loaded.
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int SERIALIZER_IGBINARY = 2;

		/**
		 * @var integer <p>The JSON serializer. This serializer deserializes JSON to an object.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int SERIALIZER_JSON = 3;

		/**
		 * @var integer The JSON array serializer. This serializer deserializes JSON to an associative array
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int SERIALIZER_JSON_ARRAY = 4;

		/**
		 * @var integer The &#xBB;&#xA0;MessagePack serializer.   This serializer is supported only if memcached is configured with <b>--enable-memcached-msgpack</b> option and the <code>msgpack</code> extension is loaded.
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int SERIALIZER_MSGPACK = 5;

		/**
		 * @var integer This can be used to create a "domain" for your item keys. The value specified here will be prefixed to each of the keys. It cannot be longer than <code>128</code> characters and will reduce the maximum available key size. The prefix is applied only to the item keys, not to the server keys.  <p>Type: <code>string</code>, default: <code>""</code>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_PREFIX_KEY = -1002;

		/**
		 * @var integer Specifies the hashing algorithm used for the item keys. The valid values are supplied via <b><code>Memcached::HASH_&#42;</code></b> constants. Each hash algorithm has its advantages and its disadvantages. Go with the default if you don't know or don't care.  <p>Type: <code>int</code>, default: <b><code>Memcached::HASH_DEFAULT</code></b></p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_HASH = 2;

		/**
		 * @var integer <p>The default (Jenkins one-at-a-time) item key hashing algorithm.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int HASH_DEFAULT = 0;

		/**
		 * @var integer <p>MD5 item key hashing algorithm.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int HASH_MD5 = 1;

		/**
		 * @var integer <p>CRC item key hashing algorithm.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int HASH_CRC = 2;

		/**
		 * @var integer <p>FNV1_64 item key hashing algorithm.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int HASH_FNV1_64 = 3;

		/**
		 * @var integer <p>FNV1_64A item key hashing algorithm.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int HASH_FNV1A_64 = 4;

		/**
		 * @var integer <p>FNV1_32 item key hashing algorithm.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int HASH_FNV1_32 = 5;

		/**
		 * @var integer <p>FNV1_32A item key hashing algorithm.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int HASH_FNV1A_32 = 6;

		/**
		 * @var integer <p>Hsieh item key hashing algorithm.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int HASH_HSIEH = 7;

		/**
		 * @var integer <p>Murmur item key hashing algorithm.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int HASH_MURMUR = 8;

		/**
		 * @var integer Specifies the method of distributing item keys to the servers. Currently supported methods are modulo and consistent hashing. Consistent hashing delivers better distribution and allows servers to be added to the cluster with minimal cache losses.  <p>Type: <code>int</code>, default: <b><code>Memcached::DISTRIBUTION_MODULA</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_DISTRIBUTION = 9;

		/**
		 * @var integer <p>Modulo-based key distribution algorithm.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int DISTRIBUTION_MODULA = 0;

		/**
		 * @var integer <p>Consistent hashing key distribution algorithm (based on libketama).</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int DISTRIBUTION_CONSISTENT = 1;

		/**
		 * @var integer Virtual Bucket hashing key distribution algorithm (also called vbucket).
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int DISTRIBUTION_VIRTUAL_BUCKET = 6;

		/**
		 * @var integer <p>Enables or disables compatibility with libketama-like behavior. When enabled, the item key hashing algorithm is set to MD5 and distribution is set to be weighted consistent hashing distribution. This is useful because other libketama-based clients (Python, Ruby, etc.) with the same server configuration will be able to access the keys transparently.</p> <p><b>Note</b>:</p><p>It is highly recommended to enable this option if you want to use consistent hashing, and it may be enabled by default in future releases.</p>  <p>Type: <code>bool</code>, default: <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_LIBKETAMA_COMPATIBLE = 16;

		/**
		 * @var integer Specifies the hashing algorithm for host mapping.  <p>Type: <code>int</code>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_LIBKETAMA_HASH = 17;

		/**
		 * @var integer Enables TCP keep alive.  <p>Type: <code>bool</code>, default: <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_TCP_KEEPALIVE = 32;

		/**
		 * @var integer Enables or disables buffered I/O. Enabling buffered I/O causes storage commands to "buffer" instead of being sent. Any action that retrieves data causes this buffer to be sent to the remote connection. Quitting the connection or closing down the connection will also cause the buffered data to be pushed to the remote connection.  <p>Type: <code>bool</code>, default: <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_BUFFER_WRITES = 10;

		/**
		 * @var integer Enable the use of the binary protocol. Please note that you cannot toggle this option on an open connection.  <p>Type: <code>bool</code>, default: <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_BINARY_PROTOCOL = 18;

		/**
		 * @var integer Enables or disables asynchronous I/O. This is the fastest transport available for storage functions.  <p>Type: <code>bool</code>, default: <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_NO_BLOCK = 0;

		/**
		 * @var integer Enables or disables ignoring the result of storage commands (set, add, replace, append, prepend, delete, increment, decrement, etc.). Storage commands will be sent without spending time waiting for a reply (there would be no reply). Retrieval commands such as <code>Memcached::get()</code> are unaffected by this setting.  <p>Type: <code>bool</code>, default: <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_NOREPLY = 26;

		/**
		 * @var integer Enables or disables the no-delay feature for connecting sockets (may be faster in some environments).  <p>Type: <code>bool</code>, default: <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_TCP_NODELAY = 1;

		/**
		 * @var integer The maximum socket send buffer in bytes. <p>Type: <code>int</code>, default: varies by platform/kernel configuration.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_SOCKET_SEND_SIZE = 4;

		/**
		 * @var integer The maximum socket receive buffer in bytes. <p>Type: <code>int</code>, default: varies by platform/kernel configuration.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_SOCKET_RECV_SIZE = 5;

		/**
		 * @var integer In non-blocking mode this set the value of the timeout during socket connection, in milliseconds.  <p>Type: <code>int</code>, default: <code>1000</code>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_CONNECT_TIMEOUT = 14;

		/**
		 * @var integer The amount of time, in seconds, to wait until retrying a failed connection attempt.  <p>Type: <code>int</code>, default: <code>2</code>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_RETRY_TIMEOUT = 15;

		/**
		 * @var integer The amount of time, in seconds, to wait until retrying dead servers. <code>0</code> means no retry.  <p>Type: <code>int</code>, default: <code>0</code>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_DEAD_TIMEOUT = 36;

		/**
		 * @var integer Socket sending timeout, in microseconds. In cases where you cannot use non-blocking I/O this will allow you to still have timeouts on the sending of data.  <p>Type: <code>int</code>, default: <code>0</code>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_SEND_TIMEOUT = 19;

		/**
		 * @var integer Socket reading timeout, in microseconds. In cases where you cannot use non-blocking I/O this will allow you to still have timeouts on the reading of data.  <p>Type: <code>int</code>, default: <code>0</code>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_RECV_TIMEOUT = 20;

		/**
		 * @var integer Timeout for connection polling, in milliseconds. <p>Type: <code>int</code>, default: <code>1000</code>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_POLL_TIMEOUT = 8;

		/**
		 * @var integer Enables or disables caching of DNS lookups. <p>Type: <code>bool</code>, default: <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_CACHE_LOOKUPS = 6;

		/**
		 * @var integer Specifies the failure limit for server connection attempts. The server will be removed after this many continuous connection failures.  <p>Type: <code>int</code>, default: <code>5</code>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_SERVER_FAILURE_LIMIT = 21;

		/**
		 * @var integer Specifies the timeout limit for server connection attempts. The server will be removed after this many continuous connection timeouts.  <p>Type: <code>int</code>, default: <code>0</code>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_SERVER_TIMEOUT_LIMIT = 37;

		/**
		 * @var integer <p>Removes disabled servers from the list. Use together with <b><code>Memcached::OPT_SERVER_FAILURE_LIMIT</code></b> and <b><code>Memcached::OPT_SERVER_TIMEOUT_LIMIT</code></b>.</p> <p><b>Note</b>:</p><p>This option is replaced by <b><code>Memcached::OPT_REMOVE_FAILED_SERVERS</code></b>.</p>  <p>Type: <code>bool</code>, default: <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_AUTO_EJECT_HOSTS = 28;

		/**
		 * @var integer Enables verification of keys by memcached.  <p>Type: <code>bool</code>, default: <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_VERIFY_KEY = 13;

		/**
		 * @var integer Uses UDP instead of TCP.  <p>Type: <code>bool</code>, default: <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_USE_UDP = 27;

		/**
		 * @var integer Stores this many replicas of each item on different servers.  <p>Type: <code>int</code>, default: <code>0</code>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_NUMBER_OF_REPLICAS = 29;

		/**
		 * @var mixed Randomizes replica read server.  <p>Type: <code>bool</code>, default: <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const mixed OPT_RANDOMIZE_REPLICA_READS = null;

		/**
		 * @var integer Removes disabled servers from the list. Use together with <b><code>Memcached::OPT_SERVER_FAILURE_LIMIT</code></b> and <b><code>Memcached::OPT_SERVER_TIMEOUT_LIMIT</code></b>.  <p>Type: <code>bool</code>, default: <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int OPT_REMOVE_FAILED_SERVERS = 35;

		/**
		 * @var boolean Indicates whether igbinary serializer support is available. <p>Type: <code>bool</code>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const bool HAVE_IGBINARY = true;

		/**
		 * @var boolean Indicates whether JSON serializer support is available. <p>Type: <code>bool</code>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const bool HAVE_JSON = true;

		/**
		 * @var boolean Indicates whether MessagePack serializer support is available. <p>Type: <code>bool</code>.</p> <p>Available as of Memcached 3.0.0.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const bool HAVE_MSGPACK = true;

		/**
		 * @var boolean Indicates whether ZSTD compression support is available. <p>Type: <code>bool</code>.</p> <p>Available as of Memcached 3.3.0.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const bool HAVE_ZSTD = false;

		/**
		 * @var boolean Indicates whether data encryption using <code>Memcached::setEncodingKey()</code> is supported.  <p>Type: <code>bool</code>.</p> <p>Available as of Memcached 3.1.0.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const bool HAVE_ENCODING = true;

		/**
		 * @var boolean <p>Type: <code>bool</code>.</p> <p>Available as of Memcached 3.0.0.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const bool HAVE_SESSION = true;

		/**
		 * @var boolean <p>Type: <code>bool</code>.</p> <p>Available as of Memcached 3.0.0.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const bool HAVE_SASL = true;

		/**
		 * @var integer <p>A flag for <code>Memcached::get()</code>, <code>Memcached::getMulti()</code> and <code>Memcached::getMultiByKey()</code> to ensure that the CAS token values are returned as well.</p> <p>Available as of Memcached 3.0.0.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int GET_EXTENDED = 2;

		/**
		 * @var integer <p>A flag for <code>Memcached::getMulti()</code> and <code>Memcached::getMultiByKey()</code> to ensure that the keys are returned in the same order as they were requested in. Non-existing keys get a default value of <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int GET_PRESERVE_ORDER = 1;

		/**
		 * @var integer <p>The operation was successful.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_SUCCESS = 0;

		/**
		 * @var integer <p>The operation failed in some fashion.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_FAILURE = 1;

		/**
		 * @var integer <p>DNS lookup failed.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_HOST_LOOKUP_FAILURE = 2;

		/**
		 * @var integer <p>Failed to read network data.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_UNKNOWN_READ_FAILURE = 7;

		/**
		 * @var integer <p>Bad command in memcached protocol.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_PROTOCOL_ERROR = 8;

		/**
		 * @var integer <p>Error on the client side.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_CLIENT_ERROR = 9;

		/**
		 * @var integer <p>Error on the server side.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_SERVER_ERROR = 10;

		/**
		 * @var integer <p>Failed to write network data.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_WRITE_FAILURE = 5;

		/**
		 * @var integer <p>Failed to do compare-and-swap: item you are trying to store has been modified since you last fetched it.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_DATA_EXISTS = 12;

		/**
		 * @var integer <p>Item was not stored: but not because of an error. This normally means that either the condition for an "add" or a "replace" command wasn't met, or that the item is in a delete queue.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_NOTSTORED = 14;

		/**
		 * @var integer <p>Item with this key was not found (with "get" operation or "cas" operations).</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_NOTFOUND = 16;

		/**
		 * @var integer <p>Partial network data read error.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_PARTIAL_READ = 18;

		/**
		 * @var integer <p>Some errors occurred during multi-get.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_SOME_ERRORS = 19;

		/**
		 * @var integer <p>Server list is empty.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_NO_SERVERS = 20;

		/**
		 * @var integer <p>End of result set.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_END = 21;

		/**
		 * @var integer <p>System error.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_ERRNO = 26;

		/**
		 * @var integer <p>The operation was buffered.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_BUFFERED = 32;

		/**
		 * @var integer <p>The operation timed out.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_TIMEOUT = 31;

		/**
		 * @var integer <p>Bad key.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_BAD_KEY_PROVIDED = 33;

		/**
		 * @var integer Item stored.
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_STORED = 15;

		/**
		 * @var integer Item deleted.
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_DELETED = 22;

		/**
		 * @var integer Statistics.
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_STAT = 24;

		/**
		 * @var integer Item.
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_ITEM = 25;

		/**
		 * @var integer Not supported.
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_NOT_SUPPORTED = 28;

		/**
		 * @var integer Fetch not finished.
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_FETCH_NOTFINISHED = 30;

		/**
		 * @var integer Server marked dead.
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_SERVER_MARKED_DEAD = 35;

		/**
		 * @var integer Unknown stat key.
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_UNKNOWN_STAT_KEY = 36;

		/**
		 * @var integer Invalid host protocol.
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_INVALID_HOST_PROTOCOL = 34;

		/**
		 * @var integer Memory allocation failure.
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_MEMORY_ALLOCATION_FAILURE = 17;

		/**
		 * @var integer <p>Failed to create network socket.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_CONNECTION_SOCKET_CREATE_FAILURE = 11;

		/**
		 * @var integer <p>Payload failure: could not compress/decompress or serialize/unserialize the value.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_PAYLOAD_FAILURE = -1001;

		/**
		 * @var integer <p>Available as of Memcached 3.0.0.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_AUTH_PROBLEM = 40;

		/**
		 * @var integer <p>Available as of Memcached 3.0.0.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_AUTH_FAILURE = 41;

		/**
		 * @var integer <p>Available as of Memcached 3.0.0.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_AUTH_CONTINUE = 42;

		/**
		 * @var integer <p>Available as of Memcached 3.0.0.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_E2BIG = 37;

		/**
		 * @var integer <p>Available as of Memcached 3.0.0.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_KEY_TOO_BIG = 39;

		/**
		 * @var integer <p>Available as of Memcached 3.0.0.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_SERVER_TEMPORARILY_DISABLED = 47;

		/**
		 * @var integer <p>Available as of Memcached 3.0.0.</p>
		 * @link https://php.net/manual/en/memcached.constants.php
		 */
		const int RES_SERVER_MEMORY_ALLOCATION_FAILURE = 48;

		/**
		 * Create a Memcached instance
		 * <p>Creates a Memcached instance representing the connection to the memcache servers.</p><p>This function is currently not documented; only its argument list is available.</p>
		 * @param ?string $persistent_id <p>By default the Memcached instances are destroyed at the end of the request. To create an instance that persists between requests, use <code>persistent_id</code> to specify a unique ID for the instance. All instances created with the same <code>persistent_id</code> will share the same connection.</p>
		 * @param ?callable $callback
		 * @param ?string $connection_str
		 * @return self
		 * @link https://php.net/manual/en/memcached.construct.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function __construct(?string $persistent_id = null, ?callable $callback = null, ?string $connection_str = null) {}

		/**
		 * Add an item under a new key
		 * <p><b>Memcached::add()</b> is similar to <code>Memcached::set()</code>, but the operation fails if the <code>key</code> already exists on the server.</p>
		 * @param string $key <p>The key under which to store the value.</p>
		 * @param mixed $value <p>The value to store.</p>
		 * @param int $expiration <p>The expiration time, defaults to 0. See Expiration Times for more info.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. The <code>Memcached::getResultCode()</code> will return <b><code>Memcached::RES_NOTSTORED</code></b> if the key already exists.</p>
		 * @link https://php.net/manual/en/memcached.add.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function add(string $key, mixed $value, int $expiration = 0): bool {}

		/**
		 * Add an item under a new key on a specific server
		 * <p><b>Memcached::addByKey()</b> is functionally equivalent to <code>Memcached::add()</code>, except that the free-form <code>server_key</code> can be used to map the <code>key</code> to a specific server. This is useful if you need to keep a bunch of related keys on a certain server.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @param string $key <p>The key under which to store the value.</p>
		 * @param mixed $value <p>The value to store.</p>
		 * @param int $expiration <p>The expiration time, defaults to 0. See Expiration Times for more info.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. The <code>Memcached::getResultCode()</code> will return <b><code>Memcached::RES_NOTSTORED</code></b> if the key already exists.</p>
		 * @link https://php.net/manual/en/memcached.addbykey.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function addByKey(string $server_key, string $key, mixed $value, int $expiration = 0): bool {}

		/**
		 * Add a server to the server pool
		 * <p><b>Memcached::addServer()</b> adds the specified server to the server pool. No connection is established to the server at this time, but if you are using consistent key distribution option (via <b><code>Memcached::DISTRIBUTION_CONSISTENT</code></b> or <b><code>Memcached::OPT_LIBKETAMA_COMPATIBLE</code></b>), some of the internal data structures will have to be updated. Thus, if you need to add multiple servers, it is better to use <code>Memcached::addServers()</code> as the update then happens only once.</p><p>The same server may appear multiple times in the server pool, because no duplication checks are made. This is not advisable; instead, use the <code>weight</code> option to increase the selection weighting of this server.</p>
		 * @param string $host <p>The hostname of the memcache server. If the hostname is invalid, data-related operations will set <b><code>Memcached::RES_HOST_LOOKUP_FAILURE</code></b> result code. As of version 2.0.0b1, this parameter may also specify the path of a unix socket filepath ex. <code>/path/to/memcached.sock</code> to use UNIX domain sockets, in this case <code>port</code> must also be set to <code>0</code>.</p>
		 * @param int $port <p>The port on which memcache is running. Usually, this is <code>11211</code>. As of version 2.0.0b1, set this parameter to <code>0</code> when using UNIX domain sockets.</p>
		 * @param int $weight <p>The weight of the server relative to the total weight of all the servers in the pool. This controls the probability of the server being selected for operations. This is used only with consistent distribution option and usually corresponds to the amount of memory available to memcache on that server.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/memcached.addserver.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function addServer(string $host, int $port, int $weight = 0): bool {}

		/**
		 * Add multiple servers to the server pool
		 * <p><b>Memcached::addServers()</b> adds <code>servers</code> to the server pool. Each entry in <code>servers</code> is supposed to be an array containing hostname, port, and, optionally, weight of the server. No connection is established to the servers at this time.</p><p>The same server may appear multiple times in the server pool, because no duplication checks are made. This is not advisable; instead, use the <code>weight</code> option to increase the selection weighting of this server.</p>
		 * @param array $servers
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/memcached.addservers.php
		 * @since PECL memcached >= 0.1.1
		 */
		public function addServers(array $servers): bool {}

		/**
		 * Append data to an existing item
		 * <p><b>Memcached::append()</b> appends the given <code>value</code> string to the value of an existing item. The reason that <code>value</code> is forced to be a string is that appending mixed types is not well-defined.</p><p><b>Note</b>:</p><p>If the <b><code>Memcached::OPT_COMPRESSION</code></b> is enabled, the operation will fail and a warning will be issued, because appending compressed data to a value that is potentially already compressed is not possible.</p>
		 * @param string $key <p>The key under which to store the value.</p>
		 * @param string $value <p>The string to append.</p>
		 * @return ?bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. Returns <b><code>null</code></b> when compression is on.</p>
		 * @link https://php.net/manual/en/memcached.append.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function append(string $key, string $value): ?bool {}

		/**
		 * Append data to an existing item on a specific server
		 * <p><b>Memcached::appendByKey()</b> is functionally equivalent to <code>Memcached::append()</code>, except that the free-form <code>server_key</code> can be used to map the <code>key</code> to a specific server.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @param string $key <p>The key under which to store the value.</p>
		 * @param string $value <p>The string to append.</p>
		 * @return ?bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. Returns <b><code>null</code></b> when compression is on.</p>
		 * @link https://php.net/manual/en/memcached.appendbykey.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function appendByKey(string $server_key, string $key, string $value): ?bool {}

		/**
		 * Compare and swap an item
		 * <p><b>Memcached::cas()</b> performs a "check and set" operation, so that the item will be stored only if no other client has updated it since it was last fetched by this client. The check is done via the <code>cas_token</code> parameter which is a unique 64-bit value assigned to the existing item by memcache. See the documentation for <b>Memcached::get&#42;()</b> methods for how to obtain this token. Note that the token is represented as a float due to the limitations of PHP's integer space.</p>
		 * @param string|int|float $cas_token <p>Unique value associated with the existing item. Generated by memcache.</p>
		 * @param string $key <p>The key under which to store the value.</p>
		 * @param mixed $value <p>The value to store.</p>
		 * @param int $expiration <p>The expiration time, defaults to 0. See Expiration Times for more info.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. The <code>Memcached::getResultCode()</code> will return <b><code>Memcached::RES_DATA_EXISTS</code></b> if the item you are trying to store has been modified since you last fetched it.</p>
		 * @link https://php.net/manual/en/memcached.cas.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function cas(string|int|float $cas_token, string $key, mixed $value, int $expiration = 0): bool {}

		/**
		 * Compare and swap an item on a specific server
		 * <p><b>Memcached::casByKey()</b> is functionally equivalent to <code>Memcached::cas()</code>, except that the free-form <code>server_key</code> can be used to map the <code>key</code> to a specific server. This is useful if you need to keep a bunch of related keys on a certain server.</p>
		 * @param string|int|float $cas_token <p>Unique value associated with the existing item. Generated by memcache.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @param string $key <p>The key under which to store the value.</p>
		 * @param mixed $value <p>The value to store.</p>
		 * @param int $expiration <p>The expiration time, defaults to 0. See Expiration Times for more info.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. The <code>Memcached::getResultCode()</code> will return <b><code>Memcached::RES_DATA_EXISTS</code></b> if the item you are trying to store has been modified since you last fetched it.</p>
		 * @link https://php.net/manual/en/memcached.casbykey.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function casByKey(string|int|float $cas_token, string $server_key, string $key, mixed $value, int $expiration = 0): bool {}

		/**
		 * Decrement numeric item's value
		 * <p><b>Memcached::decrement()</b> decrements a numeric item's value by the specified <code>offset</code>. If the item's value is not numeric, an error will result. If the operation would decrease the value below 0, the new value will be 0. <b>Memcached::decrement()</b> will set the item to the <code>initial_value</code> parameter if the key doesn't exist.</p>
		 * @param string $key <p>The key of the item to decrement.</p>
		 * @param int $offset <p>The amount by which to decrement the item's value.</p>
		 * @param int $initial_value <p>The value to set the item to if it doesn't currently exist.</p>
		 * @param int $expiry <p>The expiry time to set on the item.</p>
		 * @return int|false <p>Returns item's new value on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/memcached.decrement.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function decrement(string $key, int $offset = 1, int $initial_value = 0, int $expiry = 0): int|false {}

		/**
		 * Decrement numeric item's value, stored on a specific server
		 * <p><b>Memcached::decrementByKey()</b> decrements a numeric item's value by the specified <code>offset</code>. If the item's value is not numeric, an error will result. If the operation would decrease the value below 0, the new value will be 0. <b>Memcached::decrementByKey()</b> will set the item to the <code>initial_value</code> parameter if the key doesn't exist.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @param string $key <p>The key of the item to decrement.</p>
		 * @param int $offset <p>The amount by which to decrement the item's value.</p>
		 * @param int $initial_value <p>The value to set the item to if it doesn't currently exist.</p>
		 * @param int $expiry <p>The expiry time to set on the item.</p>
		 * @return int|false <p>Returns item's new value on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/memcached.decrementbykey.php
		 * @since PECL memcached >= 2.0.0
		 */
		public function decrementByKey(string $server_key, string $key, int $offset = 1, int $initial_value = 0, int $expiry = 0): int|false {}

		/**
		 * Delete an item
		 * <p>Delete the <code>key</code> from the server.</p>
		 * @param string $key <p>The key to be deleted.</p>
		 * @param int $time <p>The amount of time the server will wait to delete the item.</p> <p><b>Note</b>:  As of memcached 1.3.0 (released 2009) this feature is no longer supported. Passing a non-zero <code>time</code> will cause the deletion to fail. <code>Memcached::getResultCode()</code> will return <b><code>MEMCACHED_INVALID_ARGUMENTS</code></b>. </p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. The <code>Memcached::getResultCode()</code> will return <b><code>Memcached::RES_NOTFOUND</code></b> if the key does not exist.</p>
		 * @link https://php.net/manual/en/memcached.delete.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function delete(string $key, int $time = 0): bool {}

		/**
		 * Delete an item from a specific server
		 * <p><b>Memcached::deleteByKey()</b> is functionally equivalent to <code>Memcached::delete()</code>, except that the free-form <code>server_key</code> can be used to map the <code>key</code> to a specific server.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @param string $key <p>The key to be deleted.</p>
		 * @param int $time <p>The amount of time the server will wait to delete the item.</p> <p><b>Note</b>:  As of memcached 1.3.0 (released 2009) this feature is no longer supported. Passing a non-zero <code>time</code> will cause the deletion to fail. <code>Memcached::getResultCode()</code> will return <b><code>MEMCACHED_INVALID_ARGUMENTS</code></b>. </p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. The <code>Memcached::getResultCode()</code> will return <b><code>Memcached::RES_NOTFOUND</code></b> if the key does not exist.</p>
		 * @link https://php.net/manual/en/memcached.deletebykey.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function deleteByKey(string $server_key, string $key, int $time = 0): bool {}

		/**
		 * Delete multiple items
		 * <p>Delete the array of <code>keys</code> from the server.</p>
		 * @param array $keys <p>The keys to be deleted.</p>
		 * @param int $time <p>The amount of time the server will wait to delete the items.</p> <p><b>Note</b>:  As of memcached 1.3.0 (released 2009) this feature is no longer supported. Passing a non-zero <code>time</code> will cause the deletion to fail. <code>Memcached::getResultCode()</code> will return <b><code>MEMCACHED_INVALID_ARGUMENTS</code></b>. </p>
		 * @return array <p>Returns an array indexed by <code>keys</code>. Each element is <b><code>true</code></b> if the corresponding key was deleted, or one of the <b><code>Memcached::RES_&#42;</code></b> constants if the corresponding deletion failed.</p><p>The <code>Memcached::getResultCode()</code> will return the result code for the last executed delete operation, that is, the delete operation for the last element of <code>keys</code>.</p>
		 * @link https://php.net/manual/en/memcached.deletemulti.php
		 * @since PECL memcached >= 2.0.0
		 */
		public function deleteMulti(array $keys, int $time = 0): array {}

		/**
		 * Delete multiple items from a specific server
		 * <p><b>Memcached::deleteMultiByKey()</b> is functionally equivalent to <code>Memcached::deleteMulti()</code>, except that the free-form <code>server_key</code> can be used to map the <code>keys</code> to a specific server.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @param array $keys <p>The keys to be deleted.</p>
		 * @param int $time <p>The amount of time the server will wait to delete the items.</p> <p><b>Note</b>:  As of memcached 1.3.0 (released 2009) this feature is no longer supported. Passing a non-zero <code>time</code> will cause the deletion to fail. <code>Memcached::getResultCode()</code> will return <b><code>MEMCACHED_INVALID_ARGUMENTS</code></b>. </p>
		 * @return array <p>Returns an array indexed by <code>keys</code>. Each element is <b><code>true</code></b> if the corresponding key was deleted, or one of the <b><code>Memcached::RES_&#42;</code></b> constants if the corresponding deletion failed.</p><p>The <code>Memcached::getResultCode()</code> will return the result code for the last executed delete operation, that is, the delete operation for the last element of <code>keys</code>.</p>
		 * @link https://php.net/manual/en/memcached.deletemultibykey.php
		 * @since PECL memcached >= 2.0.0
		 */
		public function deleteMultiByKey(string $server_key, array $keys, int $time = 0): array {}

		/**
		 * Fetch the next result
		 * <p><b>Memcached::fetch()</b> retrieves the next result from the last request.</p>
		 * @return array|false <p>Returns the next result or <b><code>false</code></b> otherwise. The <code>Memcached::getResultCode()</code> will return <b><code>Memcached::RES_END</code></b> if result set is exhausted.</p>
		 * @link https://php.net/manual/en/memcached.fetch.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function fetch(): array|false {}

		/**
		 * Fetch all the remaining results
		 * <p><b>Memcached::fetchAll()</b> retrieves all the remaining results from the last request.</p>
		 * @return array|false <p>Returns the results or <b><code>false</code></b> on failure. Use <code>Memcached::getResultCode()</code> if necessary.</p>
		 * @link https://php.net/manual/en/memcached.fetchall.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function fetchAll(): array|false {}

		/**
		 * Invalidate all items in the cache
		 * <p><b>Memcached::flush()</b> invalidates all existing cache items immediately (by default) or after the <code>delay</code> specified. After invalidation none of the items will be returned in response to a retrieval command (unless it's stored again under the same key after <b>Memcached::flush()</b> has invalidated the items). The flush does not actually free all the memory taken up by the existing items; that will happen gradually as new items are stored.</p>
		 * @param int $delay <p>Number of seconds to wait before invalidating the items.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. Use <code>Memcached::getResultCode()</code> if necessary.</p>
		 * @link https://php.net/manual/en/memcached.flush.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function flush(int $delay = 0): bool {}

		/**
		 * Retrieve an item
		 * <p><b>Memcached::get()</b> returns the item that was previously stored under the <code>key</code>. If the item is found and the <code>get_flags</code> is given <b><code>Memcached::GET_EXTENDED</code></b>, it will also return the CAS token value for the item. See <code>Memcached::cas()</code> for how to use CAS tokens. Read-through caching callback may be specified via <code>cache_cb</code> parameter.</p><p></p>
		 * @param string $key <p>The key of the item to retrieve.</p>
		 * @param ?callable $cache_cb <p>Read-through caching callback or <b><code>null</code></b>.</p>
		 * @param int $get_flags <p>Flags to control the returned result. When <b><code>Memcached::GET_EXTENDED</code></b> is given, the function will also return the CAS token.</p>
		 * @return mixed <p>Returns the value stored in the cache or <b><code>false</code></b> otherwise. If the <code>get_flags</code> is set to <b><code>Memcached::GET_EXTENDED</code></b>, an array containing the value and the CAS token is returned instead of only the value. The <code>Memcached::getResultCode()</code> will return <b><code>Memcached::RES_NOTFOUND</code></b> if the key does not exist.</p>
		 * @link https://php.net/manual/en/memcached.get.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function get(string $key, ?callable $cache_cb = null, int $get_flags = 0): mixed {}

		/**
		 * Gets the keys stored on all the servers
		 * <p><b>Memcached::getAllKeys()</b> queries each memcache server and retrieves an array of all keys stored on them at that point in time. This is not an atomic operation, so it isn't a truly consistent snapshot of the keys at point in time. As memcache doesn't guarantee to return all keys you also cannot assume that all keys have been returned.</p><p><b>Note</b>:</p><p>This method is intended for debugging purposes and should not be used at scale!</p>
		 * @return array|false <p>Returns the keys stored on all the servers on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/memcached.getallkeys.php
		 * @since PECL memcached >= 2.0.0
		 */
		public function getAllKeys(): array|false {}

		/**
		 * Retrieve an item from a specific server
		 * <p><b>Memcached::getByKey()</b> is functionally equivalent to <code>Memcached::get()</code>, except that the free-form <code>server_key</code> can be used to map the <code>key</code> to a specific server.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @param string $key <p>The key of the item to fetch.</p>
		 * @param ?callable $cache_cb <p>Read-through caching callback or <b><code>null</code></b></p>
		 * @param int $get_flags <p>Flags to control the returned result. When value of <b><code>Memcached::GET_EXTENDED</code></b> is given will return the CAS token.</p>
		 * @return mixed <p>Returns the value stored in the cache or <b><code>false</code></b> otherwise. The <code>Memcached::getResultCode()</code> will return <b><code>Memcached::RES_NOTFOUND</code></b> if the key does not exist.</p>
		 * @link https://php.net/manual/en/memcached.getbykey.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function getByKey(string $server_key, string $key, ?callable $cache_cb = null, int $get_flags = 0): mixed {}

		/**
		 * Request multiple items
		 * <p><b>Memcached::getDelayed()</b> issues a request to memcache for multiple items the keys of which are specified in the <code>keys</code> array. The method does not wait for response and returns right away. When you are ready to collect the items, call either <code>Memcached::fetch()</code> or <code>Memcached::fetchAll()</code>. If <code>with_cas</code> is true, the CAS token values will also be requested.</p><p>Instead of fetching the results explicitly, you can specify a result callback via <code>value_cb</code> parameter.</p>
		 * @param array $keys <p>Array of keys to request.</p>
		 * @param bool $with_cas <p>Whether to request CAS token values also.</p>
		 * @param ?callable $value_cb <p>The result callback or <b><code>null</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. Use <code>Memcached::getResultCode()</code> if necessary.</p>
		 * @link https://php.net/manual/en/memcached.getdelayed.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function getDelayed(array $keys, bool $with_cas = false, ?callable $value_cb = null): bool {}

		/**
		 * Request multiple items from a specific server
		 * <p><b>Memcached::getDelayedByKey()</b> is functionally equivalent to <code>Memcached::getDelayed()</code>, except that the free-form <code>server_key</code> can be used to map the <code>keys</code> to a specific server.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @param array $keys <p>Array of keys to request.</p>
		 * @param bool $with_cas <p>Whether to request CAS token values also.</p>
		 * @param ?callable $value_cb <p>The result callback or <b><code>null</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. Use <code>Memcached::getResultCode()</code> if necessary.</p>
		 * @link https://php.net/manual/en/memcached.getdelayedbykey.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function getDelayedByKey(string $server_key, array $keys, bool $with_cas = false, ?callable $value_cb = null): bool {}

		/**
		 * Retrieve multiple items
		 * <p><b>Memcached::getMulti()</b> is similar to <code>Memcached::get()</code>, but instead of a single key item, it retrieves multiple items the keys of which are specified in the <code>keys</code> array.</p><p><b>Note</b>:</p><p>Before v3.0 a second argument <code>&amp;cas_tokens</code> was in use. It was filled with the CAS token values for the found items. The <code>&amp;cas_tokens</code> parameter was removed in v3.0 of the extension. It was replaced with a new flag <b><code>Memcached::GET_EXTENDED</code></b> that needs is to be used as the value for <code>get_flags</code>.</p><p>The <code>get_flags</code> parameter can be used to specify additional options for <b>Memcached::getMulti()</b>. <b><code>Memcached::GET_PRESERVE_ORDER</code></b> ensures that the keys are returned in the same order as they were requested in. <b><code>Memcached::GET_EXTENDED</code></b> ensures that the CAS tokens will be fetched too.</p>
		 * @param array $keys <p>Array of keys to retrieve.</p>
		 * @param int $get_flags <p>The flags for the get operation.</p>
		 * @return array|false <p>Returns the array of found items or <b><code>false</code></b> on failure. Use <code>Memcached::getResultCode()</code> if necessary.</p>
		 * @link https://php.net/manual/en/memcached.getmulti.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function getMulti(array $keys, int $get_flags = 0): array|false {}

		/**
		 * Retrieve multiple items from a specific server
		 * <p><b>Memcached::getMultiByKey()</b> is functionally equivalent to <code>Memcached::getMulti()</code>, except that the free-form <code>server_key</code> can be used to map the <code>keys</code> to a specific server.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @param array $keys <p>Array of keys to retrieve.</p>
		 * @param int $get_flags <p>The flags for the get operation.</p>
		 * @return array|false <p>Returns the array of found items or <b><code>false</code></b> on failure. Use <code>Memcached::getResultCode()</code> if necessary.</p>
		 * @link https://php.net/manual/en/memcached.getmultibykey.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function getMultiByKey(string $server_key, array $keys, int $get_flags = 0): array|false {}

		/**
		 * Retrieve a Memcached option value
		 * <p>This method returns the value of a Memcached <code>option</code>. Some options correspond to the ones defined by libmemcached, and some are specific to the extension. See Memcached Constants for more information.</p>
		 * @param int $option <p>One of the <code>Memcached::OPT_&#42;</code> constants.</p>
		 * @return mixed <p>Returns the value of the requested option, or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/memcached.getoption.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function getOption(int $option): mixed {}

		/**
		 * Return the result code of the last operation
		 * <p><b>Memcached::getResultCode()</b> returns one of the <b><code>Memcached::RES_&#42;</code></b> constants that is the result of the last executed Memcached method.</p>
		 * @return int <p>Result code of the last Memcached operation.</p>
		 * @link https://php.net/manual/en/memcached.getresultcode.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function getResultCode(): int {}

		/**
		 * Return the message describing the result of the last operation
		 * <p><b>Memcached::getResultMessage()</b> returns a string that describes the result code of the last executed Memcached method.</p>
		 * @return string <p>Message describing the result of the last Memcached operation.</p>
		 * @link https://php.net/manual/en/memcached.getresultmessage.php
		 * @since PECL memcached >= 1.0.0
		 */
		public function getResultMessage(): string {}

		/**
		 * Map a key to a server
		 * <p><b>Memcached::getServerByKey()</b> returns the server that would be selected by a particular <code>server_key</code> in all the <b>Memcached::&#42;ByKey()</b> operations.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @return array|false <p>Returns an array containing three keys of <code>host</code>, <code>port</code>, and <code>weight</code> on success or <b><code>false</code></b> on failure. Use <code>Memcached::getResultCode()</code> if necessary.</p>
		 * @link https://php.net/manual/en/memcached.getserverbykey.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function getServerByKey(string $server_key): array|false {}

		/**
		 * Get the list of the servers in the pool
		 * <p><b>Memcached::getServerList()</b> returns the list of all servers that are in its server pool.</p>
		 * @return array <p>The list of all servers in the server pool.</p>
		 * @link https://php.net/manual/en/memcached.getserverlist.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function getServerList(): array {}

		/**
		 * Get server pool statistics
		 * <p><b>Memcached::getStats()</b> returns an array containing the state of all available memcache servers. See &#xBB;&#xA0;memcache protocol specification for details on these statistics.</p>
		 * @param ?string $type <p>The type of statistics to fetch.</p>
		 * @return array|false <p>Array of server statistics, one entry per server, or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/memcached.getstats.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function getStats(?string $type = null): array|false {}

		/**
		 * Get server pool version info
		 * <p><b>Memcached::getVersion()</b> returns an array containing the version info for all available memcache servers.</p>
		 * @return array|false <p>Array of server versions, one entry per server.</p>
		 * @link https://php.net/manual/en/memcached.getversion.php
		 * @since PECL memcached >= 0.1.5
		 */
		public function getVersion(): array|false {}

		/**
		 * Increment numeric item's value
		 * <p><b>Memcached::increment()</b> increments a numeric item's value by the specified <code>offset</code>. If the item's value is not numeric, an error will result. <b>Memcached::increment()</b> will set the item to the <code>initial_value</code> parameter if the key doesn't exist.</p>
		 * @param string $key <p>The key of the item to increment.</p>
		 * @param int $offset <p>The amount by which to increment the item's value.</p>
		 * @param int $initial_value <p>The value to set the item to if it doesn't currently exist.</p>
		 * @param int $expiry <p>The expiry time to set on the item.</p>
		 * @return int|false <p>Returns new item's value on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/memcached.increment.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function increment(string $key, int $offset = 1, int $initial_value = 0, int $expiry = 0): int|false {}

		/**
		 * Increment numeric item's value, stored on a specific server
		 * <p><b>Memcached::incrementByKey()</b> increments a numeric item's value by the specified <code>offset</code>. If the item's value is not numeric, an error will result. <b>Memcached::incrementByKey()</b> will set the item to the <code>initial_value</code> parameter if the key doesn't exist.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @param string $key <p>The key of the item to increment.</p>
		 * @param int $offset <p>The amount by which to increment the item's value.</p>
		 * @param int $initial_value <p>The value to set the item to if it doesn't currently exist.</p>
		 * @param int $expiry <p>The expiry time to set on the item.</p>
		 * @return int|false <p>Returns new item's value on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/memcached.incrementbykey.php
		 * @since PECL memcached >= 2.0.0
		 */
		public function incrementByKey(string $server_key, string $key, int $offset = 1, int $initial_value = 0, int $expiry = 0): int|false {}

		/**
		 * Check if a persitent connection to memcache is being used
		 * <p><b>Memcached::isPersistent()</b> checks if the connections to the memcache servers are persistent connections.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if Memcache instance uses a persistent connection, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/memcached.ispersistent.php
		 * @since PECL memcached >= 2.0.0
		 */
		public function isPersistent(): bool {}

		/**
		 * Check if the instance was recently created
		 * <p><b>Memcached::isPristine()</b> checks if the Memcache instance was recently created.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if instance is recently created, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/memcached.ispristine.php
		 * @since PECL memcached >= 2.0.0
		 */
		public function isPristine(): bool {}

		/**
		 * Prepend data to an existing item
		 * <p><b>Memcached::prepend()</b> prepends the given <code>value</code> string to the value of an existing item. The reason that <code>value</code> is forced to be a string is that prepending mixed types is not well-defined.</p><p><b>Note</b>:</p><p>If the <b><code>Memcached::OPT_COMPRESSION</code></b> is enabled, the operation will fail and a warning will be issued, because prepending compressed data to a value that is potentially already compressed is not possible.</p>
		 * @param string $key <p>The key of the item to prepend the data to.</p>
		 * @param string $value <p>The string to prepend.</p>
		 * @return ?bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. Returns <b><code>null</code></b> when compression is on.</p>
		 * @link https://php.net/manual/en/memcached.prepend.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function prepend(string $key, string $value): ?bool {}

		/**
		 * Prepend data to an existing item on a specific server
		 * <p><b>Memcached::prependByKey()</b> is functionally equivalent to <code>Memcached::prepend()</code>, except that the free-form <code>server_key</code> can be used to map the <code>key</code> to a specific server.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @param string $key <p>The key of the item to prepend the data to.</p>
		 * @param string $value <p>The string to prepend.</p>
		 * @return ?bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. Returns <b><code>null</code></b> when compression is on.</p>
		 * @link https://php.net/manual/en/memcached.prependbykey.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function prependByKey(string $server_key, string $key, string $value): ?bool {}

		/**
		 * Close any open connections
		 * <p><b>Memcached::quit()</b> closes any open connections to the memcache servers.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/memcached.quit.php
		 * @since PECL memcached >= 2.0.0
		 */
		public function quit(): bool {}

		/**
		 * Replace the item under an existing key
		 * <p><b>Memcached::replace()</b> is similar to <code>Memcached::set()</code>, but the operation fails if the <code>key</code> does not exist on the server.</p>
		 * @param string $key <p>The key under which to store the value.</p>
		 * @param mixed $value <p>The value to store.</p>
		 * @param int $expiration <p>The expiration time, defaults to 0. See Expiration Times for more info.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. The <code>Memcached::getResultCode()</code> will return <b><code>Memcached::RES_NOTSTORED</code></b> if the key does not exist.</p>
		 * @link https://php.net/manual/en/memcached.replace.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function replace(string $key, mixed $value, int $expiration = 0): bool {}

		/**
		 * Replace the item under an existing key on a specific server
		 * <p><b>Memcached::replaceByKey()</b> is functionally equivalent to <code>Memcached::replace()</code>, except that the free-form <code>server_key</code> can be used to map the <code>key</code> to a specific server. This is useful if you need to keep a bunch of related keys on a certain server.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @param string $key <p>The key under which to store the value.</p>
		 * @param mixed $value <p>The value to store.</p>
		 * @param int $expiration <p>The expiration time, defaults to 0. See Expiration Times for more info.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. The <code>Memcached::getResultCode()</code> will return <b><code>Memcached::RES_NOTSTORED</code></b> if the key does not exist.</p>
		 * @link https://php.net/manual/en/memcached.replacebykey.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function replaceByKey(string $server_key, string $key, mixed $value, int $expiration = 0): bool {}

		/**
		 * Clears all servers from the server list
		 * <p><b>Memcached::resetserverlist()</b> removes all memcache servers from the known server list, resetting it back to empty.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/memcached.resetserverlist.php
		 * @since PECL memcached >= 2.0.0
		 */
		public function resetServerList(): bool {}

		/**
		 * Store an item
		 * <p><b>Memcached::set()</b> stores the <code>value</code> on a memcache server under the specified <code>key</code>. The <code>expiration</code> parameter can be used to control when the value is considered expired.</p><p>The value can be any valid PHP type except for resources, because those cannot be represented in a serialized form. If the <b><code>Memcached::OPT_COMPRESSION</code></b> option is turned on, the serialized value will also be compressed before storage.</p>
		 * @param string $key <p>The key under which to store the value.</p>
		 * @param mixed $value <p>The value to store.</p>
		 * @param int $expiration <p>The expiration time, defaults to 0. See Expiration Times for more info.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. Use <code>Memcached::getResultCode()</code> if necessary.</p>
		 * @link https://php.net/manual/en/memcached.set.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function set(string $key, mixed $value, int $expiration = 0): bool {}

		/**
		 * Store an item on a specific server
		 * <p><b>Memcached::setByKey()</b> is functionally equivalent to <code>Memcached::set()</code>, except that the free-form <code>server_key</code> can be used to map the <code>key</code> to a specific server. This is useful if you need to keep a bunch of related keys on a certain server.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @param string $key <p>The key under which to store the value.</p>
		 * @param mixed $value <p>The value to store.</p>
		 * @param int $expiration <p>The expiration time, defaults to 0. See Expiration Times for more info.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. Use <code>Memcached::getResultCode()</code> if necessary.</p>
		 * @link https://php.net/manual/en/memcached.setbykey.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function setByKey(string $server_key, string $key, mixed $value, int $expiration = 0): bool {}

		/**
		 * Set AES encryption key for data in Memcached
		 * <p>This method sets the AES encryption/decryption key for data written to and read from Memcached.</p>
		 * @param string $key The AES key.
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/memcached.setencodingkey.php
		 * @since PECL memcached >= 3.1.0
		 */
		public function setEncodingKey(string $key): bool {}

		/**
		 * Store multiple items
		 * <p><b>Memcached::setMulti()</b> is similar to <code>Memcached::set()</code>, but instead of a single key/value item, it works on multiple items specified in <code>items</code>. The <code>expiration</code> time applies to all the items at once.</p>
		 * @param array $items <p>An array of key/value pairs to store on the server.</p>
		 * @param int $expiration <p>The expiration time, defaults to 0. See Expiration Times for more info.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. Use <code>Memcached::getResultCode()</code> if necessary.</p>
		 * @link https://php.net/manual/en/memcached.setmulti.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function setMulti(array $items, int $expiration = 0): bool {}

		/**
		 * Store multiple items on a specific server
		 * <p><b>Memcached::setMultiByKey()</b> is functionally equivalent to <code>Memcached::setMulti()</code>, except that the free-form <code>server_key</code> can be used to map the keys from <code>items</code> to a specific server. This is useful if you need to keep a bunch of related keys on a certain server.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @param array $items <p>An array of key/value pairs to store on the server.</p>
		 * @param int $expiration <p>The expiration time, defaults to 0. See Expiration Times for more info.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. Use <code>Memcached::getResultCode()</code> if necessary.</p>
		 * @link https://php.net/manual/en/memcached.setmultibykey.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function setMultiByKey(string $server_key, array $items, int $expiration = 0): bool {}

		/**
		 * Set a Memcached option
		 * <p>This method sets the value of a Memcached <code>option</code>. Some options correspond to the ones defined by libmemcached, and some are specific to the extension.</p>
		 * @param int $option <p>One of the <b><code>Memcached::OPT_&#42;</code></b> constant. See Memcached Constants for more information.</p>
		 * @param mixed $value <p>The value to be set.</p> <p><b>Note</b>:</p><p>The options listed below require values specified via constants.</p><ul> <li> <b><code>Memcached::OPT_HASH</code></b> requires <b><code>Memcached::HASH_&#42;</code></b> values. </li> <li> <b><code>Memcached::OPT_DISTRIBUTION</code></b> requires <b><code>Memcached::DISTRIBUTION_&#42;</code></b> values. </li> <li> <b><code>Memcached::OPT_SERIALIZER</code></b> requires <b><code>Memcached::SERIALIZER_&#42;</code></b> values. </li> <li> <b><code>Memcached::OPT_COMPRESSION_TYPE</code></b> requires <b><code>Memcached::COMPRESSION_&#42;</code></b> values. </li> </ul>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/memcached.setoption.php
		 * @since PECL memcached >= 0.1.0
		 */
		public function setOption(int $option, mixed $value): bool {}

		/**
		 * Set Memcached options
		 * <p><b>Memcached::setOptions()</b> is a variation of the <code>Memcached::setOption()</code> that takes an array of options to be set.</p>
		 * @param array $options <p>An associative array of options where the key is the option to set and the value is the new value for the option.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/memcached.setoptions.php
		 * @since PECL memcached >= 2.0.0
		 */
		public function setOptions(array $options): bool {}

		/**
		 * Set the credentials to use for authentication
		 * <p><b>Memcached::setSaslAuthData()</b> sets the username and password that should be used for SASL authentication with the memcache servers.</p><p><i>This method is only available when the memcached extension is built with SASL support.</i> Please refer to Memcached setup for how to do this.</p>
		 * @param string $username <p>The username to use for authentication.</p>
		 * @param string $password <p>The password to use for authentication.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/memcached.setsaslauthdata.php
		 * @since PECL memcached >= 2.0.0
		 */
		public function setSaslAuthData(string $username, string $password): bool {}

		/**
		 * Set a new expiration on an item
		 * <p><b>Memcached::touch()</b> sets a new expiration value on the given key.</p>
		 * @param string $key <p>The key under which to store the value.</p>
		 * @param int $expiration <p>The expiration time, defaults to 0. See Expiration Times for more info.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. Use <code>Memcached::getResultCode()</code> if necessary.</p>
		 * @link https://php.net/manual/en/memcached.touch.php
		 * @since PECL memcached >= 2.0.0
		 */
		public function touch(string $key, int $expiration = 0): bool {}

		/**
		 * Set a new expiration on an item on a specific server
		 * <p><b>Memcached::touchByKey()</b> is functionally equivalent to <code>Memcached::touch()</code>, except that the free-form <code>server_key</code> can be used to map the <code>key</code> to a specific server.</p>
		 * @param string $server_key <p>The key identifying the server to store the value on or retrieve it from. Instead of hashing on the actual key for the item, we hash on the server key when deciding which memcached server to talk to. This allows related items to be grouped together on a single server for efficiency with multi operations.</p>
		 * @param string $key <p>The key under which to store the value.</p>
		 * @param int $expiration <p>The expiration time, defaults to 0. See Expiration Times for more info.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. Use <code>Memcached::getResultCode()</code> if necessary.</p>
		 * @link https://php.net/manual/en/memcached.touchbykey.php
		 * @since PECL memcached >= 2.0.0
		 */
		public function touchByKey(string $server_key, string $key, int $expiration = 0): bool {}
	}

	/**
	 * @link https://php.net/manual/en/class.memcachedexception.php
	 * @since PECL memcached >= 0.1.0
	 */
	class MemcachedException extends \RuntimeException {

		/**
		 * @var string <p>The exception message</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.message
		 */
		protected string $message = "";

		/**
		 * @var string <p>The string representation of the stack trace</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.string
		 */
		private string $string = "";

		/**
		 * @var int <p>The exception code</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.code
		 */
		protected int $code;

		/**
		 * @var string <p>The filename where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.file
		 */
		protected string $file = "";

		/**
		 * @var int <p>The line where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.line
		 */
		protected int $line;

		/**
		 * @var array <p>The stack trace as an array</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.trace
		 */
		private array $trace = [];

		/**
		 * @var ?Throwable <p>The previously thrown exception</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.previous
		 */
		private ?\Throwable $previous = null;

		/**
		 * Clone the exception
		 * <p><code>Exception</code>s cannot be cloned, and attempting to do so will throw an <code>Error</code>.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/exception.clone.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		private function __clone() {}

		/**
		 * String representation of the exception
		 * <p>Returns the <code>string</code> representation of the exception.</p>
		 * @return string <p>Returns the <code>string</code> representation of the exception.</p>
		 * @link https://php.net/manual/en/exception.tostring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __toString(): string {}

		/**
		 * Gets the Exception code
		 * <p>Returns the Exception code.</p>
		 * @return int <p>Returns the exception code as <code>int</code> in <code>Exception</code> but possibly as other type in <code>Exception</code> descendants (for example as <code>string</code> in <code>PDOException</code>).</p>
		 * @link https://php.net/manual/en/exception.getcode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getCode(): int {}

		/**
		 * Gets the file in which the exception was created
		 * <p>Get the name of the file in which the exception was created.</p>
		 * @return string <p>Returns the filename in which the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getfile.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getFile(): string {}

		/**
		 * Gets the line in which the exception was created
		 * <p>Get line number where the exception was created.</p>
		 * @return int <p>Returns the line number where the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getline.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getLine(): int {}

		/**
		 * Gets the Exception message
		 * <p>Returns the Exception message.</p>
		 * @return string <p>Returns the Exception message as a string.</p>
		 * @link https://php.net/manual/en/exception.getmessage.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getMessage(): string {}

		/**
		 * Returns previous Throwable
		 * <p>Returns previous <code>Throwable</code> (which had been passed as the third parameter of <code>Exception::__construct()</code>).</p>
		 * @return ?Throwable <p>Returns the previous <code>Throwable</code> if available or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/exception.getprevious.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		final public function getPrevious(): ?\Throwable {}

		/**
		 * Gets the stack trace
		 * <p>Returns the Exception stack trace.</p>
		 * @return array <p>Returns the Exception stack trace as an <code>array</code>.</p>
		 * @link https://php.net/manual/en/exception.gettrace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTrace(): array {}

		/**
		 * Gets the stack trace as a string
		 * <p>Returns the Exception stack trace as a string.</p>
		 * @return string <p>Returns the Exception stack trace as a string.</p>
		 * @link https://php.net/manual/en/exception.gettraceasstring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTraceAsString(): string {}
	}

}
