"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthRoutes = void 0;

var _configSchema = require("@osd/config-schema");

var _cryptiles = require("@hapi/cryptiles");

var _querystring = require("querystring");

var _helper = require("./helper");

var _next_url = require("../../../utils/next_url");

var _common = require("../../../../common");

var _cookie_splitter = require("../../../session/cookie_splitter");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   *   Copyright OpenSearch Contributors
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   *   Licensed under the Apache License, Version 2.0 (the "License").
                                                                                                                                                                                                                   *   You may not use this file except in compliance with the License.
                                                                                                                                                                                                                   *   A copy of the License is located at
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   *       http://www.apache.org/licenses/LICENSE-2.0
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   *   or in the "license" file accompanying this file. This file is distributed
                                                                                                                                                                                                                   *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
                                                                                                                                                                                                                   *   express or implied. See the License for the specific language governing
                                                                                                                                                                                                                   *   permissions and limitations under the License.
                                                                                                                                                                                                                   */

class OpenIdAuthRoutes {
  constructor(router, config, sessionStorageFactory, openIdAuthConfig, securityClient, core, wreckClient) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.openIdAuthConfig = openIdAuthConfig;
    this.securityClient = securityClient;
    this.core = core;
    this.wreckClient = wreckClient;
  }

  redirectToLogin(request, response) {
    this.sessionStorageFactory.asScoped(request).clear();
    return response.redirected({
      headers: {
        location: `${this.core.http.basePath.serverBasePath}${_common.OPENID_AUTH_LOGIN}`
      }
    });
  }

  getExtraAuthStorageOptions(logger) {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.openid.extra_storage.cookie_prefix,
      additionalCookies: this.config.openid.extra_storage.additional_cookies,
      logger
    };
  }

  setupRoutes() {
    this.router.get({
      path: _common.OPENID_AUTH_LOGIN,
      validate: {
        query: _configSchema.schema.object({
          code: _configSchema.schema.maybe(_configSchema.schema.string()),
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          })),
          state: _configSchema.schema.maybe(_configSchema.schema.string()),
          refresh: _configSchema.schema.maybe(_configSchema.schema.string())
        }, {
          unknowns: 'allow'
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      var _this$config$openid2, _this$config$openid3;

      // implementation refers to https://github.com/hapijs/bell/blob/master/lib/oauth.js
      // Sign-in initialization
      if (!request.query.code) {
        var _this$config$openid;

        const nonce = (0, _cryptiles.randomString)(OpenIdAuthRoutes.NONCE_LENGTH);
        const query = {
          client_id: (_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.client_id,
          response_type: _common.AUTH_RESPONSE_TYPE,
          redirect_uri: `${(0, _helper.getBaseRedirectUrl)(this.config, this.core, request)}${_common.OPENID_AUTH_LOGIN}`,
          state: nonce,
          scope: this.openIdAuthConfig.scope
        };
        const queryString = (0, _querystring.stringify)(query);
        const location = `${this.openIdAuthConfig.authorizationEndpoint}?${queryString}`;
        const cookie = {
          oidc: {
            state: nonce,
            nextUrl: (0, _helper.getNextUrl)(this.config, this.core, request)
          },
          authType: _common.AuthType.OPEN_ID
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location
          }
        });
      } // Authentication callback
      // validate state first


      let cookie;

      try {
        var _cookie$oidc;

        cookie = await this.sessionStorageFactory.asScoped(request).get();

        if (!cookie || !((_cookie$oidc = cookie.oidc) !== null && _cookie$oidc !== void 0 && _cookie$oidc.state) || cookie.oidc.state !== request.query.state) {
          return this.redirectToLogin(request, response);
        }
      } catch (error) {
        return this.redirectToLogin(request, response);
      }

      const nextUrl = cookie.oidc.nextUrl;
      const clientId = (_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.client_id;
      const clientSecret = (_this$config$openid3 = this.config.openid) === null || _this$config$openid3 === void 0 ? void 0 : _this$config$openid3.client_secret;
      const query = {
        grant_type: _common.AUTH_GRANT_TYPE,
        code: request.query.code,
        redirect_uri: `${(0, _helper.getBaseRedirectUrl)(this.config, this.core, request)}${_common.OPENID_AUTH_LOGIN}`,
        client_id: clientId,
        client_secret: clientSecret
      };

      try {
        var _this$config$openid4;

        const tokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient);
        const user = await this.securityClient.authenticateWithHeader(request, this.openIdAuthConfig.authHeaderName, `Bearer ${tokenResponse.idToken}`); // set to cookie

        const sessionStorage = {
          username: user.username,
          credentials: {
            authHeaderValueExtra: true,
            expires_at: (0, _helper.getExpirationDate)(tokenResponse)
          },
          authType: _common.AuthType.OPEN_ID,
          expiryTime: Date.now() + this.config.session.ttl
        };

        if ((_this$config$openid4 = this.config.openid) !== null && _this$config$openid4 !== void 0 && _this$config$openid4.refresh_tokens && tokenResponse.refreshToken) {
          Object.assign(sessionStorage.credentials, {
            refresh_token: tokenResponse.refreshToken
          });
        }

        (0, _cookie_splitter.setExtraAuthStorage)(request, `Bearer ${tokenResponse.idToken}`, this.getExtraAuthStorageOptions(context.security_plugin.logger));
        this.sessionStorageFactory.asScoped(request).set(sessionStorage);
        return response.redirected({
          headers: {
            location: nextUrl
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`OpenId authentication failed: ${error}`);

        if (error.toString().toLowerCase().includes('authentication exception')) {
          return response.unauthorized();
        } else {
          return this.redirectToLogin(request, response);
        }
      }
    });
    this.router.get({
      path: _common.OPENID_AUTH_LOGOUT,
      validate: false
    }, async (context, request, response) => {
      var _cookie$credentials, _this$config$openid5;

      const cookie = await this.sessionStorageFactory.asScoped(request).get();
      let tokenFromExtraStorage = '';
      const extraAuthStorageOptions = this.getExtraAuthStorageOptions(context.security_plugin.logger);

      if (cookie !== null && cookie !== void 0 && (_cookie$credentials = cookie.credentials) !== null && _cookie$credentials !== void 0 && _cookie$credentials.authHeaderValueExtra) {
        tokenFromExtraStorage = (0, _cookie_splitter.getExtraAuthStorageValue)(request, extraAuthStorageOptions);
      }

      (0, _cookie_splitter.clearSplitCookies)(request, extraAuthStorageOptions);
      this.sessionStorageFactory.asScoped(request).clear(); // authHeaderValue is the bearer header, e.g. "Bearer <auth_token>"

      const token = tokenFromExtraStorage.length ? tokenFromExtraStorage.split(' ')[1] : cookie === null || cookie === void 0 ? void 0 : cookie.credentials.authHeaderValue.split(' ')[1]; // get auth token

      const nextUrl = (0, _helper.getBaseRedirectUrl)(this.config, this.core, request);
      const logoutQueryParams = {
        post_logout_redirect_uri: `${nextUrl}`,
        id_token_hint: token
      };
      const endSessionUrl = (0, _helper.composeLogoutUrl)((_this$config$openid5 = this.config.openid) === null || _this$config$openid5 === void 0 ? void 0 : _this$config$openid5.logout_url, this.openIdAuthConfig.endSessionEndpoint, logoutQueryParams);
      return response.redirected({
        headers: {
          location: endSessionUrl
        }
      });
    });
  }

}

exports.OpenIdAuthRoutes = OpenIdAuthRoutes;

_defineProperty(OpenIdAuthRoutes, "NONCE_LENGTH", 22);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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