/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.performanceanalyzer.metricsdb;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.sql.Connection;
import java.sql.DriverManager;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jooq.BatchBindStep;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Query;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SQLDialect;
import org.jooq.Select;
import org.jooq.SelectField;
import org.jooq.SelectJoinStep;
import org.jooq.TableLike;
import org.jooq.exception.DataAccessException;
import org.jooq.impl.DSL;
import org.opensearch.performanceanalyzer.DBUtils;
import org.opensearch.performanceanalyzer.PerformanceAnalyzerApp;
import org.opensearch.performanceanalyzer.config.PluginSettings;
import org.opensearch.performanceanalyzer.metricsdb.Dimensions;
import org.opensearch.performanceanalyzer.metricsdb.Metric;
import org.opensearch.performanceanalyzer.rca.framework.metrics.ExceptionsAndErrors;
import org.opensearch.performanceanalyzer.reader.Removable;

public class MetricsDB
implements Removable {
    private static final Logger LOG = LogManager.getLogger(MetricsDB.class);
    private static final String DB_FILE_PREFIX_PATH_DEFAULT = "/tmp/metricsdb_";
    private static final String DB_FILE_PREFIX_PATH_CONF_NAME = "metrics-db-file-prefix-path";
    private static final String DB_URL = "jdbc:sqlite:";
    private final Connection conn;
    private final DSLContext create;
    public static final String SUM = "sum";
    public static final String AVG = "avg";
    public static final String MIN = "min";
    public static final String MAX = "max";
    public static final Set<String> AGG_VALUES = Collections.unmodifiableSet(new HashSet<String>(Arrays.asList("sum", "avg", "min", "max")));
    private long windowStartTime;

    public static String getDBFilePath(long windowStartTime) {
        return MetricsDB.getFilePrefix() + windowStartTime;
    }

    public String getDBFilePath() {
        return MetricsDB.getDBFilePath(this.windowStartTime);
    }

    public static String getFilePrefix() {
        return PluginSettings.instance().getSettingValue(DB_FILE_PREFIX_PATH_CONF_NAME, DB_FILE_PREFIX_PATH_DEFAULT);
    }

    public MetricsDB(long windowStartTime) throws Exception {
        this.windowStartTime = windowStartTime;
        String url = DB_URL + this.getDBFilePath();
        try {
            this.conn = DriverManager.getConnection(url);
            this.conn.setAutoCommit(false);
        }
        catch (Exception e) {
            PerformanceAnalyzerApp.ERRORS_AND_EXCEPTIONS_AGGREGATOR.updateStat(ExceptionsAndErrors.READER_METRICSDB_ACCESS_ERRORS, "", 1);
            throw e;
        }
        this.create = DSL.using((Connection)this.conn, (SQLDialect)SQLDialect.SQLITE);
    }

    public static MetricsDB fetchExisting(long windowStartTime) throws Exception {
        String filePath = MetricsDB.getDBFilePath(windowStartTime);
        if (!new File(filePath).exists()) {
            PerformanceAnalyzerApp.ERRORS_AND_EXCEPTIONS_AGGREGATOR.updateStat(ExceptionsAndErrors.READER_METRICSDB_ACCESS_ERRORS, "", 1);
            throw new FileNotFoundException(String.format("MetricsDB file %s could not be found.", filePath));
        }
        return new MetricsDB(windowStartTime);
    }

    public void close() throws Exception {
        this.conn.close();
    }

    public void createMetric(Metric<?> metric, List<String> dimensions) {
        if (DBUtils.checkIfTableExists(this.create, metric.getName())) {
            return;
        }
        List<Field<?>> fields = DBUtils.getFieldsFromList(dimensions);
        fields.add(DSL.field((String)SUM, metric.getValueType()));
        fields.add(DSL.field((String)AVG, metric.getValueType()));
        fields.add(DSL.field((String)MIN, metric.getValueType()));
        fields.add(DSL.field((String)MAX, metric.getValueType()));
        this.create.createTable(metric.getName()).columns(fields).execute();
    }

    public BatchBindStep startBatchPut(Metric<?> metric, List<String> dimensions) {
        ArrayList<Object> dummyValues = new ArrayList<Object>();
        for (String dim : dimensions) {
            dummyValues.add(null);
        }
        dummyValues.add(null);
        dummyValues.add(null);
        dummyValues.add(null);
        dummyValues.add(null);
        return this.create.batch((Query)this.create.insertInto(DSL.table((String)metric.getName())).values(dummyValues));
    }

    public BatchBindStep startBatchPut(String tableName, int dimNum) {
        if (dimNum < 1 || !DBUtils.checkIfTableExists(this.create, tableName)) {
            throw new IllegalArgumentException(String.format("Incorrect arguments %s, %d", tableName, dimNum));
        }
        ArrayList<Object> dummyValues = new ArrayList<Object>(dimNum);
        for (int i = 0; i < dimNum; ++i) {
            dummyValues.add(null);
        }
        return this.create.batch((Query)this.create.insertInto(DSL.table((String)tableName)).values(dummyValues));
    }

    public void putMetric(Metric<Double> metric, Dimensions dimensions, long windowStartTime) {
        this.create.insertInto(DSL.table((String)metric.getName())).set(DSL.field((String)SUM, Double.class), (Object)metric.getSum()).set(DSL.field((String)AVG, Double.class), (Object)metric.getAvg()).set(DSL.field((String)MIN, Double.class), (Object)metric.getMin()).set(DSL.field((String)MAX, Double.class), (Object)metric.getMax()).set(dimensions.getFieldMap()).execute();
    }

    @VisibleForTesting
    public void deleteMetric(String metricName) {
        if (DBUtils.checkIfTableExists(this.create, metricName)) {
            this.create.dropTable(metricName).execute();
        }
    }

    public List<TableLike<Record>> getAggregatedMetricTables(List<String> metrics, List<String> aggregations, List<String> dimensions) throws Exception {
        ArrayList<TableLike<Record>> tList = new ArrayList<TableLike<Record>>();
        List<Field<?>> groupByFields = DBUtils.getFieldsFromList(dimensions);
        for (int i = 0; i < metrics.size(); ++i) {
            Field field;
            String metric = metrics.get(i);
            List<Field<?>> selectFields = DBUtils.getFieldsFromList(dimensions);
            String aggType = aggregations.get(i);
            if (aggType.equals(SUM)) {
                field = DSL.field((String)SUM, Double.class);
                selectFields.add(DSL.sum((Field)field).as(metric));
            } else if (aggType.equals(AVG)) {
                field = DSL.field((String)AVG, Double.class);
                selectFields.add(DSL.avg((Field)field).as(metric));
            } else if (aggType.equals(MIN)) {
                field = DSL.field((String)MIN, Double.class);
                selectFields.add(DSL.min((Field)field).as(metric));
            } else if (aggType.equals(MAX)) {
                field = DSL.field((String)MAX, Double.class);
                selectFields.add(DSL.max((Field)field).as(metric));
            } else {
                throw new Exception("Unknown agg type");
            }
            if (!DBUtils.checkIfTableExists(this.create, metrics.get(i))) {
                tList.add(null);
                continue;
            }
            tList.add((TableLike<Record>)this.create.select(selectFields).from((TableLike)DSL.table((String)metric)).groupBy(groupByFields).asTable());
        }
        return tList;
    }

    public Result<Record> queryMetric(List<String> metrics, List<String> aggregations, List<String> dimensions) throws Exception {
        List<TableLike<Record>> tList = this.getAggregatedMetricTables(metrics, aggregations, dimensions);
        Object finalTable = null;
        for (int i = 0; i < tList.size(); ++i) {
            TableLike<Record> metricTable = tList.get(i);
            if (metricTable == null) {
                LOG.info(String.format("%s metric table does not exist. Returning null for the metric/dimension.", metrics.get(i)));
                continue;
            }
            List<Field<?>> selectFields = DBUtils.getSelectFieldsForMetricName(metrics.get(i), metrics, dimensions);
            SelectJoinStep curTable = this.create.select(selectFields).from((TableLike)metricTable);
            finalTable = finalTable == null ? curTable : finalTable.union((Select)curTable);
        }
        List<Field<?>> allFields = DBUtils.getFieldsFromList(dimensions);
        for (String metric : metrics) {
            allFields.add(DSL.max((Field)DSL.field((String)metric, Double.class)).as(metric));
        }
        List<Field<?>> groupByFields = DBUtils.getFieldsFromList(dimensions);
        if (finalTable == null) {
            return null;
        }
        return this.create.select(allFields).from(finalTable).groupBy(groupByFields).fetch();
    }

    public Result<Record> queryMetric(String metric) throws DataAccessException {
        return this.create.select(new SelectField[0]).from((TableLike)DSL.table((String)metric)).fetch();
    }

    public Result<Record> queryMetric(String metric, Collection<String> dimensions, int limit) throws DataAccessException {
        if (!DBUtils.checkIfTableExists(this.create, metric)) {
            return null;
        }
        if (limit < 0) {
            throw new IllegalArgumentException("Limit must be non-negative");
        }
        List<Field<?>> fields = DBUtils.getFieldsFromList(dimensions);
        fields.add(DSL.field((String)SUM, Double.class));
        fields.add(DSL.field((String)AVG, Double.class));
        fields.add(DSL.field((String)MIN, Double.class));
        fields.add(DSL.field((String)MAX, Double.class));
        return this.create.select(fields).from((TableLike)DSL.table((String)metric)).limit(limit).fetch();
    }

    public void commit() throws Exception {
        this.conn.commit();
    }

    @Override
    public void remove() throws Exception {
        this.conn.close();
    }

    public void deleteOnDiskFile() {
        MetricsDB.deleteOnDiskFile(this.windowStartTime);
    }

    public static void deleteOnDiskFile(long windowStartTime) {
        Path dbFilePath = Paths.get(MetricsDB.getDBFilePath(windowStartTime), new String[0]);
        try {
            Files.delete(dbFilePath);
        }
        catch (IOException | SecurityException e) {
            LOG.error("Failed to delete File - {} with ExceptionCode: {}", (Object)dbFilePath, (Object)ExceptionsAndErrors.READER_METRICSDB_ACCESS_ERRORS, (Object)e);
            PerformanceAnalyzerApp.ERRORS_AND_EXCEPTIONS_AGGREGATOR.updateStat(ExceptionsAndErrors.READER_METRICSDB_ACCESS_ERRORS, "", 1);
        }
    }

    public static Set<Long> listOnDiskFiles() {
        String prefix = MetricsDB.getFilePrefix();
        Path parentPath = Paths.get(prefix, new String[0]).getParent();
        HashSet<Long> found = new HashSet<Long>();
        try (Stream<Path> paths = Files.list(parentPath);){
            PathMatcher matcher = FileSystems.getDefault().getPathMatcher("regex:" + prefix + "\\d+");
            int prefixLength = prefix.length();
            paths.filter(matcher::matches).map(path -> path.toString()).forEach(s -> {
                try {
                    found.add(Long.parseUnsignedLong(s.substring(prefixLength), 10));
                }
                catch (IndexOutOfBoundsException | NumberFormatException e) {
                    LOG.error("Unexpected file in metricsdb directory - {}", s);
                }
            });
        }
        catch (IOException | SecurityException e) {
            LOG.error("Failed to access metricsdb directory - {} with ExceptionCode: {}", (Object)parentPath, (Object)ExceptionsAndErrors.READER_METRICSDB_ACCESS_ERRORS, (Object)e);
            PerformanceAnalyzerApp.ERRORS_AND_EXCEPTIONS_AGGREGATOR.updateStat(ExceptionsAndErrors.READER_METRICSDB_ACCESS_ERRORS, "", 1);
        }
        return found;
    }

    public DSLContext getDSLContext() {
        return this.create;
    }

    public boolean metricExists(String metric) {
        return DBUtils.checkIfTableExists(this.create, metric);
    }
}

