"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
Object.defineProperty(exports, "__esModule", { value: true });
var commons_1 = require("../../../utils/commons");
var dimensions_1 = require("../../../utils/dimensions");
var logger_1 = require("../../../utils/logger");
var line_1 = require("../renderer/canvas/primitives/line");
var spec_1 = require("../state/utils/spec");
var axis_type_utils_1 = require("./axis_type_utils");
var scales_1 = require("./scales");
exports.defaultTickFormatter = function (tick) { return "" + tick; };
function computeAxisTicksDimensions(axisSpec, xDomain, yDomain, totalBarsInCluster, bboxCalculator, chartRotation, _a, fallBackTickFormatter, barsPadding, enableHistogramMode) {
    var gridLine = _a.gridLine, tickLabel = _a.tickLabel;
    var _b, _c;
    if (axisSpec.hide && !gridLine.horizontal.visible && !gridLine.vertical.visible) {
        return null;
    }
    var scale = getScaleForAxisSpec(axisSpec, xDomain, yDomain, totalBarsInCluster, chartRotation, 0, 1, barsPadding, enableHistogramMode);
    if (!scale) {
        logger_1.Logger.warn("Cannot compute scale for axis spec " + axisSpec.id + ". Axis will not be displayed.");
        return null;
    }
    var dimensions = computeTickDimensions(scale, (_c = (_b = axisSpec.labelFormat, (_b !== null && _b !== void 0 ? _b : axisSpec.tickFormat)), (_c !== null && _c !== void 0 ? _c : fallBackTickFormatter)), bboxCalculator, tickLabel, { timeZone: xDomain.timeZone });
    return __assign({}, dimensions);
}
exports.computeAxisTicksDimensions = computeAxisTicksDimensions;
function isYDomain(position, chartRotation) {
    var isStraightRotation = chartRotation === 0 || chartRotation === 180;
    if (axis_type_utils_1.isVerticalAxis(position)) {
        return isStraightRotation;
    }
    return !isStraightRotation;
}
exports.isYDomain = isYDomain;
function getScaleForAxisSpec(axisSpec, xDomain, yDomain, totalBarsInCluster, chartRotation, minRange, maxRange, barsPadding, enableHistogramMode) {
    var axisIsYDomain = isYDomain(axisSpec.position, chartRotation);
    var range = [minRange, maxRange];
    if (axisIsYDomain) {
        var yScales = scales_1.computeYScales({
            yDomains: yDomain,
            range: range,
            ticks: axisSpec.ticks,
            integersOnly: axisSpec.integersOnly,
        });
        if (yScales.has(axisSpec.groupId)) {
            return yScales.get(axisSpec.groupId);
        }
        return null;
    }
    return scales_1.computeXScale({
        xDomain: xDomain,
        totalBarsInCluster: totalBarsInCluster,
        range: range,
        barsPadding: barsPadding,
        enableHistogramMode: enableHistogramMode,
        ticks: axisSpec.ticks,
        integersOnly: axisSpec.integersOnly,
    });
}
exports.getScaleForAxisSpec = getScaleForAxisSpec;
function computeRotatedLabelDimensions(unrotatedDims, degreesRotation) {
    var width = unrotatedDims.width, height = unrotatedDims.height;
    var radians = (degreesRotation * Math.PI) / 180;
    var rotatedHeight = Math.abs(width * Math.sin(radians)) + Math.abs(height * Math.cos(radians));
    var rotatedWidth = Math.abs(width * Math.cos(radians)) + Math.abs(height * Math.sin(radians));
    return {
        width: rotatedWidth,
        height: rotatedHeight,
    };
}
exports.computeRotatedLabelDimensions = computeRotatedLabelDimensions;
exports.getMaxLabelDimensions = function (bboxCalculator, _a) {
    var fontSize = _a.fontSize, fontFamily = _a.fontFamily, rotation = _a.rotation;
    return function (acc, tickLabel) {
        var bbox = bboxCalculator.compute(tickLabel, 0, fontSize, fontFamily);
        var rotatedBbox = computeRotatedLabelDimensions(bbox, rotation);
        var width = Math.ceil(rotatedBbox.width);
        var height = Math.ceil(rotatedBbox.height);
        var labelWidth = Math.ceil(bbox.width);
        var labelHeight = Math.ceil(bbox.height);
        var prevWidth = acc.maxLabelBboxWidth;
        var prevHeight = acc.maxLabelBboxHeight;
        var prevLabelWidth = acc.maxLabelTextWidth;
        var prevLabelHeight = acc.maxLabelTextHeight;
        return {
            maxLabelBboxWidth: prevWidth > width ? prevWidth : width,
            maxLabelBboxHeight: prevHeight > height ? prevHeight : height,
            maxLabelTextWidth: prevLabelWidth > labelWidth ? prevLabelWidth : labelWidth,
            maxLabelTextHeight: prevLabelHeight > labelHeight ? prevLabelHeight : labelHeight,
        };
    };
};
function computeTickDimensions(scale, tickFormat, bboxCalculator, tickLabelStyle, tickFormatOptions) {
    var tickValues = scale.ticks();
    var tickLabels = tickValues.map(function (d) { return tickFormat(d, tickFormatOptions); });
    var defaultAcc = { maxLabelBboxWidth: 0, maxLabelBboxHeight: 0, maxLabelTextWidth: 0, maxLabelTextHeight: 0 };
    var dimensions = tickLabelStyle.visible
        ? tickLabels.reduce(exports.getMaxLabelDimensions(bboxCalculator, tickLabelStyle), defaultAcc)
        : defaultAcc;
    return __assign(__assign({}, dimensions), { tickValues: tickValues,
        tickLabels: tickLabels });
}
function getUserTextOffsets(dimensions, offset) {
    var defaults = {
        x: 0,
        y: 0,
    };
    if (offset.reference === 'global') {
        return {
            local: defaults,
            global: {
                x: commons_1.getPercentageValue(offset.x, dimensions.maxLabelBboxWidth, 0),
                y: commons_1.getPercentageValue(offset.y, dimensions.maxLabelBboxHeight, 0),
            },
        };
    }
    return {
        global: defaults,
        local: {
            x: commons_1.getPercentageValue(offset.x, dimensions.maxLabelTextWidth, 0),
            y: commons_1.getPercentageValue(offset.y, dimensions.maxLabelTextHeight, 0),
        },
    };
}
function getHorizontalTextOffset(width, alignment) {
    switch (alignment) {
        case 'left':
            return -width / 2;
        case 'right':
            return width / 2;
        case 'center':
        default:
            return 0;
    }
}
function getVerticalTextOffset(height, alignment) {
    switch (alignment) {
        case 'top':
            return -height / 2;
        case 'bottom':
            return height / 2;
        case 'middle':
        default:
            return 0;
    }
}
function getHorizontalAlign(position, alignment) {
    if (alignment === void 0) { alignment = commons_1.HorizontalAlignment.Near; }
    if (alignment === commons_1.HorizontalAlignment.Center ||
        alignment === commons_1.HorizontalAlignment.Right ||
        alignment === commons_1.HorizontalAlignment.Left) {
        return alignment;
    }
    if (position === commons_1.Position.Left) {
        return alignment === commons_1.HorizontalAlignment.Near ? commons_1.HorizontalAlignment.Right : commons_1.HorizontalAlignment.Left;
    }
    if (position === commons_1.Position.Right) {
        return alignment === commons_1.HorizontalAlignment.Near ? commons_1.HorizontalAlignment.Left : commons_1.HorizontalAlignment.Right;
    }
    return 'center';
}
function getVerticalAlign(position, alignment) {
    if (alignment === void 0) { alignment = commons_1.VerticalAlignment.Middle; }
    if (alignment === commons_1.VerticalAlignment.Middle ||
        alignment === commons_1.VerticalAlignment.Top ||
        alignment === commons_1.VerticalAlignment.Bottom) {
        return alignment;
    }
    if (position === commons_1.Position.Top) {
        return alignment === commons_1.VerticalAlignment.Near ? commons_1.VerticalAlignment.Bottom : commons_1.VerticalAlignment.Top;
    }
    if (position === commons_1.Position.Bottom) {
        return alignment === commons_1.VerticalAlignment.Near ? commons_1.VerticalAlignment.Top : commons_1.VerticalAlignment.Bottom;
    }
    return commons_1.VerticalAlignment.Middle;
}
function getTickLabelProps(_a, tickPosition, position, axisPosition, tickDimensions, showTicks, textOffset, textAlignment) {
    var tickLine = _a.tickLine, tickLabel = _a.tickLabel;
    var _b, _c;
    var maxLabelBboxWidth = tickDimensions.maxLabelBboxWidth, maxLabelTextWidth = tickDimensions.maxLabelTextWidth, maxLabelBboxHeight = tickDimensions.maxLabelBboxHeight, maxLabelTextHeight = tickDimensions.maxLabelTextHeight;
    var tickDimension = showTicks ? tickLine.size + tickLine.padding : 0;
    var labelPadding = dimensions_1.getSimplePadding(tickLabel.padding);
    var isLeftAxis = position === commons_1.Position.Left;
    var isAxisTop = position === commons_1.Position.Top;
    var align = getHorizontalAlign(position, (_b = textAlignment) === null || _b === void 0 ? void 0 : _b.horizontal);
    var verticalAlign = getVerticalAlign(position, (_c = textAlignment) === null || _c === void 0 ? void 0 : _c.vertical);
    var userOffsets = getUserTextOffsets(tickDimensions, textOffset);
    var textOffsetX = getHorizontalTextOffset(maxLabelTextWidth, align) + userOffsets.local.x;
    var textOffsetY = getVerticalTextOffset(maxLabelTextHeight, verticalAlign) + userOffsets.local.y;
    if (axis_type_utils_1.isVerticalAxis(position)) {
        var x = isLeftAxis ? axisPosition.width - tickDimension - labelPadding.inner : tickDimension + labelPadding.inner;
        var offsetX = (isLeftAxis ? -1 : 1) * (maxLabelBboxWidth / 2);
        return {
            x: x,
            y: tickPosition,
            offsetX: offsetX + userOffsets.global.x,
            offsetY: userOffsets.global.y,
            textOffsetY: textOffsetY,
            textOffsetX: textOffsetX,
            align: align,
            verticalAlign: verticalAlign,
        };
    }
    var offsetY = isAxisTop ? -maxLabelBboxHeight / 2 : maxLabelBboxHeight / 2;
    return {
        x: tickPosition,
        y: isAxisTop ? axisPosition.height - tickDimension - labelPadding.inner : tickDimension + labelPadding.inner,
        offsetX: userOffsets.global.x,
        offsetY: offsetY + userOffsets.global.y,
        textOffsetX: textOffsetX,
        textOffsetY: textOffsetY,
        align: align,
        verticalAlign: verticalAlign,
    };
}
exports.getTickLabelProps = getTickLabelProps;
function getVerticalAxisTickLineProps(position, axisWidth, tickSize, tickPosition) {
    var isLeftAxis = position === commons_1.Position.Left;
    var y = tickPosition;
    var x1 = isLeftAxis ? axisWidth : 0;
    var x2 = isLeftAxis ? axisWidth - tickSize : tickSize;
    return [x1, y, x2, y];
}
exports.getVerticalAxisTickLineProps = getVerticalAxisTickLineProps;
function getHorizontalAxisTickLineProps(position, axisHeight, tickSize, tickPosition) {
    var isTopAxis = position === commons_1.Position.Top;
    var x = tickPosition;
    var y1 = isTopAxis ? axisHeight - tickSize : 0;
    var y2 = isTopAxis ? axisHeight : tickSize;
    return [x, y1, x, y2];
}
exports.getHorizontalAxisTickLineProps = getHorizontalAxisTickLineProps;
function getVerticalAxisGridLineProps(tickPosition, chartWidth) {
    return [0, tickPosition, chartWidth, tickPosition];
}
exports.getVerticalAxisGridLineProps = getVerticalAxisGridLineProps;
function getHorizontalAxisGridLineProps(tickPosition, chartHeight) {
    return [tickPosition, 0, tickPosition, chartHeight];
}
exports.getHorizontalAxisGridLineProps = getHorizontalAxisGridLineProps;
function getMinMaxRange(axisPosition, chartRotation, chartDimensions) {
    var width = chartDimensions.width, height = chartDimensions.height;
    switch (axisPosition) {
        case commons_1.Position.Bottom:
        case commons_1.Position.Top:
            return getBottomTopAxisMinMaxRange(chartRotation, width);
        case commons_1.Position.Left:
        case commons_1.Position.Right:
        default:
            return getLeftAxisMinMaxRange(chartRotation, height);
    }
}
exports.getMinMaxRange = getMinMaxRange;
function getBottomTopAxisMinMaxRange(chartRotation, width) {
    switch (chartRotation) {
        case 90:
            return { minRange: 0, maxRange: width };
        case -90:
            return { minRange: width, maxRange: 0 };
        case 180:
            return { minRange: width, maxRange: 0 };
        case 0:
        default:
            return { minRange: 0, maxRange: width };
    }
}
function getLeftAxisMinMaxRange(chartRotation, height) {
    switch (chartRotation) {
        case 90:
            return { minRange: 0, maxRange: height };
        case -90:
            return { minRange: height, maxRange: 0 };
        case 180:
            return { minRange: 0, maxRange: height };
        case 0:
        default:
            return { minRange: height, maxRange: 0 };
    }
}
function getAvailableTicks(axisSpec, scale, totalBarsInCluster, enableHistogramMode, fallBackTickFormatter, rotationOffset, tickFormatOptions) {
    var _a, _b;
    var ticks = scale.ticks();
    var isSingleValueScale = scale.domain[0] - scale.domain[1] === 0;
    var hasAdditionalTicks = enableHistogramMode && scale.bandwidth > 0;
    if (hasAdditionalTicks) {
        var lastComputedTick = ticks[ticks.length - 1];
        if (!isSingleValueScale) {
            var penultimateComputedTick = ticks[ticks.length - 2];
            var computedTickDistance = lastComputedTick - penultimateComputedTick;
            var numTicks = scale.minInterval / computedTickDistance;
            for (var i = 1; i <= numTicks; i++) {
                ticks.push(i * computedTickDistance + lastComputedTick);
            }
        }
    }
    var shift = totalBarsInCluster > 0 ? totalBarsInCluster : 1;
    var band = scale.bandwidth / (1 - scale.barsPadding);
    var halfPadding = (band - scale.bandwidth) / 2;
    var offset = (enableHistogramMode ? -halfPadding : (scale.bandwidth * shift) / 2) + (scale.isSingleValue() ? 0 : rotationOffset);
    var tickFormatter = (_a = axisSpec.tickFormat, (_a !== null && _a !== void 0 ? _a : fallBackTickFormatter));
    if (isSingleValueScale && hasAdditionalTicks) {
        var _c = __read(ticks, 1), firstTickValue = _c[0];
        var firstTick = {
            value: firstTickValue,
            label: tickFormatter(firstTickValue, tickFormatOptions),
            position: (_b = scale.scale(firstTickValue), (_b !== null && _b !== void 0 ? _b : 0)) + offset,
        };
        var lastTickValue = firstTickValue + scale.minInterval;
        var lastTick = {
            value: lastTickValue,
            label: tickFormatter(lastTickValue, tickFormatOptions),
            position: scale.bandwidth + halfPadding * 2,
        };
        return [firstTick, lastTick];
    }
    return enableDuplicatedTicks(axisSpec, scale, offset, fallBackTickFormatter, tickFormatOptions);
}
exports.getAvailableTicks = getAvailableTicks;
function enableDuplicatedTicks(axisSpec, scale, offset, fallBackTickFormatter, tickFormatOptions) {
    var ticks = scale.ticks();
    var allTicks = ticks.map(function (tick) {
        var _a, _b;
        return ({
            value: tick,
            label: (_a = axisSpec.tickFormat, (_a !== null && _a !== void 0 ? _a : fallBackTickFormatter))(tick, tickFormatOptions),
            position: (_b = scale.scale(tick), (_b !== null && _b !== void 0 ? _b : 0)) + offset,
        });
    });
    if (axisSpec.showDuplicatedTicks === true) {
        return allTicks;
    }
    return commons_1.getUniqueValues(allTicks, 'label', true);
}
exports.enableDuplicatedTicks = enableDuplicatedTicks;
function getVisibleTicks(allTicks, axisSpec, axisDim) {
    allTicks.sort(function (a, b) { return a.position - b.position; });
    var showOverlappingTicks = axisSpec.showOverlappingTicks, showOverlappingLabels = axisSpec.showOverlappingLabels;
    var maxLabelBboxHeight = axisDim.maxLabelBboxHeight, maxLabelBboxWidth = axisDim.maxLabelBboxWidth;
    var requiredSpace = axis_type_utils_1.isVerticalAxis(axisSpec.position) ? maxLabelBboxHeight / 2 : maxLabelBboxWidth / 2;
    var previousOccupiedSpace = 0;
    var visibleTicks = [];
    for (var i = 0; i < allTicks.length; i++) {
        var position = allTicks[i].position;
        if (i === 0) {
            visibleTicks.push(allTicks[i]);
            previousOccupiedSpace = position + requiredSpace;
        }
        else if (position - requiredSpace >= previousOccupiedSpace) {
            visibleTicks.push(allTicks[i]);
            previousOccupiedSpace = position + requiredSpace;
        }
        else {
            if (showOverlappingTicks || showOverlappingLabels) {
                var overlappingTick = __assign(__assign({}, allTicks[i]), { label: showOverlappingLabels ? allTicks[i].label : '' });
                visibleTicks.push(overlappingTick);
            }
        }
    }
    return visibleTicks;
}
exports.getVisibleTicks = getVisibleTicks;
function getAxisPosition(chartDimensions, chartMargins, axisTitleHeight, axisTitleStyle, axisSpec, axisDim, cumTopSum, cumBottomSum, cumLeftSum, cumRightSum, tickDimension, labelPaddingSum, showLabels) {
    var titlePadding = dimensions_1.getSimplePadding(axisTitleStyle.padding);
    var titleDimension = axisTitleStyle.visible && axisTitleHeight > 0 ? titlePadding.inner + axisTitleHeight + titlePadding.outer : 0;
    var position = axisSpec.position;
    var maxLabelBboxHeight = axisDim.maxLabelBboxHeight, maxLabelBboxWidth = axisDim.maxLabelBboxWidth;
    var top = chartDimensions.top, left = chartDimensions.left, height = chartDimensions.height, width = chartDimensions.width;
    var dimensions = {
        top: top,
        left: left,
        width: width,
        height: height,
    };
    var topIncrement = 0;
    var bottomIncrement = 0;
    var leftIncrement = 0;
    var rightIncrement = 0;
    if (axis_type_utils_1.isVerticalAxis(position)) {
        var dimWidth = labelPaddingSum + (showLabels ? maxLabelBboxWidth : 0) + tickDimension + titleDimension;
        if (position === commons_1.Position.Left) {
            leftIncrement = chartMargins.left + dimWidth;
            dimensions.left = cumLeftSum + chartMargins.left;
        }
        else {
            rightIncrement = dimWidth + chartMargins.right;
            dimensions.left = left + width + cumRightSum;
        }
        dimensions.width = dimWidth;
    }
    else {
        var dimHeight = labelPaddingSum + (showLabels ? maxLabelBboxHeight : 0) + tickDimension + titleDimension;
        if (position === commons_1.Position.Top) {
            topIncrement = dimHeight + chartMargins.top;
            dimensions.top = cumTopSum + chartMargins.top;
        }
        else {
            bottomIncrement = dimHeight + chartMargins.bottom;
            dimensions.top = top + height + cumBottomSum;
        }
        dimensions.height = dimHeight;
    }
    return { dimensions: dimensions, topIncrement: topIncrement, bottomIncrement: bottomIncrement, leftIncrement: leftIncrement, rightIncrement: rightIncrement };
}
exports.getAxisPosition = getAxisPosition;
function shouldShowTicks(_a, axisHidden) {
    var visible = _a.visible, strokeWidth = _a.strokeWidth, size = _a.size;
    return !axisHidden && visible && size > 0 && strokeWidth >= line_1.MIN_STROKE_WIDTH;
}
exports.shouldShowTicks = shouldShowTicks;
function getAxisTicksPositions(computedChartDims, _a, chartRotation, axisSpecs, axisDimensions, axesStyles, xDomain, yDomain, totalGroupsCount, enableHistogramMode, fallBackTickFormatter, barsPadding) {
    var chartPaddings = _a.chartPaddings, chartMargins = _a.chartMargins, sharedAxesStyle = _a.axes;
    var axisPositions = new Map();
    var axisVisibleTicks = new Map();
    var axisTicks = new Map();
    var axisGridLinesPositions = new Map();
    var chartDimensions = computedChartDims.chartDimensions;
    var cumTopSum = 0;
    var cumBottomSum = chartPaddings.bottom;
    var cumLeftSum = computedChartDims.leftMargin;
    var cumRightSum = chartPaddings.right;
    axisDimensions.forEach(function (axisDim, id) {
        var _a, _b;
        var axisSpec = spec_1.getSpecsById(axisSpecs, id);
        if (!axisSpec) {
            return;
        }
        var minMaxRanges = getMinMaxRange(axisSpec.position, chartRotation, chartDimensions);
        var scale = getScaleForAxisSpec(axisSpec, xDomain, yDomain, totalGroupsCount, chartRotation, minMaxRanges.minRange, minMaxRanges.maxRange, barsPadding, enableHistogramMode);
        if (!scale) {
            throw new Error("Cannot compute scale for axis spec " + axisSpec.id);
        }
        var tickFormatOptions = {
            timeZone: xDomain.timeZone,
        };
        var _c = (_a = axesStyles.get(id), (_a !== null && _a !== void 0 ? _a : sharedAxesStyle)), axisTitle = _c.axisTitle, tickLine = _c.tickLine, tickLabel = _c.tickLabel, gridLine = _c.gridLine;
        var isVertical = axis_type_utils_1.isVerticalAxis(axisSpec.position);
        var rotationOffset = enableHistogramMode &&
            ((isVertical && [-90].includes(chartRotation)) || (!isVertical && [180].includes(chartRotation)))
            ? scale.step
            : 0;
        var allTicks = getAvailableTicks(axisSpec, scale, totalGroupsCount, enableHistogramMode, isVertical ? fallBackTickFormatter : exports.defaultTickFormatter, rotationOffset, tickFormatOptions);
        var visibleTicks = getVisibleTicks(allTicks, axisSpec, axisDim);
        var axisSpecConfig = axisSpec.gridLine;
        var gridLineThemeStyles = isVertical ? gridLine.vertical : gridLine.horizontal;
        var gridLineStyles = axisSpecConfig ? commons_1.mergePartial(gridLineThemeStyles, axisSpecConfig) : gridLineThemeStyles;
        if (_b = axisSpec.showGridLines, (_b !== null && _b !== void 0 ? _b : gridLineStyles.visible)) {
            var gridLines = visibleTicks.map(function (tick) { return computeAxisGridLinePositions(isVertical, tick.position, chartDimensions); });
            axisGridLinesPositions.set(id, gridLines);
        }
        var labelPadding = dimensions_1.getSimplePadding(tickLabel.padding);
        var showTicks = shouldShowTicks(tickLine, axisSpec.hide);
        var axisTitleHeight = axisSpec.title !== undefined ? axisTitle.fontSize : 0;
        var tickDimension = showTicks ? tickLine.size + tickLine.padding : 0;
        var labelPaddingSum = tickLabel.visible ? labelPadding.inner + labelPadding.outer : 0;
        var axisPosition = getAxisPosition(chartDimensions, chartMargins, axisTitleHeight, axisTitle, axisSpec, axisDim, cumTopSum, cumBottomSum, cumLeftSum, cumRightSum, labelPaddingSum, tickDimension, tickLabel.visible);
        cumTopSum += axisPosition.topIncrement;
        cumBottomSum += axisPosition.bottomIncrement;
        cumLeftSum += axisPosition.leftIncrement;
        cumRightSum += axisPosition.rightIncrement;
        axisPositions.set(id, axisPosition.dimensions);
        axisVisibleTicks.set(id, visibleTicks);
        axisTicks.set(id, allTicks);
    });
    return {
        axisPositions: axisPositions,
        axisTicks: axisTicks,
        axisVisibleTicks: axisVisibleTicks,
        axisGridLinesPositions: axisGridLinesPositions,
    };
}
exports.getAxisTicksPositions = getAxisTicksPositions;
function computeAxisGridLinePositions(isVerticalAxis, tickPosition, chartDimensions) {
    var positions = isVerticalAxis
        ? getVerticalAxisGridLineProps(tickPosition, chartDimensions.width)
        : getHorizontalAxisGridLineProps(tickPosition, chartDimensions.height);
    return positions;
}
exports.computeAxisGridLinePositions = computeAxisGridLinePositions;
exports.isDuplicateAxis = function (_a, _b, tickMap, specs) {
    var position = _a.position, title = _a.title;
    var tickLabels = _b.tickLabels;
    var firstTickLabel = tickLabels[0];
    var lastTickLabel = tickLabels.slice(-1)[0];
    var hasDuplicate = false;
    tickMap.forEach(function (_a, axisId) {
        var axisTickLabels = _a.tickLabels;
        if (!hasDuplicate &&
            axisTickLabels &&
            tickLabels.length === axisTickLabels.length &&
            firstTickLabel === axisTickLabels[0] &&
            lastTickLabel === axisTickLabels.slice(-1)[0]) {
            var spec = spec_1.getSpecsById(specs, axisId);
            if (spec && spec.position === position && title === spec.title) {
                hasDuplicate = true;
            }
        }
    });
    return hasDuplicate;
};
//# sourceMappingURL=axis_utils.js.map