/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { get, find } from 'lodash';
import React, { useEffect } from 'react';
import { EuiFormRow, EuiIconTip, EuiComboBox } from '@elastic/eui';
import { i18n } from '@kbn/i18n';
import { FormattedMessage } from '@kbn/i18n/react';
import { search } from '../../../../data/public';
var _search$aggs = search.aggs,
    parseEsInterval = _search$aggs.parseEsInterval,
    InvalidEsCalendarIntervalError = _search$aggs.InvalidEsCalendarIntervalError; // we check if Elasticsearch interval is valid to show a user appropriate error message
// e.g. there is the case when a user inputs '14d' but it's '2w' in expression equivalent and the request will fail
// we don't check it for 0ms because the overall time range has not yet been set

function isValidCalendarInterval(interval) {
  if (interval === '0ms') {
    return {
      isValidCalendarValue: true
    };
  }

  try {
    parseEsInterval(interval);
    return {
      isValidCalendarValue: true
    };
  } catch (e) {
    if (e instanceof InvalidEsCalendarIntervalError) {
      return {
        isValidCalendarValue: false,
        error: e.message
      };
    }

    return {
      isValidCalendarValue: true
    };
  }
}

var errorMessage = i18n.translate('visDefaultEditor.controls.timeInterval.invalidFormatErrorMessage', {
  defaultMessage: 'Invalid interval format.'
});

function validateInterval(agg, value, definedOption, timeBase) {
  var _agg$buckets2;

  if (definedOption) {
    var _agg$buckets;

    return {
      isValid: true,
      interval: (_agg$buckets = agg.buckets) === null || _agg$buckets === void 0 ? void 0 : _agg$buckets.getInterval()
    };
  }

  if (!value) {
    return {
      isValid: false
    };
  }

  if (!timeBase) {
    // we check if Elasticsearch interval is valid ES interval to show a user appropriate error message
    // we don't check if there is timeBase
    var _isValidCalendarInter = isValidCalendarInterval(value),
        _isValidCalendarValue = _isValidCalendarInter.isValidCalendarValue,
        _error = _isValidCalendarInter.error;

    if (!_isValidCalendarValue) {
      return {
        isValid: false,
        error: _error
      };
    }
  }

  var isValid = search.aggs.isValidInterval(value, timeBase);

  if (!isValid) {
    return {
      isValid: false,
      error: errorMessage
    };
  }

  var interval = (_agg$buckets2 = agg.buckets) === null || _agg$buckets2 === void 0 ? void 0 : _agg$buckets2.getInterval();

  var _isValidCalendarInter2 = isValidCalendarInterval(interval.expression),
      isValidCalendarValue = _isValidCalendarInter2.isValidCalendarValue,
      error = _isValidCalendarInter2.error;

  if (!isValidCalendarValue) {
    return {
      isValid: false,
      error: error
    };
  }

  return {
    isValid: isValid,
    interval: interval
  };
}

function TimeIntervalParamEditor(_ref) {
  var agg = _ref.agg,
      aggParam = _ref.aggParam,
      editorConfig = _ref.editorConfig,
      value = _ref.value,
      setValue = _ref.setValue,
      showValidation = _ref.showValidation,
      setTouched = _ref.setTouched,
      setValidity = _ref.setValidity;
  var timeBase = get(editorConfig, 'interval.timeBase');
  var options = timeBase ? [] : (aggParam.options || []).reduce(function (filtered, option) {
    if (option.enabled ? option.enabled(agg) : true) {
      filtered.push({
        label: option.display,
        key: option.val
      });
    }

    return filtered;
  }, []);
  var selectedOptions = [];
  var definedOption;

  if (value) {
    definedOption = find(options, {
      key: value
    });
    selectedOptions = definedOption ? [definedOption] : [{
      label: value,
      key: 'custom'
    }];
  }

  var _validateInterval = validateInterval(agg, value, definedOption, timeBase),
      isValid = _validateInterval.isValid,
      error = _validateInterval.error,
      interval = _validateInterval.interval;

  var scaledHelpText = interval && interval.scaled ? /*#__PURE__*/React.createElement("strong", {
    "data-test-subj": "currentlyScaledText",
    className: "eui-displayBlock"
  }, /*#__PURE__*/React.createElement(FormattedMessage, {
    id: "visDefaultEditor.controls.timeInterval.scaledHelpText",
    defaultMessage: "Currently scaled to {bucketDescription}",
    values: {
      bucketDescription: get(interval, 'description') || ''
    }
  }), ' ', /*#__PURE__*/React.createElement(EuiIconTip, {
    position: "right",
    type: "questionInCircle",
    content: interval.scale <= 1 ? tooManyBucketsTooltip : tooLargeBucketsTooltip
  })) : null;
  var helpText = /*#__PURE__*/React.createElement(React.Fragment, null, scaledHelpText, get(editorConfig, 'interval.help') || selectOptionHelpText);

  var onCustomInterval = function onCustomInterval(customValue) {
    return setValue(customValue.trim());
  };

  var onChange = function onChange(opts) {
    var selectedOpt = get(opts, '0');
    setValue(selectedOpt ? selectedOpt.key : '');
  };

  useEffect(function () {
    setValidity(isValid);
  }, [isValid, setValidity]);
  return /*#__PURE__*/React.createElement(EuiFormRow, {
    compressed: true,
    error: error,
    fullWidth: true,
    helpText: helpText,
    isInvalid: showValidation && !isValid,
    label: i18n.translate('visDefaultEditor.controls.timeInterval.minimumIntervalLabel', {
      defaultMessage: 'Minimum interval'
    })
  }, /*#__PURE__*/React.createElement(EuiComboBox, {
    compressed: true,
    fullWidth: true,
    "data-test-subj": "visEditorInterval",
    isInvalid: showValidation && !isValid,
    noSuggestions: !!timeBase,
    onChange: onChange,
    onCreateOption: onCustomInterval,
    options: options,
    selectedOptions: selectedOptions,
    singleSelection: {
      asPlainText: true
    },
    placeholder: i18n.translate('visDefaultEditor.controls.timeInterval.selectIntervalPlaceholder', {
      defaultMessage: 'Select an interval'
    }),
    onBlur: setTouched
  }));
}

var tooManyBucketsTooltip = /*#__PURE__*/React.createElement(FormattedMessage, {
  id: "visDefaultEditor.controls.timeInterval.createsTooManyBucketsTooltip",
  defaultMessage: "This interval creates too many buckets to show in the selected time range, so it has been scaled up."
});
var tooLargeBucketsTooltip = /*#__PURE__*/React.createElement(FormattedMessage, {
  id: "visDefaultEditor.controls.timeInterval.createsTooLargeBucketsTooltip",
  defaultMessage: "This interval creates buckets that are too large to show in the selected time range, so it has been scaled down."
});
var selectOptionHelpText = /*#__PURE__*/React.createElement(FormattedMessage, {
  id: "visDefaultEditor.controls.timeInterval.selectOptionHelpText",
  defaultMessage: "Select an option or create a custom value. Examples: 30s, 20m, 24h, 2d, 1w, 1M"
});
export { TimeIntervalParamEditor };