# Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
# file Copyright.txt or https://cmake.org/licensing for details.

#.rst:
# FindTCL
# -------
#
# TK_INTERNAL_PATH was removed.
#
# This module finds if Tcl is installed and determines where the include
# files and libraries are.  It also determines what the name of the
# library is.  This code sets the following variables:
#
# ::
#
#   TCL_FOUND              = Tcl was found
#   TK_FOUND               = Tk was found
#   TCLTK_FOUND            = Tcl and Tk were found
#   TCL_LIBRARY            = path to Tcl library (tcl tcl80)
#   TCL_INCLUDE_PATH       = path to where tcl.h can be found
#   TCL_TCLSH              = path to tclsh binary (tcl tcl80)
#   TK_LIBRARY             = path to Tk library (tk tk80 etc)
#   TK_INCLUDE_PATH        = path to where tk.h can be found
#   TK_WISH                = full path to the wish executable
#
#
#
# In an effort to remove some clutter and clear up some issues for
# people who are not necessarily Tcl/Tk gurus/developpers, some
# variables were moved or removed.  Changes compared to CMake 2.4 are:
#
# ::
#
#    => they were only useful for people writing Tcl/Tk extensions.
#    => these libs are not packaged by default with Tcl/Tk distributions.
#       Even when Tcl/Tk is built from source, several flavors of debug libs
#       are created and there is no real reason to pick a single one
#       specifically (say, amongst tcl84g, tcl84gs, or tcl84sgx).
#       Let's leave that choice to the user by allowing him to assign
#       TCL_LIBRARY to any Tcl library, debug or not.
#    => this ended up being only a Win32 variable, and there is a lot of
#       confusion regarding the location of this file in an installed Tcl/Tk
#       tree anyway (see 8.5 for example). If you need the internal path at
#       this point it is safer you ask directly where the *source* tree is
#       and dig from there.

include(${CMAKE_CURRENT_LIST_DIR}/CMakeFindFrameworks.cmake)
include(${CMAKE_CURRENT_LIST_DIR}/FindTclsh.cmake)
include(${CMAKE_CURRENT_LIST_DIR}/FindWish.cmake)

set(MODTCL_VERSION $ENV{MODTCL_VERSION})
set(MODTK_VERSION $ENV{MODTK_VERSION})
set(MODTCL_INCDIR $ENV{MODTCL_INCDIR})
set(MODTK_INCDIR $ENV{MODTK_INCDIR})
set(MODTCL_LIB $ENV{MODTCL_LIB})
set(MODTK_LIB $ENV{MODTK_LIB})
set(MODTCL_LIBDIR $ENV{MODTCL_LIBDIR})
set(MODTK_LIBDIR $ENV{MODTK_LIBDIR})

if(TCLSH_VERSION_STRING)
  set(TCL_TCLSH_VERSION "${TCLSH_VERSION_STRING}")
else()
  get_filename_component(TCL_TCLSH_PATH "${TCL_TCLSH}" PATH)
  get_filename_component(TCL_TCLSH_PATH_PARENT "${TCL_TCLSH_PATH}" PATH)
  if(MODTCL_VERSION)
    set(TCL_TCLSH_VERSION "${MODTCL_VERSION}")
  else(MODTCL_VERSION)
    string(REGEX REPLACE
      "^.*tclsh([0-9]\\.*[0-9]).*$" "\\1" TCL_TCLSH_VERSION "${TCL_TCLSH}")
  endif(MODTCL_VERSION)
endif()

get_filename_component(TK_WISH_PATH "${TK_WISH}" PATH)
get_filename_component(TK_WISH_PATH_PARENT "${TK_WISH_PATH}" PATH)
if(MODTK_VERSION)
  set(TK_WISH_VERSION "${MODTK_VERSION}")
else(MODTK_VERSION)
  string(REGEX REPLACE
    "^.*wish([0-9]\\.*[0-9]).*$" "\\1" TK_WISH_VERSION "${TK_WISH}")
endif(MODTK_VERSION)

get_filename_component(TCL_INCLUDE_PATH_PARENT "${TCL_INCLUDE_PATH}" PATH)
get_filename_component(TK_INCLUDE_PATH_PARENT "${TK_INCLUDE_PATH}" PATH)

get_filename_component(TCL_LIBRARY_PATH "${TCL_LIBRARY}" PATH)
get_filename_component(TCL_LIBRARY_PATH_PARENT "${TCL_LIBRARY_PATH}" PATH)
string(REGEX REPLACE
  "^.*tcl([0-9]\\.*[0-9]).*$" "\\1" TCL_LIBRARY_VERSION "${TCL_LIBRARY}")

get_filename_component(TK_LIBRARY_PATH "${TK_LIBRARY}" PATH)
get_filename_component(TK_LIBRARY_PATH_PARENT "${TK_LIBRARY_PATH}" PATH)
string(REGEX REPLACE
  "^.*tk([0-9]\\.*[0-9]).*$" "\\1" TK_LIBRARY_VERSION "${TK_LIBRARY}")

set(TCLTK_OPENBSD_LIB_PATHS)

if(MODTCL_LIBDIR)
  list(APPEND TCLTK_OPENBSD_LIB_PATHS "${MODTCL_LIBDIR}")
endif()

if(MODTK_LIBDIR)
  list(APPEND TCLTK_OPENBSD_LIB_PATHS "${MODTK_LIBDIR}")
endif()

if(NOT "${TCLTK_OPENBSD_LIB_PATHS}" STREQUAL "")
  set(TCLTK_POSSIBLE_LIB_PATHS
    "${TCLTK_OPENBSD_LIB_PATHS}"
    )
else()
  set(TCLTK_POSSIBLE_LIB_PATHS
    "${TCL_INCLUDE_PATH_PARENT}/lib"
    "${TK_INCLUDE_PATH_PARENT}/lib"
    "${TCL_LIBRARY_PATH}"
    "${TK_LIBRARY_PATH}"
    "${TCL_TCLSH_PATH_PARENT}/lib"
    "${TK_WISH_PATH_PARENT}/lib"
    /usr/local/lib/tcl/tcl8.6
    /usr/local/lib/tcl/tk8.6
    /usr/local/lib/tcl/tcl8.5
    /usr/local/lib/tcl/tk8.5
    /usr/local/lib/tcl/tcl8.4
    /usr/local/lib/tcl/tk8.4
    )
endif()

if(WIN32)
  get_filename_component(
    ActiveTcl_CurrentVersion
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\ActiveState\\ActiveTcl;CurrentVersion]"
    NAME)
  set(TCLTK_POSSIBLE_LIB_PATHS ${TCLTK_POSSIBLE_LIB_PATHS}
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\ActiveState\\ActiveTcl\\${ActiveTcl_CurrentVersion}]/lib"
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\Scriptics\\Tcl\\8.6;Root]/lib"
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\Scriptics\\Tcl\\8.5;Root]/lib"
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\Scriptics\\Tcl\\8.4;Root]/lib"
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\Scriptics\\Tcl\\8.3;Root]/lib"
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\Scriptics\\Tcl\\8.2;Root]/lib"
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\Scriptics\\Tcl\\8.0;Root]/lib"
    "$ENV{ProgramFiles}/Tcl/Lib"
    "C:/Program Files/Tcl/lib"
    "C:/Tcl/lib"
    )
endif()

if(MODTCL_LIB)
  find_library(TCL_LIBRARY
    NAMES
    ${MODTCL_LIB}
    PATHS ${TCLTK_POSSIBLE_LIB_PATHS}
    )
else(MODTCL_LIB)
  find_library(TCL_LIBRARY
    NAMES
    tcl
    tcl${TCL_LIBRARY_VERSION} tcl${TCL_TCLSH_VERSION} tcl${TK_WISH_VERSION}
    tcl86 tcl8.6
    tcl85 tcl8.5
    tcl84 tcl8.4
    tcl83 tcl8.3
    tcl82 tcl8.2
    tcl80 tcl8.0
    PATHS ${TCLTK_POSSIBLE_LIB_PATHS}
    )
endif(MODTCL_LIB)

if(MODTK_LIB)
  find_library(TK_LIBRARY
    NAMES
    ${MODTK_LIB}
    PATHS ${TCLTK_POSSIBLE_LIB_PATHS}
    )
else(MODTK_LIB)
  find_library(TK_LIBRARY
    NAMES
    tk
    tk${TK_LIBRARY_VERSION} tk${TCL_TCLSH_VERSION} tk${TK_WISH_VERSION}
    tk86 tk8.6
    tk85 tk8.5
    tk84 tk8.4
    tk83 tk8.3
    tk82 tk8.2
    tk80 tk8.0
    PATHS ${TCLTK_POSSIBLE_LIB_PATHS}
    )
endif(MODTK_LIB)

CMAKE_FIND_FRAMEWORKS(Tcl)
CMAKE_FIND_FRAMEWORKS(Tk)

set(TCL_FRAMEWORK_INCLUDES)
if(Tcl_FRAMEWORKS)
  if(NOT TCL_INCLUDE_PATH)
    foreach(dir ${Tcl_FRAMEWORKS})
      set(TCL_FRAMEWORK_INCLUDES ${TCL_FRAMEWORK_INCLUDES} ${dir}/Headers)
    endforeach()
  endif()
endif()

set(TK_FRAMEWORK_INCLUDES)
if(Tk_FRAMEWORKS)
  if(NOT TK_INCLUDE_PATH)
    foreach(dir ${Tk_FRAMEWORKS})
      set(TK_FRAMEWORK_INCLUDES ${TK_FRAMEWORK_INCLUDES}
        ${dir}/Headers ${dir}/PrivateHeaders)
    endforeach()
  endif()
endif()

set(TCLTK_OPENBSD_INCLUDE_PATHS)

if(MODTCL_INCDIR)
  list(APPEND TCLTK_OPENBSD_INCLUDE_PATHS "${MODTCL_INCDIR}")
endif()

if(MODTK_INCDIR)
  list(APPEND TCLTK_OPENBSD_INCLUDE_PATHS "${MODTK_INCDIR}")
endif()

if(NOT "${TCLTK_OPENBSD_INCLUDE_PATHS}" STREQUAL "")
  set(TCLTK_POSSIBLE_INCLUDE_PATHS
    "${TCLTK_OPENBSD_INCLUDE_PATHS}"
    )
else()
  set(TCLTK_POSSIBLE_INCLUDE_PATHS
    "${TCL_LIBRARY_PATH_PARENT}/include"
    "${TK_LIBRARY_PATH_PARENT}/include"
    "${TCL_INCLUDE_PATH}"
    "${TK_INCLUDE_PATH}"
    ${TCL_FRAMEWORK_INCLUDES}
    ${TK_FRAMEWORK_INCLUDES}
    "${TCL_TCLSH_PATH_PARENT}/include"
    "${TK_WISH_PATH_PARENT}/include"
    /usr/include/tcl${TK_LIBRARY_VERSION}
    /usr/include/tcl${TCL_LIBRARY_VERSION}
    /usr/include/tcl8.6
    /usr/include/tcl8.5
    /usr/include/tcl8.4
    /usr/include/tcl8.3
    /usr/include/tcl8.2
    /usr/include/tcl8.0
    /usr/local/include/tcl8.6
    /usr/local/include/tk8.6
    /usr/local/include/tcl8.5
    /usr/local/include/tk8.5
    /usr/local/include/tcl8.4
    /usr/local/include/tk8.4
    )
endif()

if(WIN32)
  set(TCLTK_POSSIBLE_INCLUDE_PATHS ${TCLTK_POSSIBLE_INCLUDE_PATHS}
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\ActiveState\\ActiveTcl\\${ActiveTcl_CurrentVersion}]/include"
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\Scriptics\\Tcl\\8.6;Root]/include"
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\Scriptics\\Tcl\\8.5;Root]/include"
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\Scriptics\\Tcl\\8.4;Root]/include"
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\Scriptics\\Tcl\\8.3;Root]/include"
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\Scriptics\\Tcl\\8.2;Root]/include"
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\Scriptics\\Tcl\\8.0;Root]/include"
    "$ENV{ProgramFiles}/Tcl/include"
    "C:/Program Files/Tcl/include"
    "C:/Tcl/include"
    )
endif()

find_path(TCL_INCLUDE_PATH
  NAMES tcl.h
  HINTS ${TCLTK_POSSIBLE_INCLUDE_PATHS}
  )

find_path(TK_INCLUDE_PATH
  NAMES tk.h
  HINTS ${TCLTK_POSSIBLE_INCLUDE_PATHS}
  )

include(${CMAKE_CURRENT_LIST_DIR}/FindPackageHandleStandardArgs.cmake)

FIND_PACKAGE_HANDLE_STANDARD_ARGS(TCL DEFAULT_MSG TCL_LIBRARY TCL_INCLUDE_PATH)
set(TCLTK_FIND_REQUIRED ${TCL_FIND_REQUIRED})
set(TCLTK_FIND_QUIETLY  ${TCL_FIND_QUIETLY})
FIND_PACKAGE_HANDLE_STANDARD_ARGS(TCLTK DEFAULT_MSG TCL_LIBRARY TCL_INCLUDE_PATH TK_LIBRARY TK_INCLUDE_PATH)
set(TK_FIND_REQUIRED ${TCL_FIND_REQUIRED})
set(TK_FIND_QUIETLY  ${TCL_FIND_QUIETLY})
FIND_PACKAGE_HANDLE_STANDARD_ARGS(TK DEFAULT_MSG TK_LIBRARY TK_INCLUDE_PATH)

mark_as_advanced(
  TCL_INCLUDE_PATH
  TK_INCLUDE_PATH
  TCL_LIBRARY
  TK_LIBRARY
  )
