/*
 This file is part of GNU Taler
 (C) 2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  withdrawViaBankV2,
  makeTestPaymentV2,
  useSharedTestkudosEnvironment,
} from "../harness/helpers.js";
import { TalerMerchantApi } from "@gnu-taler/taler-util";

/**
 * Test stored backup wallet-core API.
 */
export async function runStoredBackupsTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bank, exchange, merchant } =
    await useSharedTestkudosEnvironment(t);

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await wres;

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const sb1Resp = await walletClient.call(
    WalletApiOperation.CreateStoredBackup,
    {},
  );
  const sbList = await walletClient.call(
    WalletApiOperation.ListStoredBackups,
    {},
  );
  t.assertTrue(sbList.storedBackups.length === 1);
  t.assertTrue(sbList.storedBackups[0].name === sb1Resp.name);

  const order = {
    summary: "Buy me!",
    amount: "TESTKUDOS:5",
    fulfillment_url: "taler://fulfillment-success/thx",
  } satisfies TalerMerchantApi.Order;

  await makeTestPaymentV2(t, { walletClient, merchant, order });
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const txn1 = await walletClient.call(WalletApiOperation.GetTransactions, {});
  t.assertDeepEqual(txn1.transactions.length, 2);

  // Recover from the stored backup now.

  const sb2Resp = await walletClient.call(
    WalletApiOperation.CreateStoredBackup,
    {},
  );

  console.log("recovering backup");

  await walletClient.call(WalletApiOperation.RecoverStoredBackup, {
    name: sb1Resp.name,
  });

  console.log("first recovery done");

  // Recovery went well, now we can delete the backup
  // of the old database we stored before importing.
  {
    const sbl1 = await walletClient.call(
      WalletApiOperation.ListStoredBackups,
      {},
    );
    t.assertTrue(sbl1.storedBackups.length === 2);

    await walletClient.call(WalletApiOperation.DeleteStoredBackup, {
      name: sb1Resp.name,
    });
    const sbl2 = await walletClient.call(
      WalletApiOperation.ListStoredBackups,
      {},
    );
    t.assertTrue(sbl2.storedBackups.length === 1);
  }

  const txn2 = await walletClient.call(WalletApiOperation.GetTransactions, {});
  // We only have the withdrawal after restoring
  t.assertDeepEqual(txn2.transactions.length, 1);
}

runStoredBackupsTest.suites = ["wallet"];
