/* pagefuncs.h: "Functions" in HTML pages?  Sure, why not! */

/* Author: Brian J. Fox (bfox@ai.mit.edu) Tue Jul 18 21:23:00 1995.

   This file is part of <Meta-HTML>(tm), a system for the rapid deployment
   of Internet and Intranet applications via the use of the Meta-HTML
   language.

   Copyright (c) 1995, 1996, Brian J. Fox (bfox@ai.mit.edu).
   Copyright (c) 1996, Universal Access Inc. (http://www.ua.com).

   Meta-HTML is free software; you can redistribute it and/or modify
   it under the terms of the UAI Free Software License as published
   by Universal Access Inc.; either version 1, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   UAI Free Software License for more details.

   You should have received a copy of the UAI Free Software License
   along with this program; if you have not, you may obtain one by
   writing to:

   Universal Access Inc.
   129 El Paseo Court
   Santa Barbara, CA
   93101  */

#if !defined (_PAGEFUNCS_H_)
#define _PAGEFUNCS_H_

#define MHTML_STREAMS 1
#define MHTML_DATABASE 1
#define MHTML_CGI_EXEC 1
#define MHTML_ARITHMETIC 1
#define MHTML_BYTE_COMPILE 1

/* Functions which should really be moved to some other file. */
extern Package *alist_to_package (char *string);
extern char *package_to_alist (Package *package, int strip);

/* Globally known variable holds onto the reserved words. */
extern Package *pagefunc_function_package;

/* Globally known variable holds onto the user defined function names. */
extern Package *mhtml_user_keywords;

/* Sequentialy process PAGE. */
extern void page_process_page (volatile PAGE *page);

/* Syntax check PAGE. */
extern int page_check_syntax (PAGE *page);

/* Add a variable to the list of page variables. */
extern void pagefunc_set_variable (char *tag, char *value);

/* Add a read only variable to the list of page variables. */
extern void pagefunc_set_variable_readonly (char *tag, char *value);

/* Return the value of the variable associated with TAG in this environment. */
extern char *pagefunc_get_variable (char *tag);

/* Canonicalize the filename given such that it is a complete path to a
   file. */
extern char *pagefunc_canonicalize_file_name (char *input);

/* If you want to delete a package, you should probably call this function
   rather than calling symbol_destroy_package () from symbols.c.  This 
   allows the engine to reset a bunch of internal variables if necessary. */
extern void pagefunc_destroy_package (char *package_name);

/* What a pagefunc takes as arguments.
   PAGE is the entire page, as passed to page_fun_function ().
   BODY is the contents of the tag body, exclusive of the tags.
   VARS is the list of variables found in the start tag.
   START is the absolute start of this tag in the page.
   END is the absolute end of this tag in the page.
   NEWSTART is a pointer to an integer.  Change this to reset the
            location of the parser pointer.
   DEBUG_LEVEL is an integer which says how much debugging to do for
               this function. */
#define PFunArgs PAGE *page, PAGE *body, Package *vars, \
		 int start, int end, int *newstart, int debug_level

typedef void PFunHandler (PFunArgs);

/* A structure which is used to describe a PAGEFUNC. */
typedef struct
{
  char *tag;		/* The name of the function. */
  int complexp;		/* Non-zero indicates <foo> ... </foo>.  */
  int debug_level;	/* How much debugging to do. */
  PFunHandler *fun;	/* The code that handles this PAGEFUNC. */
} PFunDesc;

typedef struct
{
  int type;		/* Either user_MACRO, user_SUBST, or user_DEFUN. */
  int flags;		/* Interesting bits about this function. */
  int debug_level;	/* How much debugging to do. */
  char *name;		/* The name of this macro or subst. */
  char *body;		/* The body of the macro or subst. */
  char *packname;	/* Default package for the scope of this function. */
  char **named_parameters; /* Variables to bind during function invocation. */
} UserFunction;

#define user_MACRO 1
#define user_SUBST 2
#define user_DEFUN 3

#define user_WHITESPACE_DELETED	0x01

#if defined (COMPILING_PAGEFUNCS_C)

/************************************************************/
/*							    */
/*		Language Manipulation Functions		    */
/*							    */
/************************************************************/

#if defined (MHTML_ARITHMETIC)
static void pf_gt (PFunArgs);
static void pf_lt (PFunArgs);
static void pf_eq (PFunArgs);
static void pf_add (PFunArgs);
static void pf_sub (PFunArgs);
static void pf_mul (PFunArgs);
static void pf_div (PFunArgs);
static void pf_mod (PFunArgs);
#endif /* MHTML_ARITHMETIC */

/* <randomize [seed]> */
static void pf_randomize (PFunArgs);

/* <random [max]> --> A random number between 0 and max. */
static void pf_random (PFunArgs);

/* <set-var name=value> */
static void pf_set_variable (PFunArgs);

/* <get-var name> --> value */
static void pf_get_variable (PFunArgs);

/* <get-var-once name> --> value, with parser advanced past value. */
static void pf_get_variable_once (PFunArgs);

/* <unset-var name> */
static void pf_unset_variable (PFunArgs);

/* <package-vars PACKAGE...> --> newline separated list of variable names. */
static void pf_package_vars (PFunArgs);

/* <package-delete PACKAGE...> --> Delete the named packages. */
static void pf_package_delete (PFunArgs);

/* <package-names> --> newline separated list of package names. */
static void pf_package_names (PFunArgs);

/* <package-to-alist packname> --> (( packname::foo . bar) ...) */
static void pf_package_to_alist (PFunArgs);

/* <alist-to-package alist [packname]> --> package */
static void pf_alist_to_package (PFunArgs);

static void pf_in_package (PFunArgs);
static void pf_with_local_package (PFunArgs);

/* <if test then else> If eval (test) produces non-zero length text, then... */
static void pf_if (PFunArgs);

/* <ifeq this that then else> If eval (this) = eval (that), then, else */
static void pf_ifeq (PFunArgs);

/* <when TEST> body </when> */
static void pf_when (PFunArgs);

/* <not TEST> --> "true" if TEST is empty, "" otherwise. */
static void pf_not (PFunArgs);

/* <and ...> --> Result of evaluating tests if all are true, else "". */
static void pf_and (PFunArgs);

/* <or ...> --> Result of evaluating first true test, else "" if
   all are false. */
static void pf_or (PFunArgs);

/* <var-case [this=that do-this] ...> Do-this where <get-var THIS> == THAT. */
static void pf_var_case (PFunArgs);

/* <defmacro macro-name> body </defmacro>
   Store MACRO-NAME as a complex tag.
   Do replacements on %1..%9 and %body in BODY at invocation time. */
static void pf_defmacro (PFunArgs);

/* <defun fun-name> body </defun>
   Store FUN-NAME as a complex tag.
   By default, execution takes place in the local package with
   surrounding whitespace deleted, and arguments evalled.
   Do replacements on %1..%9 and %body in BODY at invocation time. */
static void pf_defun (PFunArgs);

/* <defsubst macro-name> body </defsubst>
   Store MACRO-NAME as a sinple tag.
   Do replacements on %1..%9 and %body in BODY at invocation time. */
static void pf_defsubst (PFunArgs);

/* <undef macro-name ...>  Remove a macro definition. */
static void pf_undef (PFunArgs);

/* <function-def foo bar> --> Prints out function defs. */
static void pf_function_def (PFunArgs);

/* <while test>  body-text ... </while> Do BODY-TEXT while TEST
   produces non-zero length text. */
#define PAGE_ITERATOR_MAX_COUNT 500 /* No more than this many times. */
static void pf_while (PFunArgs);

/* <increment VAR [BY=xx]>  Change the value of VAR by XX. */
static void pf_increment (PFunArgs);

/* <decrement VAR [BY=xx]>  Change the value of VAR by XX. */
static void pf_decrement (PFunArgs);

/* <comment> ... </comment> */
static void pf_comment (PFunArgs);

/* <verbatim> .... </verbatim> Insert the contents verbatim. */
static void pf_verbatim (PFunArgs);

/* <plain-text [first-char=CODE]> Modify paragraph starts. */
static void pf_plain_text (PFunArgs);

/* <include filename alt="page value if FILENAME isn't found"> */
static void pf_include (PFunArgs);

/* <replace-page filename> Replace the current page with FILENAME. */
static void pf_replace_page (PFunArgs);

/* <redirect URL> returns an HTTP Location: directive.  The returned
   URL will include the SID if the SID is present and has a value. */
static void pf_redirect (PFunArgs);

/* <server-push>TEXT</server-push> makes TEXT go down the line immediately. */
static void pf_server_push (PFunArgs);

/* <cgi-encode var1 var2 varN> --> VAR1=VAL1&VAR2=VAL2... */
static void pf_cgi_encode (PFunArgs);

/* <cgi-decode string> --> STRING to PageVars. */
static void pf_cgi_decode (PFunArgs);

/* <small-caps [upper="+0"] [lower="-1"]>This is a list</small-caps> */
static void pf_small_caps (PFunArgs);

/* <pad <get-var foo> 12 align=right>  Pad STRING on right, left, or both. */
static void pf_pad (PFunArgs);

/* <subst-in-page this that> Replace occurrences of THIS with THAT. */
static void pf_subst_in_page (PFunArgs);

/* <subst-in-var varname this that> Replace occurrences of THIS with THAT. */
static void pf_subst_in_var (PFunArgs);

/* <subst-in-string string this that> Replace occurrences of THIS with THAT. */
static void pf_subst_in_string (PFunArgs);

/* <sort array-var [sort-fun] [caseless=true]> */
static void pf_sort (PFunArgs);

/* <date> --> Mon Feb 12 04:28:15 PST 1996 */
static void pf_date (PFunArgs);

/* <symbol-info name> -->
   "STRING\ncount" or "FUNCTION\n0" or "BINARY\nlength" */
static void pf_symbol_info (PFunArgs);

/* <match string regex
      [action=[delete | extract | report | startpos | endpos]]>
   When action is "report" (the default), returns "true" if REGEX matched.
   When action is "extract", returns the substring of STRING matching REGEX.
   When action is "delete", returns the unmatched part STRING.
   When action is "startpos", returns the numeric offset of the start of
   the matched substring.
   When action is "endtpos", returns the numeric offset of the end of the
   matched substring. */
static void pf_match (PFunArgs);

/* Return "less", "equal" or "greater" depending on string comparision of two
   arguments.  Argument "caseless=true" means do the comparison with case
   being insignificant. */
static void pf_string_compare (PFunArgs);

/* <substring string start [end]> */
static void pf_substring (PFunArgs);

/* <upcase string> */
static void pf_upcase (PFunArgs);

/* <downcase string> */
static void pf_downcase (PFunArgs);

/* <capitalize string> */
static void pf_capitalize (PFunArgs);

/* <word-wrap <get-var text> [width=60]> */
static void pf_word_wrap (PFunArgs);

/* <set-form-input-values> For every page variable, do a replacement. */
static void pf_set_form_input_values (PFunArgs);

/* <page-insert location string> */
static void pf_page_insert (PFunArgs);

/* <page-search start-offset regex> --> location */
static void pf_page_search (PFunArgs);

#if defined (DEPRECATED)
/* <html-quote "<this is an mhtml command>"> -> &ltthis is ...and&gt */
static void pf_html_quote (PFunArgs);

#endif /* DEPRECATED */

/************************************************************/
/*							    */
/*			HTML "Helper" Functions		    */
/*							    */
/************************************************************/

#if defined (DEPRECATED)
/* <input-item Name [flags for INPUT tag]>. */
static void pf_input_item (PFunArgs);
#endif /* DEPRECATED */

/* <debugging-on set-var=2 with-open-database=3 if=0> */
static void pf_debugging_on (PFunArgs);
static void pf_page_debug (PFunArgs);

static void pf_time (PFunArgs);
static void pf_pid (PFunArgs);

PFunDesc pagefunc_table[] = {

  /* General use functions: variable manipulation and iteration operators. */
  { "SET-VAR",		0, 0, pf_set_variable },
  { "GET-VAR",		0, 0, pf_get_variable },
  { "GET-VAR-ONCE",	0, 0, pf_get_variable_once },
  { "UNSET-VAR",	0, 0, pf_unset_variable },
  { "SUBST-IN-VAR",	0, 0, pf_subst_in_var },
  { "SUBST-IN-STRING",	0, 0, pf_subst_in_string },
  { "SYMBOL-INFO",	0, 0, pf_symbol_info },
  { "PACKAGE-VARS",	0, 0, pf_package_vars },
  { "PACKAGE-NAMES",	0, 0, pf_package_names },
  { "IN-PACKAGE",	1, 0, pf_in_package },
  { "WITH-LOCAL-PACKAGE",	1, 0, pf_with_local_package },
  { "PACKAGE-TO-ALIST", 0, 0, pf_package_to_alist },
  { "ALIST-TO-PACKAGE", 0, 0, pf_alist_to_package },
  { "PACKAGE-DELETE",	0, 0, pf_package_delete },
  { "INCREMENT",	0, 0, pf_increment },
  { "DECREMENT",	0, 0, pf_decrement },
  { "IF",		0, 0, pf_if },
  { "IFEQ",		0, 0, pf_ifeq },
  { "WHEN",		1, 0, pf_when },
  { "NOT",		0, 0, pf_not },
  { "AND",		0, 0, pf_and },
  { "OR",		0, 0, pf_or },
  { "VAR-CASE",		0, 0, pf_var_case },
  { "WHILE",		1, 0, pf_while },
  { "DEFMACRO",		1, 0, pf_defmacro },
  { "DEFINE-CONTAINER",	1, 0, pf_defmacro },
  { "DEFSUBST",		1, 0, pf_defsubst },
  { "DEFINE-TAG",	1, 0, pf_defsubst },
  { "DEFUN",		1, 0, pf_defun },
  { "DEFINE-FUNCTION",	1, 0, pf_defun },
  { "UNDEF",		0, 0, pf_undef },
  { "FUNCTION-DEF",	0, 0, pf_function_def },
  { "DEBUGGING-ON",	0, 0, pf_debugging_on },
  { "PAGE-DEBUG",	0, 0, pf_page_debug },
  { "SORT",		0, 0, pf_sort },
  { "DATE",		0, 0, pf_date },

  /* Random string operations. */
  { "MATCH",		0, 0, pf_match },
  { "STRING-COMPARE",	0, 0, pf_string_compare },
  { "SUBSTRING",	0, 0, pf_substring },
  { "UPCASE",		0, 0, pf_upcase },
  { "DOWNCASE",		0, 0, pf_downcase },
  { "CAPITALIZE",	0, 0, pf_capitalize },
  { "WORD-WRAP",	0, 0, pf_word_wrap },

  /* File manipulation functions. */
  { "INCLUDE",		0, 0, pf_include },
  { "REPLACE-PAGE",	0, 0, pf_replace_page },
  { "REDIRECT",		0, 0, pf_redirect },
  { "SERVER-PUSH",	1, 0, pf_server_push },
  { "CGI-ENCODE",	0, 0, pf_cgi_encode },
  { "CGI-DECODE",	0, 0, pf_cgi_decode },

  /* Block manipulation. */
  { "COMMENT",		1, 0, pf_comment },
  { "VERBATIM",		1, 0, pf_verbatim },
  { "PLAIN-TEXT",	1, 0, pf_plain_text },

#if defined (DEPRECATED)
  /* HTML "Helper" Functions. */
  { "INPUT-ITEM",	0, 0, pf_input_item },
#endif /* DEPRECATED */

  /* HTML Character manipulation. */
  { "SMALL-CAPS",	1, 0, pf_small_caps },

  /* Page manipulation.  The following functions operate on the page as it
     exists in its current state at the time the function was called. */
  { "SUBST-IN-PAGE",	0, 0, pf_subst_in_page },
  { "PAD",		0, 0, pf_pad },
  { "SET-FORM-INPUT-VALUES", 0, 0, pf_set_form_input_values },
  { "PAGE-INSERT",	0, 0, pf_page_insert },
  { "PAGE-SEARCH",	0, 0, pf_page_search },

  /* The following functions are deprecated.  They can easily be
     written as macros. */
#if defined (DEPRECATED)
  { "HTML-QUOTE",	0, 0, pf_html_quote },
#endif /* DEPRECATED */


#if defined (MHTML_ARITHMETIC)
  { "GT",		0, 0, pf_gt },
  { "LT",		0, 0, pf_lt },
  { "EQ",		0, 0, pf_eq },
  { "ADD",		0, 0, pf_add },
  { "SUB",		0, 0, pf_sub },
  { "MUL",		0, 0, pf_mul },
  { "DIV",		0, 0, pf_div },
  { "MOD",		0, 0, pf_mod },
#endif /* MHTML_ARITHMETIC */

  { "RANDOMIZE",	0, 0, pf_randomize },
  { "RANDOM",		0, 0, pf_random },

  { "TIME",		     0, 0, pf_time },
  { "PID",		     0, 0, pf_pid },

  { (char *)NULL,	0, 0, (PFunHandler *)NULL }
};

#endif /* COMPILING_PAGEFUNCS_C */

#endif /* !_PAGEFUNCS_H_ */
