# -*- coding: utf-8 -*-
##############################################################################
#
#    GNU Health: The Free Health and Hospital Information System
#    Copyright (C) 2008-2013  Luis Falcon <lfalcon@gnusolidario.org>
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
##############################################################################
from datetime import datetime
from trytond.model import ModelView, ModelSingleton, ModelSQL, fields
from trytond.transaction import Transaction
from trytond.pool import Pool


__all__ = ['GnuHealthSequences', 'PatientData', 'TestType', 'Lab',
    'GnuHealthLabTestUnits', 'GnuHealthTestCritearea',
    'GnuHealthPatientLabTest']


class GnuHealthSequences(ModelSingleton, ModelSQL, ModelView):
    "Standard Sequences for GNU Health"
    __name__ = "gnuhealth.sequences"

    lab_sequence = fields.Property(fields.Many2One('ir.sequence',
        'Lab Sequence', domain=[('code', '=', 'gnuhealth.lab')],
        required=True))


class PatientData(ModelSQL, ModelView):
    'Patient lab tests'
    __name__ = 'gnuhealth.patient'

    lab_test_ids = fields.One2Many('gnuhealth.patient.lab.test', 'patient_id',
        'Lab Tests Required')


class TestType(ModelSQL, ModelView):
    'Type of Lab test'
    __name__ = 'gnuhealth.lab.test_type'

    name = fields.Char('Test',
        help="Test type, eg X-Ray, hemogram,biopsy...", required=True,
        select=True)
    code = fields.Char('Code',
        help="Short name - code for the test", required=True, select=True)
    info = fields.Text('Description')
    product_id = fields.Many2One('product.product', 'Service', required=True)
    critearea = fields.One2Many('gnuhealth.lab.test.critearea', 'test_type_id',
        'Test Cases')

    @classmethod
    def __setup__(cls):
        super(TestType, cls).__setup__()
        cls._sql_constraints = [
            ('code_uniq', 'unique(name)', 'The Lab Test code must be unique'),
        ]


class Lab(ModelSQL, ModelView):
    'Lab Test'
    __name__ = 'gnuhealth.lab'

    name = fields.Char('ID', help="Lab result ID", readonly=True)
    test = fields.Many2One('gnuhealth.lab.test_type', 'Test type',
        help="Lab test type", required=True, select=True)
    patient = fields.Many2One('gnuhealth.patient', 'Patient',
     help="Patient ID", required=True, select=True)
    pathologist = fields.Many2One('gnuhealth.physician', 'Pathologist',
        help="Pathologist", select=True)
    requestor = fields.Many2One('gnuhealth.physician', 'Physician',
        help="Doctor who requested the test", select=True)
    results = fields.Text('Results')
    diagnosis = fields.Text('Diagnosis')
    critearea = fields.One2Many('gnuhealth.lab.test.critearea',
        'gnuhealth_lab_id', 'Lab Test Critearea')
    date_requested = fields.DateTime('Date requested', required=True,
        select=True)
    date_analysis = fields.DateTime('Date of the Analysis', select=True)

    @classmethod
    def __setup__(cls):
        super(Lab, cls).__setup__()
        cls._sql_constraints += [
            ('id_uniq', 'unique (name)', 'The test ID code must be unique'),
        ]

    @staticmethod
    def default_date_requested():
        return datetime.now()

    @staticmethod
    def default_analysis():
        return datetime.now()

    @classmethod
    def create(cls, values):
        Sequence = Pool().get('ir.sequence')
        Config = Pool().get('gnuhealth.sequences')

        values = values.copy()
        if not values.get('name'):
            config = Config(1)
            values['name'] = Sequence.get_id(
                config.lab_sequence.id)

        return super(Lab, cls).create(values)


class GnuHealthLabTestUnits(ModelSQL, ModelView):
    'Lab Test Units'
    __name__ = 'gnuhealth.lab.test.units'

    name = fields.Char('Unit', select=True)
    code = fields.Char('Code', select=True)

    @classmethod
    def __setup__(cls):
        super(GnuHealthLabTestUnits, cls).__setup__()
        cls._sql_constraints = [
            ('name_uniq', 'unique(name)', 'The Unit name must be unique'),
        ]


class GnuHealthTestCritearea(ModelSQL, ModelView):
    'Lab Test Critearea'
    __name__ = 'gnuhealth.lab.test.critearea'

    name = fields.Char('Analyte', required=True, select=True)
    excluded = fields.Boolean('Excluded', help='Select this option when'
        ' this analyte is excluded from the test')
    result = fields.Float('Value')
    result_text = fields.Char('Result - Text', help='Non-numeric results. For '
        'example qualitative values, morphological, colors ...')
    remarks = fields.Char('Remarks')
    normal_range = fields.Text('Reference')
    lower_limit = fields.Float('Lower Limit')
    upper_limit = fields.Float('Upper Limit')
    warning = fields.Boolean('Warn', help='Warns the patient about this '
        ' analyte result'
        ' It is useful to contextualize the result to each patient status '
        ' like age, sex, comorbidities, ...',
         on_change_with=['result', 'lower_limit', 'upper_limit'])
    units = fields.Many2One('gnuhealth.lab.test.units', 'Units')
    test_type_id = fields.Many2One('gnuhealth.lab.test_type', 'Test type',
     select=True)
    gnuhealth_lab_id = fields.Many2One('gnuhealth.lab', 'Test Cases',
        select=True)
    sequence = fields.Integer('Sequence')

    @classmethod
    def __setup__(cls):
        super(GnuHealthTestCritearea, cls).__setup__()
        cls._order.insert(0, ('sequence', 'ASC'))

    @staticmethod
    def default_sequence():
        return 1

    @staticmethod
    def default_excluded():
        return False

    def on_change_with_warning(self):
        if (self.result < self.lower_limit or self.result > self.upper_limit):
            return True
        return False


class GnuHealthPatientLabTest(ModelSQL, ModelView):
    'Patient Lab Test'
    __name__ = 'gnuhealth.patient.lab.test'

    name = fields.Many2One('gnuhealth.lab.test_type', 'Test Type',
        required=True, select=True)
    date = fields.DateTime('Date', select=True)
    state = fields.Selection([
        ('draft', 'Draft'),
        ('tested', 'Tested'),
        ('ordered', 'Ordered'),
        ('cancel', 'Cancel'),
        ], 'State', readonly=True, select=True)
    patient_id = fields.Many2One('gnuhealth.patient', 'Patient', required=True,
     select=True)
    doctor_id = fields.Many2One('gnuhealth.physician', 'Doctor',
        help="Doctor who Request the lab test.", select=True)

    @staticmethod
    def default_date():
        return datetime.now()

    @staticmethod
    def default_state():
        return 'draft'

    @staticmethod
    def default_doctor_id():
        User = Pool().get('res.user')
        user = User(Transaction().user)
        uid = int(user.id)

        parties = Pool().get('party.party').search([
                ('internal_user', '=', uid)])
        if parties:
            doctors = Pool().get('gnuhealth.physician').search([
                    ('name', '=', parties[0].id)])
            if doctors:
                return doctors[0].id
        else:
            return False

