// @HEADER
// *****************************************************************************
//        MueLu: A package for multigrid based preconditioning
//
// Copyright 2012 NTESS and the MueLu contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER

#ifndef MUELU_MAPTRANSFERFACTORY_DECL_HPP_
#define MUELU_MAPTRANSFERFACTORY_DECL_HPP_

#include "MueLu_ConfigDefs.hpp"
#include "MueLu_TwoLevelFactoryBase.hpp"

namespace MueLu {

/*!
  @class MapTransferFactory class.
  @brief Factory to transfer a map from a fine to a coarse level

  Factory that transfers a map (given by a variable name and a generating factory) for building
  a coarse version of the map. The coarse map is stored on the coarse level using the same variable name
  and generating factory than the original fine level map.

  The transfer is based on the prolongator maps. The prolongator/nullspace can also contain rotational modes,
  that might not be of interest for the map of interest. Use the option "nullspace vectors: limit to" to exclude such modes.

  ## Input/output ##

  ### User parameters ###
  Parameter | type | default | master.xml | validated | requested | description
  ----------|------|---------|:----------:|:---------:|:---------:|------------
  map: name                   | string  | ""     |   | * | * | Name of the map
  map: factory                | string  | "null" |   | * | * | Name of the generating factory
  P                           | Factory | null   |   | * | * | Generating factory of prolongator
  nullspace vectors: limit to | string  | "all"  |   | * | * | Use only these nullspace vectors/columns of P to transfer the map (e.g. to drop rotations)

  The * in the @c master.xml column denotes that the parameter is defined in the @c master.xml file.<br>
  The * in the @c validated column means that the parameter is declared in the list of valid input parameters (see @c GetValidParameters() ).<br>
  The * in the @c requested column states that the data is requested as input with all dependencies (see @c DeclareInput() ).

  ### Variables provided by this factory ###

  After \c Build() , the following data is available (if requested):

  Parameter | generated by | description
  ----------|--------------|------------
  | map: name | MapTransferFactory | Coarse version of the input map

*/

template <class Scalar        = DefaultScalar,
          class LocalOrdinal  = DefaultLocalOrdinal,
          class GlobalOrdinal = DefaultGlobalOrdinal,
          class Node          = DefaultNode>
class MapTransferFactory : public TwoLevelFactoryBase {
#undef MUELU_MAPTRANSFERFACTORY_SHORT
#include "MueLu_UseShortNames.hpp"

 public:
  //! Input
  //@{

  RCP<const ParameterList> GetValidParameterList() const override;

  void DeclareInput(Level& fineLevel, Level& coarseLevel) const override;

  //@}

  //@{
  //! @name Build methods.

  //! Build an object with this factory.
  void Build(Level& fineLevel, Level& coarseLevel) const override;

  //@}

 private:
  /*!
    @brief Get the max number of entries per row of P to be considered for map transfer

    To exclude some nullspace vectors (e.g. rotations in 2D or 3D elasticity), when doing the map transfer,
    this routine translates the user wish to a maximal number of entries per row of P to be considered during the map transfer.

    \warning Rows of P are looped from left to right, so we rely on the usual ordering of the nullspace vectors (translations in x/y/z, then rotations around x,y,z).

    @param[in] pL Parameter list with user-given configuration
    @return Number of entries per row of the prolongator to be used for the map transfer
  */
  int GetLimitOfProlongatorColumns(const ParameterList& pL) const;

  //! Generating factory of input variable
  mutable RCP<const FactoryBase> mapFact_;

};  // class MapTransferFactory

}  // namespace MueLu

#define MUELU_MAPTRANSFERFACTORY_SHORT
#endif /* MUELU_MAPTRANSFERFACTORY_DECL_HPP_ */
