//===================================================================================================================
//
// file :		Device_6.h
//
// description :	Include for the Device root classes in its release 6 (IDL release 6, Tango release 10)
//
// project :		TANGO
//
// author(s) :		E.Taurel
//
// Copyright (C) :      2014,2015
//						European Synchrotron Radiation Facility
//                      BP 220, Grenoble 38043
//                      FRANCE
//
// This file is part of Tango.
//
// Tango is free software: you can redistribute it and/or modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Tango is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
// of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with Tango.
// If not, see <http://www.gnu.org/licenses/>.
//
//
//
//===================================================================================================================

#ifndef _DEVICE_6_H
#define _DEVICE_6_H

#include <tango/common/tango_const.h>
#include <tango/server/device_5.h>

#include <string>
#include <memory>

namespace Tango
{
class DeviceClass;

//==================================================================================================================
//
//			The Device_6Impl class
//
//
// description :
// 		This class is derived directly from the Tango::Device_skel class generated by CORBA. It represents the
// CORBA
//		servant which will be accessed by the client. It implements all the methods and attributes defined in
// the
// 		IDL interface for Device.
//
//==================================================================================================================

/**
 * Base class for all TANGO device since version 6.
 *
 * This class inherits from DeviceImpl class which itself inherits from
 * CORBA classes where all the network layer is implemented.
 * This class has been created since release 7 of Tango library where the IDL
 * Tango module has been modified in order to create a Device_5 interface
 * which inherits from the original Device interface
 *
 *
 *
 *
 * @headerfile tango.h
 * @ingroup Server
 */

// NOLINTNEXTLINE(readability-identifier-naming)
class Device_6Impl : public virtual POA_Tango::Device_6, public Device_5Impl
{
  public:
    /**@name Constructors
     * Miscellaneous constructors */
    //@{
    /**
     * Constructs a newly allocated Device_6Impl object from its name.
     *
     * The device description field is set to <i>A Tango device</i>. The device
     * state is set to unknown and the device status is set to
     * <b>Not Initialised</b>
     *
     * @param 	device_class	Pointer to the device class object
     * @param	dev_name	The device name
     *
     */
    Device_6Impl(DeviceClass *device_class, const std::string &dev_name);

    /**
     * Constructs a newly allocated Device_6Impl object from its name and its description.
     *
     * The device
     * state is set to unknown and the device status is set to
     * <i>Not Initialised</i>
     *
     * @param 	device_class	Pointer to the device class object
     * @param	dev_name	The device name
     * @param	desc	The device description
     *
     */
    Device_6Impl(DeviceClass *device_class, const std::string &dev_name, const std::string &desc);

    /**
     * Constructs a newly allocated Device_6Impl object from all its creation
     * parameters.
     *
     * The device is constructed from its name, its description, an original state
     * and status
     *
     * @param 	device_class	Pointer to the device class object
     * @param	dev_name	The device name
     * @param	desc 		The device description
     * @param	dev_state 	The device initial state
     * @param	dev_status	The device initial status
     *
     */
    Device_6Impl(DeviceClass *device_class,
                 const std::string &dev_name,
                 const std::string &desc,
                 Tango::DevState dev_state,
                 const std::string &dev_status);

    /**
     * Constructs a newly allocated Device_6Impl object from all its creation
     * parameters with some default values.
     *
     * The device is constructed from its name, its description, an original state
     * and status. This constructor defined default values for the description,
     * state and status parameters. The default device description is <i>A TANGO device</i>.
     * The default device state is <i>UNKNOWN</i> and the default device status
     * is <i>Not initialised</i>.
     *
     * @param 	device_class	Pointer to the device class object
     * @param	dev_name	The device name
     * @param	desc	The device desc
     * @param	dev_state 	The device initial state
     * @param	dev_status	The device initial status
     *
     */
    Device_6Impl(DeviceClass *device_class,
                 const char *dev_name,
                 const char *desc = "A TANGO device",
                 Tango::DevState dev_state = Tango::UNKNOWN,
                 const char *dev_status = StatusNotSet);

    //@}

    /**@name Destructor
     * Only one desctructor is defined for this class */
    //@{
    /**
     * The device desctructor.
     */
    ~Device_6Impl() override { }

    //@}

    /**@name CORBA operation methods
     * Method defined to implement TANGO device CORBA operation */
    //@{
    /**
     * Get device info.
     *
     * Invoked when the client request the info CORBA operation.
     * It updates the black box and returns a DevInfo_6 object
     * with miscellaneous device info
     *
     * @return A DevInfo object
     */
    Tango::DevInfo_6 *info_6() override;
    //@}

    /// @privatesection

  private:
    class Device_6ImplExt
    {
      public:
        Device_6ImplExt() { }

        ~Device_6ImplExt() { }
    };

    std::unique_ptr<Device_6ImplExt> ext_6; // Class extension
};

} // namespace Tango

#endif // _DEVICE_6_H
