/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "CptecAccess.h"

//#include <string.h>

void CptecAccess::serve(MvRequest& in, MvRequest& out)
{
    std::cout << "CPTEC Access::serve in..." << std::endl;
    in.print();

    int i;

    // Build the path
    std::string spath;
    if (!buildPath(in, spath))
        return;

    // Build the filename(s)
    std::vector<std::string> vfiles;
    if (!buildFilenames(in, vfiles))
        return;

    // Handling Observation data
    const char* type = in("TYPE");
    if (strcmp(type, "OBSERVATION") == 0) {
        // Check if file exists
        std::string ffile = spath + vfiles[0];
        FILE* f           = fopen(ffile.c_str(), "r");
        if (!f) {
            std::string error = "CptecAccess-> FILE NOT FOUND: ";
            error += ffile.c_str();
            setError(1, error.c_str());
            return;
        }

        // Create output request
        out.setVerb("BUFR");
        out("PATH") = ffile.c_str();
        out.print();
        return;
    }

    // Handling model data
    // Get the requested PARAMs
    const char* caux;
    int nparams = in.iterInit("PARAMETER");
    std::map<std::string, int> params;
    std::string param;
    for (i = 0; i < nparams; ++i) {
        in.iterGetNextValue(caux);
        param         = caux;
        params[param] = 1;
    }

    // Get the requested LEVELs
    int nlevels = in.iterInit("LEVEL_LIST");
    std::map<long, int> levels;
    double dlevel;
    for (i = 0; i < nlevels; ++i) {
        in.iterGetNextValue(dlevel);
        levels[(long)dlevel] = 1;
    }

    // Build the requested fieldset
    // For each input file filter the requested fields by PARAM/LEVEL
    // and add them to the output GRIB file

    // Auxilliary variables for GribApi
    char shortName[20];
    size_t len;
    int error       = 0;
    grib_handle* h  = nullptr;
    grib_context* c = grib_context_get_default();

    // Create the output file name
    std::string outname(marstmp());
    for (i = 0; i < (signed)vfiles.size(); ++i) {
        // Full filename
        std::string ffile = spath + vfiles[i];

        // Open the input file
        FILE* f = fopen(ffile.c_str(), "r");
        if (!f) {
            std::string error = "CptecAccess-> FILE NOT FOUND: ";
            error += ffile.c_str();
            setError(1, error.c_str());
            return;
        }

        // Loop on all GRIB messages in file
        long level;
        while ((h = grib_handle_new_from_file(c, f, &error)) != nullptr) {
            len = 20;
            grib_get_string(h, "shortName", shortName, &len);
            param = std::string(shortName);
            grib_get_long(h, "level", &level);
            if (params.find(param) != params.end()) {
                if (levels.find(level) != levels.end()) {
                    grib_write_message(h, outname.c_str(), "a");
                }
            }

            grib_handle_delete(h);
        }

        // Create output request
        out.setVerb("GRIB");
        out("PATH") = outname.c_str();

        fclose(f);
    }

    std::cout << "CPTEC Access::serve out..." << std::endl;
    out.print();

    return;
}

bool CptecAccess::buildPath(MvRequest& in, string& spath)
{
    // Build the path
    // 1. Date/time related string
    const char* date = in("DATE");
    const char* time = in("TIME");
    std::string yyyymm(date, 6);
    std::string dd(date, 6, 2);
    std::string hh(time, 2);
    std::string sdate = yyyymm + '/' + dd + hh;

    //eta 20
    // /rede/eva01/modoper/tempo/regional/etap_20km/grbctl/2012/Jan/11
    // /rede/tupa_tempo/Eta/ams_15km/temp/201202/1700/grib

    //eta 15 - 1h
    // /rede/contg02/stornext/oper/tempo/Eta/ams_15km/temp/201201/1100/grib

    // 2. Fixed part of the path. Try to get it from the
    // environment variable: METVIEW_CPTEC_ACCESS_PATH.
    // If not defined, use a default value.
    const char* fpath = getenv("METVIEW_CPTEC_ACCESS_PATH");
    spath             = fpath ? fpath : "/rede/tupa_tempo/";
    //  spath = fpath ? fpath : "/mnt/";

    // 3. Part of the path related to the model
    const char* model       = in("MODEL");
    const char* observation = in("OBSERVATION");
    const char* type        = in("TYPE");

    if (strcmp(type, "MODEL") == 0) {
        if (strcmp(model, "ETA151") == 0)
            spath += "Eta/ams_15km/temp/" + sdate + "/grib/";
        //    spath += "eta15/stornext/oper/tempo/Eta/ams_15km/temp/" + sdate + "/grib/";
        else if (strcmp(model, "ETANOR") == 0)
            spath += "Eta/nor_10km/" + sdate + "/grbctl/";
        else if (strcmp(model, "ETAVALE") == 0)
            spath += "Eta/smar_5km/" + sdate + "/grbctl/";
        else if (strcmp(model, "GLOBAL299") == 0)
            spath += "MCGA/TQ0299L064/" + sdate + "/pos/";
        else if (strcmp(model, "GLOBAL213") == 0)
            spath += "MCGA/TQ0213L042/" + sdate + "/pos/";
        else if (strcmp(model, "BRAMS05") == 0)
            spath += "BRAMS/ams_05km/" + sdate + "/grads/";
        else if (strcmp(model, "MCGA3DVAR") == 0)
            spath += "G3DVAR/" + sdate + "/mcga/pos/dataout/";
    }
    else if (strcmp(type, "OBSERVATION") == 0) {
        if (strcmp(observation, "PREPBUFR") == 0)
            spath = "/rede/tupa_tempo/externos/Download/NCEP/" + sdate + "/";
        else if (strcmp(observation, "EMA_INMET") == 0)
            spath = "/rede/dds_trabalho/HDS/BUFR/";
        else if (strcmp(observation, "SYNOP") == 0)
            spath = "/rede/dds_trabalho/GEMPAK_BUFR/";
        else if (strcmp(observation, "SYNOP") == 0)
            spath = "/rede/dds_trabalho/GEMPAK_BUFR/";
        else if (strcmp(observation, "SHIP") == 0)
            spath = "/rede/dds_trabalho/GEMPAK_BUFR/";
        else if (strcmp(observation, "UPPERAIR") == 0)
            spath = "/rede/dds_trabalho/GEMPAK_BUFR/";
        else if (strcmp(observation, "AIRCRAFT") == 0)
            spath = "/rede/dds_trabalho/GEMPAK_BUFR/";
        else if (strcmp(observation, "1bamua") == 0)
            spath = "/rede/tupa_tempo/externos/Download/NCEP/" + sdate + "/";
        else if (strcmp(observation, "1bamub") == 0)
            spath = "/rede/tupa_tempo/externos/Download/NCEP/" + sdate + "/";
        else if (strcmp(observation, "1bhrs3") == 0)
            spath = "/rede/tupa_tempo/externos/Download/NCEP/" + sdate + "/";
        else if (strcmp(observation, "airsev") == 0)
            spath = "/rede/tupa_tempo/externos/Download/NCEP/" + sdate + "/";
        else if (strcmp(observation, "gpsro") == 0)
            spath = "/rede/tupa_tempo/externos/Download/NCEP/" + sdate + "/";
        else if (strcmp(observation, "mtiasi") == 0)
            spath = "/rede/tupa_tempo/externos/Download/NCEP/" + sdate + "/";
        else if (strcmp(observation, "satwnd") == 0)
            spath = "/rede/tupa_tempo/externos/Download/NCEP/" + sdate + "/";
        else if (strcmp(observation, "SAnl") == 0)
            spath = "/rede/tupa_tempo/externos/Download/NCEP/" + sdate + "/";
    }
    else {
        std::string error = "CptecAccess-> ERROR: MODEL not recognized ";
        setError(1, error.c_str());
        return false;
    }

    return true;
}

bool CptecAccess::buildFilenames(MvRequest& in, std::vector<std::string>& vfn)
{
    int i, j, k;

    // Part of the filename related to the model
    const char* model       = in("MODEL");
    const char* type        = in("TYPE");
    const char* observation = in("OBSERVATION");

    // Get all values related to parameters DATE/TIME/STEP
    double xx;
    int ndates = in.iterInit("DATE");
    std::vector<double> dates;
    dates.reserve(ndates);
    for (i = 0; i < ndates; ++i) {
        in.iterGetNextValue(xx);
        dates.push_back(xx);
    }

    int ntimes = in.iterInit("TIME");
    std::vector<double> times;
    times.reserve(ntimes);
    for (i = 0; i < ntimes; ++i) {
        in.iterGetNextValue(xx);
        times.push_back(xx / 100.);  //hhmm
    }

    int nsteps = in.iterInit("STEP");
    std::vector<double> steps;
    steps.reserve(nsteps);
    for (i = 0; i < nsteps; ++i) {
        in.iterGetNextValue(xx);
        steps.push_back(xx);
    }

    char buf[12];
    char buf2[12];

    if (strcmp(type, "MODEL") == 0) {
        if (strcmp(model, "ETA151") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    // Analise time stamp string
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmddHH", buf);
                    std::string cs = "eta_15km_";
                    cs             = cs + buf + '+';

                    // Forecast time stamp string
                    for (k = 0; k < nsteps; k++) {
                        // Forecast date
                        MvDate dfore = danal + steps[k] / 24.;
                        dfore.Format("yyyymmddHH", buf);
                        vfn.push_back(cs + buf + ".grb");
                    }
                }
            }
        }
        else if (strcmp(model, "ETANOR") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    // Analise time stamp string
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmddHH", buf);
                    std::string cs = "eta_10km_";
                    cs             = cs + buf + '+';

                    // Forecast time stamp string
                    for (k = 0; k < nsteps; k++) {
                        // Forecast date
                        MvDate dfore = danal + steps[k] / 24.;
                        dfore.Format("yyyymmddHH", buf);
                        vfn.push_back(cs + buf + ".grb");
                    }
                }
            }
        }
        else if (strcmp(model, "ETAVALE") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    // Analise time stamp string
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmddHH", buf);
                    std::string cs = "eta_05km_";
                    cs             = cs + buf + '+';

                    // Forecast time stamp string
                    for (k = 0; k < nsteps; k++) {
                        // Forecast date
                        MvDate dfore = danal + steps[k] / 24.;
                        dfore.Format("yyyymmddHH", buf);
                        vfn.push_back(cs + buf + ".grb");
                    }
                }
            }
        }
        else if (strcmp(model, "GLOBAL299") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    // Analise time stamp string
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmddHH", buf);
                    std::string cs = "GPOSNMC";
                    cs             = cs + buf;

                    // Forecast time stamp string
                    for (k = 0; k < nsteps; k++) {
                        // Forecast date
                        MvDate dfore = danal + steps[k] / 24.;
                        dfore.Format("yyyymmddHH", buf);
                        vfn.push_back(cs + buf + "P.fct.TQ0299L064.grb");
                    }
                }
            }
        }
        else if (strcmp(model, "GLOBAL213") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    // Analise time stamp string
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmddHH", buf);
                    std::string cs = "GPOSNMC";
                    cs             = cs + buf;

                    // Forecast time stamp string
                    for (k = 0; k < nsteps; k++) {
                        // Forecast date
                        MvDate dfore = danal + steps[k] / 24.;
                        dfore.Format("yyyymmddHH", buf);
                        vfn.push_back(cs + buf + "P.fct.TQ0213L042.grb");
                    }
                }
            }
        }
        else if (strcmp(model, "MCGA3DVAR") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    // Analise time stamp string
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmddHH", buf);
                    std::string cs = "GPOSCPT";
                    cs             = cs + buf;

                    // Forecast time stamp string
                    for (k = 0; k < nsteps; k++) {
                        // Forecast date
                        MvDate dfore = danal + steps[k] / 24.;
                        dfore.Format("yyyymmddHH", buf);
                        vfn.push_back(cs + buf + "P.fct.TQ0299L064.grb");
                    }
                }
            }
        }
        else if (strcmp(model, "BRAMS05") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    // Analise time stamp string
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmddHH", buf);
                    std::string cs = "JULES_BRAMS05km_";
                    cs             = cs + buf + "_";

                    // Forecast time stamp string
                    for (k = 0; k < nsteps; k++) {
                        // Forecast date
                        MvDate dfore = danal + steps[k] / 24.;
                        dfore.Format("yyyymmddHH", buf);
                        vfn.push_back(cs + buf + ".grib2");
                    }
                }
            }
        }
    }
    else if (strcmp(type, "OBSERVATION") == 0) {
        if (strcmp(observation, "PREPBUFR") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmdd", buf);
                    danal.Format("HH", buf2);
                    std::string rodada    = buf2;
                    std::string anomesdia = buf;
                    //gdas1.t00z.1bamua.tm00.bufr_d.20121129   vfn.push_back("gdas1.T" + rodada + "Z.prepbufr.nr." + anomesdia + rodada);
                    vfn.push_back("gdas1.t" + rodada + "z.prepbufr.nr." + anomesdia);
                }
            }
        }
        else if (strcmp(observation, "1bamua") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmdd", buf);
                    danal.Format("HH", buf2);
                    std::string rodada    = buf2;
                    std::string anomesdia = buf;
                    //vfn.push_back("gdas1.T" + rodada + "Z.prepbufr.nr." + anomesdia + rodada);
                    vfn.push_back("gdas1.t" + rodada + "z.1bamua.tm" + rodada + ".bufr_d." + anomesdia);
                }
            }
        }
        else if (strcmp(observation, "1bamub") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmdd", buf);
                    danal.Format("HH", buf2);
                    std::string rodada    = buf2;
                    std::string anomesdia = buf;
                    //vfn.push_back("gdas1.T" + rodada + "Z.prepbufr.nr." + anomesdia + rodada);
                    vfn.push_back("gdas1.t" + rodada + "z.1bamub.tm" + rodada + ".bufr_d." + anomesdia);
                }
            }
        }
        else if (strcmp(observation, "1bhrs3") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmdd", buf);
                    danal.Format("HH", buf2);
                    std::string rodada    = buf2;
                    std::string anomesdia = buf;
                    //vfn.push_back("gdas1.T" + rodada + "Z.prepbufr.nr." + anomesdia + rodada);
                    vfn.push_back("gdas1.t" + rodada + "z.1bhrs3.tm" + rodada + ".bufr_d." + anomesdia);
                }
            }
        }
        else if (strcmp(observation, "airsev") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmdd", buf);
                    danal.Format("HH", buf2);
                    std::string rodada    = buf2;
                    std::string anomesdia = buf;
                    //vfn.push_back("gdas1.T" + rodada + "Z.prepbufr.nr." + anomesdia + rodada);
                    vfn.push_back("gdas1.t" + rodada + "z.airsev.tm" + rodada + ".bufr_d." + anomesdia);
                }
            }
        }
        else if (strcmp(observation, "gpsro") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmdd", buf);
                    danal.Format("HH", buf2);
                    std::string rodada    = buf2;
                    std::string anomesdia = buf;
                    //vfn.push_back("gdas1.T" + rodada + "Z.prepbufr.nr." + anomesdia + rodada);
                    vfn.push_back("gdas1.t" + rodada + "z.gpsro.tm" + rodada + ".bufr_d." + anomesdia);
                }
            }
        }
        else if (strcmp(observation, "mtiasi") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmdd", buf);
                    danal.Format("HH", buf2);
                    std::string rodada    = buf2;
                    std::string anomesdia = buf;
                    //vfn.push_back("gdas1.T" + rodada + "Z.prepbufr.nr." + anomesdia + rodada);
                    vfn.push_back("gdas1.t" + rodada + "z.mtiasi.tm" + rodada + ".bufr_d." + anomesdia);
                }
            }
        }
        else if (strcmp(observation, "satwnd") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmdd", buf);
                    danal.Format("HH", buf2);
                    std::string rodada    = buf2;
                    std::string anomesdia = buf;
                    //vfn.push_back("gdas1.T" + rodada + "Z.prepbufr.nr." + anomesdia + rodada);
                    vfn.push_back("gdas1.t" + rodada + "z.satwnd.tm" + rodada + ".bufr_d." + anomesdia);
                }
            }
        }
        else if (strcmp(observation, "SAnl") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmdd", buf);
                    danal.Format("HH", buf2);
                    std::string rodada    = buf2;
                    std::string anomesdia = buf;
                    //vfn.push_back("gdas1.T" + rodada + "Z.prepbufr.nr." + anomesdia + rodada);
                    vfn.push_back("gdas2.T" + rodada + "Z.SAnl." + anomesdia + rodada);
                }
            }
        }
        else if (strcmp(observation, "EMA_INMET") == 0) {
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmdd", buf);
                    danal.Format("HH", buf2);
                    std::string rodada    = buf2;
                    std::string anomesdia = buf;
                    vfn.push_back("ISAI_" + anomesdia + rodada + ".wmo");
                }
            }
        }
        else {
            std::string tobs = (const char*)in("OBSERVATION");
            std::transform(tobs.begin(), tobs.end(), tobs.begin(), ::tolower);

            // Build all filenames
            // char buf[12];
            for (i = 0; i < ndates; i++) {
                MvDate dd(dates[i]);
                for (j = 0; j < ntimes; j++) {
                    // Analise time stamp string
                    MvDate danal = dd + times[j] / 24.;
                    danal.Format("yyyymmddHH", buf);
                    std::string cs = buf;
                    vfn.push_back(cs + "_" + tobs + ".bufr");
                }
            }
        }
    }
    else {
        std::string error = "CptecAccess-> ERROR: MODEL not recognized ";
        setError(1, error.c_str());
        return false;
    }

    return true;
}

int main(int argc, char** argv)
{
    MvApplication theApp(argc, argv);
    CptecAccess data("CPTEC_ACCESS");

    theApp.run();
}
