// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package util

import (
	"context"
	"path/filepath"
	"strings"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestCommand(t *testing.T) {
	something := "SOMETHING"
	assert.EqualValues(t, something, Command(context.Background(), nil, "echo", "-n", something))

	assert.Panics(t, func() { Command(context.Background(), nil, "false") })
}

func TestCommandWithOptions(t *testing.T) {
	assert.EqualValues(t, "", CommandWithOptions(context.Background(), CommandOptions{
		ExitCodes: []int{0, 1},
	}, "false"))
}

func TestCommandWithErr(t *testing.T) {
	assert.NoError(t, CommandWithErr(context.Background(), CommandOptions{}, "true"))
	assert.Error(t, CommandWithErr(context.Background(), CommandOptions{}, "false"))
}

func TestCommandTimeout(t *testing.T) {
	tmp := t.TempDir()

	ctx, cancel := context.WithCancel(context.Background())

	go func() {
		// blocks forever because of the firewall at 4.4.4.4 and
		// the git clone process forks a git-remote-https child process
		assert.PanicsWithValue(t, "killed", func() {
			_ = Command(ctx, nil, "git", "clone", "https://4.4.4.4", filepath.Join(tmp, "something"))
		})
	}()

	pattern := "git-remote-https origin https://4.4.4.4"
	ps := []string{"-x", "-o", "pid,ppid,pgid,args"}

	Retry(func() {
		out := Command(context.Background(), nil, "ps", ps...)
		if !strings.Contains(out, pattern) {
			panic(out + " does not contain " + pattern)
		}
	}, 5)

	cancel()
	<-ctx.Done()

	Retry(func() {
		out := Command(context.Background(), nil, "ps", ps...)
		if strings.Contains(out, pattern) {
			panic(out + " contains " + pattern)
		}
	}, 5)
}
