; Copyright (C) 2022, ForrestHunt, Inc.
; Written by Matt Kaufmann
; License: A 3-clause BSD license.  See the LICENSE file distributed with ACL2.

; This is a trivial example of an abstract stobj with a child stobj fields that
; are arrays, one of which contains stobjs.  See absstobj-with-hash-table.lisp
; for an analogous example using hash-table fields instead.

; I say "trivial" above because the only difference between the abstract stobj
; and its foundational stobj is that while both have two fields, they form a
; two-element list in the foundational stobj but they are consed together in
; the abstract stobj.

(in-package "ACL2")

(defstobj st$c fld$c)

(defstobj top$c
  (ht1 :type (array rational (8)) :initially 0)
  (ht2 :type (array st$c (5))))

(defun top$ap (x)
  (declare (xargs :guard t))
  (and (consp x)
       (let ((a (car x)))
         (and (rational-listp a)
              (equal (len a) 8)))
       (let ((a (cdr x))) ; ((a0) (a1) ... (a4))
         (and (alistp a)
              (all-nils (strip-cdrs a))
              (equal (len a) 5)))))

(defun-nx top-corr (x y)
  (and (top$cp x)
       (top$ap y)
       (equal (car x) (car y))
       (equal (cadr x) (cdr y))))

(defun create-top$a ()
  (declare (xargs :guard t))
  (cons (make-list 8 :initial-element 0)
        (make-list 5 :initial-element '(nil))))

(defun lookup1$a (key s)
  (declare (xargs :guard (and (natp key)
                              (< key 8)
                              (top$ap s))))
  (nth key (car s)))

(defun update1$a (key val s)
  (declare (xargs :guard (and (natp key)
                              (< key 8)
                              (rationalp val)
                              (top$ap s))))
  (cons (update-nth key val (car s))
        (cdr s)))

(defun lookup2$a (key s)
  (declare (xargs :guard (and (natp key)
                              (< key 5)
                              (top$ap s))))
  (nth key (cdr s)))

(defun update2$a (key st$c s)
  (declare (xargs :guard (and (natp key)
                              (< key 5)
                              (top$ap s))
                  :stobjs st$c))
  (cons (car s)
        (non-exec (update-nth key st$c (cdr s)))))

(defthm nth-open
  (implies (natp n)
           (equal (nth (1+ n) l)
                  (if (endp l)
                      nil
                    (nth n (cdr l))))))

; The capitalized events below have been generated by first running the
; defabsstobj form below.

(DEFTHM CREATE-TOP{CORRESPONDENCE}
        (TOP-CORR (CREATE-TOP$C) (CREATE-TOP$A))
        :RULE-CLASSES NIL)

(DEFTHM CREATE-TOP{PRESERVED}
        (TOP$AP (CREATE-TOP$A))
        :RULE-CLASSES NIL)

(DEFTHM LOOKUP1{CORRESPONDENCE}
        (IMPLIES (AND (TOP-CORR TOP$C TOP)
                      (NATP I)
                      (< I 8)
                      (TOP$AP TOP))
                 (EQUAL (HT1I I TOP$C)
                        (LOOKUP1$A I TOP)))
        :RULE-CLASSES NIL)

(DEFTHM LOOKUP1{GUARD-THM}
        (IMPLIES (AND (TOP-CORR TOP$C TOP)
                      (NATP I)
                      (< I 8)
                      (TOP$AP TOP))
                 (AND (INTEGERP I)
                      (<= 0 I)
                      (< I (HT1-LENGTH TOP$C))))
        :RULE-CLASSES NIL)

(defthm ht1p-update-nth
  (implies (and (ht1p x)
                (rationalp v)
                (natp i)
                (< i (len x)))
           (ht1p (update-nth i V x))))

(defthm rational-listp-update-nth
  (implies (and (rational-listp x)
                (rationalp v)
                (natp i)
                (< i (len x)))
           (rational-listp (update-nth i V x))))

(DEFTHM UPDATE1{CORRESPONDENCE}
        (IMPLIES (AND (TOP-CORR TOP$C TOP)
                      (NATP I)
                      (< I 8)
                      (RATIONALP V)
                      (TOP$AP TOP))
                 (TOP-CORR (UPDATE-HT1I I V TOP$C)
                           (UPDATE1$A I V TOP)))
        :RULE-CLASSES NIL)

(DEFTHM UPDATE1{GUARD-THM}
        (IMPLIES (AND (TOP-CORR TOP$C TOP)
                      (NATP I)
                      (< I 8)
                      (RATIONALP V)
                      (TOP$AP TOP))
                 (AND (INTEGERP I)
                      (<= 0 I)
                      (< I (HT1-LENGTH TOP$C))
                      (RATIONALP V)))
        :RULE-CLASSES NIL)

(DEFTHM UPDATE1{PRESERVED}
        (IMPLIES (AND (NATP I)
                      (< I 8)
                      (RATIONALP V)
                      (TOP$AP TOP))
                 (TOP$AP (UPDATE1$A I V TOP)))
        :RULE-CLASSES NIL)

(DEFTHM LOOKUP2{CORRESPONDENCE}
        (IMPLIES (AND (TOP-CORR TOP$C TOP)
                      (NATP I)
                      (< I 5)
                      (TOP$AP TOP))
                 (EQUAL (HT2I I TOP$C)
                        (LOOKUP2$A I TOP)))
        :RULE-CLASSES NIL)

(DEFTHM LOOKUP2{GUARD-THM}
        (IMPLIES (AND (TOP-CORR TOP$C TOP)
                      (NATP I)
                      (< I 5)
                      (TOP$AP TOP))
                 (AND (INTEGERP I)
                      (<= 0 I)
                      (< I (HT2-LENGTH TOP$C))))
        :RULE-CLASSES NIL)

(defthm update2{correspondence}-lemma
  (implies (and (all-nils (strip-cdrs x))
                (st$cp st$c)
                (natp i)
                (< i (len x)))
           (all-nils (strip-cdrs (update-nth i st$c x)))))

(defthm alistp-update-nth
  (implies (and (alistp x)
                (st$cp st$c)
                (natp i)
                (< i (len x)))
           (alistp (update-nth i st$c x))))

(defthm ht2p-is-alistp
  (equal (ht2p x)
         (and (alistp x)
              (all-nils (strip-cdrs x)))))

(DEFTHM UPDATE2{CORRESPONDENCE}
        (IMPLIES (AND (TOP-CORR TOP$C TOP)
                      (ST$CP ST$C)
                      (NATP I)
                      (< I 5)
                      (TOP$AP TOP))
                 (TOP-CORR (UPDATE-HT2I I ST$C TOP$C)
                           (UPDATE2$A I ST$C TOP)))
        :RULE-CLASSES NIL)

(DEFTHM UPDATE2{GUARD-THM}
        (IMPLIES (AND (TOP-CORR TOP$C TOP)
                      (ST$CP ST$C)
                      (NATP I)
                      (< I 5)
                      (TOP$AP TOP))
                 (AND (INTEGERP I)
                      (<= 0 I)
                      (< I (HT2-LENGTH TOP$C))))
        :RULE-CLASSES NIL)

(DEFTHM UPDATE2{PRESERVED}
        (IMPLIES (AND (ST$CP ST$C)
                      (NATP I)
                      (< I 5)
                      (TOP$AP TOP))
                 (TOP$AP (UPDATE2$A I ST$C TOP)))
        :RULE-CLASSES NIL)

(defabsstobj top
  :foundation top$c
  :corr-fn top-corr
  :recognizer (topp :logic top$ap :exec top$cp)
  :creator (create-top :logic create-top$a :exec create-top$c)
  :exports ((lookup1 :logic lookup1$a :exec ht1i)
            (update1 :logic update1$a :exec update-ht1i)
            (lookup2 :logic lookup2$a :exec ht2i :updater update2)
            (update2 :logic update2$a :exec update-ht2i)))
