/*
 * Copyright (C) 1999 Lars Knoll (knoll@kde.org)
 *           (C) 1999 Antti Koivisto (koivisto@kde.org)
 *           (C) 2005 Allan Sandfeld Jensen (kde@carewolf.com)
 *           (C) 2005, 2006 Samuel Weinig (sam.weinig@gmail.com)
 * Copyright (C) 2005-2025 Apple Inc. All rights reserved.
 * Copyright (C) 2010-2015 Google Inc. All rights reserved.
 * Copyright (C) 2023, 2024 Igalia S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include "RenderLayerModelObject.h"

#include "BlendingKeyframes.h"
#include "ContainerNodeInlines.h"
#include "InspectorInstrumentation.h"
#include "MotionPath.h"
#include "ReferenceFilterOperation.h"
#include "ReferencedSVGResources.h"
#include "RenderDescendantIterator.h"
#include "RenderElementInlines.h"
#include "RenderLayer.h"
#include "RenderLayerBacking.h"
#include "RenderLayerCompositor.h"
#include "RenderLayerScrollableArea.h"
#include "RenderMultiColumnSet.h"
#include "RenderObjectInlines.h"
#include "RenderSVGBlock.h"
#include "RenderSVGModelObject.h"
#include "RenderSVGResourceClipper.h"
#include "RenderSVGResourceFilter.h"
#include "RenderSVGResourceLinearGradient.h"
#include "RenderSVGResourceMarker.h"
#include "RenderSVGResourceMasker.h"
#include "RenderSVGResourceRadialGradient.h"
#include "RenderSVGText.h"
#include "RenderStyleInlines.h"
#include "RenderView.h"
#include "SVGClipPathElement.h"
#include "SVGFilterElement.h"
#include "SVGGraphicsElement.h"
#include "SVGMarkerElement.h"
#include "SVGMaskElement.h"
#include "SVGTextElement.h"
#include "SVGURIReference.h"
#include "Settings.h"
#include "TransformOperationData.h"
#include "TransformState.h"
#include <wtf/MathExtras.h>
#include <wtf/TZoneMallocInlines.h>

namespace WebCore {

WTF_MAKE_TZONE_OR_ISO_ALLOCATED_IMPL(RenderLayerModelObject);

bool RenderLayerModelObject::s_wasFloating = false;
bool RenderLayerModelObject::s_hadLayer = false;
bool RenderLayerModelObject::s_wasTransformed = false;
bool RenderLayerModelObject::s_layerWasSelfPainting = false;

RenderLayerModelObject::RenderLayerModelObject(Type type, Element& element, RenderStyle&& style, OptionSet<TypeFlag> baseTypeFlags, TypeSpecificFlags typeSpecificFlags)
    : RenderElement(type, element, WTFMove(style), baseTypeFlags | TypeFlag::IsLayerModelObject, typeSpecificFlags)
{
    ASSERT(isRenderLayerModelObject());
}

RenderLayerModelObject::RenderLayerModelObject(Type type, Document& document, RenderStyle&& style, OptionSet<TypeFlag> baseTypeFlags, TypeSpecificFlags typeSpecificFlags)
    : RenderElement(type, document, WTFMove(style), baseTypeFlags | TypeFlag::IsLayerModelObject, typeSpecificFlags)
{
    ASSERT(isRenderLayerModelObject());
}

// Do not add any code in below destructor. Add it to willBeDestroyed() instead.
RenderLayerModelObject::~RenderLayerModelObject() = default;

void RenderLayerModelObject::willBeDestroyed()
{
    if (isPositioned()) {
        if (style().hasViewportConstrainedPosition())
            view().frameView().removeViewportConstrainedObject(*this);
    }

    destroyLayer();

    RenderElement::willBeDestroyed();
}

void RenderLayerModelObject::destroyLayer()
{
    setHasLayer(false);
    m_layer = nullptr;
}

void RenderLayerModelObject::createLayer()
{
    ASSERT(!m_layer);
    m_layer = RenderLayer::create(*this);
    setHasLayer(true);
    m_layer->insertOnlyThisLayer();
}

bool RenderLayerModelObject::hasSelfPaintingLayer() const
{
    return m_layer && m_layer->isSelfPaintingLayer();
}

void RenderLayerModelObject::styleWillChange(Style::Difference diff, const RenderStyle& newStyle)
{
    s_wasFloating = isFloating();
    s_hadLayer = hasLayer();
    s_wasTransformed = isTransformed();
    if (s_hadLayer)
        s_layerWasSelfPainting = layer()->isSelfPaintingLayer();

    auto* oldStyle = hasInitializedStyle() ? &style() : nullptr;
    if (diff == Style::DifferenceResult::RepaintLayer && parent() && oldStyle && oldStyle->clip() != newStyle.clip())
        layer()->clearClipRectsIncludingDescendants();
    RenderElement::styleWillChange(diff, newStyle);
}

void RenderLayerModelObject::styleDidChange(Style::Difference diff, const RenderStyle* oldStyle)
{
    updateFromStyle();
    RenderElement::styleDidChange(diff, oldStyle);

    // When an out-of-flow-positioned element changes its display between block and inline-block,
    // then an incremental layout on the element's containing block lays out the element through
    // LayoutPositionedObjects, which skips laying out the element's parent.
    // The element's parent needs to relayout so that it calls
    // RenderBlockFlow::setStaticInlinePositionForChild with the out-of-flow-positioned child, so
    // that when it's laid out, its RenderBox::computeOutOfFlowPositionedLogicalWidth/Height takes into
    // account its new inline/block position rather than its old block/inline position.
    // Position changes and other types of display changes are handled elsewhere.
    if ((oldStyle && isOutOfFlowPositioned() && parent() && (parent() != containingBlock()))
        && (style().position() == oldStyle->position())
        && (style().isOriginalDisplayInlineType() != oldStyle->isOriginalDisplayInlineType())
        && ((style().isOriginalDisplayBlockType()) || (style().isOriginalDisplayInlineType()))
        && ((oldStyle->isOriginalDisplayBlockType()) || (oldStyle->isOriginalDisplayInlineType())))
            parent()->setChildNeedsLayout();

    bool gainedOrLostLayer = false;
    if (requiresLayer()) {
        if (!layer() && layerCreationAllowedForSubtree()) {
            gainedOrLostLayer = true;
            if (s_wasFloating && isFloating())
                setChildNeedsLayout();
            createLayer();
            if (parent() && !needsLayout())
                layer()->setRepaintStatus(RepaintStatus::NeedsFullRepaint);
        }
    } else if (layer() && layer()->parent()) {
        gainedOrLostLayer = true;
        if (oldStyle && oldStyle->hasBlendMode())
            layer()->willRemoveChildWithBlendMode();
        setHasTransformRelatedProperty(false); // All transform-related properties force layers, so we know we don't have one or the object doesn't support them.
        setHasSVGTransform(false); // Same reason as for setHasTransformRelatedProperty().
        setHasReflection(false);

        // Repaint the about to be destroyed self-painting layer when style change also triggers repaint.
        if (layer()->isSelfPaintingLayer() && layer()->repaintStatus() == RepaintStatus::NeedsFullRepaint && layer()->cachedClippedOverflowRect())
            repaintUsingContainer(containerForRepaint().renderer.get(), *(layer()->cachedClippedOverflowRect()));

        layer()->removeOnlyThisLayer(); // calls destroyLayer() which clears m_layer
        if (s_wasFloating && isFloating())
            setChildNeedsLayout();
        if (s_wasTransformed)
            setNeedsLayoutAndPreferredWidthsUpdate();
    }

    if (gainedOrLostLayer)
        InspectorInstrumentation::didAddOrRemoveScrollbars(*this);

    if (layer()) {
        layer()->styleChanged(diff, oldStyle);
        if (s_hadLayer && layer()->isSelfPaintingLayer() != s_layerWasSelfPainting)
            setChildNeedsLayout();
    }

    bool newStyleIsViewportConstrained = style().hasViewportConstrainedPosition();
    bool oldStyleIsViewportConstrained = oldStyle && oldStyle->hasViewportConstrainedPosition();
    if (newStyleIsViewportConstrained != oldStyleIsViewportConstrained) {
        if (newStyleIsViewportConstrained && layer())
            view().frameView().addViewportConstrainedObject(*this);
        else
            view().frameView().removeViewportConstrainedObject(*this);
    }

    if (oldStyle && !oldStyle->scrollPaddingEqual(style())) {
        if (isDocumentElementRenderer()) {
            LocalFrameView& frameView = view().frameView();
            frameView.updateScrollbarSteps();
        } else if (RenderLayer* renderLayer = layer())
            renderLayer->updateScrollbarSteps();
    }

    if (oldStyle && !oldStyle->scrollSnapDataEquivalent(style())) {
        if (auto* scrollSnapBox = enclosingScrollableContainer())
            scrollSnapBox->setNeedsLayout();
    }
}

bool RenderLayerModelObject::applyCachedClipAndScrollPosition(RepaintRects&, const RenderLayerModelObject*, VisibleRectContext) const
{
    return false;
}

bool RenderLayerModelObject::shouldPlaceVerticalScrollbarOnLeft() const
{
// RTL Scrollbars require some system support, and this system support does not exist on certain versions of macOS. iOS uses a separate mechanism.
#if PLATFORM(IOS_FAMILY)
    return false;
#else
    switch (settings().userInterfaceDirectionPolicy()) {
    case UserInterfaceDirectionPolicy::Content:
        return style().shouldPlaceVerticalScrollbarOnLeft();
    case UserInterfaceDirectionPolicy::System:
        return settings().systemLayoutDirection() == TextDirection::RTL;
    }
    ASSERT_NOT_REACHED();
    return style().shouldPlaceVerticalScrollbarOnLeft();
#endif
}

std::optional<LayoutRect> RenderLayerModelObject::cachedLayerClippedOverflowRect() const
{
    return hasLayer() ? layer()->cachedClippedOverflowRect() : std::nullopt;
}

bool RenderLayerModelObject::startAnimation(double timeOffset, const GraphicsLayerAnimation& animation, const BlendingKeyframes& keyframes)
{
    if (!layer() || !layer()->backing())
        return false;
    return layer()->backing()->startAnimation(timeOffset, animation, keyframes);
}

void RenderLayerModelObject::animationPaused(double timeOffset, const BlendingKeyframes& keyframes)
{
    if (!layer() || !layer()->backing())
        return;
    layer()->backing()->animationPaused(timeOffset, keyframes.acceleratedAnimationName());
}

void RenderLayerModelObject::animationFinished(const BlendingKeyframes& keyframes)
{
    if (!layer() || !layer()->backing())
        return;
    layer()->backing()->animationFinished(keyframes.acceleratedAnimationName());
}

void RenderLayerModelObject::transformRelatedPropertyDidChange()
{
    if (!layer() || !layer()->backing())
        return;
    layer()->backing()->transformRelatedPropertyDidChange();
}

void RenderLayerModelObject::suspendAnimations(MonotonicTime time)
{
    if (!layer() || !layer()->backing())
        return;
    layer()->backing()->suspendAnimations(time);
}

TransformationMatrix* RenderLayerModelObject::layerTransform() const
{
    if (hasLayer())
        return layer()->transform();
    return nullptr;
}

void RenderLayerModelObject::updateLayerTransform()
{
    if (auto* box = dynamicDowncast<RenderBox>(this); box && MotionPath::needsUpdateAfterContainingBlockLayout(style().offsetPath())) {
        if (auto* containingBlock = this->containingBlock()) {
            view().frameView().layoutContext().setBoxNeedsTransformUpdateAfterContainerLayout(*box, *containingBlock);
            return;
        }
    }
    // Transform-origin depends on box size, so we need to update the layer transform after layout.
    if (hasLayer())
        layer()->updateTransform();
}

bool RenderLayerModelObject::shouldPaintSVGRenderer(const PaintInfo& paintInfo, const OptionSet<PaintPhase> relevantPaintPhases) const
{
    if (paintInfo.context().paintingDisabled())
        return false;

    if (!relevantPaintPhases.isEmpty() && !relevantPaintPhases.contains(paintInfo.phase))
        return false;

    if (!paintInfo.shouldPaintWithinRoot(*this))
        return false;

    if (style().usedVisibility() == Visibility::Hidden || style().display() == DisplayType::None)
        return false;

    return true;
}

auto RenderLayerModelObject::computeVisibleRectsInSVGContainer(const RepaintRects& rects, const RenderLayerModelObject* container, VisibleRectContext context) const -> std::optional<RepaintRects>
{
    ASSERT(is<RenderSVGModelObject>(this) || is<RenderSVGBlock>(this));
    ASSERT(!style().hasInFlowPosition());
    ASSERT(!view().frameView().layoutContext().isPaintOffsetCacheEnabled());

    if (container == this)
        return rects;

    bool containerIsSkipped;
    auto* localContainer = this->container(container, containerIsSkipped);
    if (!localContainer)
        return rects;

    ASSERT_UNUSED(containerIsSkipped, !containerIsSkipped);

    auto adjustedRects = rects;

    LayoutSize locationOffset;
    if (CheckedPtr modelObject = dynamicDowncast<RenderSVGModelObject>(this))
        locationOffset = modelObject->locationOffsetEquivalent();
    else if (CheckedPtr svgBlock = dynamicDowncast<RenderSVGBlock>(this))
        locationOffset = svgBlock->locationOffset();


    // We are now in our parent container's coordinate space. Apply our transform to obtain a bounding box
    // in the parent's coordinate space that encloses us.
    if (hasLayer() && layer()->transform())
        adjustedRects.transform(*layer()->transform());

    adjustedRects.move(locationOffset);

    if (localContainer->hasNonVisibleOverflow()) {
        bool isEmpty = !downcast<RenderLayerModelObject>(*localContainer).applyCachedClipAndScrollPosition(adjustedRects, container, context);
        if (isEmpty) {
            if (context.options.contains(VisibleRectContext::Option::UseEdgeInclusiveIntersection))
                return std::nullopt;
            return adjustedRects;
        }
    }

    return localContainer->computeVisibleRectsInContainer(adjustedRects, container, context);
}

void RenderLayerModelObject::mapLocalToSVGContainer(const RenderLayerModelObject* ancestorContainer, TransformState& transformState, OptionSet<MapCoordinatesMode> mode, bool* wasFixed) const
{
    ASSERT(is<RenderSVGModelObject>(this) || is<RenderSVGBlock>(this));
    ASSERT(style().position() == PositionType::Static);

    if (ancestorContainer == this)
        return;

    ASSERT(!view().frameView().layoutContext().isPaintOffsetCacheEnabled());

    bool ancestorSkipped;
    auto* container = this->container(ancestorContainer, ancestorSkipped);
    if (!container)
        return;

    ASSERT_UNUSED(ancestorSkipped, !ancestorSkipped);

    // If this box has a transform, it acts as a fixed position container for fixed descendants,
    // and may itself also be fixed position. So propagate 'fixed' up only if this box is fixed position.
    if (isTransformed())
        mode.remove(IsFixed);

    if (wasFixed)
        *wasFixed = mode.contains(IsFixed);

    auto containerOffset = offsetFromContainer(*container, LayoutPoint(transformState.mappedPoint()));

    pushOntoTransformState(transformState, mode, nullptr, container, containerOffset, false);

    mode.remove(ApplyContainerFlip);

    container->mapLocalToContainer(ancestorContainer, transformState, mode, wasFixed);
}

void RenderLayerModelObject::applySVGTransform(TransformationMatrix& transform, const SVGGraphicsElement& graphicsElement, const RenderStyle& style, const FloatRect& boundingBox, const std::optional<AffineTransform>& preApplySVGTransformMatrix, const std::optional<AffineTransform>& postApplySVGTransformMatrix, OptionSet<RenderStyle::TransformOperationOption> options) const
{
    auto svgTransform = graphicsElement.transform().concatenate();
    auto* supplementalTransform = graphicsElement.supplementalTransform(); // SMIL <animateMotion>

    // This check does not use style.hasTransformRelatedProperty() on purpose -- we only want to know if either the 'transform' property, an
    // offset path, or the individual transform operations are set (perspective / transform-style: preserve-3d are not relevant here).
    bool hasCSSTransform = style.hasTransform() || style.hasRotate() || style.hasTranslate() || style.hasScale();
    bool hasSVGTransform = !svgTransform.isIdentity() || preApplySVGTransformMatrix || postApplySVGTransformMatrix || supplementalTransform;

    // Common case: 'viewBox' set on outermost <svg> element -> 'preApplySVGTransformMatrix'
    // passed by RenderSVGViewportContainer::applyTransform(), the anonymous single child
    // of RenderSVGRoot, that wraps all direct children from <svg> as present in DOM. All
    // other transformations are unset (no need to compute transform-origin, etc. in that case).
    if (!hasCSSTransform && !hasSVGTransform)
        return;

    auto affectedByTransformOrigin = [&]() {
        if (preApplySVGTransformMatrix && !preApplySVGTransformMatrix->isIdentityOrTranslation())
            return true;
        if (postApplySVGTransformMatrix && !postApplySVGTransformMatrix->isIdentityOrTranslation())
            return true;
        if (supplementalTransform && !supplementalTransform->isIdentityOrTranslation())
            return true;
        if (hasCSSTransform)
            return style.affectedByTransformOrigin();
        return !svgTransform.isIdentityOrTranslation();
    };

    FloatPoint3D originTranslate;
    if (options.contains(RenderStyle::TransformOperationOption::TransformOrigin) && affectedByTransformOrigin())
        originTranslate = style.computeTransformOrigin(boundingBox);

    style.applyTransformOrigin(transform, originTranslate);

    if (supplementalTransform)
        transform.multiplyAffineTransform(*supplementalTransform);

    if (preApplySVGTransformMatrix)
        transform.multiplyAffineTransform(preApplySVGTransformMatrix.value());

    // CSS transforms take precedence over SVG transforms.
    if (hasCSSTransform)
        style.applyCSSTransform(transform, TransformOperationData(boundingBox, this), options);
    else if (!svgTransform.isIdentity())
        transform.multiplyAffineTransform(svgTransform);

    if (postApplySVGTransformMatrix)
        transform.multiplyAffineTransform(postApplySVGTransformMatrix.value());

    style.unapplyTransformOrigin(transform, originTranslate);
}

void RenderLayerModelObject::updateHasSVGTransformFlags()
{
    ASSERT(document().settings().layerBasedSVGEngineEnabled());

    bool hasSVGTransform = needsHasSVGTransformFlags();
    setHasTransformRelatedProperty(hasSVGTransform || style().hasTransformRelatedProperty());
    setHasSVGTransform(hasSVGTransform);
}

RenderSVGResourceClipper* RenderLayerModelObject::svgClipperResourceFromStyle() const
{
    if (!document().settings().layerBasedSVGEngineEnabled())
        return nullptr;

    return WTF::switchOn(style().clipPath(),
        [&](const Style::ReferencePath& clipPath) -> RenderSVGResourceClipper* {
            if (RefPtr referencedClipPathElement = ReferencedSVGResources::referencedClipPathElement(treeScopeForSVGReferences(), clipPath)) {
                if (auto* referencedClipperRenderer = dynamicDowncast<RenderSVGResourceClipper>(referencedClipPathElement->renderer()))
                    return referencedClipperRenderer;
            }

            if (auto* svgElement = dynamicDowncast<SVGElement>(this->element()))
                document().addPendingSVGResource(clipPath.fragment(), *svgElement);

            return nullptr;

        },
        [&](const auto&) -> RenderSVGResourceClipper* {
            return nullptr;
        }
    );
}

RenderSVGResourceFilter* RenderLayerModelObject::svgFilterResourceFromStyle() const
{
    if (!document().settings().layerBasedSVGEngineEnabled())
        return nullptr;

    const auto& filter = style().filter();
    if (filter.size() != 1)
        return nullptr;

    RefPtr referenceFilterOperation = dynamicDowncast<Style::ReferenceFilterOperation>(filter[0].value);
    if (!referenceFilterOperation)
        return nullptr;

    if (RefPtr referencedFilterElement = ReferencedSVGResources::referencedFilterElement(treeScopeForSVGReferences(), *referenceFilterOperation)) {
        if (auto* referencedFilterRenderer = dynamicDowncast<RenderSVGResourceFilter>(referencedFilterElement->renderer()))
            return referencedFilterRenderer;
    }

    if (auto* svgElement = dynamicDowncast<SVGElement>(this->element()))
        document().addPendingSVGResource(referenceFilterOperation->fragment(), *svgElement);

    return nullptr;
}

RenderSVGResourceMasker* RenderLayerModelObject::svgMaskerResourceFromStyle() const
{
    if (!document().settings().layerBasedSVGEngineEnabled())
        return nullptr;

    RefPtr maskImage = style().maskLayers().usedFirst().image().tryStyleImage();
    if (!maskImage)
        return nullptr;

    auto resourceID = SVGURIReference::fragmentIdentifierFromIRIString(maskImage->url(), protectedDocument());

    if (RefPtr referencedMaskElement = ReferencedSVGResources::referencedMaskElement(treeScopeForSVGReferences(), *maskImage)) {
        if (auto* referencedMaskerRenderer = dynamicDowncast<RenderSVGResourceMasker>(referencedMaskElement->renderer()))
            return referencedMaskerRenderer;
    }

    if (auto* element = this->element())
        document().addPendingSVGResource(resourceID, downcast<SVGElement>(*element));

    return nullptr;
}

RenderSVGResourceMarker* RenderLayerModelObject::svgMarkerStartResourceFromStyle() const
{
    return svgMarkerResourceFromStyle(style().markerStart());
}

RenderSVGResourceMarker* RenderLayerModelObject::svgMarkerMidResourceFromStyle() const
{
    return svgMarkerResourceFromStyle(style().markerMid());
}

RenderSVGResourceMarker* RenderLayerModelObject::svgMarkerEndResourceFromStyle() const
{
    return svgMarkerResourceFromStyle(style().markerEnd());
}

RenderSVGResourceMarker* RenderLayerModelObject::svgMarkerResourceFromStyle(const Style::SVGMarkerResource& markerResource) const
{
    if (!document().settings().layerBasedSVGEngineEnabled())
        return nullptr;

    auto markerResourceURL = markerResource.tryURL();
    if (!markerResourceURL)
        return nullptr;

    if (RefPtr referencedMarkerElement = ReferencedSVGResources::referencedMarkerElement(treeScopeForSVGReferences(), *markerResourceURL)) {
        if (auto* referencedMarkerRenderer = dynamicDowncast<RenderSVGResourceMarker>(referencedMarkerElement->renderer()))
            return referencedMarkerRenderer;
    }

    if (auto* element = dynamicDowncast<SVGElement>(this->element()))
        document().addPendingSVGResource(AtomString(markerResourceURL->resolved.string()), *element);

    return nullptr;
}

RenderSVGResourcePaintServer* RenderLayerModelObject::svgFillPaintServerResourceFromStyle(const RenderStyle& style) const
{
    if (!document().settings().layerBasedSVGEngineEnabled())
        return nullptr;

    auto fillURL = style.fill().tryAnyURL();
    if (!fillURL)
        return nullptr;

    if (RefPtr referencedElement = ReferencedSVGResources::referencedPaintServerElement(treeScopeForSVGReferences(), *fillURL)) {
        if (auto* referencedPaintServerRenderer = dynamicDowncast<RenderSVGResourcePaintServer>(referencedElement->renderer()))
            return referencedPaintServerRenderer;
    }

    if (auto* element = this->element())
        document().addPendingSVGResource(AtomString(fillURL->resolved.string()), downcast<SVGElement>(*element));

    return nullptr;
}

RenderSVGResourcePaintServer* RenderLayerModelObject::svgStrokePaintServerResourceFromStyle(const RenderStyle& style) const
{
    if (!document().settings().layerBasedSVGEngineEnabled())
        return nullptr;

    auto strokeURL = style.stroke().tryAnyURL();
    if (!strokeURL)
        return nullptr;

    if (RefPtr referencedElement = ReferencedSVGResources::referencedPaintServerElement(treeScopeForSVGReferences(), *strokeURL)) {
        if (auto* referencedPaintServerRenderer = dynamicDowncast<RenderSVGResourcePaintServer>(referencedElement->renderer()))
            return referencedPaintServerRenderer;
    }

    if (auto* element = this->element())
        document().addPendingSVGResource(AtomString(strokeURL->resolved.string()), downcast<SVGElement>(*element));

    return nullptr;
}

LegacyRenderSVGResourceClipper* RenderLayerModelObject::legacySVGClipperResourceFromStyle() const
{
    return WTF::switchOn(style().clipPath(),
        [&](const Style::ReferencePath& clipPath) -> LegacyRenderSVGResourceClipper* {
            return ReferencedSVGResources::referencedClipperRenderer(treeScopeForSVGReferences(), clipPath);
        },
        [&](const auto&) -> LegacyRenderSVGResourceClipper* {
            return nullptr;
        }
    );
}

bool RenderLayerModelObject::pointInSVGClippingArea(const FloatPoint& point) const
{
    auto clipPathReferenceBox = [&](CSSBoxType boxType) -> FloatRect {
        FloatRect referenceBox;
        switch (boxType) {
        case CSSBoxType::BorderBox:
        case CSSBoxType::MarginBox:
        case CSSBoxType::StrokeBox:
        case CSSBoxType::BoxMissing:
            // FIXME: strokeBoundingBox() takes dasharray into account but shouldn't.
            referenceBox = strokeBoundingBox();
            break;
        case CSSBoxType::ViewBox:
            if (element()) {
                // FIXME: [LBSE] This should not need to use SVGLengthContext, RenderSVGRoot holds that information.
                auto viewportSize = SVGLengthContext(downcast<SVGElement>(element())).viewportSize();
                if (viewportSize)
                    referenceBox.setSize(*viewportSize);
                break;
            }
            [[fallthrough]];
        case CSSBoxType::ContentBox:
        case CSSBoxType::FillBox:
        case CSSBoxType::PaddingBox:
            referenceBox = objectBoundingBox();
            break;
        }
        return referenceBox;
    };

    return WTF::switchOn(style().clipPath(),
        [&](const Style::BasicShapePath& clipPath) {
            auto referenceBox = clipPathReferenceBox(clipPath.referenceBox());
            if (!referenceBox.contains(point))
                return false;
            return Style::path(clipPath.shape(), referenceBox).contains(point, Style::windRule(clipPath.shape()));
        },
        [&](const Style::BoxPath& clipPath) {
            auto referenceBox = clipPathReferenceBox(clipPath.referenceBox());
            if (!referenceBox.contains(point))
                return false;
            return FloatRoundedRect { referenceBox }.path().contains(point);
        },
        [&](const auto&) {
            if (auto* referencedClipperRenderer = svgClipperResourceFromStyle())
                return referencedClipperRenderer->hitTestClipContent(objectBoundingBox(), LayoutPoint(point));
            return true;
        }
    );
}

CheckedPtr<RenderLayer> RenderLayerModelObject::checkedLayer() const
{
    return m_layer.get();
}

void RenderLayerModelObject::repaintOrRelayoutAfterSVGTransformChange()
{
    ASSERT(document().settings().layerBasedSVGEngineEnabled());

    updateHasSVGTransformFlags();

    // LBSE shares the text rendering code with the legacy SVG engine, largely unmodified.
    // At present text layout depends on transformations ('screen font scaling factor' is used to
    // determine which font to use for layout / painting). Therefore if the x/y scaling factors
    // of the transformation matrix changes due to the transform update, we have to recompute the text metrics
    // of all RenderSVGText descendants of the renderer in the ancestor chain, that will receive the transform
    // update.
    //
    // There is no intrinsic reason for that, besides historical ones. If we decouple
    // the 'font size screen scaling factor' from layout and only use it during painting
    // we can optimize transformations for text, simply by avoid the need for layout.
    auto previousTransform = layerTransform() ? layerTransform()->toAffineTransform() : identity;
    updateLayerTransform();

    auto currentTransform = layerTransform() ? layerTransform()->toAffineTransform() : identity;

    // We have to force a stacking context if we did not have a transform before. Normally
    // RenderLayer::styleChanged does this for us but repaintOrRelayoutAfterSVGTransformChange
    // does not end up calling it.
    if (previousTransform.isIdentity() && !currentTransform.isIdentity()) {
        if (hasLayer())
            layer()->forceStackingContextIfNeeded();
    }

    auto determineIfLayerTransformChangeModifiesScale = [&]() -> bool {
        if (previousTransform == currentTransform)
            return false;

        // Only if the effective x/y scale changes, a re-layout is necessary, due to changed on-screen scaling factors.
        // The next RenderSVGText layout will see a different 'screen font scaling factor', different text metrics etc.
        if (!WTF::areEssentiallyEqual(previousTransform.xScale(), currentTransform.xScale()))
            return true;

        if (!WTF::areEssentiallyEqual(previousTransform.yScale(), currentTransform.yScale()))
            return true;

        return false;
    }();

    if (determineIfLayerTransformChangeModifiesScale) {
        if (auto* textAffectedByTransformChange = dynamicDowncast<RenderSVGText>(this)) {
            // Mark text metrics for update, and only trigger a relayout and not an explicit repaint.
            textAffectedByTransformChange->setNeedsTextMetricsUpdate();
            textAffectedByTransformChange->textElement().updateSVGRendererForElementChange();
            return;
        }

        // Recursively mark text metrics for update in all descendant RenderSVGText objects.
        bool markedAny = false;
        for (auto& textDescendantAffectedByTransformChange : descendantsOfType<RenderSVGText>(*this)) {
            textDescendantAffectedByTransformChange.setNeedsTextMetricsUpdate();
            textDescendantAffectedByTransformChange.textElement().updateSVGRendererForElementChange();
            if (!markedAny)
                markedAny = true;
        }

        // If we marked a text descendant for relayout, we are expecting a relayout ourselves, so no reason for an explicit repaint().
        if (markedAny)
            return;
    }

    // Instead of performing a full-fledged layout (issuing repaints), just recompute the layer transform, and repaint.
    // In LBSE transformations do not affect the layout (except for text, where it still does!) -- SVG follows closely the CSS/HTML route, to avoid costly layouts.
    repaintRendererOrClientsOfReferencedSVGResources();
}

void RenderLayerModelObject::paintSVGClippingMask(PaintInfo& paintInfo, const FloatRect& objectBoundingBox) const
{
    ASSERT(paintInfo.phase == PaintPhase::ClippingMask);
    auto& context = paintInfo.context();
    if (!paintInfo.shouldPaintWithinRoot(*this) || style().usedVisibility() != Visibility::Visible || context.paintingDisabled())
        return;

    ASSERT(document().settings().layerBasedSVGEngineEnabled());
    if (auto* referencedClipperRenderer = svgClipperResourceFromStyle())
        referencedClipperRenderer->applyMaskClipping(paintInfo, *this, objectBoundingBox);
}

void RenderLayerModelObject::paintSVGMask(PaintInfo& paintInfo, const LayoutPoint& adjustedPaintOffset) const
{
    ASSERT(paintInfo.phase == PaintPhase::Mask);
    auto& context = paintInfo.context();
    if (!paintInfo.shouldPaintWithinRoot(*this) || context.paintingDisabled())
        return;

    ASSERT(isSVGLayerAwareRenderer());
    if (auto* referencedMaskerRenderer = svgMaskerResourceFromStyle())
        referencedMaskerRenderer->applyMask(paintInfo, *this, adjustedPaintOffset);
}

bool rendererNeedsPixelSnapping(const RenderLayerModelObject& renderer)
{
    if (renderer.document().settings().layerBasedSVGEngineEnabled() && renderer.isSVGLayerAwareRenderer() && !renderer.isRenderSVGRoot())
        return false;
    return true;
}

FloatRect snapRectToDevicePixelsIfNeeded(const LayoutRect& rect, const RenderLayerModelObject& renderer)
{
    if (!rendererNeedsPixelSnapping(renderer))
        return rect;
    return snapRectToDevicePixels(rect, renderer.document().deviceScaleFactor());
}

FloatRect snapRectToDevicePixelsIfNeeded(const FloatRect& rect, const RenderLayerModelObject& renderer)
{
    if (!rendererNeedsPixelSnapping(renderer))
        return rect;
    return snapRectToDevicePixels(LayoutRect { rect }, renderer.document().deviceScaleFactor());
}

} // namespace WebCore

