///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2018 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//! @examplefile zalesak_dg_adapt.cc The Zalesak slotted disk benchmark -- adaptive post-treatment
// usage exemple:
//   cat leveque-o2-full-100-P2d-6000-e-0.field leveque-o2-full-100-P2d-6000-e-3000.field > in.branch
//   ./zalesak_dg_adapt < in.branch > adapt.branch
// TODO: test zalesak L1 & mass error ; cf ctrl paras (field_error,subdivide)
# include "rheolef_seq.h"
using namespace rheolef;
using namespace std;
#include "zalesak.h"
Float heaviside (const Float& x) { return (x <= 0) ? 0 : 1; }
Float criterion (const Float& x, const Float& y) { return (x+y)/2; }
int main(int argc, char**argv) {
  environment rheolef (argc, argv);
  size_t subdivide   = (argc > 1) ? atoi(argv[1]) : 3;
  Float  field_error = (argc > 2) ? atof(argv[2]) : 1e-3;
  Float  geo_error   = 1e-7;
  bool   do_verbose  = true;
  bool   do_clean    = true;
  string tmp = do_clean ? get_tmpdir() + "/" : "";
  string cleanlist;
  check_macro (communicator().size() == 1, "zalesak_dg_adapt: command may be used as mono-process only");
  // ----------------------------------------
  // 1. put criterion and phi0, phi in a file
  // ----------------------------------------
  Float t0, tf;
  field_basic<Float,sequential> phi0_h, phi_h;
  branch_basic<Float,sequential> ievent ("t","phi");
  din >> ievent (t0, phi0_h);
  while (din >> ievent (tf, phi_h));
  const space& Xh = phi_h.get_space();
  field_basic<Float,sequential> ch = interpolate (Xh, compose(criterion, compose (heaviside, phi0_h), compose (heaviside, phi_h)));
  branch_basic<Float,sequential> oevent ("t", "c", "phi0", "phi");
  dout << vtk << setbasename(tmp+"criterion")
       << oevent (t0,   ch,  phi0_h, phi_h); 
  cleanlist += tmp+"criterion-0.vtk";
  // ----------------------------------------
  // 2. run pvbatch
  // ----------------------------------------
  string py_name = tmp+"adapt.py";
  odiststream py (py_name, io::nogz);
  cleanlist += " "+py_name;
  bool view_2d = phi_h.get_geo().map_dimension() < 3;
  py << "#!/usr/bin/env pvbatch --script=" << endl
     << "# This is a paraview script automatically generated by rheolef." << endl
     << endl
     << "from paraview.simple  import *" << endl
     << "from paraview_rheolef import *  # load rheolef specific functions" << endl
     << endl
     << "opt = {                     \\" << endl
     << "    'mark'        : 'c',    \\" << endl
     << "    'view_2d'     : " << view_2d << ",      \\" << endl
     << "    'geo_error'   : " << geo_error << ",   \\" << endl
     << "    'field_error' : " << field_error << ",   \\" << endl
     << "    'subdivide'   : " << subdivide << "       \\" << endl
     << "    }" << endl
     << endl
     << "adapt_Pk_iso_P1 (paraview, \""<<tmp<<"criterion-0.vtk\", \""<<tmp<<"adapt.vtk\", opt)" << endl
     << endl
    ;
  string prog = "pvbatch --force-offscreen-rendering ";
  string command = "DISPLAY=:0.0 LANG=C PYTHONPATH=" + string(_RHEOLEF_PKGDATADIR) + " " + prog + py_name;
  if (do_verbose) derr << "! " << command << endl;
  check_macro (dis_system (command) == 0, "unix command failed");
  cleanlist += " "+tmp+"adapt.vtk";
  // ----------------------------------------------------
  // 4. load phi0 & phi from adapt.vtk & put it on stdout
  // ----------------------------------------------------
  Float t;
  field_basic<Float,sequential> c_ha, phi0_ha, phi_ha;
  idiststream in_vtk (tmp+"adapt.vtk");
  in_vtk.is() >> vtk;
  iorheo::setbasename(in_vtk.is(),"adapt");
  branch_basic<Float,sequential> ia_event ("t", "c", "phi0", "phi");
  in_vtk >> ia_event (t, c_ha, phi0_ha, phi_ha);
  c_ha.get_geo().save();
  branch oa_event ("t", "phi");
  dout << rheo 
       << oa_event (t0, phi0_ha)
       << oa_event (tf, phi_ha); 
  // ----------------------------------------------------
  // 5. clean tmp & statistics
  // ----------------------------------------------------
  if (do_clean) {
    command = "rm -f " + cleanlist;
    if (do_verbose) derr << "! " << command << endl;
    check_macro (dis_system (command) == 0, "unix command failed");
  }
  if (do_verbose) {
    cerr << "zalesak_dg_adapt: mesh size " << Xh.get_geo().size()
         << " -> "          << phi_ha.get_geo().size() << endl;
  }
}
