// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package generic

import (
	"context"

	"code.forgejo.org/f3/gof3/v3/kind"
	"code.forgejo.org/f3/gof3/v3/logger"

	"github.com/google/go-cmp/cmp"
)

type TreeDriverInterface interface {
	logger.Interface

	GetTree() TreeInterface
	SetTree(TreeInterface)

	GetPageSize() int
	SetPageSize(int)

	AllocateID() bool

	Init()

	Diff(a, b NodeDriverInterface) string

	Factory(ctx context.Context, kind kind.Kind) NodeDriverInterface
}

var DefaultPageSize = int(25)

type NullTreeDriver struct {
	logger.Logger

	tree TreeInterface

	pageSize int
}

func (o *NullTreeDriver) Init() {
	o.pageSize = DefaultPageSize
}

func NewNullTreeDriver() TreeDriverInterface {
	d := &NullTreeDriver{}
	d.Init()
	return d
}

func (o *NullTreeDriver) SetTree(tree TreeInterface) {
	o.tree = tree
	if tree != nil {
		o.SetLogger(tree)
	}
}
func (o *NullTreeDriver) GetTree() TreeInterface { return o.tree }

func (o *NullTreeDriver) GetPageSize() int         { return o.pageSize }
func (o *NullTreeDriver) SetPageSize(pageSize int) { o.pageSize = pageSize }

func (o *NullTreeDriver) AllocateID() bool { return true }

func (o *NullTreeDriver) Diff(a, b NodeDriverInterface) string {
	aFormat := a.ToFormat()
	bFormat := b.ToFormat()

	return cmp.Diff(aFormat, bFormat)
}

func (o *NullTreeDriver) Factory(ctx context.Context, kind kind.Kind) NodeDriverInterface {
	d := NewNullDriver()
	d.SetTreeDriver(o)
	return d
}
