/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008-2018 Red Hat and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.byteman.contrib.bmunit;

import java.lang.reflect.Method;

/**
 * Class bundling sone utility ethods used by both JUnit and TestNG runner classes
 */
public class BMRunnerUtil {

    public static String normaliseLoadDirectory(BMScript annotation) {
        String loadDirectory = annotation.dir();
        if (loadDirectory != null && loadDirectory.length() > 0) {
            return loadDirectory;
        }

        return null;
    }

    /**
     * construct the text of a rule script from a set of BMRule annotations
     *
     * @param bmRules annotations describing the rules to be installed
     * @return the equivalent rule text
     */
    public static String constructScriptText(BMRule[] bmRules) {
        StringBuilder builder = new StringBuilder();
        builder.append("# BMUnit autogenerated script ");
        for (BMRule bmRule : bmRules) {
            builder.append("\nRULE ");
            builder.append(bmRule.name());
            if (bmRule.isInterface()) {
                builder.append("\nINTERFACE ");
            } else {
                builder.append("\nCLASS ");
            }
            if (bmRule.isOverriding()) {
                builder.append("^");
            }
            builder.append(bmRule.targetClass());
            builder.append("\nMETHOD ");
            builder.append(bmRule.targetMethod());
            String location = bmRule.targetLocation();
            if (location != null && location.length() > 0) {
                builder.append('\n');
                // if we have AT XXX or AFTER XXX then use it
                // otherwise prepend AT
                if (!location.startsWith("AFTER ") && !location.startsWith("AT ") ) {
                    builder.append("AT ");
                }
                builder.append(location);
            }
            String helper = bmRule.helper();
            if (helper != null && helper.length() > 0) {
                builder.append("\nHELPER ");
                builder.append(helper);
            }
            String binding = bmRule.binding();
            if (binding != null && binding.length() > 0) {
                builder.append("\nBIND " + binding);
            }
            builder.append("\nIF ");
            builder.append(bmRule.condition());
            builder.append("\nDO ");
            builder.append(bmRule.action());
            builder.append("\nENDRULE\n");
        }
        return builder.toString();
    }

    /**
     * method which computes the name of the BMRules file for a method test if it is not supplied in the
     * method annotation
     *
     * @param name   the value supplied in the annotation or "" if it has been defaulted
     * @param method the Framework method annotated with an @BMRules annotation
     * @return by default this returns the annotation value or the the bare method name if the annotation
     *         value is null or empty
     */
    public static String computeBMScriptName(String name, Method method) {
        // if the annotation has a real name  then use  it
        if (name != null && name.length() > 0) {
            return name;
        }
        // use the method name

        return method.getName();
    }

    /**
     * method which computes the name of the BMRules file for a test class if it is not supplied in the
     * class annotation
     *
     * @param name      the value supplied in the annotation or "" if it has been defaulted
     * @return by default this returns the annotation value or null if the annotation value is null or empty.
     */
    public static String computeBMScriptName(String name) {
        if (name != null && name.length() > 0) {
            return name;
        }

        return null;
    }
}
