/* 
 * lex_auxiliary.c
 *
 * Variables to maintain lexer state (apart from the actual DFA that is!)
 * and functions to manipulate that state and the lexing buffer.
 *
 * Functions used elsewhere (i.e. called from the lexer):
 *   newline ()             [called from almost every lexer class]
 *   newline_in_string ()   [called from LEX_STRING class]
 *   set_entry_type ()      [called from START class]
 *   open_entry ()          [called from START class]
 *   close_entry ()         [called from LEX_VALUE class]
 *   start_string ()        [called from LEX_VALUE class]
 *   end_string ()          [called from LEX_STRING class]
 *   open_brace ()          [called from LEX_STRING class]
 *   close_brace ()         [called from LEX_STRING class]
 *   quote_in_string ()     [called from LEX_STRING class]
 *
 * GPW 1996/07/25-28
 */


#include <stdlib.h>
#include <stdarg.h>
#include <assert.h>

#include "lex_auxiliary.h"
#include "stdpccts.h"
#include "error.h"
#include "prototypes.h"

#define DEBUG 1


/* Global data: */

static int     TexDepth = 0;          /* depth of brace-nesting */
static char    EntryOpener = '\0';    /* '(' or '{' */
static char    StringOpener = '\0';   /* '{' or '"' */
static char   *EntryType = NULL;      /* eg. `string', `article', `inbook', ... */
entry_metatype_t
               EntryMetatype;


/*
 * Utility functions called from the lexer (or each other)
 */



/* This one is just for debugging */
void lex_info (void)
{
   printf ("LA(1) = \"%s\" token %d, %s\n", LATEXT(1), LA(1), zztokens[LA(1)]);
#ifdef LL_K
   printf ("LA(2) = \"%s\" token %d, %s\n", LATEXT(2), LA(2), zztokens[LA(2)]);
#endif
}


void zzcr_attr (Attrib *a, int tok, char *txt)
{
   a->text = strdup (txt);
   a->token = tok;
   a->line = zzline;
   a->offset = zzbegcol;
#if DEBUG > 1
   dprintf ("zzcr_attr: input txt = %p (%s)\n", txt, txt);
   dprintf ("           dupe txt  = %p (%s)\n", a->text, a->text);
#endif
}


/* 
 * newline ()
 * 
 * Does everything needed to handle newline outside of a quoted string:
 * increments line counter, zeros end-column counter, and skips the newline.
 */
void newline (void)
{
   zzline++;
   zzskip();
}


/* 
 * newline_in_string ()
 * 
 * Does everything needed to handle newline in a quoted string:
 * increments line counter, zeros end-column counter, replaces newline
 * with a space, and continues slurping characters into the same buffer.
 */
void newline_in_string (void)
{
   zzline++;
   zzreplchar (' ');
   zzmore ();
}



/*
 * set_entry_type ()
 *
 * stores the keyword used as the "entry type" when starting an entry.  Used
 * by open_entry to determine the next lex class to enter -- if it's a
 * "string" entry, we will go straight to LEX_FIELD, otherwise we go
 * to LEX_KEY.  Could also be used for structural validation.
 */
void set_entry_type (char *entry_type)
{
   if (EntryType != NULL) free (EntryType);
   EntryType = strdup (entry_type);
}


/* 
 * open_entry ()
 * 
 * called when the '{' or '(' that starts an entry is seen.  Just
 * records which character was actually used so we can check that the
 * entry close matches it.
 */
void open_entry (char start_char)
{
   EntryOpener = start_char;
   if (strcasecmp (EntryType, "string") == 0)
   {
      zzmode (LEX_FIELD);
      EntryMetatype = ETYPE_MACRODEF;
   }

   /* Hmmm, need to do something about @preamble and @comment here... */

   else
   {
      zzmode (LEX_KEY);
      EntryMetatype = ETYPE_STRUCTURED;
   }
}


/* 
 * close_entry ()
 * 
 * called when the '}' or ')' that starts an entry is seen.  Checks that
 * the character used goes with the opening char stored by open_entry().
 * If they don't match, prints an error and continues as though they 
 * had matched.
 */        
void close_entry (char end_char)
{
   char start_char = EntryOpener;

   if ((end_char == '}' && start_char == '{') ||
       (end_char == ')' && start_char == '('))
   {
      zzmode (START);
   }
   else
   {
      lexical_error ("token mismatch: entry started with %c, "
                     "but ended with %c",
                     start_char, end_char);
      zzmode (START);
   }
}


/*
 * start_string ()
 *
 * Called when we see a '{' or '"' in the field data.  Records which
 * quote character was used, and increments the depth counter if it was
 * a '{'.  Switches to LEX_STRING mode, and tells the DFA to continue
 * slurping characters into the same buffer.
 */
void start_string (char start_char)
{
   StringOpener = start_char;
   if (start_char == '{')
      open_brace ();

   zzmore ();
   zzmode (LEX_STRING);
}


/*
 * end_string ()
 *
 * Called when we see either a '"' or '}' (if it brings us down to depth 
 * 0) in a quoted string.  Just makes sure that braces are balanced, and
 * then goes back to the LEX_VALUE mode.
 */
void end_string (char end_char)
{
   /*
    * If we're at non-zero TexDepth, that probably means mismatched braces
    * somewhere -- complain about it and reset TexDepth to minimize future
    * confusion.
    */

   if (TexDepth > 0)
   {
      lexical_error ("token mismatch: too many {'s");
      TexDepth = 0;
   }

   zzmode (LEX_FIELD);
}


/*
 * open_brace ()
 * 
 * Called when we see a '{', either to start a string (in which case 
 * it's called from start_string()) or inside a string (called directly
 * from the lexer).
 */
void open_brace (void)
{
   TexDepth++;
   zzmore ();
}


/*
 * close_brace ()
 *
 * Called when we see a '}' inside a string.  Decrements the depth counter
 * and checks to see if we are down to depth 0, in which case the string is
 * ended and the current lookahead token is set to STRING.  Otherwise,
 * just tells the DFA to keep slurping characters into the buffer.
 */
void close_brace (void)
{
   TexDepth--;
   if (StringOpener == '{' && TexDepth == 0)
   {
      NLA = STRING;
      end_string ('}');
   }

   else if (TexDepth < 0)
   {
      lexical_error ("token mismatch: too many }'s");
      TexDepth = 0;
      zzmore ();
   }

   else
   {
      zzmore ();
   }
}


void quote_in_string (void)
{
   if (StringOpener == '"')
   {
      end_string ('"');
   }
   else
   {
      lexical_warning ("found \" in string -- you probably mean `` or ''");
      zzmore ();
   }
}
