﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/location/LocationServiceRequest.h>
#include <aws/location/LocationService_EXPORTS.h>
#include <aws/location/model/TrackingFilterGeometry.h>

#include <utility>

namespace Aws {
namespace LocationService {
namespace Model {

/**
 */
class ListDevicePositionsRequest : public LocationServiceRequest {
 public:
  AWS_LOCATIONSERVICE_API ListDevicePositionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListDevicePositions"; }

  AWS_LOCATIONSERVICE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The tracker resource containing the requested devices.</p>
   */
  inline const Aws::String& GetTrackerName() const { return m_trackerName; }
  inline bool TrackerNameHasBeenSet() const { return m_trackerNameHasBeenSet; }
  template <typename TrackerNameT = Aws::String>
  void SetTrackerName(TrackerNameT&& value) {
    m_trackerNameHasBeenSet = true;
    m_trackerName = std::forward<TrackerNameT>(value);
  }
  template <typename TrackerNameT = Aws::String>
  ListDevicePositionsRequest& WithTrackerName(TrackerNameT&& value) {
    SetTrackerName(std::forward<TrackerNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional limit for the number of entries returned in a single call.</p>
   * <p>Default value: <code>100</code> </p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListDevicePositionsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pagination token specifying which page of results to return in the
   * response. If no token is provided, the default page is the first page.</p>
   * <p>Default value: <code>null</code> </p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListDevicePositionsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The geometry used to filter device positions.</p>
   */
  inline const TrackingFilterGeometry& GetFilterGeometry() const { return m_filterGeometry; }
  inline bool FilterGeometryHasBeenSet() const { return m_filterGeometryHasBeenSet; }
  template <typename FilterGeometryT = TrackingFilterGeometry>
  void SetFilterGeometry(FilterGeometryT&& value) {
    m_filterGeometryHasBeenSet = true;
    m_filterGeometry = std::forward<FilterGeometryT>(value);
  }
  template <typename FilterGeometryT = TrackingFilterGeometry>
  ListDevicePositionsRequest& WithFilterGeometry(FilterGeometryT&& value) {
    SetFilterGeometry(std::forward<FilterGeometryT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_trackerName;

  int m_maxResults{0};

  Aws::String m_nextToken;

  TrackingFilterGeometry m_filterGeometry;
  bool m_trackerNameHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_filterGeometryHasBeenSet = false;
};

}  // namespace Model
}  // namespace LocationService
}  // namespace Aws
