/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sedona.common.geometrySerde;

import org.apache.sedona.common.geometrySerde.CoordinateType;
import org.apache.sedona.common.geometrySerde.GeometryBuffer;
import org.apache.sedona.common.geometrySerde.GeometryBufferFactory;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.CoordinateSequence;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.PrecisionModel;

public class GeometrySerializer {
    private static final Coordinate NULL_COORDINATE = new Coordinate(Double.NaN, Double.NaN);
    private static final PrecisionModel PRECISION_MODEL = new PrecisionModel();

    public static byte[] serialize(Geometry geometry) {
        GeometryBuffer buffer;
        if (geometry instanceof Point) {
            buffer = GeometrySerializer.serializePoint((Point)geometry);
        } else if (geometry instanceof MultiPoint) {
            buffer = GeometrySerializer.serializeMultiPoint((MultiPoint)geometry);
        } else if (geometry instanceof LineString) {
            buffer = GeometrySerializer.serializeLineString((LineString)geometry);
        } else if (geometry instanceof MultiLineString) {
            buffer = GeometrySerializer.serializeMultiLineString((MultiLineString)geometry);
        } else if (geometry instanceof Polygon) {
            buffer = GeometrySerializer.serializePolygon((Polygon)geometry);
        } else if (geometry instanceof MultiPolygon) {
            buffer = GeometrySerializer.serializeMultiPolygon((MultiPolygon)geometry);
        } else if (geometry instanceof GeometryCollection) {
            buffer = GeometrySerializer.serializeGeometryCollection((GeometryCollection)geometry);
        } else {
            throw new UnsupportedOperationException("Geometry type is not supported: " + geometry.getClass().getSimpleName());
        }
        return buffer.toByteArray();
    }

    public static Geometry deserialize(byte[] bytes) {
        GeometryBuffer buffer = GeometryBufferFactory.wrap(bytes);
        return GeometrySerializer.deserialize(buffer);
    }

    public static Geometry deserialize(GeometryBuffer buffer) {
        return GeometrySerializer.deserialize(buffer, null);
    }

    public static Geometry deserialize(GeometryBuffer buffer, GeometryFactory factory) {
        GeometrySerializer.checkBufferSize(buffer, 8);
        int preambleByte = buffer.getByte(0) & 0xFF;
        int wkbType = preambleByte >> 4;
        CoordinateType coordType = CoordinateType.valueOf((preambleByte & 0xF) >> 1);
        boolean hasSrid = (preambleByte & 1) != 0;
        buffer.setCoordinateType(coordType);
        int srid = 0;
        if (hasSrid) {
            int srid2 = (buffer.getByte(1) & 0xFF) << 16;
            int srid1 = (buffer.getByte(2) & 0xFF) << 8;
            int srid0 = buffer.getByte(3) & 0xFF;
            srid = srid2 | srid1 | srid0;
        }
        if (factory == null) {
            factory = GeometrySerializer.createGeometryFactory(srid);
        }
        return GeometrySerializer.deserialize(buffer, wkbType, factory);
    }

    private static Geometry deserialize(GeometryBuffer buffer, int wkbType, GeometryFactory factory) {
        switch (wkbType) {
            case 1: {
                return GeometrySerializer.deserializePoint(buffer, factory);
            }
            case 4: {
                return GeometrySerializer.deserializeMultiPoint(buffer, factory);
            }
            case 2: {
                return GeometrySerializer.deserializeLineString(buffer, factory);
            }
            case 5: {
                return GeometrySerializer.deserializeMultiLineString(buffer, factory);
            }
            case 3: {
                return GeometrySerializer.deserializePolygon(buffer, factory);
            }
            case 6: {
                return GeometrySerializer.deserializeMultiPolygon(buffer, factory);
            }
            case 7: {
                return GeometrySerializer.deserializeGeometryCollection(buffer, factory);
            }
        }
        throw new IllegalArgumentException("Cannot deserialize buffer containing unknown geometry type ID: " + wkbType);
    }

    private static GeometryBuffer serializePoint(Point point) {
        Coordinate coordinate = point.getCoordinate();
        if (coordinate == null) {
            return GeometrySerializer.createGeometryBuffer(1, CoordinateType.XY, point.getSRID(), 8, 0);
        }
        CoordinateType coordType = GeometrySerializer.getCoordinateType(coordinate);
        int bufferSize = 8 + coordType.bytes;
        GeometryBuffer buffer = GeometrySerializer.createGeometryBuffer(1, coordType, point.getSRID(), bufferSize, 1);
        buffer.putCoordinate(8, coordinate);
        return buffer;
    }

    private static Point deserializePoint(GeometryBuffer buffer, GeometryFactory factory) {
        Point point;
        CoordinateType coordType = buffer.getCoordinateType();
        int numCoordinates = GeometrySerializer.getBoundedInt(buffer, 4);
        if (numCoordinates == 0) {
            point = factory.createPoint();
            buffer.mark(8);
        } else {
            int bufferSize = 8 + coordType.bytes;
            GeometrySerializer.checkBufferSize(buffer, bufferSize);
            CoordinateSequence coordinates = buffer.getCoordinate(8);
            point = factory.createPoint(coordinates);
            buffer.mark(bufferSize);
        }
        return point;
    }

    private static GeometryBuffer serializeMultiPoint(MultiPoint multiPoint) {
        int numPoints = multiPoint.getNumGeometries();
        if (numPoints == 0) {
            return GeometrySerializer.createGeometryBuffer(4, CoordinateType.XY, multiPoint.getSRID(), 8, 0);
        }
        CoordinateType coordType = GeometrySerializer.getCoordinateType(multiPoint);
        int bufferSize = 8 + numPoints * coordType.bytes;
        GeometryBuffer buffer = GeometrySerializer.createGeometryBuffer(4, coordType, multiPoint.getSRID(), bufferSize, numPoints);
        for (int k = 0; k < numPoints; ++k) {
            Point point = (Point)multiPoint.getGeometryN(k);
            Coordinate coordinate = point.getCoordinate();
            int coordinateOffset = 8 + k * coordType.bytes;
            if (coordinate == null) {
                buffer.putCoordinate(coordinateOffset, NULL_COORDINATE);
                continue;
            }
            buffer.putCoordinate(coordinateOffset, coordinate);
        }
        return buffer;
    }

    private static MultiPoint deserializeMultiPoint(GeometryBuffer buffer, GeometryFactory factory) {
        CoordinateType coordType = buffer.getCoordinateType();
        int numPoints = GeometrySerializer.getBoundedInt(buffer, 4);
        int bufferSize = 8 + numPoints * coordType.bytes;
        GeometrySerializer.checkBufferSize(buffer, bufferSize);
        Point[] points = new Point[numPoints];
        for (int i = 0; i < numPoints; ++i) {
            CoordinateSequence coordinates = buffer.getCoordinate(8 + i * coordType.bytes);
            Coordinate coordinate = coordinates.getCoordinate(0);
            points[i] = Double.isNaN(coordinate.x) ? factory.createPoint() : factory.createPoint(coordinates);
        }
        buffer.mark(bufferSize);
        return factory.createMultiPoint(points);
    }

    private static GeometryBuffer serializeLineString(LineString lineString) {
        CoordinateSequence coordinates = lineString.getCoordinateSequence();
        int numCoordinates = coordinates.size();
        if (numCoordinates == 0) {
            return GeometrySerializer.createGeometryBuffer(2, CoordinateType.XY, lineString.getSRID(), 8, 0);
        }
        CoordinateType coordType = GeometrySerializer.getCoordinateType(coordinates.getCoordinate(0));
        int bufferSize = 8 + numCoordinates * coordType.bytes;
        GeometryBuffer buffer = GeometrySerializer.createGeometryBuffer(2, coordType, lineString.getSRID(), bufferSize, numCoordinates);
        buffer.putCoordinates(8, coordinates);
        return buffer;
    }

    private static LineString deserializeLineString(GeometryBuffer buffer, GeometryFactory factory) {
        CoordinateType coordType = buffer.getCoordinateType();
        int numCoordinates = GeometrySerializer.getBoundedInt(buffer, 4);
        int bufferSize = 8 + numCoordinates * coordType.bytes;
        GeometrySerializer.checkBufferSize(buffer, bufferSize);
        CoordinateSequence coordinates = buffer.getCoordinates(8, numCoordinates);
        buffer.mark(bufferSize);
        return factory.createLineString(coordinates);
    }

    private static GeometryBuffer serializeMultiLineString(MultiLineString multiLineString) {
        int numLineStrings = multiLineString.getNumGeometries();
        CoordinateType coordType = GeometrySerializer.getCoordinateType(multiLineString);
        int numCoordinates = multiLineString.getNumPoints();
        int coordsOffset = 8;
        int numOffset = 8 + numCoordinates * coordType.bytes;
        int bufferSize = numOffset + 4 + numLineStrings * 4;
        GeometryBuffer buffer = GeometrySerializer.createGeometryBuffer(5, coordType, multiLineString.getSRID(), bufferSize, numCoordinates);
        GeomPartSerializer serializer = new GeomPartSerializer(buffer, coordsOffset, numOffset, multiLineString.getFactory());
        serializer.writeInt(numLineStrings);
        for (int k = 0; k < numLineStrings; ++k) {
            LineString ls = (LineString)multiLineString.getGeometryN(k);
            serializer.write(ls);
        }
        assert (bufferSize == serializer.intsOffset);
        return buffer;
    }

    private static MultiLineString deserializeMultiLineString(GeometryBuffer buffer, GeometryFactory factory) {
        CoordinateType coordType = buffer.getCoordinateType();
        int numCoordinates = GeometrySerializer.getBoundedInt(buffer, 4);
        int coordsOffset = 8;
        int numOffset = 8 + numCoordinates * coordType.bytes;
        GeomPartSerializer serializer = new GeomPartSerializer(buffer, coordsOffset, numOffset, factory);
        int numLineStrings = serializer.checkedReadBoundedInt();
        serializer.checkRemainingIntsAtLeast(numLineStrings);
        LineString[] lineStrings = new LineString[numLineStrings];
        for (int k = 0; k < numLineStrings; ++k) {
            LineString ls;
            lineStrings[k] = ls = serializer.readLineString();
        }
        serializer.markEndOfBuffer();
        return factory.createMultiLineString(lineStrings);
    }

    private static GeometryBuffer serializePolygon(Polygon polygon) {
        LinearRing exteriorRing = polygon.getExteriorRing();
        if (exteriorRing == null || exteriorRing.isEmpty()) {
            return GeometrySerializer.createGeometryBuffer(3, CoordinateType.XY, polygon.getSRID(), 8, 0);
        }
        CoordinateSequence coordinates = exteriorRing.getCoordinateSequence();
        CoordinateType coordType = GeometrySerializer.getCoordinateType(coordinates.getCoordinate(0));
        int numCoordinates = polygon.getNumPoints();
        int numInteriorRings = polygon.getNumInteriorRing();
        int coordsOffset = 8;
        int numRingsOffset = 8 + numCoordinates * coordType.bytes;
        int bufferSize = numRingsOffset + 4 + 4 * (numInteriorRings + 1);
        GeometryBuffer buffer = GeometrySerializer.createGeometryBuffer(3, coordType, polygon.getSRID(), bufferSize, numCoordinates);
        GeomPartSerializer serializer = new GeomPartSerializer(buffer, coordsOffset, numRingsOffset, polygon.getFactory());
        serializer.write(polygon);
        assert (bufferSize == serializer.intsOffset);
        return buffer;
    }

    private static Polygon deserializePolygon(GeometryBuffer buffer, GeometryFactory factory) {
        CoordinateType coordType = buffer.getCoordinateType();
        int numCoordinates = GeometrySerializer.getBoundedInt(buffer, 4);
        if (numCoordinates == 0) {
            buffer.mark(8);
            return factory.createPolygon();
        }
        int coordsOffset = 8;
        int numRingsOffset = 8 + numCoordinates * coordType.bytes;
        GeomPartSerializer serializer = new GeomPartSerializer(buffer, coordsOffset, numRingsOffset, factory);
        Polygon polygon = serializer.readPolygon();
        serializer.markEndOfBuffer();
        return polygon;
    }

    private static GeometryBuffer serializeMultiPolygon(MultiPolygon multiPolygon) {
        int numPolygons = multiPolygon.getNumGeometries();
        int numCoordinates = 0;
        CoordinateType coordType = GeometrySerializer.getCoordinateType(multiPolygon);
        int totalRings = 0;
        for (int k = 0; k < numPolygons; ++k) {
            Polygon polygon = (Polygon)multiPolygon.getGeometryN(k);
            if (polygon.isEmpty()) continue;
            int numRings = polygon.getNumInteriorRing() + 1;
            totalRings += numRings;
            numCoordinates += polygon.getNumPoints();
        }
        int coordsOffset = 8;
        int numPolygonsOffset = 8 + numCoordinates * coordType.bytes;
        int bufferSize = numPolygonsOffset + 4 + numPolygons * 4 + totalRings * 4;
        GeometryBuffer buffer = GeometrySerializer.createGeometryBuffer(6, coordType, multiPolygon.getSRID(), bufferSize, numCoordinates);
        GeomPartSerializer serializer = new GeomPartSerializer(buffer, coordsOffset, numPolygonsOffset, multiPolygon.getFactory());
        serializer.writeInt(numPolygons);
        for (int k = 0; k < numPolygons; ++k) {
            Polygon polygon = (Polygon)multiPolygon.getGeometryN(k);
            serializer.write(polygon);
        }
        assert (bufferSize == serializer.intsOffset);
        return buffer;
    }

    private static MultiPolygon deserializeMultiPolygon(GeometryBuffer buffer, GeometryFactory factory) {
        CoordinateType coordType = buffer.getCoordinateType();
        int numCoordinates = GeometrySerializer.getBoundedInt(buffer, 4);
        int coordsOffset = 8;
        int numPolygonsOffset = 8 + numCoordinates * coordType.bytes;
        GeomPartSerializer serializer = new GeomPartSerializer(buffer, coordsOffset, numPolygonsOffset, factory);
        int numPolygons = serializer.checkedReadBoundedInt();
        Polygon[] polygons = new Polygon[numPolygons];
        for (int k = 0; k < numPolygons; ++k) {
            Polygon polygon;
            polygons[k] = polygon = serializer.readPolygon();
        }
        serializer.markEndOfBuffer();
        return factory.createMultiPolygon(polygons);
    }

    private static GeometryBuffer serializeGeometryCollection(GeometryCollection geometryCollection) {
        int numGeometries = geometryCollection.getNumGeometries();
        if (numGeometries == 0) {
            return GeometrySerializer.createGeometryBuffer(7, CoordinateType.XY, geometryCollection.getSRID(), 8, 0);
        }
        byte[][] buffers = new byte[numGeometries][];
        int totalBytes = 0;
        for (int k = 0; k < numGeometries; ++k) {
            byte[] buf = GeometrySerializer.serialize(geometryCollection.getGeometryN(k));
            buffers[k] = buf;
            totalBytes += GeometrySerializer.alignedOffset(buf.length);
        }
        int bufferSize = 8 + totalBytes;
        GeometryBuffer buffer = GeometrySerializer.createGeometryBuffer(7, CoordinateType.XY, geometryCollection.getSRID(), bufferSize, numGeometries);
        int offset = 8;
        for (int k = 0; k < numGeometries; ++k) {
            byte[] buf = buffers[k];
            buffer.putBytes(offset, buf);
            offset += GeometrySerializer.alignedOffset(buf.length);
        }
        assert (offset == bufferSize);
        return buffer;
    }

    private static GeometryCollection deserializeGeometryCollection(GeometryBuffer buffer, GeometryFactory factory) {
        int numGeometries = GeometrySerializer.getBoundedInt(buffer, 4);
        if (numGeometries == 0) {
            buffer.mark(8);
            return factory.createGeometryCollection();
        }
        Geometry[] geometries = new Geometry[numGeometries];
        int offset = 8;
        for (int k = 0; k < numGeometries; ++k) {
            GeometryBuffer geomBuffer = buffer.slice(offset);
            Geometry geometry = GeometrySerializer.deserialize(geomBuffer, factory);
            int geomLength = GeometrySerializer.alignedOffset(geomBuffer.getMark());
            geometries[k] = geometry;
            offset += geomLength;
        }
        buffer.mark(offset);
        return factory.createGeometryCollection(geometries);
    }

    private static GeometryBuffer createGeometryBuffer(int wkbType, CoordinateType coordType, int srid, int bufferSize, int numCoordinates) {
        GeometryBuffer buffer = GeometryBufferFactory.create(bufferSize);
        buffer.setCoordinateType(coordType);
        int hasSridBit = srid != 0 ? 1 : 0;
        int preambleByte = wkbType << 4 | coordType.value << 1 | hasSridBit;
        buffer.putByte(0, (byte)preambleByte);
        if (srid != 0) {
            buffer.putByte(1, (byte)(srid >> 16));
            buffer.putByte(2, (byte)(srid >> 8));
            buffer.putByte(3, (byte)srid);
        }
        buffer.putInt(4, numCoordinates);
        return buffer;
    }

    private static void checkBufferSize(GeometryBuffer buffer, int minimumSize) {
        if (buffer.getLength() < minimumSize) {
            throw new IllegalArgumentException("Buffer to be deserialized is incomplete");
        }
    }

    private static int getBoundedInt(GeometryBuffer buffer, int offset) {
        int value = buffer.getInt(offset);
        if (value < 0) {
            throw new IllegalArgumentException("Unexpected negative value encountered: " + value);
        }
        if (value > buffer.getLength()) {
            throw new IllegalArgumentException("Unexpected large value encountered: " + value);
        }
        return value;
    }

    private static CoordinateType getCoordinateType(Coordinate coordinate) {
        boolean hasZ = !Double.isNaN(coordinate.getZ());
        boolean hasM = !Double.isNaN(coordinate.getM());
        return GeometrySerializer.getCoordinateType(hasZ, hasM);
    }

    private static CoordinateType getCoordinateType(Geometry geometry) {
        Coordinate coord = geometry.getCoordinate();
        if (coord != null) {
            return GeometrySerializer.getCoordinateType(coord);
        }
        return CoordinateType.XY;
    }

    private static CoordinateType getCoordinateType(boolean hasZ, boolean hasM) {
        if (hasZ && hasM) {
            return CoordinateType.XYZM;
        }
        if (hasZ) {
            return CoordinateType.XYZ;
        }
        if (hasM) {
            return CoordinateType.XYM;
        }
        return CoordinateType.XY;
    }

    private static int alignedOffset(int offset) {
        return offset + 7 & 0xFFFFFFF8;
    }

    private static GeometryFactory createGeometryFactory(int srid) {
        return new GeometryFactory(PRECISION_MODEL, srid);
    }

    static class GeomPartSerializer {
        final GeometryBuffer buffer;
        int coordsOffset;
        final int coordsEndOffset;
        int intsOffset;
        final GeometryFactory factory;

        GeomPartSerializer(GeometryBuffer buffer, int coordsOffset, int intsOffset, GeometryFactory factory) {
            this.buffer = buffer;
            this.coordsOffset = coordsOffset;
            this.coordsEndOffset = intsOffset;
            this.intsOffset = intsOffset;
            this.factory = factory;
        }

        LineString readLineString() {
            CoordinateSequence coordinates = this.readCoordinates();
            return this.factory.createLineString(coordinates);
        }

        LinearRing readRing() {
            CoordinateSequence coordinates = this.readCoordinates();
            return this.factory.createLinearRing(coordinates);
        }

        Polygon readPolygon() {
            int numRings = this.checkedReadBoundedInt();
            if (numRings == 0) {
                return this.factory.createPolygon();
            }
            this.checkRemainingIntsAtLeast(numRings);
            int numInteriorRings = numRings - 1;
            LinearRing shell = this.readRing();
            LinearRing[] holes = new LinearRing[numInteriorRings];
            for (int k = 0; k < numInteriorRings; ++k) {
                holes[k] = this.readRing();
            }
            return this.factory.createPolygon(shell, holes);
        }

        CoordinateSequence readCoordinates() {
            int numCoordinates = GeometrySerializer.getBoundedInt(this.buffer, this.intsOffset);
            int newCoordsOffset = this.coordsOffset + this.buffer.getCoordinateType().bytes * numCoordinates;
            if (newCoordsOffset > this.coordsEndOffset) {
                throw new IllegalStateException("Number of coordinates exceeds the capacity of buffer: " + numCoordinates);
            }
            CoordinateSequence coordinates = this.buffer.getCoordinates(this.coordsOffset, numCoordinates);
            this.coordsOffset = newCoordsOffset;
            this.intsOffset += 4;
            return coordinates;
        }

        int readBoundedInt() {
            int value = GeometrySerializer.getBoundedInt(this.buffer, this.intsOffset);
            this.intsOffset += 4;
            return value;
        }

        int checkedReadBoundedInt() {
            GeometrySerializer.checkBufferSize(this.buffer, this.intsOffset + 4);
            return this.readBoundedInt();
        }

        void checkRemainingIntsAtLeast(int num) {
            GeometrySerializer.checkBufferSize(this.buffer, this.intsOffset + 4 * num);
        }

        void write(LineString lineString) {
            CoordinateSequence coordinates = lineString.getCoordinateSequence();
            int numCoordinates = coordinates.size();
            this.buffer.putCoordinates(this.coordsOffset, coordinates);
            this.buffer.putInt(this.intsOffset, numCoordinates);
            this.coordsOffset += numCoordinates * this.buffer.getCoordinateType().bytes;
            this.intsOffset += 4;
        }

        void write(Polygon polygon) {
            LinearRing exteriorRing = polygon.getExteriorRing();
            if (exteriorRing.isEmpty()) {
                this.writeInt(0);
                return;
            }
            int numInteriorRings = polygon.getNumInteriorRing();
            this.writeInt(numInteriorRings + 1);
            this.write(exteriorRing);
            for (int k = 0; k < numInteriorRings; ++k) {
                this.write(polygon.getInteriorRingN(k));
            }
        }

        void writeInt(int value) {
            this.buffer.putInt(this.intsOffset, value);
            this.intsOffset += 4;
        }

        void markEndOfBuffer() {
            this.buffer.mark(this.intsOffset);
        }
    }
}

