////////////////////////////////////////////////////////////////////////////////
//
//  Licensed to the Apache Software Foundation (ASF) under one or more
//  contributor license agreements.  See the NOTICE file distributed with
//  this work for additional information regarding copyright ownership.
//  The ASF licenses this file to You under the Apache License, Version 2.0
//  (the "License"); you may not use this file except in compliance with
//  the License.  You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
////////////////////////////////////////////////////////////////////////////////

package mx.containers
{

/**
 *  The VDividedBox container lays out its children vertically
 *  in the same way as the VBox container, but it inserts
 *  a draggable divider in the gap between each child.
 *  A user can drag the divider to resize the area allotted to each child.
 *  The <code>&lt;mx:VDividedBox/&gt;</code> tag is the same as
 *  <code>&lt;mx:DividedBox direction="vertical"/&gt;</code>.
 *  
 *  <p>A VDividedBox container has the following default sizing characteristics:</p>
 *     <table class="innertable">
 *        <tr>
 *           <th>Characteristic</th>
 *           <th>Description</th>
 *        </tr>
 *        <tr>
 *           <td>Default size</td>
 *           <td>Height is large enough to hold all of its children at the default or explicit 
 *               heights of the children, plus any vertical gap between the children, plus the top and bottom padding of the 
 *               container. The width is the default or explicit width of the widest child plus the left and right padding of 
 *               the container.</td>
 *        </tr>
 *        <tr>
 *           <td>Default padding</td>
 *           <td>0 pixels for the top, bottom, left, and right values.</td>
 *        </tr>
 *        <tr>
 *           <td>Default gap</td>
 *           <td>10 pixels for the horizontal and vertical gaps.</td>
 *        </tr>
 *     </table>
 *
 *  @mxml
 *  
 *  <p>The <code>&lt;mx:VDividedBox&gt;</code> tag inherits all of the tag 
 *  attributes of its superclass, except <code>direction</code>, and adds 
 *  no new tag attributes.</p>
 *  
 *  @includeExample examples/VDividedBoxExample.mxml
 *
 *  @see mx.containers.DividedBox
 *  @see mx.containers.HDividedBox
 *  
 *  @langversion 3.0
 *  @playerversion Flash 9
 *  @playerversion AIR 1.1
 *  @productversion Flex 3
 */
public class VDividedBox extends DividedBox
{
	
	//--------------------------------------------------------------------------
	//
	//  Constructor
	//
	//--------------------------------------------------------------------------

	import org.apache.royale.events.MouseEvent;
	/**
	 *  Constructor.
	 *  
	 *  @langversion 3.0
	 *  @playerversion Flash 9
	 *  @playerversion AIR 1.1
	 *  @productversion Flex 3
	 */
	public function VDividedBox()
	{
		super();
		typeNames = "VDividedBox";
		super.direction = "vertical";
		addEventListener(MouseEvent.MOUSE_DOWN, mouseDownHandler);
	}

	protected function mouseDownHandler(event:MouseEvent):void
	{
		if (event.target != this)
		{
			return;
		}
		addEventListener(MouseEvent.MOUSE_MOVE, mouseMoveHandler);
		topMostEventDispatcher.addEventListener(MouseEvent.MOUSE_UP, mouseUpHandler);
	}

	protected function mouseUpHandler(event:MouseEvent):void
	{
		removeEventListener(MouseEvent.MOUSE_MOVE, mouseMoveHandler);
		topMostEventDispatcher.removeEventListener(MouseEvent.MOUSE_UP, mouseUpHandler);
	}

	protected function mouseMoveHandler(event:MouseEvent):void
	{
		var h1:int = Math.floor(100 * event.localY / height);
		h1 = Math.min(h1, 95);
		h1 = Math.max(h1, 5);
		getChildAt(0).percentHeight = h1;
		getChildAt(1).percentHeight = 100 - h1;
		_layout.layout();
	}


	//--------------------------------------------------------------------------
	//
	//  Overridden properties
	//
	//--------------------------------------------------------------------------

	//----------------------------------
	//  direction
	//----------------------------------
	
	[Inspectable(environment="none")]	

	/**
	 *  @private
	 *  Don't allow user to change the direction
	 */
	override public function set direction(value:String):void
	{
	}
}

}
