/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service.plugin;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import java.io.File;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import lombok.Generated;
import org.apache.commons.lang3.StringUtils;
import org.apache.pulsar.broker.ServiceConfiguration;
import org.apache.pulsar.broker.service.plugin.EntryFilter;
import org.apache.pulsar.broker.service.plugin.EntryFilterDefinition;
import org.apache.pulsar.broker.service.plugin.EntryFilterMetaData;
import org.apache.pulsar.broker.service.plugin.EntryFilterWithClassLoader;
import org.apache.pulsar.broker.service.plugin.InvalidEntryFilterException;
import org.apache.pulsar.common.nar.NarClassLoader;
import org.apache.pulsar.common.nar.NarClassLoaderBuilder;
import org.apache.pulsar.common.policies.data.EntryFilters;
import org.apache.pulsar.common.util.ObjectMapperFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EntryFilterProvider
implements AutoCloseable {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(EntryFilterProvider.class);
    @VisibleForTesting
    static final String ENTRY_FILTER_DEFINITION_FILE = "entry_filter";
    private final ServiceConfiguration serviceConfiguration;
    @VisibleForTesting
    protected Map<String, EntryFilterMetaData> definitions;
    @VisibleForTesting
    protected Map<String, NarClassLoader> cachedClassLoaders;
    @VisibleForTesting
    protected List<EntryFilter> brokerEntryFilters;

    public EntryFilterProvider(ServiceConfiguration conf) throws IOException {
        this.serviceConfiguration = conf;
        this.initialize();
        this.initializeBrokerEntryFilters();
    }

    protected void initializeBrokerEntryFilters() throws IOException {
        this.brokerEntryFilters = !this.serviceConfiguration.getEntryFilterNames().isEmpty() ? this.loadEntryFilters(this.serviceConfiguration.getEntryFilterNames()) : Collections.emptyList();
    }

    public void validateEntryFilters(String entryFilterNames) throws InvalidEntryFilterException {
        if (StringUtils.isBlank((CharSequence)entryFilterNames)) {
            return;
        }
        List<String> entryFilterList = this.readEntryFiltersString(entryFilterNames);
        for (String filterName : entryFilterList) {
            EntryFilterMetaData metaData = this.definitions.get(filterName);
            if (metaData != null) continue;
            throw new InvalidEntryFilterException("Entry filter '" + filterName + "' not found");
        }
    }

    private List<String> readEntryFiltersString(String entryFilterNames) {
        List<String> entryFilterList = Arrays.stream(entryFilterNames.split(",")).filter(n -> StringUtils.isNotBlank((CharSequence)n)).toList();
        return entryFilterList;
    }

    public List<EntryFilter> loadEntryFiltersForPolicy(EntryFilters policy) throws IOException {
        String names = policy.getEntryFilterNames();
        if (StringUtils.isBlank((CharSequence)names)) {
            return Collections.emptyList();
        }
        List<String> entryFilterList = this.readEntryFiltersString(names);
        return this.loadEntryFilters(entryFilterList);
    }

    private List<EntryFilter> loadEntryFilters(Collection<String> entryFilterNames) throws IOException {
        ImmutableMap.Builder builder = ImmutableMap.builder();
        for (String filterName : entryFilterNames) {
            EntryFilterMetaData metaData = this.definitions.get(filterName);
            if (null == metaData) {
                throw new RuntimeException("No entry filter is found for name `" + filterName + "`. Available entry filters are : " + String.valueOf(this.definitions.keySet()));
            }
            EntryFilter entryFilter = this.load(metaData);
            builder.put((Object)filterName, (Object)entryFilter);
            log.info("Successfully loaded entry filter `{}`", (Object)filterName);
        }
        return builder.build().values().asList();
    }

    public List<EntryFilter> getBrokerEntryFilters() {
        return this.brokerEntryFilters;
    }

    private void initialize() throws IOException {
        String entryFiltersDirectory = this.serviceConfiguration.getEntryFiltersDirectory();
        Path path = Paths.get(entryFiltersDirectory, new String[0]).toAbsolutePath().normalize();
        log.info("Searching for entry filters in {}", (Object)path);
        if (!path.toFile().exists()) {
            log.info("Pulsar entry filters directory not found");
            this.definitions = Collections.emptyMap();
            this.cachedClassLoaders = Collections.emptyMap();
            return;
        }
        HashMap<String, EntryFilterMetaData> entryFilterDefinitions = new HashMap<String, EntryFilterMetaData>();
        this.cachedClassLoaders = new HashMap<String, NarClassLoader>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(path, "*.nar");){
            for (Path archive : stream) {
                try {
                    NarClassLoader narClassLoader = this.loadNarClassLoader(archive);
                    EntryFilterDefinition def = EntryFilterProvider.getEntryFilterDefinition(narClassLoader);
                    log.info("Found entry filter from {} : {}", (Object)archive, (Object)def);
                    Preconditions.checkArgument((boolean)StringUtils.isNotBlank((CharSequence)def.getName()));
                    Preconditions.checkArgument((boolean)StringUtils.isNotBlank((CharSequence)def.getEntryFilterClass()));
                    EntryFilterMetaData metadata = new EntryFilterMetaData();
                    metadata.setDefinition(def);
                    metadata.setArchivePath(archive);
                    entryFilterDefinitions.put(def.getName(), metadata);
                }
                catch (Throwable t) {
                    log.warn("Failed to load entry filters from {}. It is OK however if you want to use this entry filters, please make sure you put the correct entry filter NAR package in the entry filter directory.", (Object)archive, (Object)t);
                }
            }
        }
        this.definitions = Collections.unmodifiableMap(entryFilterDefinitions);
        this.cachedClassLoaders = Collections.unmodifiableMap(this.cachedClassLoaders);
    }

    @VisibleForTesting
    static EntryFilterDefinition getEntryFilterDefinition(NarClassLoader ncl) throws IOException {
        String configStr;
        try {
            configStr = ncl.getServiceDefinition("entry_filter.yaml");
        }
        catch (NoSuchFileException e) {
            configStr = ncl.getServiceDefinition("entry_filter.yml");
        }
        return (EntryFilterDefinition)ObjectMapperFactory.getYamlMapper().reader().readValue(configStr, EntryFilterDefinition.class);
    }

    protected EntryFilter load(EntryFilterMetaData metadata) throws IOException {
        EntryFilterDefinition def = metadata.getDefinition();
        if (StringUtils.isBlank((CharSequence)def.getEntryFilterClass())) {
            throw new RuntimeException("Entry filter `" + def.getName() + "` does NOT provide a entry filters implementation");
        }
        try {
            NarClassLoader ncl = this.getNarClassLoader(metadata.getArchivePath());
            if (ncl == null) {
                throw new RuntimeException("Entry filter `" + def.getName() + "` cannot be loaded, see the broker logs for further details");
            }
            Class entryFilterClass = ncl.loadClass(def.getEntryFilterClass());
            Object filter = entryFilterClass.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
            if (!(filter instanceof EntryFilter)) {
                throw new IOException("Class " + def.getEntryFilterClass() + " does not implement entry filter interface");
            }
            EntryFilter pi = (EntryFilter)filter;
            return new EntryFilterWithClassLoader(pi, ncl, false);
        }
        catch (Throwable e) {
            if (e instanceof IOException) {
                throw (IOException)e;
            }
            log.error("Failed to load class {}", (Object)metadata.getDefinition().getEntryFilterClass(), (Object)e);
            throw new IOException(e);
        }
    }

    private NarClassLoader getNarClassLoader(Path archivePath) {
        return this.cachedClassLoaders.get(EntryFilterProvider.classLoaderKey(archivePath));
    }

    private NarClassLoader loadNarClassLoader(Path archivePath) {
        String absolutePath = EntryFilterProvider.classLoaderKey(archivePath);
        return this.cachedClassLoaders.computeIfAbsent(absolutePath, narFilePath -> {
            try {
                File narFile = archivePath.toAbsolutePath().normalize().toFile();
                return NarClassLoaderBuilder.builder().narFile(narFile).parentClassLoader(EntryFilter.class.getClassLoader()).extractionDirectory(this.serviceConfiguration.getNarExtractionDirectory()).build();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
    }

    private static String classLoaderKey(Path archivePath) {
        return archivePath.toString();
    }

    @Override
    public void close() throws Exception {
        this.brokerEntryFilters.forEach(filter -> {
            try {
                filter.close();
            }
            catch (Throwable e) {
                log.warn("Error shutting down entry filter {}", filter, (Object)e);
            }
        });
        this.cachedClassLoaders.forEach((name, ncl) -> {
            try {
                ncl.close();
            }
            catch (Throwable e) {
                log.warn("Error closing entry filter class loader {}", name, (Object)e);
            }
        });
    }
}

