/*
 * Decompiled with CFR 0.152.
 */
package com.twitter.common.objectsize;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Sets;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryPoolMXBean;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.Deque;
import java.util.LinkedList;
import java.util.Set;

public class ObjectSizeCalculator {
    private final int arrayHeaderSize;
    private final int objectHeaderSize;
    private final int objectPadding;
    private final int referenceSize;
    private final int superclassFieldPadding;
    private final LoadingCache<Class<?>, ClassSizeInfo> classSizeInfos = CacheBuilder.newBuilder().build(new CacheLoader<Class<?>, ClassSizeInfo>(){

        public ClassSizeInfo load(Class<?> clazz) {
            return new ClassSizeInfo(clazz);
        }
    });
    private final Set<Object> alreadyVisited = Sets.newIdentityHashSet();
    private final Deque<Object> pending = new ArrayDeque<Object>(16384);
    private long size;

    public static long getObjectSize(Object object) throws UnsupportedOperationException {
        return object == null ? 0L : new ObjectSizeCalculator(CurrentLayout.SPEC).calculateObjectSize(object);
    }

    public ObjectSizeCalculator(MemoryLayoutSpecification memoryLayoutSpecification) {
        Preconditions.checkNotNull((Object)memoryLayoutSpecification);
        this.arrayHeaderSize = memoryLayoutSpecification.getArrayHeaderSize();
        this.objectHeaderSize = memoryLayoutSpecification.getObjectHeaderSize();
        this.objectPadding = memoryLayoutSpecification.getObjectPadding();
        this.referenceSize = memoryLayoutSpecification.getReferenceSize();
        this.superclassFieldPadding = memoryLayoutSpecification.getSuperclassFieldPadding();
    }

    public synchronized long calculateObjectSize(Object object) {
        try {
            while (true) {
                this.visit(object);
                if (this.pending.isEmpty()) {
                    long l = this.size;
                    return l;
                }
                object = this.pending.removeFirst();
            }
        }
        finally {
            this.alreadyVisited.clear();
            this.pending.clear();
            this.size = 0L;
        }
    }

    private void visit(Object object) {
        if (this.alreadyVisited.contains(object)) {
            return;
        }
        Class<?> clazz = object.getClass();
        if (clazz == ArrayElementsVisitor.class) {
            ((ArrayElementsVisitor)object).visit(this);
        } else {
            this.alreadyVisited.add(object);
            if (clazz.isArray()) {
                this.visitArray(object);
            } else {
                ((ClassSizeInfo)this.classSizeInfos.getUnchecked(clazz)).visit(object, this);
            }
        }
    }

    private void visitArray(Object object) {
        Class<?> clazz = object.getClass().getComponentType();
        int n = Array.getLength(object);
        if (clazz.isPrimitive()) {
            this.increaseByArraySize(n, ObjectSizeCalculator.getPrimitiveFieldSize(clazz));
        } else {
            this.increaseByArraySize(n, this.referenceSize);
            switch (n) {
                case 0: {
                    break;
                }
                case 1: {
                    this.enqueue(Array.get(object, 0));
                    break;
                }
                default: {
                    this.enqueue(new ArrayElementsVisitor((Object[])object));
                }
            }
        }
    }

    private void increaseByArraySize(int n, long l) {
        this.increaseSize(ObjectSizeCalculator.roundTo((long)this.arrayHeaderSize + (long)n * l, this.objectPadding));
    }

    void enqueue(Object object) {
        if (object != null) {
            this.pending.addLast(object);
        }
    }

    void increaseSize(long l) {
        this.size += l;
    }

    @VisibleForTesting
    static long roundTo(long l, int n) {
        return (l + (long)n - 1L) / (long)n * (long)n;
    }

    private static long getPrimitiveFieldSize(Class<?> clazz) {
        if (clazz == Boolean.TYPE || clazz == Byte.TYPE) {
            return 1L;
        }
        if (clazz == Character.TYPE || clazz == Short.TYPE) {
            return 2L;
        }
        if (clazz == Integer.TYPE || clazz == Float.TYPE) {
            return 4L;
        }
        if (clazz == Long.TYPE || clazz == Double.TYPE) {
            return 8L;
        }
        throw new AssertionError((Object)("Encountered unexpected primitive type " + clazz.getName()));
    }

    @VisibleForTesting
    static MemoryLayoutSpecification getEffectiveMemoryLayoutSpecification() {
        String string = System.getProperty("java.vm.name");
        if (string == null || !string.startsWith("Java HotSpot(TM) ") && !string.startsWith("OpenJDK") && !string.startsWith("TwitterJDK")) {
            throw new UnsupportedOperationException("ObjectSizeCalculator only supported on HotSpot VM");
        }
        String string2 = System.getProperty("sun.arch.data.model");
        if ("32".equals(string2)) {
            return new MemoryLayoutSpecification(){

                @Override
                public int getArrayHeaderSize() {
                    return 12;
                }

                @Override
                public int getObjectHeaderSize() {
                    return 8;
                }

                @Override
                public int getObjectPadding() {
                    return 8;
                }

                @Override
                public int getReferenceSize() {
                    return 4;
                }

                @Override
                public int getSuperclassFieldPadding() {
                    return 4;
                }
            };
        }
        if (!"64".equals(string2)) {
            throw new UnsupportedOperationException("Unrecognized value '" + string2 + "' of sun.arch.data.model system property");
        }
        String string3 = System.getProperty("java.vm.version");
        int n = Integer.parseInt(string3.substring(0, string3.indexOf(46)));
        if (n >= 17) {
            long l = 0L;
            for (MemoryPoolMXBean memoryPoolMXBean : ManagementFactory.getMemoryPoolMXBeans()) {
                l += memoryPoolMXBean.getUsage().getMax();
            }
            if (l < 0x780000000L) {
                return new MemoryLayoutSpecification(){

                    @Override
                    public int getArrayHeaderSize() {
                        return 16;
                    }

                    @Override
                    public int getObjectHeaderSize() {
                        return 12;
                    }

                    @Override
                    public int getObjectPadding() {
                        return 8;
                    }

                    @Override
                    public int getReferenceSize() {
                        return 4;
                    }

                    @Override
                    public int getSuperclassFieldPadding() {
                        return 4;
                    }
                };
            }
        }
        return new MemoryLayoutSpecification(){

            @Override
            public int getArrayHeaderSize() {
                return 24;
            }

            @Override
            public int getObjectHeaderSize() {
                return 16;
            }

            @Override
            public int getObjectPadding() {
                return 8;
            }

            @Override
            public int getReferenceSize() {
                return 8;
            }

            @Override
            public int getSuperclassFieldPadding() {
                return 8;
            }
        };
    }

    private class ClassSizeInfo {
        private final long objectSize;
        private final long fieldsSize;
        private final Field[] referenceFields;

        public ClassSizeInfo(Class<?> clazz) {
            long l = 0L;
            LinkedList<Field> linkedList = new LinkedList<Field>();
            for (Field field : clazz.getDeclaredFields()) {
                if (Modifier.isStatic(field.getModifiers())) continue;
                Class<?> clazz2 = field.getType();
                if (clazz2.isPrimitive()) {
                    l += ObjectSizeCalculator.getPrimitiveFieldSize(clazz2);
                    continue;
                }
                field.setAccessible(true);
                linkedList.add(field);
                l += (long)ObjectSizeCalculator.this.referenceSize;
            }
            Class<?> clazz3 = clazz.getSuperclass();
            if (clazz3 != null) {
                ClassSizeInfo classSizeInfo = (ClassSizeInfo)ObjectSizeCalculator.this.classSizeInfos.getUnchecked(clazz3);
                l += ObjectSizeCalculator.roundTo(classSizeInfo.fieldsSize, ObjectSizeCalculator.this.superclassFieldPadding);
                linkedList.addAll(Arrays.asList(classSizeInfo.referenceFields));
            }
            this.fieldsSize = l;
            this.objectSize = ObjectSizeCalculator.roundTo((long)ObjectSizeCalculator.this.objectHeaderSize + l, ObjectSizeCalculator.this.objectPadding);
            this.referenceFields = linkedList.toArray(new Field[linkedList.size()]);
        }

        void visit(Object object, ObjectSizeCalculator objectSizeCalculator) {
            objectSizeCalculator.increaseSize(this.objectSize);
            this.enqueueReferencedObjects(object, objectSizeCalculator);
        }

        public void enqueueReferencedObjects(Object object, ObjectSizeCalculator objectSizeCalculator) {
            for (Field field : this.referenceFields) {
                try {
                    objectSizeCalculator.enqueue(field.get(object));
                }
                catch (IllegalAccessException illegalAccessException) {
                    AssertionError assertionError = new AssertionError((Object)("Unexpected denial of access to " + field));
                    ((Throwable)((Object)assertionError)).initCause(illegalAccessException);
                    throw assertionError;
                }
            }
        }
    }

    private static class ArrayElementsVisitor {
        private final Object[] array;

        ArrayElementsVisitor(Object[] objectArray) {
            this.array = objectArray;
        }

        public void visit(ObjectSizeCalculator objectSizeCalculator) {
            for (Object object : this.array) {
                if (object == null) continue;
                objectSizeCalculator.visit(object);
            }
        }
    }

    private static class CurrentLayout {
        private static final MemoryLayoutSpecification SPEC = ObjectSizeCalculator.getEffectiveMemoryLayoutSpecification();

        private CurrentLayout() {
        }
    }

    public static interface MemoryLayoutSpecification {
        public int getArrayHeaderSize();

        public int getObjectHeaderSize();

        public int getObjectPadding();

        public int getReferenceSize();

        public int getSuperclassFieldPadding();
    }
}

