/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.maven.it;

import java.io.File;
import java.util.Properties;

import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.assertNull;

/**
 * This is a test set for <a href="https://issues.apache.org/jira/browse/MNG-4273">MNG-4273</a>.
 *
 * @author Benjamin Bentmann
 */
public class MavenITmng4273RestrictedCoreRealmAccessForPluginTest extends AbstractMavenIntegrationTestCase {

    public MavenITmng4273RestrictedCoreRealmAccessForPluginTest() {
        super("[2.0.6,)");
    }

    /**
     * Verify that internal utility/implementation classes used by the Maven core do not leak into the plugin realm.
     * Otherwise, we risk linkage errors when a plugin creates a custom class loader with parent-first delegation on
     * top of the plugin realm because the custom class loader will prefer potentially incompatible classes from the
     * core realm over classes from itself.
     *
     * @throws Exception in case of failure
     */
    @Test
    public void testit() throws Exception {
        File testDir = extractResources("/mng-4273");

        Verifier verifier = newVerifier(testDir.getAbsolutePath());
        verifier.setAutoclean(false);
        verifier.addCliArgument("validate");
        verifier.execute();
        verifier.verifyErrorFreeLog();

        Properties props = verifier.loadProperties("target/class.properties");

        assertNull(props.getProperty("org.codehaus.plexus.util.FileUtils$FilterWrapper"));

        try {
            // some IBM JRE's ship with Xerces (xml.jar) so a plugin can load this class from the bootstrap loader
            ClassLoader.getSystemClassLoader().loadClass("org.apache.xerces.util.ParserConfigurationSettings");
        } catch (ClassNotFoundException e) {
            // not provided by JRE, and must not be provided by Maven's core realm either
            assertNull(props.getProperty("org.apache.xerces.util.ParserConfigurationSettings"));
        }
    }
}
