// ***************************************************************************************************************************
// * Licensed to the Apache Software Foundation (ASF) under one or more contributor license agreements.  See the NOTICE file *
// * distributed with this work for additional information regarding copyright ownership.  The ASF licenses this file        *
// * to you under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance            *
// * with the License.  You may obtain a copy of the License at                                                              *
// *                                                                                                                         *
// *  http://www.apache.org/licenses/LICENSE-2.0                                                                             *
// *                                                                                                                         *
// * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on an  *
// * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the License for the        *
// * specific language governing permissions and limitations under the License.                                              *
// ***************************************************************************************************************************
package org.apache.juneau.http.resource;

import static org.apache.juneau.internal.ThrowableUtils.*;
import static org.apache.juneau.internal.StringUtils.*;

import java.io.*;
import java.util.*;
import java.util.function.*;

import org.apache.http.*;
import org.apache.juneau.http.entity.*;
import org.apache.juneau.http.header.*;
import org.apache.juneau.internal.*;

/**
 * Builder for {@link HttpEntity} beans.
 *
 * <ul class='seealso'>
 * 	<li class='link'>{@doc jm.HttpParts}
 * 	<li class='extlink'>{@source}
 * </ul>
 *
 * @param <T> The bean type to create for this builder.
 */
@FluentSetters(returns="HttpResourceBuilder<T>")
public class HttpResourceBuilder<T extends BasicResource> {

	HeaderList headers = HeaderList.EMPTY;
	HeaderList.Builder headersBuilder;

	BasicHttpEntity entity;
	HttpEntityBuilder<?> entityBuilder;

	/** The HttpEntity implementation class. */
	protected final Class<? extends BasicResource> implClass;

	/** The HttpEntity implementation class. */
	protected final Class<? extends BasicHttpEntity> entityImplClass;

	/**
	 * Constructor.
	 *
	 * @param implClass
	 * 	The subclass of {@link HttpResponse} to create.
	 * 	<br>This must contain a public constructor that takes in an {@link HttpResourceBuilder} object.
	 * @param entityImplClass
	 * 	The subclass of {@link BasicHttpEntity} to create.
	 * 	<br>This must contain a public constructor that takes in an {@link HttpEntityBuilder} object.
	 */
	public HttpResourceBuilder(Class<T> implClass, Class<? extends BasicHttpEntity> entityImplClass) {
		this.implClass = implClass;
		this.entityImplClass = entityImplClass;
	}

	/**
	 * Copy constructor.
	 *
	 * @param impl
	 * 	The implementation object of {@link HttpEntity} to copy from.
	 * 	<br>This must contain a public constructor that takes in an {@link HttpResourceBuilder} object.
	 */
	public HttpResourceBuilder(T impl) {
		implClass = impl.getClass();
		headers = impl.headers;
		entity = impl.entity;
		this.entityImplClass = entity.getClass();
	}

	/**
	 * Instantiates the entity bean from the settings in this builder.
	 *
	 * @return A new {@link HttpEntity} bean.
	 */
	@SuppressWarnings("unchecked")
	public T build() {
		try {
			return (T) implClass.getConstructor(HttpResourceBuilder.class).newInstance(this);
		} catch (Exception e) {
			throw runtimeException(e);
		}
	}

	HeaderList headers() {
		if (headersBuilder != null)
			return headersBuilder.build();
		if (headers == null)
			return HeaderList.EMPTY;
		return headers;
	}

	BasicHttpEntity entity() {
		if (entityBuilder != null)
			return entityBuilder.build();
		if (entity == null)
			return BasicHttpEntity.EMPTY;
		return entity;
	}

	/**
	 * Copies the contents of the specified HTTP response to this builder.
	 *
	 * @param response The response to copy from.  Must not be null.
	 * @return This object.
	 * @throws IOException If content could not be retrieved.
	 */
	public HttpResourceBuilder<?> copyFrom(HttpResponse response) throws IOException {
		headers(response.getAllHeaders());
		content(response.getEntity().getContent());
		return this;
	}

	//-----------------------------------------------------------------------------------------------------------------
	// HttpEntityBuilder setters.
	//-----------------------------------------------------------------------------------------------------------------

	/**
	 * Returns access to the underlying builder for the HTTP entity.
	 *
	 * @return The underlying builder for the HTTP entity.
	 */
	public HttpEntityBuilder<?> getEntity() {
		if (entityBuilder == null) {
			entityBuilder = entity == null ? BasicHttpEntity.create(entityImplClass) : entity.copy();
			entity = null;
		}
		return entityBuilder;
	}

	/**
	 * Sets the content on this entity bean.
	 *
	 * @param value The entity content, can be <jk>null</jk>.
	 * @return This object.
	 */
	@FluentSetter
	public HttpResourceBuilder<T> content(Object value) {
		getEntity().content(value);
		return this;
	}

	/**
	 * Sets the content on this entity bean from a supplier.
	 *
	 * <p>
	 * Repeatable entities such as {@link StringEntity} use this to allow the entity content to be resolved at
	 * serialization time.
	 *
	 * @param value The entity content, can be <jk>null</jk>.
	 * @return This object.
	 */
	@FluentSetter
	public HttpResourceBuilder<T> content(Supplier<?> value) {
		getEntity().content(value);
		return this;
	}

	/**
	 * Sets the content type on this entity bean.
	 *
	 * @param value The new <c>Content-Type</c> header, or <jk>null</jk> to unset.
	 * @return This object.
	 */
	@FluentSetter
	public HttpResourceBuilder<T> contentType(String value) {
		getEntity().contentType(value);
		return this;
	}

	/**
	 * Sets the content type on this entity bean.
	 *
	 * @param value The new <c>Content-Type</c> header, or <jk>null</jk> to unset.
	 * @return This object.
	 */
	@FluentSetter
	public HttpResourceBuilder<T> contentType(ContentType value) {
		getEntity().contentType(value);
		return this;
	}

	/**
	 * Sets the content length on this entity bean.
	 *
	 * @param value The new <c>Content-Length</c> header value, or <c>-1</c> to unset.
	 * @return This object.
	 */
	@FluentSetter
	public HttpResourceBuilder<T> contentLength(long value) {
		getEntity().contentLength(value);
		return this;
	}

	/**
	 * Sets the content encoding header on this entity bean.
	 *
	 * @param value The new <c>Content-Encoding</c> header, or <jk>null</jk> to unset.
	 * @return This object.
	 */
	@FluentSetter
	public HttpResourceBuilder<T> contentEncoding(String value) {
		getEntity().contentEncoding(value);
		return this;
	}

	/**
	 * Sets the content encoding header on this entity bean.
	 *
	 * @param value The new <c>Content-Encoding</c> header, or <jk>null</jk> to unset.
	 * @return This object.
	 */
	@FluentSetter
	public HttpResourceBuilder<T> contentEncoding(ContentEncoding value) {
		getEntity().contentEncoding(value);
		return this;
	}

	/**
	 * Sets the 'chunked' flag value to <jk>true</jk>.
	 *
	 * <ul class='notes'>
	 * 	<li>If the {@link HttpEntity#getContentLength()} method returns a negative value, the HttpClient code will always
	 * 		use chunked encoding.
	 * </ul>
	 *
	 * @return This object.
	 */
	@FluentSetter
	public HttpResourceBuilder<T> chunked() {
		getEntity().chunked();
		return this;
	}

	/**
	 * Sets the 'chunked' flag value.
	 *
	 * <ul class='notes'>
	 * 	<li class='note'>If the {@link HttpEntity#getContentLength()} method returns a negative value, the HttpClient code will always
	 * 		use chunked encoding.
	 * </ul>
	 *
	 * @param value The new value for this flag.
	 * @return This object.
	 */
	@FluentSetter
	public HttpResourceBuilder<T> chunked(boolean value) {
		getEntity().chunked(value);
		return this;
	}

	/**
	 * Specifies that the contents of this resource should be cached into an internal byte array so that it can
	 * be read multiple times.
	 *
	 * @return This object.
	 * @throws IOException If entity could not be read into memory.
	 */
	@FluentSetter
	public HttpResourceBuilder<T> cached() throws IOException {
		getEntity().cached();
		return this;
	}

	//-----------------------------------------------------------------------------------------------------------------
	// BasicHeaderGroup setters.
	//-----------------------------------------------------------------------------------------------------------------

	/**
	 * Returns access to the underlying builder for the headers.
	 *
	 * @return The underlying builder for the headers.
	 */
	public HeaderList.Builder getHeaders() {
		if (headersBuilder == null) {
			headersBuilder = headers == null ? HeaderList.create() : headers.copy();
			headers = null;
		}
		return headersBuilder;
	}

	/**
	 * Sets the specified headers in this builder.
	 *
	 * @param value The new value.
	 * @return This object.
	 */
	@FluentSetter
	public HttpResourceBuilder<T> headers(HeaderList value) {
		headers = value;
		headersBuilder = null;
		return this;
	}

	/**
	 * Adds the specified header to the end of the headers in this builder.
	 *
	 * @param value The header to add.  <jk>null</jk> values are ignored.
	 * @return This object.
	 */
	public HttpResourceBuilder<T> header(Header value) {
		if (value != null)
			getHeaders().append(value);
		return this;
	}

	/**
	 * Adds the specified header to the end of the headers in this builder.
	 *
	 * <p>
	 * This is a no-op if either the name or value is <jk>null</jk>.
	 *
	 * @param name The header name.
	 * @param value The header value.
	 * @return This object.
	 */
	public HttpResourceBuilder<T> header(String name, String value) {
		if (name != null && value != null)
			getHeaders().append(name, value);
		return this;
	}

	/**
	 * Adds the specified headers to the end of the headers in this builder.
	 *
	 * @param values The headers to add.  <jk>null</jk> headers and headers with <jk>null</jk> names or values are ignored.
	 * @return This object.
	 */
	public HttpResourceBuilder<T> headers(Header...values) {
		for (Header h : values) {
			if (h != null) {
				String n = h.getName();
				String v = h.getValue();
				if (isNotEmpty(n)) {
					if (n.equalsIgnoreCase("content-type"))
						contentType(v);
					else if (n.equalsIgnoreCase("content-encoding"))
						contentEncoding(v);
					else if (n.equalsIgnoreCase("content-length"))
						contentLength(Long.parseLong(v));
					else
						getHeaders().append(h);
				}
			}
		}
		return this;
	}

	/**
	 * Adds the specified headers to the end of the headers in this builder.
	 *
	 * @param values The headers to add.  <jk>null</jk> values are ignored.
	 * @return This object.
	 */
	public HttpResourceBuilder<T> headers(List<Header> values) {
		getHeaders().append(values);
		return this;
	}

	//-----------------------------------------------------------------------------------------------------------------
	// Other methods
	//-----------------------------------------------------------------------------------------------------------------

	// <FluentSetters>

	// </FluentSetters>
}
