/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.util.bkd;

import java.io.IOException;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.PointValues;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.MathUtil;
import org.apache.lucene.util.bkd.BKDConfig;
import org.apache.lucene.util.bkd.DocIdsWriter;

public class BKDReader
extends PointValues {
    final BKDConfig config;
    final int numLeaves;
    final IndexInput in;
    final byte[] minPackedValue;
    final byte[] maxPackedValue;
    final long pointCount;
    final int docCount;
    final int version;
    final long minLeafBlockFP;
    private final long indexStartPointer;
    private final int numIndexBytes;
    private final IndexInput indexIn;
    private final boolean isTreeBalanced;

    public BKDReader(IndexInput metaIn, IndexInput indexIn, IndexInput dataIn) throws IOException {
        this.version = CodecUtil.checkHeader(metaIn, "BKD", 4, 9);
        int numDims = metaIn.readVInt();
        int numIndexDims = this.version >= 6 ? metaIn.readVInt() : numDims;
        int maxPointsInLeafNode = metaIn.readVInt();
        int bytesPerDim = metaIn.readVInt();
        this.config = new BKDConfig(numDims, numIndexDims, bytesPerDim, maxPointsInLeafNode);
        this.numLeaves = metaIn.readVInt();
        assert (this.numLeaves > 0);
        this.minPackedValue = new byte[this.config.packedIndexBytesLength];
        this.maxPackedValue = new byte[this.config.packedIndexBytesLength];
        metaIn.readBytes(this.minPackedValue, 0, this.config.packedIndexBytesLength);
        metaIn.readBytes(this.maxPackedValue, 0, this.config.packedIndexBytesLength);
        ArrayUtil.ByteArrayComparator comparator = ArrayUtil.getUnsignedComparator(this.config.bytesPerDim);
        for (int dim = 0; dim < this.config.numIndexDims; ++dim) {
            if (comparator.compare(this.minPackedValue, dim * this.config.bytesPerDim, this.maxPackedValue, dim * this.config.bytesPerDim) <= 0) continue;
            throw new CorruptIndexException("minPackedValue " + new BytesRef(this.minPackedValue) + " is > maxPackedValue " + new BytesRef(this.maxPackedValue) + " for dim=" + dim, metaIn);
        }
        this.pointCount = metaIn.readVLong();
        this.docCount = metaIn.readVInt();
        this.numIndexBytes = metaIn.readVInt();
        if (this.version >= 9) {
            this.minLeafBlockFP = metaIn.readLong();
            this.indexStartPointer = metaIn.readLong();
        } else {
            this.indexStartPointer = indexIn.getFilePointer();
            this.minLeafBlockFP = indexIn.readVLong();
            indexIn.seek(this.indexStartPointer);
        }
        this.indexIn = indexIn;
        this.in = dataIn;
        this.isTreeBalanced = this.numLeaves != 1 && this.isTreeBalanced();
    }

    private boolean isTreeBalanced() throws IOException {
        if (this.version >= 9) {
            return false;
        }
        if (this.config.numDims > 1) {
            assert (1 << MathUtil.log(this.numLeaves, 2) == this.numLeaves);
            return true;
        }
        if (1 << MathUtil.log(this.numLeaves, 2) != this.numLeaves) {
            return false;
        }
        int lastLeafNodePointCount = Math.toIntExact(this.pointCount % (long)this.config.maxPointsInLeafNode);
        PointValues.PointTree pointTree = this.getPointTree();
        while (pointTree.moveToSibling() || pointTree.moveToChild()) {
        }
        final int[] count = new int[]{0};
        pointTree.visitDocIDs(new PointValues.IntersectVisitor(){

            @Override
            public void visit(int docID) {
                count[0] = count[0] + 1;
            }

            @Override
            public void visit(int docID, byte[] packedValue) {
                throw new AssertionError();
            }

            @Override
            public PointValues.Relation compare(byte[] minPackedValue, byte[] maxPackedValue) {
                throw new AssertionError();
            }
        });
        return count[0] != lastLeafNodePointCount;
    }

    @Override
    public PointValues.PointTree getPointTree() throws IOException {
        return new BKDPointTree(this.indexIn.slice("packedIndex", this.indexStartPointer, this.numIndexBytes), this.in.clone(), this.config, this.numLeaves, this.version, this.pointCount, this.minPackedValue, this.maxPackedValue, this.isTreeBalanced);
    }

    @Override
    public byte[] getMinPackedValue() {
        return (byte[])this.minPackedValue.clone();
    }

    @Override
    public byte[] getMaxPackedValue() {
        return (byte[])this.maxPackedValue.clone();
    }

    @Override
    public int getNumDimensions() throws IOException {
        return this.config.numDims;
    }

    @Override
    public int getNumIndexDimensions() throws IOException {
        return this.config.numIndexDims;
    }

    @Override
    public int getBytesPerDimension() throws IOException {
        return this.config.bytesPerDim;
    }

    @Override
    public long size() {
        return this.pointCount;
    }

    @Override
    public int getDocCount() {
        return this.docCount;
    }

    private static class BKDReaderDocIDSetIterator
    extends DocIdSetIterator {
        private int idx;
        private int length;
        private int offset;
        private int docID;
        final int[] docIDs;
        private final DocIdsWriter docIdsWriter;

        public BKDReaderDocIDSetIterator(int maxPointsInLeafNode) {
            this.docIDs = new int[maxPointsInLeafNode];
            this.docIdsWriter = new DocIdsWriter(maxPointsInLeafNode);
        }

        @Override
        public int docID() {
            return this.docID;
        }

        private void reset(int offset, int length) {
            this.offset = offset;
            this.length = length;
            assert (offset + length <= this.docIDs.length);
            this.docID = -1;
            this.idx = 0;
        }

        @Override
        public int nextDoc() throws IOException {
            if (this.idx == this.length) {
                this.docID = Integer.MAX_VALUE;
            } else {
                this.docID = this.docIDs[this.offset + this.idx];
                ++this.idx;
            }
            return this.docID;
        }

        @Override
        public int advance(int target) throws IOException {
            return this.slowAdvance(target);
        }

        @Override
        public long cost() {
            return this.length;
        }
    }

    private static class BKDPointTree
    implements PointValues.PointTree {
        private int nodeID;
        private final int nodeRoot;
        private int level;
        private final IndexInput innerNodes;
        private final IndexInput leafNodes;
        private final long[] leafBlockFPStack;
        private final int[] readNodeDataPositions;
        private final int[] rightNodePositions;
        private final int[] splitDimsPos;
        private final boolean[] negativeDeltas;
        private final byte[][] splitValuesStack;
        private final byte[] minPackedValue;
        private final byte[] maxPackedValue;
        private final byte[][] splitDimValueStack;
        private final BKDConfig config;
        private final int leafNodeOffset;
        private final int version;
        final long pointCount;
        private final int lastLeafNodePointCount;
        private final int rightMostLeafNode;
        private final byte[] scratchDataPackedValue;
        private final byte[] scratchMinIndexPackedValue;
        private final byte[] scratchMaxIndexPackedValue;
        private final int[] commonPrefixLengths;
        private final BKDReaderDocIDSetIterator scratchIterator;
        private final DocIdsWriter docIdsWriter;
        private final boolean isTreeBalanced;

        private BKDPointTree(IndexInput innerNodes, IndexInput leafNodes, BKDConfig config, int numLeaves, int version2, long pointCount, byte[] minPackedValue, byte[] maxPackedValue, boolean isTreeBalanced) throws IOException {
            this(innerNodes, leafNodes, config, numLeaves, version2, pointCount, 1, 1, minPackedValue, maxPackedValue, new BKDReaderDocIDSetIterator(config.maxPointsInLeafNode), new byte[config.packedBytesLength], new byte[config.packedIndexBytesLength], new byte[config.packedIndexBytesLength], new int[config.numDims], isTreeBalanced);
            this.readNodeData(false);
        }

        private BKDPointTree(IndexInput innerNodes, IndexInput leafNodes, BKDConfig config, int numLeaves, int version2, long pointCount, int nodeID, int level, byte[] minPackedValue, byte[] maxPackedValue, BKDReaderDocIDSetIterator scratchIterator, byte[] scratchDataPackedValue, byte[] scratchMinIndexPackedValue, byte[] scratchMaxIndexPackedValue, int[] commonPrefixLengths, boolean isTreeBalanced) {
            this.config = config;
            this.version = version2;
            this.nodeID = nodeID;
            this.nodeRoot = nodeID;
            this.level = level;
            this.isTreeBalanced = isTreeBalanced;
            this.leafNodeOffset = numLeaves;
            this.innerNodes = innerNodes;
            this.leafNodes = leafNodes;
            this.minPackedValue = (byte[])minPackedValue.clone();
            this.maxPackedValue = (byte[])maxPackedValue.clone();
            int treeDepth = this.getTreeDepth(numLeaves);
            this.splitDimValueStack = new byte[treeDepth][];
            this.splitValuesStack = new byte[treeDepth][];
            this.splitValuesStack[0] = new byte[config.packedIndexBytesLength];
            this.leafBlockFPStack = new long[treeDepth + 1];
            this.readNodeDataPositions = new int[treeDepth + 1];
            this.rightNodePositions = new int[treeDepth];
            this.splitDimsPos = new int[treeDepth];
            this.negativeDeltas = new boolean[config.numIndexDims * treeDepth];
            this.pointCount = pointCount;
            this.rightMostLeafNode = (1 << treeDepth - 1) - 1;
            int lastLeafNodePointCount = Math.toIntExact(pointCount % (long)config.maxPointsInLeafNode);
            this.lastLeafNodePointCount = lastLeafNodePointCount == 0 ? config.maxPointsInLeafNode : lastLeafNodePointCount;
            this.scratchIterator = scratchIterator;
            this.commonPrefixLengths = commonPrefixLengths;
            this.scratchDataPackedValue = scratchDataPackedValue;
            this.scratchMinIndexPackedValue = scratchMinIndexPackedValue;
            this.scratchMaxIndexPackedValue = scratchMaxIndexPackedValue;
            this.docIdsWriter = scratchIterator.docIdsWriter;
        }

        @Override
        public PointValues.PointTree clone() {
            BKDPointTree index = new BKDPointTree(this.innerNodes.clone(), this.leafNodes.clone(), this.config, this.leafNodeOffset, this.version, this.pointCount, this.nodeID, this.level, this.minPackedValue, this.maxPackedValue, this.scratchIterator, this.scratchDataPackedValue, this.scratchMinIndexPackedValue, this.scratchMaxIndexPackedValue, this.commonPrefixLengths, this.isTreeBalanced);
            index.leafBlockFPStack[index.level] = this.leafBlockFPStack[this.level];
            if (!this.isLeafNode()) {
                index.rightNodePositions[index.level] = this.rightNodePositions[this.level];
                index.readNodeDataPositions[index.level] = this.readNodeDataPositions[this.level];
                index.splitValuesStack[index.level] = (byte[])this.splitValuesStack[this.level].clone();
                System.arraycopy(this.negativeDeltas, this.level * this.config.numIndexDims, index.negativeDeltas, this.level * this.config.numIndexDims, this.config.numIndexDims);
                index.splitDimsPos[this.level] = this.splitDimsPos[this.level];
            }
            return index;
        }

        @Override
        public byte[] getMinPackedValue() {
            return this.minPackedValue;
        }

        @Override
        public byte[] getMaxPackedValue() {
            return this.maxPackedValue;
        }

        @Override
        public boolean moveToChild() throws IOException {
            if (this.isLeafNode()) {
                return false;
            }
            this.resetNodeDataPosition();
            this.pushBoundsLeft();
            this.pushLeft();
            return true;
        }

        private void resetNodeDataPosition() throws IOException {
            assert ((long)this.readNodeDataPositions[this.level] <= this.innerNodes.getFilePointer());
            this.innerNodes.seek(this.readNodeDataPositions[this.level]);
        }

        private void pushBoundsLeft() {
            int splitDimPos = this.splitDimsPos[this.level];
            if (this.splitDimValueStack[this.level] == null) {
                this.splitDimValueStack[this.level] = new byte[this.config.bytesPerDim];
            }
            System.arraycopy(this.maxPackedValue, splitDimPos, this.splitDimValueStack[this.level], 0, this.config.bytesPerDim);
            assert (ArrayUtil.getUnsignedComparator(this.config.bytesPerDim).compare(this.maxPackedValue, splitDimPos, this.splitValuesStack[this.level], splitDimPos) >= 0) : "config.bytesPerDim=" + this.config.bytesPerDim + " splitDimPos=" + this.splitDimsPos[this.level] + " config.numIndexDims=" + this.config.numIndexDims + " config.numDims=" + this.config.numDims;
            System.arraycopy(this.splitValuesStack[this.level], splitDimPos, this.maxPackedValue, splitDimPos, this.config.bytesPerDim);
        }

        private void pushLeft() throws IOException {
            this.nodeID *= 2;
            ++this.level;
            this.readNodeData(true);
        }

        private void pushBoundsRight() {
            int splitDimPos = this.splitDimsPos[this.level];
            assert (this.splitDimValueStack[this.level] != null);
            System.arraycopy(this.minPackedValue, splitDimPos, this.splitDimValueStack[this.level], 0, this.config.bytesPerDim);
            assert (ArrayUtil.getUnsignedComparator(this.config.bytesPerDim).compare(this.minPackedValue, splitDimPos, this.splitValuesStack[this.level], splitDimPos) <= 0) : "config.bytesPerDim=" + this.config.bytesPerDim + " splitDimPos=" + this.splitDimsPos[this.level] + " config.numIndexDims=" + this.config.numIndexDims + " config.numDims=" + this.config.numDims;
            System.arraycopy(this.splitValuesStack[this.level], splitDimPos, this.minPackedValue, splitDimPos, this.config.bytesPerDim);
        }

        private void pushRight() throws IOException {
            int nodePosition = this.rightNodePositions[this.level];
            assert ((long)nodePosition >= this.innerNodes.getFilePointer()) : "nodePosition = " + nodePosition + " < currentPosition=" + this.innerNodes.getFilePointer();
            this.innerNodes.seek(nodePosition);
            this.nodeID = 2 * this.nodeID + 1;
            ++this.level;
            this.readNodeData(false);
        }

        @Override
        public boolean moveToSibling() throws IOException {
            if (!this.isLeftNode() || this.isRootNode()) {
                return false;
            }
            this.pop();
            this.popBounds(this.maxPackedValue);
            this.pushBoundsRight();
            this.pushRight();
            assert (this.nodeExists());
            return true;
        }

        private void pop() {
            this.nodeID /= 2;
            --this.level;
        }

        private void popBounds(byte[] packedValue) {
            System.arraycopy(this.splitDimValueStack[this.level], 0, packedValue, this.splitDimsPos[this.level], this.config.bytesPerDim);
        }

        @Override
        public boolean moveToParent() {
            if (this.isRootNode()) {
                return false;
            }
            byte[] packedValue = this.isLeftNode() ? this.maxPackedValue : this.minPackedValue;
            this.pop();
            this.popBounds(packedValue);
            return true;
        }

        private boolean isRootNode() {
            return this.nodeID == this.nodeRoot;
        }

        private boolean isLeftNode() {
            return (this.nodeID & 1) == 0;
        }

        private boolean isLeafNode() {
            return this.nodeID >= this.leafNodeOffset;
        }

        private boolean nodeExists() {
            return this.nodeID - this.leafNodeOffset < this.leafNodeOffset;
        }

        private long getLeafBlockFP() {
            assert (this.isLeafNode()) : "nodeID=" + this.nodeID + " is not a leaf";
            return this.leafBlockFPStack[this.level];
        }

        @Override
        public long size() {
            int leftMostLeafNode;
            for (leftMostLeafNode = this.nodeID; leftMostLeafNode < this.leafNodeOffset; leftMostLeafNode *= 2) {
            }
            int rightMostLeafNode = this.nodeID;
            while (rightMostLeafNode < this.leafNodeOffset) {
                rightMostLeafNode = rightMostLeafNode * 2 + 1;
            }
            int numLeaves = rightMostLeafNode >= leftMostLeafNode ? rightMostLeafNode - leftMostLeafNode + 1 : rightMostLeafNode - leftMostLeafNode + 1 + this.leafNodeOffset;
            assert (numLeaves == this.getNumLeavesSlow(this.nodeID)) : numLeaves + " " + this.getNumLeavesSlow(this.nodeID);
            if (this.isTreeBalanced) {
                return this.sizeFromBalancedTree(leftMostLeafNode, rightMostLeafNode);
            }
            return rightMostLeafNode == this.rightMostLeafNode ? (long)(numLeaves - 1) * (long)this.config.maxPointsInLeafNode + (long)this.lastLeafNodePointCount : (long)numLeaves * (long)this.config.maxPointsInLeafNode;
        }

        private long sizeFromBalancedTree(int leftMostLeafNode, int rightMostLeafNode) {
            int extraPoints = Math.toIntExact((long)this.config.maxPointsInLeafNode * (long)this.leafNodeOffset - this.pointCount);
            assert (extraPoints < this.leafNodeOffset) : "point excess should be lower than leafNodeOffset";
            int nodeOffset = this.leafNodeOffset - extraPoints;
            long count = 0L;
            for (int node = leftMostLeafNode; node <= rightMostLeafNode; ++node) {
                if (this.balanceTreeNodePosition(0, this.leafNodeOffset, node - this.leafNodeOffset, 0, 0) < nodeOffset) {
                    count += (long)this.config.maxPointsInLeafNode;
                    continue;
                }
                count += (long)(this.config.maxPointsInLeafNode - 1);
            }
            return count;
        }

        private int balanceTreeNodePosition(int minNode, int maxNode, int node, int position, int level) {
            if (maxNode - minNode == 1) {
                return position;
            }
            int mid = minNode + maxNode + 1 >>> 1;
            if (mid > node) {
                return this.balanceTreeNodePosition(minNode, mid, node, position, level + 1);
            }
            return this.balanceTreeNodePosition(mid, maxNode, node, position + (1 << level), level + 1);
        }

        @Override
        public void visitDocIDs(PointValues.IntersectVisitor visitor) throws IOException {
            this.resetNodeDataPosition();
            this.addAll(visitor, false);
        }

        public void addAll(PointValues.IntersectVisitor visitor, boolean grown) throws IOException {
            long size;
            if (!grown && (size = this.size()) <= Integer.MAX_VALUE) {
                visitor.grow((int)size);
                grown = true;
            }
            if (this.isLeafNode()) {
                this.leafNodes.seek(this.getLeafBlockFP());
                int count = this.leafNodes.readVInt();
                this.docIdsWriter.readInts(this.leafNodes, count, visitor);
            } else {
                this.pushLeft();
                this.addAll(visitor, grown);
                this.pop();
                this.pushRight();
                this.addAll(visitor, grown);
                this.pop();
            }
        }

        @Override
        public void visitDocValues(PointValues.IntersectVisitor visitor) throws IOException {
            this.resetNodeDataPosition();
            this.visitLeavesOneByOne(visitor);
        }

        private void visitLeavesOneByOne(PointValues.IntersectVisitor visitor) throws IOException {
            if (this.isLeafNode()) {
                this.visitDocValues(visitor, this.getLeafBlockFP());
            } else {
                this.pushLeft();
                this.visitLeavesOneByOne(visitor);
                this.pop();
                this.pushRight();
                this.visitLeavesOneByOne(visitor);
                this.pop();
            }
        }

        private void visitDocValues(PointValues.IntersectVisitor visitor, long fp) throws IOException {
            int count = this.readDocIDs(this.leafNodes, fp, this.scratchIterator);
            if (this.version >= 7) {
                this.visitDocValuesWithCardinality(this.commonPrefixLengths, this.scratchDataPackedValue, this.scratchMinIndexPackedValue, this.scratchMaxIndexPackedValue, this.leafNodes, this.scratchIterator, count, visitor);
            } else {
                this.visitDocValuesNoCardinality(this.commonPrefixLengths, this.scratchDataPackedValue, this.scratchMinIndexPackedValue, this.scratchMaxIndexPackedValue, this.leafNodes, this.scratchIterator, count, visitor);
            }
        }

        private int readDocIDs(IndexInput in, long blockFP, BKDReaderDocIDSetIterator iterator) throws IOException {
            in.seek(blockFP);
            int count = in.readVInt();
            this.docIdsWriter.readInts(in, count, iterator.docIDs);
            return count;
        }

        private int getNumLeavesSlow(int node) {
            if (node >= 2 * this.leafNodeOffset) {
                return 0;
            }
            if (node >= this.leafNodeOffset) {
                return 1;
            }
            int leftCount = this.getNumLeavesSlow(node * 2);
            int rightCount = this.getNumLeavesSlow(node * 2 + 1);
            return leftCount + rightCount;
        }

        private void readNodeData(boolean isLeft) throws IOException {
            this.leafBlockFPStack[this.level] = this.leafBlockFPStack[this.level - 1];
            if (!isLeft) {
                int n = this.level;
                this.leafBlockFPStack[n] = this.leafBlockFPStack[n] + this.innerNodes.readVLong();
            }
            if (!this.isLeafNode()) {
                System.arraycopy(this.negativeDeltas, (this.level - 1) * this.config.numIndexDims, this.negativeDeltas, this.level * this.config.numIndexDims, this.config.numIndexDims);
                this.negativeDeltas[this.level * this.config.numIndexDims + this.splitDimsPos[this.level - 1] / this.config.bytesPerDim] = isLeft;
                if (this.splitValuesStack[this.level] == null) {
                    this.splitValuesStack[this.level] = (byte[])this.splitValuesStack[this.level - 1].clone();
                } else {
                    System.arraycopy(this.splitValuesStack[this.level - 1], 0, this.splitValuesStack[this.level], 0, this.config.packedIndexBytesLength);
                }
                int code = this.innerNodes.readVInt();
                int splitDim = code % this.config.numIndexDims;
                this.splitDimsPos[this.level] = splitDim * this.config.bytesPerDim;
                int prefix = (code /= this.config.numIndexDims) % (1 + this.config.bytesPerDim);
                int suffix = this.config.bytesPerDim - prefix;
                if (suffix > 0) {
                    int firstDiffByteDelta = code / (1 + this.config.bytesPerDim);
                    if (this.negativeDeltas[this.level * this.config.numIndexDims + splitDim]) {
                        firstDiffByteDelta = -firstDiffByteDelta;
                    }
                    int startPos = this.splitDimsPos[this.level] + prefix;
                    int oldByte = this.splitValuesStack[this.level][startPos] & 0xFF;
                    this.splitValuesStack[this.level][startPos] = (byte)(oldByte + firstDiffByteDelta);
                    this.innerNodes.readBytes(this.splitValuesStack[this.level], startPos + 1, suffix - 1);
                }
                int leftNumBytes = this.nodeID * 2 < this.leafNodeOffset ? this.innerNodes.readVInt() : 0;
                this.rightNodePositions[this.level] = Math.toIntExact(this.innerNodes.getFilePointer()) + leftNumBytes;
                this.readNodeDataPositions[this.level] = Math.toIntExact(this.innerNodes.getFilePointer());
            }
        }

        private int getTreeDepth(int numLeaves) {
            return MathUtil.log(numLeaves, 2) + 2;
        }

        private void visitDocValuesNoCardinality(int[] commonPrefixLengths, byte[] scratchDataPackedValue, byte[] scratchMinIndexPackedValue, byte[] scratchMaxIndexPackedValue, IndexInput in, BKDReaderDocIDSetIterator scratchIterator, int count, PointValues.IntersectVisitor visitor) throws IOException {
            this.readCommonPrefixes(commonPrefixLengths, scratchDataPackedValue, in);
            if (this.config.numIndexDims != 1 && this.version >= 5) {
                byte[] minPackedValue = scratchMinIndexPackedValue;
                System.arraycopy(scratchDataPackedValue, 0, minPackedValue, 0, this.config.packedIndexBytesLength);
                byte[] maxPackedValue = scratchMaxIndexPackedValue;
                System.arraycopy(minPackedValue, 0, maxPackedValue, 0, this.config.packedIndexBytesLength);
                this.readMinMax(commonPrefixLengths, minPackedValue, maxPackedValue, in);
                PointValues.Relation r = visitor.compare(minPackedValue, maxPackedValue);
                if (r == PointValues.Relation.CELL_OUTSIDE_QUERY) {
                    return;
                }
                visitor.grow(count);
                if (r == PointValues.Relation.CELL_INSIDE_QUERY) {
                    for (int i = 0; i < count; ++i) {
                        visitor.visit(scratchIterator.docIDs[i]);
                    }
                    return;
                }
            } else {
                visitor.grow(count);
            }
            int compressedDim = this.readCompressedDim(in);
            if (compressedDim == -1) {
                this.visitUniqueRawDocValues(scratchDataPackedValue, scratchIterator, count, visitor);
            } else {
                this.visitCompressedDocValues(commonPrefixLengths, scratchDataPackedValue, in, scratchIterator, count, visitor, compressedDim);
            }
        }

        private void visitDocValuesWithCardinality(int[] commonPrefixLengths, byte[] scratchDataPackedValue, byte[] scratchMinIndexPackedValue, byte[] scratchMaxIndexPackedValue, IndexInput in, BKDReaderDocIDSetIterator scratchIterator, int count, PointValues.IntersectVisitor visitor) throws IOException {
            this.readCommonPrefixes(commonPrefixLengths, scratchDataPackedValue, in);
            int compressedDim = this.readCompressedDim(in);
            if (compressedDim == -1) {
                visitor.grow(count);
                this.visitUniqueRawDocValues(scratchDataPackedValue, scratchIterator, count, visitor);
            } else {
                if (this.config.numIndexDims != 1) {
                    byte[] minPackedValue = scratchMinIndexPackedValue;
                    System.arraycopy(scratchDataPackedValue, 0, minPackedValue, 0, this.config.packedIndexBytesLength);
                    byte[] maxPackedValue = scratchMaxIndexPackedValue;
                    System.arraycopy(minPackedValue, 0, maxPackedValue, 0, this.config.packedIndexBytesLength);
                    this.readMinMax(commonPrefixLengths, minPackedValue, maxPackedValue, in);
                    PointValues.Relation r = visitor.compare(minPackedValue, maxPackedValue);
                    if (r == PointValues.Relation.CELL_OUTSIDE_QUERY) {
                        return;
                    }
                    visitor.grow(count);
                    if (r == PointValues.Relation.CELL_INSIDE_QUERY) {
                        for (int i = 0; i < count; ++i) {
                            visitor.visit(scratchIterator.docIDs[i]);
                        }
                        return;
                    }
                } else {
                    visitor.grow(count);
                }
                if (compressedDim == -2) {
                    this.visitSparseRawDocValues(commonPrefixLengths, scratchDataPackedValue, in, scratchIterator, count, visitor);
                } else {
                    this.visitCompressedDocValues(commonPrefixLengths, scratchDataPackedValue, in, scratchIterator, count, visitor, compressedDim);
                }
            }
        }

        private void readMinMax(int[] commonPrefixLengths, byte[] minPackedValue, byte[] maxPackedValue, IndexInput in) throws IOException {
            for (int dim = 0; dim < this.config.numIndexDims; ++dim) {
                int prefix = commonPrefixLengths[dim];
                in.readBytes(minPackedValue, dim * this.config.bytesPerDim + prefix, this.config.bytesPerDim - prefix);
                in.readBytes(maxPackedValue, dim * this.config.bytesPerDim + prefix, this.config.bytesPerDim - prefix);
            }
        }

        private void visitSparseRawDocValues(int[] commonPrefixLengths, byte[] scratchPackedValue, IndexInput in, BKDReaderDocIDSetIterator scratchIterator, int count, PointValues.IntersectVisitor visitor) throws IOException {
            int i;
            int length;
            for (i = 0; i < count; i += length) {
                length = in.readVInt();
                for (int dim = 0; dim < this.config.numDims; ++dim) {
                    int prefix = commonPrefixLengths[dim];
                    in.readBytes(scratchPackedValue, dim * this.config.bytesPerDim + prefix, this.config.bytesPerDim - prefix);
                }
                scratchIterator.reset(i, length);
                visitor.visit(scratchIterator, scratchPackedValue);
            }
            if (i != count) {
                throw new CorruptIndexException("Sub blocks do not add up to the expected count: " + count + " != " + i, in);
            }
        }

        private void visitUniqueRawDocValues(byte[] scratchPackedValue, BKDReaderDocIDSetIterator scratchIterator, int count, PointValues.IntersectVisitor visitor) throws IOException {
            scratchIterator.reset(0, count);
            visitor.visit(scratchIterator, scratchPackedValue);
        }

        private void visitCompressedDocValues(int[] commonPrefixLengths, byte[] scratchPackedValue, IndexInput in, BKDReaderDocIDSetIterator scratchIterator, int count, PointValues.IntersectVisitor visitor, int compressedDim) throws IOException {
            int i;
            int runLen;
            int compressedByteOffset = compressedDim * this.config.bytesPerDim + commonPrefixLengths[compressedDim];
            int n = compressedDim;
            commonPrefixLengths[n] = commonPrefixLengths[n] + 1;
            for (i = 0; i < count; i += runLen) {
                scratchPackedValue[compressedByteOffset] = in.readByte();
                runLen = Byte.toUnsignedInt(in.readByte());
                for (int j = 0; j < runLen; ++j) {
                    for (int dim = 0; dim < this.config.numDims; ++dim) {
                        int prefix = commonPrefixLengths[dim];
                        in.readBytes(scratchPackedValue, dim * this.config.bytesPerDim + prefix, this.config.bytesPerDim - prefix);
                    }
                    visitor.visit(scratchIterator.docIDs[i + j], scratchPackedValue);
                }
            }
            if (i != count) {
                throw new CorruptIndexException("Sub blocks do not add up to the expected count: " + count + " != " + i, in);
            }
        }

        private int readCompressedDim(IndexInput in) throws IOException {
            byte compressedDim = in.readByte();
            if (compressedDim < -2 || compressedDim >= this.config.numDims || this.version < 7 && compressedDim == -2) {
                throw new CorruptIndexException("Got compressedDim=" + compressedDim, in);
            }
            return compressedDim;
        }

        private void readCommonPrefixes(int[] commonPrefixLengths, byte[] scratchPackedValue, IndexInput in) throws IOException {
            for (int dim = 0; dim < this.config.numDims; ++dim) {
                int prefix;
                commonPrefixLengths[dim] = prefix = in.readVInt();
                if (prefix <= 0) continue;
                in.readBytes(scratchPackedValue, dim * this.config.bytesPerDim, prefix);
            }
        }

        public String toString() {
            return "nodeID=" + this.nodeID;
        }
    }
}

