/// Licensed to the Apache Software Foundation (ASF) under one
/// or more contributor license agreements.  See the NOTICE file
/// distributed with this work for additional information
/// regarding copyright ownership.  The ASF licenses this file
/// to you under the Apache License, Version 2.0 (the
/// "License"); you may not use this file except in compliance
/// with the License.  You may obtain a copy of the License at
///
///   http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing,
/// software distributed under the License is distributed on an
/// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
/// KIND, either express or implied.  See the License for the
/// specific language governing permissions and limitations
/// under the License.

#pragma once

#include <cstdint>
#include <limits>
#include <vector>

namespace impala {

/// The following are pairs of plain and delta encoded versions of datasets. Most of them
/// were generated by parquet-mr to test the correctness and compatibility.
const std::vector<int32_t> header_only_data = {1};
const std::vector<uint8_t> header_only_expected = {
  0x80,0x01 /* 128 - block size */,
  0x04 /* 4 - miniblocks in block*/,
  0x01 /* 1 - total value count */,
  0x02 /* 2 - first value (ZigZag of 1) */};

const std::vector<int32_t> short_block_plain = {2, 4, 8, -10};

const std::vector<uint8_t> short_block_encoded = {0x80, 0x01, 0x04, 0x04, 0x04, 0x23,
  0x05, 0x00, 0x00, 0x00, 0xd4, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

const std::vector<int32_t> block_aligned_plain = {-1155484576, -723955400, 1033096058,
  -1690734402, -1557280266, 1327362106, -1930858313, 502539523, -1728529858, -938301587,
  1431162155, 1085665355, 1654374947, -1661998771, -65105105, -73789608, -518907128,
  99135751, -252332814, 755814641, 1180918287, 1344049776, 553609048, 1580443894,
  629649304, -1266264776, 99807007, 5955764, -1946737912, 39620447, -152527805,
  -1877116806, 448784075, 1086124775, -1609984092, 1227951724, 1764356251, 64111306,
  -960731419, -100082026, -39845375, -1339022546, 2092649110, -568315836, -1089884900,
  -81839914, -1146103148, -1846688624, -784703072, 55004124, -691960657, 1770461755,
  -2032810463, -1177788003, -432352882, -65824064, 575267217, -1949367821, 356750287,
  798819494, -92022521, 1318001577, -1192467086, -1412716779, -1223932479, 276053035,
  615126903, 1542603436, 1988388716, 1177882237, 19265476, -1430871151, 307082914,
  -1333570194, 1496453452, -790542135, 1455004595, -1690249972, -604059026, -290476856,
  -122204761, -1097539750, -576617062, -2002123957, -1663951485, 193034304, 768747578,
  1337360486, 934457803, 873612482, -624972850, 355564760, 41547336, 1781447028,
  -1321591463, 1081281446, -982203381, -222544851, -1233998085, -1331702554, 907249073,
  206843130, -931610657, 1697401307, -234791961, -996341162, 61144662, -1638496702,
  1693091376, -1667947369, -627963836, 105673451, -918940699, -300903254, -28037702,
  1358495114, -502062723, 2056481282, 731396580, 617380302, -162912754, 818693069,
  -1184143835, -410943565, -1385753324, 1233308008, -840115328, -1474241772, 1895913584,
  1376826901, 1984652857, 412901507, -632101198, 2025024491, -2139603013, -1233011876,
  -34604646, -1854979571, -131984708, -1866233577, 1516579035, 1612513822, 203005342,
  -911714850, 307621155, 13453556, 125015832, 483675743, 2077347671, 37011083, -120473323,
  2069609186, -46741469, 1502566715, -998828580, -1206282339, -2141482063, -1550311379,
  1097472880, 1449582151, 1327824311, -1105846005, -651630279, -1174121238, 223701431,
  810978550, 43703267, 835843581, 1519786882, -772329283, 372536197, -317962831,
  -642907139, -706495414, 15792134, -1116133237, 1322391079, -1231319713, -2011725892,
  1299386162, -348690342, 1341976962, 1190607957, -262501494, -540039728, -986836170,
  -1675546118, -1948015872, -392391576, 638256747, 191085745, -97160665, -1517309270,
  1375971382, 2133755906, 891811204, -2118700536, 2110473668, -2058625827, 861211557,
  1573507430, 1625767564, 2051434938, -1922104200, -1271440420, 927536580, 1386277121,
  1184911833, 50057146, -185358571, -1284027987, -1025692308, -1093701172, -1116573039,
  -1686684117, 2068400044, 1009668318, -1847249948, 1670927751, 408998980, -380240510,
  1439774921, 718930634, -1800042595, 16808464, -1743967552, 1137718941, 252008887,
  -460347667, -1053605891, 1278046001, 947528252, 1842130704, 1599145891, -1341955486,
  1631478226, 1754478786, -1370798799, 1516348954, 730123440, -1409786204, 1491046204,
  -1114899725, -656053644, -453397258, -694901410, -1804067552, 13393714, -2101289195,
  176533870, 1339899789, -1511804464, -1974699707, -1660688399, -115501757, 86156176,
  -1659413839, 1775057871, 116690193, -524129043, -1335659601, -1387390647, -1587895448,
  1291940779, -782382009, -129016790, -1111888250, 1369132585, -1582995723, 1775301948,
  1352452311, -898816007, -1072231506, 98890493, 816987543, 1763924914, 526139168,
  -312090222, -1231414341, 977688068, -287577538, -1465993732, 1336202001, -1765140980,
  -2020641285, 95008905, 768704422, -1985683736, 489119572, 1303323276, -1357461657,
  -869863794, 1628866452, -779918512, -1750503926, 219565529, -255081261, 1102069833,
  -2000829650, 187633124, 919959231, -1628224372, 1433358016, -2066119013, 1076044702,
  837759872, -1135845471, 1374513806, 391471507, 980775637, 1136222341, 1920058376,
  810323999, 1940365237, 1278875422, 330772320, -1037359638, 562086714, 200094072,
  -1255479666, 369661635, -782706145, -462895706, -1256820224, -1998711323, -252927047,
  367830942, -1133964626, -2012825476, -1180767193, 315210081, 753219948, 759217255,
  -146291256, -500459170, -539123842, 1505097534, -298228243, -60533992, 38982769,
  -1821527659, 174692559, -1699744250, -1761913560, 1172992642, 1026162883, -2090846535,
  1556971572, -237338788, -1631021909, 935231044, 1514404713, -387312699, 1628770735,
  83168215, -1073062227, 1177720299, -1503701971, -1422521171, -230082541, 1236421386,
  -514995219, 274845224, 1537588519, 971788340, -332759143, 601583161, -646267509,
  -1702306988, -1388817557, -1169314299, -1961514302, 1305086064, 2022278952, -1609053499,
  521914699, 153728813, 971743052, 2050261229, 1072226033, 398015640, 168570246,
  -1242849904, -352718239, 1864625443, 473771110, -775896574, -1785991171, -1698577684,
  -713629100, -1407697474, 1522485231, 1933026373, 1956675524, 1479877258, -1156654721,
  1157041776, -1895306073, -1094584616, -218513495, 885467306, 92567844, 642230895,
  -1264266518, -341998742, 910648711, -1341345462, 1194456154, -357758998, 2014868857,
  152070336, 869823999, 838196198, 1022100081, -1821037852, -844417953, -150389341,
  648233322, 1424864302, -357681834, 593534002, 1755230166, 417906522, 212262954,
  -774410251, -1093280814, -1417675888, 1261380316, 1842059106, 1051130778, 2061353156,
  1883576512, -731890682, 834930750, -1977670049, -1709832625, -1784970291, -540389483,
  -1729208290, 559726140, 1607963264, -1301499534, 7046458, -1734818876, -1333551218,
  -102237163, -1229445687, 652525130, 297411598, -151249948, -2133044636, -829943262,
  -1325570812, -602140069, -855508472, -188173652, -1195647518, -1942886337, 1466810945,
  -1421901403, 273829714, -915206375, 1500167556, 1279446969, -2057734606, -845879838,
  -187653156, 1226822235, -1318474016, 1568167162, 522351049, -82500903, -2124478282,
  6206815, -423784092, -1668441430, 718048728, 891909914, 1500508615, -1740889846,
  -1309293780, 575224572, 139309227, 1018052040, 1116841129, 370693002, 1996831313,
  -705395362, 1745527253, 950526261, 796591014, -1050856894, 1260609160, 2093350350,
  -1822145887, 1647214459, 723532789, 1333177144, 1596606704, -418146186, 1007807978,
  -763332258, -1982438190, -1748487655, 1524473942, -481749145, 1331719238, 1186734360,
  -698126068, -1974647768, -1413982619, 1948877192, 468468166, 608358874, -276461812,
  1957512113, -535550357, 195735820, -635653955, 1578840615, 1693905812, 1535391450,
  1624731846, -951859941, -2034716340, -1590554017, 17238435, 1274238702, 148633111,
  -1078680217, -915546208, -623407610, 1165258293, -1738230679, -1195271230, -1066332187,
  1801720615, 815047091, 1248991463, 637849975, 1052104562, 1596827623, 537886508,
  -421422934, 638634658, -1530804315, 1449931150, -1769766076, -697632367, -1293788023,
  -667061805, -798447888, 1121160937, -1877570216, -1631068502, 1007099545, 2107635686,
  -473166987, 1783836689, 1971722490, -624946848, 743429529, -772614666, 2078325877,
  213491976, -947654206, 1122458913, -1473055708, -687104271, 1491227244, 1840692816,
  626883118, -1111616686, 866933502, -531104861, -1808813221, 1256950921, -1171671161,
  -72306630, 68890601, -732073629, -1043301463, -982944228, -1000479329, 1562686362,
  -614744875, 671630938, 986543907, 1168942750, 745872312, 1083062131, -448848760,
  6160024, 1508438260, 1153973754, -970129109, 1036270507, -861662957, 509137782,
  -1510790466, -913832580, -1942660168, -159733880, 84622757, 901069556, -1636005354,
  -337985709, 709866112, 199998020, -1462570842, -118839529, 130306312, 1753621801,
  -1164826604, -2081318369, 2045445150, 45229425, 782750073, 623756509, -1232512223,
  -634784323, 1629809103, 1086653778, 959728767, 938947207, 833249441, 609668531,
  -1824715378, -811387034, -1954814462, 1358368073, -1024257124, -1984328560, 991765439,
  -2072755304, -119762364, -1950013280, 1995232727, 1581851075, 1774717699};

const std::vector<uint8_t> block_aligned_encoded = {0x80, 0x01, 0x04, 0x80, 0x05, 0xbf,
  0xae, 0xfa, 0xcd, 0x08, 0xcf, 0xb7, 0xaf, 0xb9, 0x0f, 0x20, 0x20, 0x20, 0x20, 0xc0,
  0x8a, 0x4e, 0x95, 0x2a, 0x68, 0x50, 0xe4, 0x2c, 0x93, 0x3b, 0xd9, 0x20, 0x47, 0x8a,
  0x83, 0x2c, 0x14, 0x86, 0x27, 0x65, 0x6e, 0x61, 0xb9, 0x34, 0xa2, 0xa0, 0x0c, 0x23,
  0x83, 0x9a, 0xf6, 0x17, 0xdb, 0xaf, 0xaa, 0xa6, 0x13, 0xd1, 0x08, 0x08, 0x11, 0xfe,
  0x66, 0xc0, 0xbf, 0x7b, 0x9d, 0x12, 0x15, 0xea, 0xb5, 0xca, 0x97, 0xc4, 0xda, 0x11,
  0x6a, 0x11, 0x7b, 0x98, 0xf9, 0x0d, 0x61, 0xe7, 0x83, 0x6c, 0xa0, 0xd3, 0xf1, 0xa2,
  0x66, 0xe7, 0x09, 0xad, 0xb7, 0x06, 0x7f, 0xec, 0x94, 0x49, 0x1e, 0x4f, 0x85, 0xd0,
  0xc2, 0x78, 0x4c, 0x86, 0x2f, 0xca, 0xb8, 0x8a, 0xf4, 0xe9, 0x42, 0x88, 0x93, 0x94,
  0x0a, 0xcf, 0x87, 0x02, 0xcd, 0x7d, 0xdf, 0xfd, 0x75, 0x3c, 0x30, 0x32, 0x07, 0x3f,
  0x5a, 0xfb, 0xf1, 0xcc, 0xfa, 0x21, 0x70, 0x1f, 0xca, 0xca, 0x14, 0x39, 0x5c, 0x38,
  0x06, 0x04, 0xfa, 0x92, 0xa1, 0xa5, 0x92, 0xe2, 0xda, 0xb0, 0x64, 0xbd, 0x24, 0x17,
  0xd0, 0x8e, 0x9b, 0x17, 0x40, 0x3e, 0x16, 0x03, 0x12, 0x80, 0x3e, 0x99, 0x65, 0xe2,
  0xae, 0x53, 0x11, 0x2d, 0x7f, 0x15, 0x0f, 0x26, 0x2e, 0x50, 0x15, 0x21, 0x48, 0x96,
  0xd3, 0xfa, 0xdc, 0xc0, 0x6a, 0x7f, 0x5c, 0xa2, 0x79, 0xab, 0xb7, 0xa6, 0x8f, 0x26,
  0x3c, 0xe4, 0xd7, 0xd3, 0x51, 0xf8, 0x8a, 0xe2, 0xba, 0x24, 0xd8, 0xa2, 0xad, 0xbb,
  0x26, 0x10, 0x4f, 0x74, 0x83, 0x5b, 0x0e, 0xce, 0x99, 0xe4, 0x98, 0x64, 0x89, 0x8c,
  0xae, 0xd9, 0x5d, 0x04, 0xa8, 0x1a, 0xb7, 0x6e, 0x91, 0xb9, 0x34, 0xcc, 0xa1, 0x4a,
  0x0e, 0x1b, 0xe5, 0xc4, 0x7f, 0x0a, 0x05, 0xbf, 0x5e, 0xef, 0x95, 0x49, 0xc2, 0x7c,
  0x46, 0x8a, 0x30, 0xa1, 0xcf, 0xb1, 0x37, 0xf3, 0xe5, 0x8b, 0x2f, 0x75, 0x6e, 0x94,
  0x8c, 0xd6, 0x86, 0x52, 0xe4, 0xfd, 0xd4, 0x34, 0xc9, 0xcb, 0x8f, 0x1d, 0x19, 0xde,
  0xb2, 0xa8, 0x12, 0x28, 0x96, 0xf9, 0x94, 0x46, 0x4b, 0x2f, 0xd7, 0x86, 0x36, 0xb5,
  0x99, 0x26, 0x25, 0xf9, 0x00, 0x2d, 0xe3, 0xb4, 0x8c, 0xcb, 0x19, 0xc6, 0xa9, 0x44,
  0x24, 0x65, 0x25, 0x45, 0xf3, 0xd2, 0x40, 0x6e, 0x01, 0x41, 0x27, 0x1d, 0xc0, 0x4a,
  0xe3, 0x53, 0xbc, 0x42, 0xd0, 0x46, 0x8e, 0xc7, 0x8e, 0x9d, 0x85, 0x9b, 0x7f, 0x73,
  0x41, 0x28, 0x94, 0xa2, 0x9a, 0x99, 0x6b, 0x9e, 0x26, 0x20, 0x08, 0xbe, 0x8f, 0xa5,
  0x48, 0x45, 0xea, 0xe2, 0x9d, 0xe6, 0x9d, 0x14, 0x46, 0x7a, 0x9d, 0x4d, 0x1f, 0x92,
  0x63, 0xdf, 0x80, 0xf5, 0x77, 0xf4, 0x54, 0x43, 0x22, 0xf2, 0xbe, 0x07, 0xb6, 0x58,
  0x67, 0xde, 0x68, 0x14, 0xb1, 0x4a, 0xe3, 0xcd, 0x51, 0xa1, 0xc2, 0x35, 0xdc, 0xce,
  0x0a, 0x4d, 0xa6, 0x97, 0x00, 0x0a, 0x66, 0xdd, 0xa8, 0xb6, 0x60, 0x4c, 0x3f, 0xd3,
  0x13, 0xc3, 0x75, 0xb3, 0x9e, 0x09, 0x01, 0x31, 0x95, 0xd6, 0x51, 0xcd, 0x80, 0xba,
  0x37, 0xe4, 0x77, 0x49, 0x18, 0xf4, 0xff, 0x6a, 0x08, 0x57, 0x9c, 0x31, 0x4e, 0xe8,
  0xe1, 0x9d, 0xba, 0xd4, 0x75, 0x47, 0x16, 0xd6, 0xed, 0x29, 0x42, 0x4f, 0x8c, 0x40,
  0xb3, 0x95, 0xd1, 0x92, 0xb9, 0x8f, 0x58, 0x50, 0xa7, 0xe2, 0x8e, 0x83, 0x3e, 0xad,
  0x6e, 0x6c, 0xa0, 0xf8, 0x86, 0xd9, 0x8b, 0xb8, 0xbd, 0x3a, 0xce, 0xdb, 0x11, 0xb0,
  0x0c, 0x6d, 0x36, 0x16, 0x14, 0xca, 0xbd, 0x9a, 0x2c, 0xd2, 0x2d, 0xca, 0x74, 0x28,
  0x9a, 0x13, 0x4d, 0xa7, 0x0b, 0x18, 0xb6, 0x40, 0x10, 0x35, 0x04, 0x76, 0x07, 0xac,
  0xa9, 0x49, 0x83, 0x7b, 0x41, 0x3c, 0xa2, 0xb1, 0x17, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xed, 0xc9, 0x55, 0x44, 0x6b, 0x76, 0x44, 0x97, 0xb8, 0xe3, 0xf8, 0x0f, 0x20, 0x20,
  0x20, 0x20, 0xb1, 0xcb, 0x9b, 0x60, 0x30, 0x1e, 0xc7, 0xa3, 0x56, 0x68, 0xdd, 0x21,
  0x3b, 0xf4, 0x42, 0x41, 0x45, 0xf3, 0xec, 0x1d, 0xdc, 0x41, 0x51, 0x87, 0xad, 0xe9,
  0x95, 0xb5, 0x4a, 0xac, 0xfa, 0xc6, 0x7f, 0xb6, 0x0b, 0x13, 0xbb, 0x3e, 0x3f, 0xe6,
  0x67, 0xe4, 0x2d, 0x18, 0xd0, 0x0d, 0x2e, 0x49, 0x4f, 0x47, 0x44, 0x85, 0x8c, 0x09,
  0x89, 0x2b, 0x4c, 0x26, 0x1b, 0x3d, 0x51, 0x05, 0x3a, 0xc8, 0xdd, 0xc9, 0x03, 0x6e,
  0x30, 0xbc, 0x32, 0x86, 0x53, 0x25, 0xed, 0x94, 0x04, 0xef, 0x89, 0xde, 0x40, 0x5d,
  0xef, 0x05, 0x96, 0x68, 0x29, 0x76, 0xd9, 0x6f, 0x16, 0x02, 0x4d, 0x7b, 0x67, 0x01,
  0x24, 0xff, 0xe4, 0xdb, 0xad, 0x2a, 0x74, 0xea, 0xcd, 0xef, 0x2e, 0x73, 0x20, 0x6a,
  0xce, 0x47, 0x88, 0xfa, 0xc8, 0xa2, 0x4f, 0x69, 0x5e, 0x1d, 0xe3, 0x30, 0x89, 0x94,
  0x7c, 0x8d, 0x4a, 0x78, 0x60, 0x91, 0x7d, 0xee, 0x3a, 0x36, 0x9f, 0x9a, 0xbd, 0xd9,
  0x67, 0x60, 0xd9, 0x82, 0xdd, 0xd2, 0x4b, 0x91, 0x8d, 0xa2, 0xf9, 0xbc, 0xd0, 0x51,
  0x26, 0x88, 0xc3, 0xae, 0x91, 0x93, 0x50, 0xa8, 0x47, 0x83, 0xed, 0xf6, 0xd4, 0xb0,
  0xc9, 0xc3, 0x38, 0x40, 0x64, 0x56, 0x58, 0x2c, 0x2e, 0x6c, 0x59, 0x26, 0xc2, 0x7b,
  0xc8, 0xa9, 0x99, 0xaa, 0x91, 0x9e, 0x14, 0x3c, 0xa8, 0x5b, 0xe5, 0x10, 0x44, 0xe5,
  0x55, 0xe7, 0x69, 0x60, 0x08, 0x51, 0x82, 0xfd, 0xe7, 0x44, 0x34, 0xc7, 0x50, 0x1d,
  0x34, 0xf7, 0x51, 0xe4, 0xdf, 0xb8, 0x86, 0x76, 0x41, 0xbd, 0xef, 0x28, 0x52, 0x88,
  0x01, 0x6f, 0x72, 0xdb, 0xea, 0x64, 0xd0, 0x8c, 0x7f, 0x56, 0x12, 0xdf, 0x4e, 0x6f,
  0x74, 0x5f, 0x45, 0xdc, 0x0f, 0xe0, 0xfa, 0xbc, 0x52, 0x24, 0xe5, 0x64, 0x82, 0x23,
  0x5e, 0x6e, 0x8f, 0xae, 0xe6, 0x2a, 0x98, 0x63, 0x00, 0x2c, 0xd8, 0x4d, 0xb7, 0xac,
  0x8e, 0xdb, 0x85, 0x35, 0x90, 0xaa, 0x1b, 0xcc, 0xc8, 0x81, 0xa0, 0x7b, 0x25, 0x05,
  0x0d, 0x87, 0xd4, 0x9c, 0x95, 0x2d, 0xcd, 0x33, 0x01, 0xaa, 0x32, 0xdb, 0xa9, 0x82,
  0x3a, 0x99, 0xeb, 0x98, 0xca, 0x08, 0xb5, 0x92, 0x70, 0xc5, 0x54, 0xa6, 0xf4, 0x27,
  0x9e, 0x02, 0x49, 0x41, 0xe4, 0x9a, 0xe4, 0xd6, 0x8b, 0x73, 0xed, 0xeb, 0xe7, 0x3b,
  0x67, 0x45, 0x84, 0x71, 0xa4, 0x10, 0x10, 0x3e, 0xcb, 0x51, 0xf2, 0x8e, 0x6c, 0xb2,
  0x7e, 0x7b, 0xd1, 0x6e, 0x2f, 0x7e, 0xa6, 0x39, 0x91, 0x5d, 0x8d, 0x77, 0x5e, 0x5f,
  0x3e, 0x77, 0x71, 0x40, 0x12, 0x51, 0x43, 0xd5, 0xaf, 0x8f, 0x3f, 0x51, 0xc9, 0xec,
  0x54, 0x34, 0x4a, 0x97, 0x81, 0x50, 0x53, 0xa9, 0x07, 0xec, 0x0d, 0x38, 0x95, 0x54,
  0xdf, 0xf2, 0x67, 0xe9, 0x7f, 0x60, 0xd7, 0xeb, 0xbc, 0x1e, 0x99, 0x16, 0xe9, 0x79,
  0x4f, 0x2b, 0x26, 0x91, 0xc1, 0x4a, 0x42, 0xbb, 0x16, 0x55, 0x1c, 0x07, 0x30, 0x5c,
  0x40, 0x9d, 0x86, 0x0a, 0x17, 0x21, 0xd9, 0x6b, 0xe0, 0xfa, 0xde, 0xb4, 0x9f, 0xc6,
  0x10, 0x71, 0xcb, 0xc8, 0x3e, 0xd0, 0x7c, 0x6d, 0xc7, 0x30, 0xfc, 0x44, 0xe1, 0x86,
  0x7b, 0x7a, 0x44, 0xc5, 0xf5, 0xce, 0xa2, 0x2b, 0xa2, 0x94, 0xaf, 0x50, 0x00, 0x00,
  0x00, 0x00, 0xa4, 0x9e, 0x73, 0x2c, 0xc3, 0xd9, 0x38, 0xe4, 0x8d, 0xdd, 0xe5, 0x9a,
  0x8e, 0xb8, 0xa0, 0x8b, 0x74, 0x5e, 0x27, 0x71, 0xce, 0xe5, 0x6f, 0x3d, 0x1e, 0xb0,
  0xe0, 0xeb, 0xef, 0xed, 0x80, 0x01, 0x65, 0x40, 0x51, 0x07, 0x2b, 0xfc, 0xe3, 0xc4,
  0x4f, 0xe0, 0x92, 0xd5, 0x81, 0x35, 0xf5, 0x63, 0xb8, 0xdc, 0x43, 0x92, 0x5e, 0x1b,
  0xa6, 0xdb, 0x59, 0x7c, 0x91, 0x8b, 0x2d, 0x25, 0x81, 0x17, 0x93, 0xd5, 0xf5, 0xd8,
  0x0f, 0x20, 0x20, 0x20, 0x20, 0x68, 0x96, 0x44, 0x4a, 0x8c, 0x06, 0xb6, 0x1a, 0x26,
  0x95, 0x5c, 0x57, 0x0c, 0xbc, 0x2f, 0x4d, 0xe4, 0x5a, 0x79, 0x7a, 0x69, 0x3f, 0x9b,
  0x71, 0x8d, 0x85, 0x35, 0x29, 0xe6, 0x0d, 0xeb, 0x01, 0x2d, 0x45, 0x80, 0xa4, 0xa6,
  0x47, 0xf9, 0x42, 0xed, 0x14, 0x70, 0x11, 0x16, 0xce, 0x98, 0xcd, 0x91, 0x43, 0xba,
  0x45, 0xe5, 0x88, 0x5a, 0x64, 0x6c, 0x14, 0x5f, 0xf7, 0xff, 0x98, 0x38, 0x73, 0x99,
  0x9c, 0x5c, 0xc3, 0xe4, 0xff, 0x5b, 0xa8, 0x65, 0xd3, 0xff, 0xb5, 0xb8, 0x98, 0xc7,
  0x33, 0xbc, 0x57, 0x98, 0x4b, 0x73, 0xef, 0xc2, 0x46, 0x93, 0xef, 0x3a, 0x01, 0x84,
  0x49, 0x24, 0x32, 0x08, 0x81, 0x51, 0x37, 0x5f, 0xd2, 0x94, 0x24, 0x45, 0xf8, 0xb3,
  0xc4, 0x39, 0x15, 0x54, 0x6e, 0xd8, 0xf7, 0xa8, 0xfb, 0x67, 0x7c, 0xb6, 0xa5, 0x8c,
  0x14, 0x62, 0xd9, 0xb6, 0x35, 0x11, 0x11, 0x82, 0x78, 0x16, 0xae, 0x25, 0x5a, 0xf6,
  0xde, 0x71, 0xdc, 0x9e, 0x9a, 0x50, 0x4e, 0x7e, 0x12, 0x06, 0x91, 0xfb, 0xed, 0x04,
  0xc0, 0xb4, 0x43, 0x19, 0x95, 0xfb, 0xf2, 0x44, 0x2c, 0x44, 0x61, 0xc0, 0x30, 0x73,
  0xce, 0x2f, 0x38, 0x9c, 0xc4, 0x00, 0x00, 0x00, 0x00, 0x25, 0x1e, 0x35, 0xa9, 0x17,
  0x83, 0xac, 0xe5, 0x7e, 0xbf, 0x0a, 0x34, 0x25, 0xed, 0xf8, 0xac, 0x4d, 0x52, 0xd8,
  0x38, 0x2c, 0xc5, 0x5a, 0x6f, 0x6b, 0xe1, 0xeb, 0x07, 0x37, 0xc0, 0x2f, 0x13, 0x4f,
  0xac, 0xf6, 0x42, 0x8c, 0xc6, 0xae, 0xa0, 0xfa, 0xa2, 0xd2, 0x86, 0xcd, 0x18, 0x47,
  0xac, 0x61, 0x81, 0x69, 0x3b, 0xe0, 0xc4, 0xe9, 0xc0, 0xb3, 0x2c, 0x21, 0x56, 0x8c,
  0xcd, 0x0b, 0x45, 0xd4, 0xab, 0x02, 0x2c, 0x9a, 0x52, 0xe4, 0xdc, 0x88, 0x23, 0xfb,
  0x67, 0x60, 0x6a, 0xcc, 0x26, 0x7f, 0x65, 0x6c, 0xde, 0xa6, 0xf8, 0xde, 0x38, 0xd1,
  0xa0, 0x9e, 0x90, 0xa4, 0x61, 0x3c, 0x4e, 0x2f, 0x59, 0x56, 0x51, 0x1e, 0x43, 0x9d,
  0xe5, 0x2f, 0xb9, 0x8e, 0xa2, 0x5a, 0x20, 0x0b, 0x24, 0x18, 0x5b, 0x2c, 0x49, 0xf5,
  0xe8, 0x26, 0xaf, 0x84, 0x82, 0xb9, 0xd6, 0x55, 0x35, 0xaa, 0x97, 0x45, 0x38, 0xea,
  0x7d, 0xab, 0xbe, 0x95, 0x47, 0xe0, 0x88, 0x72, 0x68, 0x6a, 0xbb, 0x40, 0x7b, 0x0a,
  0x0d, 0x67, 0xf7, 0xf9, 0x23, 0x12, 0x12, 0x75, 0xa2, 0xb9, 0x8b, 0xa3, 0x36, 0x7d,
  0x83, 0x6e, 0x92, 0xa9, 0x0e, 0x84, 0x9c, 0x8a, 0xf4, 0x81, 0x12, 0xd5, 0x0d, 0x6c,
  0x14, 0xda, 0x79, 0xa4, 0xd2, 0x7d, 0x2c, 0x8b, 0x43, 0xce, 0x74, 0x40, 0xeb, 0xc4,
  0xc3, 0xc5, 0xfe, 0xfb, 0x56, 0x72, 0xb4, 0x9b, 0x12, 0x99, 0xca, 0x7c, 0x2a, 0xe3,
  0x9e, 0x84, 0x16, 0x6f, 0x32, 0x14, 0xa0, 0xa6, 0xcd, 0x34, 0x0c, 0x34, 0xb3, 0xb9,
  0xf5, 0x72, 0xaf, 0x6e, 0x21, 0x20, 0x08, 0xa4, 0x38, 0x88, 0xec, 0xb6, 0x03, 0x8c,
  0x73, 0xbb, 0xdd, 0xca, 0x6d, 0x65, 0x82, 0xb0, 0xe0, 0xa1, 0xc4, 0x41, 0xc8, 0xf7,
  0xd4, 0x2d, 0x36, 0x2a, 0x15, 0x85, 0xb7, 0xa2, 0xac, 0x49, 0xa4, 0xd2, 0xc8, 0x57,
  0x48, 0xd5, 0x5b, 0xaf, 0xe4, 0xcc, 0x2f, 0xea, 0xa3, 0x3f, 0xb5, 0x9c, 0x04, 0x2e,
  0x33, 0x13, 0xd3, 0x9c, 0x3e, 0x61, 0x2d, 0x3e, 0x90, 0xe4, 0x0f, 0xa4, 0x8a, 0x07,
  0xb2, 0x56, 0x4e, 0xf8, 0x12, 0x43, 0x40, 0x02, 0x34, 0x4e, 0xa8, 0xe7, 0xf8, 0x1c,
  0xa5, 0xd0, 0xb3, 0x92, 0xfc, 0x2c, 0xa3, 0x9c, 0x67, 0x69, 0x9d, 0x50, 0xae, 0xeb,
  0x96, 0xd7, 0xbd, 0x4e, 0x13, 0x43, 0x43, 0xf1, 0x12, 0x5f, 0x55, 0x38, 0xa6, 0xe1,
  0x6f, 0x54, 0x25, 0x6e, 0x29, 0x1b, 0x0a, 0x9d, 0xb2, 0x0c, 0xb0, 0xb8, 0x01, 0xc5,
  0xe8, 0xb0, 0x9b, 0x0f, 0x20, 0x20, 0x20, 0x20, 0x66, 0x59, 0xcf, 0x26, 0xbf, 0xaf,
  0x39, 0x2f, 0x1e, 0x48, 0x81, 0x3d, 0x12, 0xed, 0xeb, 0x7e, 0x8b, 0x39, 0x6b, 0xb4,
  0x8d, 0x75, 0x57, 0x50, 0x54, 0x24, 0x5d, 0x28, 0x79, 0x76, 0x2e, 0x92, 0xa2, 0xf5,
  0x1e, 0x7b, 0xe9, 0xbc, 0x4a, 0x5d, 0x18, 0xd1, 0x8f, 0xdc, 0x14, 0x4f, 0x9e, 0x03,
  0x5a, 0xf8, 0xc6, 0xc3, 0x54, 0x24, 0x70, 0xa9, 0xf4, 0xe2, 0xed, 0xad, 0x24, 0x83,
  0x83, 0xbb, 0x9d, 0x6a, 0x73, 0x4a, 0x6e, 0x4b, 0x79, 0x9a, 0x9e, 0x42, 0x13, 0x08,
  0xa3, 0xca, 0xae, 0xb0, 0x40, 0xfc, 0x5f, 0xc4, 0xe6, 0x65, 0x7b, 0xf3, 0x33, 0x5d,
  0xdb, 0x10, 0xb3, 0x2d, 0x31, 0x1d, 0xb2, 0x87, 0x21, 0x07, 0x6a, 0x0d, 0xae, 0x0a,
  0x62, 0x27, 0x7e, 0xa4, 0x0a, 0x80, 0xd3, 0x77, 0xae, 0x40, 0xac, 0x84, 0x96, 0x42,
  0x3f, 0xd0, 0x9e, 0x23, 0xec, 0xb3, 0x67, 0x23, 0x14, 0xa3, 0xea, 0x1d, 0x50, 0xa9,
  0xe7, 0x8c, 0x00, 0xa8, 0x4b, 0x9b, 0x76, 0x0f, 0xff, 0x80, 0x68, 0xb2, 0xc7, 0x2d,
  0xf4, 0xbe, 0xa7, 0x29, 0x00, 0x2a, 0xf3, 0x3a, 0x74, 0x6d, 0xee, 0xa9, 0xe6, 0x3e,
  0x00, 0x86, 0xb4, 0x66, 0xe1, 0x39, 0x60, 0x66, 0x6f, 0x41, 0x65, 0x19, 0xa9, 0x8e,
  0x52, 0x9c, 0x5b, 0x7e, 0x91, 0x4a, 0x4d, 0xdf, 0xec, 0xb5, 0x1f, 0x72, 0x1d, 0x6f,
  0x69, 0x3d, 0xd1, 0xdd, 0x5b, 0xe6, 0x19, 0xd7, 0x44, 0x38, 0x11, 0xd2, 0x13, 0xfa,
  0xac, 0x89, 0xa1, 0x97, 0x3b, 0x75, 0xeb, 0xe5, 0xe4, 0xc3, 0xac, 0x2a, 0xda, 0x32,
  0x41, 0x78, 0x24, 0x02, 0x67, 0xee, 0x30, 0xb8, 0x15, 0x39, 0x4b, 0xcc, 0xeb, 0xed,
  0xb4, 0xc7, 0xad, 0x58, 0xe3, 0x11, 0xed, 0xf5, 0xa0, 0x91, 0xaa, 0x76, 0x1a, 0xc3,
  0xd7, 0x43, 0x86, 0x36, 0xa4, 0xb2, 0xe2, 0xe9, 0xe7, 0x7f, 0x8b, 0x64, 0xf9, 0x11,
  0xf8, 0x5e, 0xa3, 0x50, 0x96, 0x03, 0xe1, 0xd9, 0x61, 0xc7, 0x05, 0x6d, 0x2b, 0x5c,
  0x7a, 0xc7, 0xd4, 0xa4, 0xd0, 0x01, 0x9c, 0x6a, 0xc7, 0xd2, 0x7c, 0xa1, 0x59, 0x45,
  0xa9, 0x3d, 0x80, 0x24, 0x2c, 0x4d, 0x25, 0xf4, 0xf1, 0x44, 0x87, 0xd9, 0x87, 0xcd,
  0xd0, 0xe7, 0xc8, 0xde, 0xea, 0xd9, 0xd6, 0x32, 0x8e, 0xc8, 0xad, 0x09, 0x58, 0x2c,
  0x8e, 0x6c, 0x9c, 0xc0, 0xcc, 0xb2, 0xd3, 0x89, 0xf1, 0xc1, 0x1d, 0xd8, 0xf1, 0xa0,
  0xa2, 0x48, 0x05, 0xce, 0xa8, 0xf6, 0xff, 0xe1, 0x3d, 0xc0, 0xc4, 0x25, 0xf2, 0x36,
  0x60, 0x3b, 0x33, 0xcb, 0xa8, 0x55, 0x00, 0x00, 0x00, 0x00, 0xcc, 0xc6, 0xb5, 0xf8,
  0x28, 0xf6, 0x14, 0x60, 0x69, 0x23, 0x86, 0x2f, 0x51, 0x0d, 0xf5, 0x07, 0x65, 0x03,
  0x13, 0x84, 0xd0, 0x94, 0xfc, 0x9d, 0x66, 0x49, 0x82, 0xb8, 0x45, 0xbc, 0x6f, 0x93,
  0xf3, 0x91, 0x09, 0xea, 0xd2, 0x8f, 0xba, 0x5f, 0x40, 0xa7, 0x16, 0xae, 0x04, 0x81,
  0x99, 0x7f, 0x04, 0xc9, 0x3c, 0x4d, 0xea, 0x00, 0xa3, 0xda, 0x30, 0x65, 0xa5, 0xd8,
  0x9a, 0x36, 0xcc, 0x0b, 0x83, 0x59, 0x53, 0x4a, 0x94, 0x3c, 0x89, 0x70, 0xbf, 0x48,
  0x98, 0x0b, 0x69, 0x46, 0x7c, 0x03, 0x69, 0xb9, 0x58, 0xab, 0xf6, 0x5e, 0x54, 0x90,
  0xfd, 0x56, 0x80, 0x48, 0x9d, 0xd6, 0xa7, 0x42, 0x66, 0x89, 0x0c, 0x9e, 0xdb, 0xb7,
  0x69, 0x89, 0xa9, 0x69, 0x9f, 0x01, 0x97, 0x6f, 0xb4, 0xce, 0x97, 0xa5, 0x24, 0x10,
  0x97, 0x05, 0x0c, 0x31, 0x6a, 0xe7, 0xa7, 0x87, 0x60, 0x88, 0xcb, 0x3c, 0x34, 0x91,
  0x21, 0x02, 0x02, 0x6f, 0xcd, 0xe5, 0xf5, 0xce, 0x11, 0x71, 0x17, 0x6a, 0x5d, 0x09,
  0x3f, 0xed, 0x9f, 0x2d, 0x60, 0x2c, 0x21, 0x9b, 0x46, 0x45, 0x27, 0x42, 0x61, 0xda,
  0x78, 0x21, 0x9d, 0xf3, 0xd9, 0xe9, 0x0f, 0x20, 0x20, 0x20, 0x20, 0xe3, 0xcc, 0xf1,
  0x86, 0x01, 0xf2, 0xdd, 0x49, 0x74, 0xf9, 0xc2, 0x03, 0x89, 0x1f, 0x02, 0xea, 0x70,
  0xc7, 0x31, 0xaa, 0x80, 0x3c, 0x0d, 0x4d, 0x39, 0xbe, 0x99, 0x02, 0x3c, 0xfe, 0x76,
  0x85, 0x15, 0x80, 0x28, 0x75, 0xbb, 0x76, 0xee, 0x83, 0x24, 0x91, 0x07, 0xe5, 0x00,
  0x29, 0x10, 0x3e, 0xe2, 0x9d, 0x14, 0x99, 0x13, 0x34, 0x70, 0xde, 0x1a, 0x8a, 0x87,
  0xc9, 0xf8, 0xdb, 0x83, 0x3b, 0x1f, 0xec, 0x73, 0x35, 0x08, 0x77, 0x54, 0x88, 0x35,
  0xeb, 0x04, 0x90, 0xfe, 0x1c, 0x38, 0xe9, 0x03, 0x83, 0x8b, 0xd1, 0x28, 0x24, 0xf8,
  0x9e, 0xf2, 0xb0, 0x4a, 0x86, 0x11, 0x40, 0x8e, 0x29, 0x5b, 0xcb, 0xcb, 0x43, 0x03,
  0xb4, 0x78, 0x98, 0x5f, 0xf5, 0x2d, 0x5a, 0xca, 0x42, 0x4c, 0x97, 0x44, 0x0d, 0x13,
  0x9f, 0x14, 0x14, 0x7d, 0x3f, 0x4d, 0x56, 0x6d, 0x45, 0xc7, 0x6e, 0xca, 0xbd, 0xd2,
  0x39, 0x4c, 0xfd, 0xb8, 0xa6, 0x45, 0x30, 0x85, 0x8e, 0xb2, 0xbe, 0x1c, 0xb3, 0x82,
  0xbe, 0xc7, 0x9f, 0x12, 0x5b, 0x19, 0x52, 0xf6, 0xa3, 0xec, 0x71, 0xc6, 0x76, 0xc8,
  0x24, 0x06, 0xf1, 0x3e, 0x3b, 0xde, 0xcb, 0x21, 0x8e, 0x4c, 0x8d, 0xbe, 0x7c, 0xda,
  0x1b, 0x1c, 0x16, 0x34, 0xc0, 0x5e, 0x50, 0xc7, 0xe4, 0x6b, 0x61, 0x22, 0x05, 0xb8,
  0x25, 0xce, 0x89, 0x35, 0x35, 0xd5, 0xe3, 0x08, 0x01, 0x2b, 0xd0, 0x2c, 0x3d, 0x3e,
  0x24, 0x4e, 0x23, 0x89, 0x28, 0x62, 0x21, 0x74, 0x0f, 0x89, 0x8d, 0x65, 0x39, 0x2e,
  0xa8, 0xfe, 0xf9, 0xd2, 0xd3, 0xe6, 0xe3, 0x9c, 0xe7, 0x73, 0xad, 0x4a, 0xf0, 0x71,
  0x00, 0xb3, 0xa8, 0x6f, 0x93, 0xad, 0xf8, 0x41, 0x36, 0xf3, 0xd5, 0xfb, 0x16, 0x7b,
  0x84, 0x89, 0xf4, 0x74, 0xdd, 0x46, 0x2b, 0x87, 0xf4, 0x72, 0x32, 0xfd, 0x15, 0x57,
  0x35, 0xcd, 0x67, 0xd9, 0xed, 0x82, 0x35, 0x22, 0xc0, 0x7e, 0xbc, 0x05, 0x87, 0x49,
  0x7e, 0xdd, 0x4e, 0x15, 0x47, 0x0e, 0x6c, 0x42, 0x37, 0x34, 0x82, 0x52, 0xac, 0x8f,
  0x7d, 0xca, 0x0a, 0x62, 0x17, 0x0a, 0x46, 0xd2, 0xfc, 0x54, 0xc6, 0x47, 0xcb, 0x98,
  0x6d, 0x60, 0x91, 0x4a, 0x6c, 0x7a, 0x89, 0xe9, 0xb1, 0x63, 0x65, 0x8a, 0x58, 0xb4,
  0x92, 0xe4, 0x21, 0x4c, 0x23, 0xdf, 0x1e, 0xba, 0x99, 0x2b, 0x2f, 0x26, 0xd8, 0xd5,
  0x89, 0x7a, 0x69, 0x00, 0x00, 0x00, 0x00, 0x4f, 0x77, 0x32, 0xf6, 0x37, 0x12, 0x7b,
  0x0d, 0x32, 0xff, 0x4f, 0xd0, 0x17, 0x94, 0x35, 0x06, 0x8d, 0x17, 0x30, 0xa2, 0x0b,
  0x93, 0x48, 0x41, 0x9f, 0x88, 0xe0, 0xe8, 0xec, 0xd2, 0x2b, 0x8d, 0x1e, 0x3a, 0x45,
  0xaf, 0xf1, 0x8b, 0x62, 0xe7, 0x0c, 0x72, 0xf9, 0xcb, 0xfc, 0x2f, 0x10, 0xbd, 0x93,
  0x44, 0x37, 0x60, 0x31, 0x73, 0x82, 0x1b, 0x40, 0xf3, 0xb2, 0xce, 0xc0, 0xe6, 0x74,
  0x8d, 0xf0, 0x10, 0x5d, 0xdf, 0xba, 0x3f, 0xa7, 0xd0, 0xda, 0xae, 0xfa, 0x47, 0xce,
  0xa6, 0x94, 0x74, 0x22, 0x5e, 0x62, 0x07, 0xd7, 0xe8, 0x90, 0xaa, 0x33, 0x30, 0x21,
  0x75, 0x13, 0xd3, 0xf6, 0x0f, 0x6b, 0xd7, 0x3b, 0xa2, 0xe1, 0x30, 0x96, 0x05, 0x52,
  0x58, 0x3b, 0x5e, 0xfc, 0x83, 0x0a, 0x77, 0xd7, 0x22, 0x5e, 0x7d, 0xe9, 0x6a, 0x4e,
  0x78, 0xe1, 0xa9, 0x47, 0x71, 0xaa, 0x7c, 0x81, 0xed, 0xa7, 0x66, 0x01, 0xbb, 0x6b,
  0xeb, 0x73, 0x3a, 0x16, 0x64, 0x16, 0x44, 0x22, 0x43, 0x97, 0xf0, 0xc3, 0xb5, 0x61,
  0x45, 0xfe, 0xd3, 0xfe, 0x2f, 0xa8, 0x5c, 0xf2, 0xc7, 0x7b, 0x8b, 0x03, 0xf3, 0x2b,
  0xd3, 0x83, 0x11, 0x06, 0xea, 0xc2, 0x69, 0xbb, 0x88, 0xf7, 0x65, 0x0f, 0x26, 0x1a,
  0x8a, 0x00, 0x00, 0x00, 0x00};

const std::vector<int32_t> block_not_fully_written_plain = {-1155484576, -723955400,
  1033096058, -1690734402, -1557280266, 1327362106, -1930858313, 502539523, -1728529858,
  -938301587, 1431162155, 1085665355, 1654374947, -1661998771, -65105105, -73789608,
  -518907128, 99135751, -252332814, 755814641, 1180918287, 1344049776, 553609048,
  1580443894, 629649304, -1266264776, 99807007, 5955764, -1946737912, 39620447,
  -152527805, -1877116806, 448784075, 1086124775, -1609984092, 1227951724, 1764356251,
  64111306, -960731419, -100082026, -39845375, -1339022546, 2092649110, -568315836,
  -1089884900, -81839914, -1146103148, -1846688624, -784703072, 55004124, -691960657,
  1770461755, -2032810463, -1177788003, -432352882, -65824064, 575267217, -1949367821,
  356750287, 798819494, -92022521, 1318001577, -1192467086, -1412716779, -1223932479,
  276053035, 615126903, 1542603436, 1988388716, 1177882237, 19265476, -1430871151,
  307082914, -1333570194, 1496453452, -790542135, 1455004595, -1690249972, -604059026,
  -290476856, -122204761, -1097539750, -576617062, -2002123957, -1663951485, 193034304,
  768747578, 1337360486, 934457803, 873612482, -624972850, 355564760, 41547336,
  1781447028, -1321591463, 1081281446, -982203381, -222544851, -1233998085, -1331702554,
  907249073, 206843130, -931610657, 1697401307, -234791961, -996341162, 61144662,
  -1638496702, 1693091376, -1667947369, -627963836, 105673451, -918940699, -300903254,
  -28037702, 1358495114, -502062723, 2056481282, 731396580, 617380302, -162912754,
  818693069, -1184143835, -410943565, -1385753324};

const std::vector<uint8_t> block_not_fully_written_encoded = {0x80, 0x01, 0x04, 0x7d,
  0xbf, 0xae, 0xfa, 0xcd, 0x08, 0xb5, 0x9e, 0xf2, 0xaf, 0x0f, 0x20, 0x20, 0x20, 0x20,
  0x73, 0xe4, 0xb6, 0x94, 0xdd, 0xc1, 0xb8, 0xe3, 0xdf, 0xec, 0xa3, 0xd8, 0xd3, 0xa0,
  0xf2, 0x82, 0xdf, 0x6d, 0xee, 0x26, 0x18, 0xc8, 0xc9, 0xb8, 0xe7, 0xfb, 0x08, 0x0c,
  0xd6, 0xdc, 0x02, 0xf6, 0xca, 0x34, 0x18, 0xaa, 0x59, 0x6d, 0x39, 0x08, 0xbb, 0x6a,
  0x66, 0x66, 0x73, 0x19, 0xe4, 0x9c, 0xc5, 0x6e, 0x52, 0xb5, 0x7d, 0xf1, 0x2c, 0xda,
  0xc4, 0xc3, 0x79, 0x7a, 0x4b, 0x53, 0x76, 0x60, 0x9a, 0xdd, 0xd4, 0x9f, 0x86, 0x4b,
  0x0b, 0x66, 0x9a, 0x63, 0x15, 0xb7, 0xb9, 0xd8, 0x54, 0x94, 0xfc, 0x77, 0xb7, 0x84,
  0x83, 0x1c, 0xe1, 0x4b, 0x39, 0x89, 0x32, 0xb8, 0x3d, 0x4e, 0x52, 0x42, 0x3b, 0xed,
  0xfc, 0x09, 0x82, 0xe1, 0x6a, 0xcc, 0x30, 0x39, 0x66, 0x75, 0xef, 0x89, 0x9a, 0x06,
  0xf2, 0xb3, 0x63, 0xf1, 0x7f, 0x54, 0x8a, 0x6f, 0xd2, 0x23, 0x33, 0x14, 0xec, 0xb5,
  0xa0, 0x05, 0xb7, 0x53, 0xfb, 0xa0, 0x58, 0xec, 0x4a, 0xda, 0x63, 0xbe, 0x25, 0x24,
  0xca, 0x29, 0xf7, 0x9a, 0xca, 0x99, 0xa6, 0x15, 0xb6, 0x6b, 0xe8, 0x3d, 0x4c, 0xbf,
  0x4a, 0xae, 0x06, 0x6b, 0x95, 0x7e, 0xc8, 0x68, 0x8e, 0x2d, 0x03, 0x6f, 0x89, 0x47,
  0x49, 0x2d, 0x63, 0xdc, 0x73, 0xc4, 0xe7, 0x5b, 0x55, 0xd3, 0x13, 0xb7, 0x59, 0xe9,
  0x8e, 0x3b, 0x97, 0x31, 0x3c, 0x51, 0xab, 0xe4, 0x4a, 0xba, 0xd7, 0x31, 0x0b, 0xad,
  0x6e, 0x80, 0x78, 0x4e, 0x27, 0xdd, 0xc3, 0x0d, 0x81, 0xf3, 0x4c, 0x98, 0x17, 0xe3,
  0xf4, 0xad, 0x8c, 0xb7, 0x6c, 0xa7, 0xcd, 0x10, 0xd7, 0x90, 0x6c, 0x8e, 0x34, 0xa1,
  0xfd, 0x67, 0x83, 0xe4, 0x77, 0xd9, 0x72, 0x04, 0x72, 0xb8, 0x57, 0x95, 0xfc, 0x1b,
  0xe5, 0x45, 0x3d, 0x8a, 0x09, 0xcf, 0x64, 0x91, 0x5b, 0xe5, 0x3e, 0x89, 0xdd, 0x6d,
  0x47, 0xe6, 0x3e, 0x86, 0x05, 0x3e, 0x66, 0xd4, 0xe7, 0x22, 0x34, 0x8f, 0xd0, 0x72,
  0x46, 0xb2, 0x5b, 0x6c, 0x90, 0x95, 0xac, 0xee, 0xae, 0x4a, 0xe2, 0x30, 0xef, 0x35,
  0x68, 0xf3, 0x8e, 0x24, 0xac, 0x5a, 0x95, 0xe2, 0x67, 0xe6, 0x33, 0x19, 0x79, 0x03,
  0xad, 0x23, 0x18, 0x7f, 0xad, 0xf2, 0x85, 0x9a, 0xd6, 0x00, 0xf4, 0x80, 0x85, 0xbf,
  0xfd, 0x3c, 0xbc, 0xbb, 0xf5, 0x29, 0xaf, 0x8d, 0x7a, 0xe8, 0x05, 0x85, 0x4e, 0xd9,
  0xdb, 0x40, 0xdb, 0xed, 0x0a, 0x9a, 0x4c, 0xc5, 0x06, 0x26, 0xd3, 0x61, 0x26, 0x8f,
  0x58, 0xa2, 0xad, 0xe9, 0x95, 0xf7, 0x4e, 0x9d, 0xc7, 0x9f, 0xe2, 0x9c, 0x00, 0x79,
  0xfa, 0x62, 0x92, 0xda, 0x5d, 0x77, 0xa7, 0xae, 0xab, 0x21, 0xa5, 0x18, 0x70, 0xb5,
  0x0b, 0xc1, 0x46, 0x68, 0xc7, 0x0a, 0xb3, 0xe2, 0x80, 0xab, 0x09, 0xc2, 0xe8, 0x35,
  0x37, 0x0a, 0x00, 0x00, 0x00, 0x00, 0xbd, 0xbf, 0x45, 0xa8, 0x69, 0xba, 0xb4, 0x3e,
  0x86, 0x6d, 0x2b, 0x75, 0x66, 0xf8, 0x71, 0x00, 0xe4, 0xee, 0x3e, 0x51, 0x80, 0xda,
  0x22, 0x37, 0x97, 0xd1, 0xb1, 0x17, 0xa7, 0x59, 0xd3, 0x07, 0x0a, 0xf6, 0x99, 0x4d,
  0x9b, 0x3b, 0x06, 0xba, 0x87, 0xcf, 0xaf, 0x15, 0x89, 0x47, 0x92, 0x41, 0x02, 0xe6,
  0xa8, 0xb2, 0x48, 0x2b, 0xfb, 0xb8, 0x42, 0xb2, 0xb8, 0xa6, 0x95, 0xe8, 0xeb, 0x3d,
  0x60, 0xc8, 0xd4, 0x9f, 0xab, 0xe0, 0x41, 0x8b, 0x6b, 0x17, 0xa3, 0xcd, 0x8e, 0x6b,
  0x18, 0x0c, 0x20, 0x90, 0x7e, 0x13, 0x7d, 0x17, 0x03, 0x2c, 0x85, 0x87, 0x32, 0x74,
  0xdb, 0xf3, 0x7b, 0x4c, 0x5a, 0x65, 0x80, 0xb5, 0xf3, 0x69, 0x9d, 0x03, 0x29, 0x61,
  0x14, 0xa9, 0xfc, 0xdc, 0xe3, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

const std::vector<int32_t> miniblock_not_fully_written_plain = {-1155484576, -723955400,
  1033096058, -1690734402, -1557280266, 1327362106, -1930858313, 502539523, -1728529858,
  -938301587, 1431162155, 1085665355, 1654374947, -1661998771, -65105105, -73789608,
  -518907128, 99135751, -252332814, 755814641, 1180918287, 1344049776, 553609048,
  1580443894, 629649304, -1266264776, 99807007, 5955764, -1946737912};

const std::vector<uint8_t> miniblock_not_fully_written_encoded = {0x80, 0x01, 0x04, 0x1d,
  0xbf, 0xae, 0xfa, 0xcd, 0x08, 0xd7, 0xf6, 0x9d, 0xc6, 0x0e, 0x20, 0x00, 0x00, 0x00,
  0x84, 0x5a, 0x1c, 0x8e, 0xee, 0x37, 0x1e, 0xdd, 0xf0, 0x62, 0x09, 0xd2, 0xe4, 0x16,
  0x58, 0x7c, 0xf0, 0xe3, 0x53, 0x20, 0x29, 0x3e, 0x2f, 0xb2, 0xf8, 0x71, 0x6e, 0x05,
  0xe7, 0x52, 0x68, 0xef, 0xdb, 0xaa, 0x7d, 0xa3, 0x6a, 0xe3, 0x9e, 0x01, 0xcc, 0xe0,
  0xcb, 0x5f, 0x84, 0x8f, 0x49, 0x96, 0xd6, 0xe4, 0xb7, 0xae, 0x8e, 0x67, 0x92, 0xd3,
  0xd5, 0x39, 0xdf, 0x73, 0x5c, 0xc9, 0xdb, 0x59, 0xab, 0x53, 0x3a, 0x99, 0x97, 0xc1,
  0x70, 0x5f, 0xab, 0xd9, 0x7a, 0xb0, 0xca, 0x4e, 0xba, 0x8d, 0x0d, 0xee, 0x1c, 0x7e,
  0x94, 0x92, 0x46, 0x45, 0x4a, 0xff, 0x97, 0xb1, 0x4e, 0xc4, 0xb7, 0x3b, 0x4c, 0x63,
  0x62, 0x03, 0x93, 0x57, 0xd0, 0xc5, 0x41, 0xaf, 0xcb, 0x6e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00};

const std::vector<int32_t> negative_deltas_plain = {10, -18, -53, -81, -113, -145, -177,
  -211, -243, -276, -305, -337, -369, -402, -433, -468, -498, -531, -561, -596, -629,
  -662, -692, -721, -756, -786, -819, -850, -886, -913, -945, -977, -1009, -1041, -1076,
  -1108, -1139, -1169, -1202, -1235, -1268, -1301, -1329, -1364, -1394, -1427, -1458,
  -1490, -1522, -1558, -1589, -1619, -1652, -1686, -1715, -1746, -1778, -1809, -1843,
  -1877, -1905, -1938, -1973, -2006, -2038, -2067, -2101, -2132, -2162, -2198, -2228,
  -2258, -2289, -2321, -2356, -2390, -2417, -2454, -2485, -2514, -2549, -2579, -2611,
  -2645, -2675, -2710, -2741, -2773, -2803, -2837, -2867, -2898, -2934, -2966, -2996,
  -3029, -3059, -3094, -3123, -3155, -3188, -3219, -3249, -3281, -3315, -3349, -3379,
  -3411, -3446, -3477, -3506, -3537, -3570, -3605, -3637, -3669, -3702, -3733, -3766,
  -3795, -3827, -3860, -3892, -3925, -3954, -3988, -4018, -4052};

const std::vector<uint8_t> negative_deltas_encoded = {0x80, 0x01, 0x04, 0x80, 0x01, 0x14,
  0x49, 0x04, 0x04, 0x04, 0x04, 0x29, 0x59, 0x55, 0x53, 0x84, 0x55, 0x64, 0x72, 0x74,
  0x42, 0x74, 0x28, 0x47, 0x16, 0x5a, 0x55, 0x25, 0x65, 0x47, 0x44, 0x94, 0x72, 0x64,
  0x55, 0x61, 0x47, 0x83, 0x56, 0x36, 0x93, 0x24, 0x54, 0x38, 0x76, 0x71, 0x67, 0x25,
  0xa3, 0x60, 0x28, 0x57, 0x73, 0x62, 0x75, 0x73, 0x16, 0x75, 0x74, 0x82, 0x45, 0x76,
  0x35, 0x73, 0x25, 0x86, 0x46, 0x52, 0x45, 0x46, 0x58, 0x54, 0x84, 0x73, 0x03};

const std::vector<int32_t> deltas_are_the_same_plain = {0, 32, 64, 96, 128, 160, 192, 224,
  256, 288, 320, 352, 384, 416, 448, 480, 512, 544, 576, 608, 640, 672, 704, 736, 768,
  800, 832, 864, 896, 928, 960, 992, 1024, 1056, 1088, 1120, 1152, 1184, 1216, 1248, 1280,
  1312, 1344, 1376, 1408, 1440, 1472, 1504, 1536, 1568, 1600, 1632, 1664, 1696, 1728,
  1760, 1792, 1824, 1856, 1888, 1920, 1952, 1984, 2016, 2048, 2080, 2112, 2144, 2176,
  2208, 2240, 2272, 2304, 2336, 2368, 2400, 2432, 2464, 2496, 2528, 2560, 2592, 2624,
  2656, 2688, 2720, 2752, 2784, 2816, 2848, 2880, 2912, 2944, 2976, 3008, 3040, 3072,
  3104, 3136, 3168, 3200, 3232, 3264, 3296, 3328, 3360, 3392, 3424, 3456, 3488, 3520,
  3552, 3584, 3616, 3648, 3680, 3712, 3744, 3776, 3808, 3840, 3872, 3904, 3936, 3968,
  4000, 4032, 4064, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

const std::vector<uint8_t> deltas_are_the_same_encoded = {0x80, 0x01, 0x04, 0x80, 0x02,
  0x00, 0xbf, 0x3f, 0x0d, 0x0d, 0x0d, 0x0d, 0x00, 0x10, 0x00, 0x02, 0x40, 0x00, 0x08,
  0x00, 0x01, 0x20, 0x00, 0x04, 0x80, 0x00, 0x10, 0x00, 0x02, 0x40, 0x00, 0x08, 0x00,
  0x01, 0x20, 0x00, 0x04, 0x80, 0x00, 0x10, 0x00, 0x02, 0x40, 0x00, 0x08, 0x00, 0x01,
  0x20, 0x00, 0x04, 0x80, 0x00, 0x10, 0x00, 0x02, 0x40, 0x00, 0x08, 0x00, 0x01, 0x20,
  0x00, 0x04, 0x80, 0x00, 0x10, 0x00, 0x02, 0x40, 0x00, 0x08, 0x00, 0x01, 0x20, 0x00,
  0x04, 0x80, 0x00, 0x10, 0x00, 0x02, 0x40, 0x00, 0x08, 0x00, 0x01, 0x20, 0x00, 0x04,
  0x80, 0x00, 0x10, 0x00, 0x02, 0x40, 0x00, 0x08, 0x00, 0x01, 0x20, 0x00, 0x04, 0x80,
  0x00, 0x10, 0x00, 0x02, 0x40, 0x00, 0x08, 0x00, 0x01, 0x20, 0x00, 0x04, 0x80, 0x00,
  0x10, 0x00, 0x02, 0x40, 0x00, 0x08, 0x00, 0x01, 0x20, 0x00, 0x04, 0x80, 0x00, 0x10,
  0x00, 0x02, 0x40, 0x00, 0x08, 0x00, 0x01, 0x20, 0x00, 0x04, 0x80, 0x00, 0x10, 0x00,
  0x02, 0x40, 0x00, 0x08, 0x00, 0x01, 0x20, 0x00, 0x04, 0x80, 0x00, 0x10, 0x00, 0x02,
  0x40, 0x00, 0x08, 0x00, 0x01, 0x20, 0x00, 0x04, 0x80, 0x00, 0x10, 0x00, 0x02, 0x40,
  0x00, 0x08, 0x00, 0x01, 0x20, 0x00, 0x04, 0x80, 0x00, 0x10, 0x00, 0x02, 0x40, 0x00,
  0x08, 0x00, 0x01, 0x20, 0x00, 0x04, 0x80, 0x00, 0x10, 0x00, 0x02, 0x40, 0x00, 0x08,
  0x00, 0x01, 0x20, 0x00, 0x04, 0x80, 0x00, 0x10, 0x00, 0x02, 0x40, 0x00, 0x08, 0x00,
  0x01, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

const std::vector<int32_t> values_are_the_same_plain = {3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

const std::vector<uint8_t> values_are_the_same_encoded = {0x80, 0x01, 0x04, 0x80, 0x02,
  0x06, 0x05, 0x02, 0x02, 0x02, 0x02, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x3f, 0x00, 0x00, 0x00, 0x00,
  0x00};

const std::vector<int32_t> delta_is_zero_for_each_block_plain = {0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
  4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
  4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
  4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
  4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4};

const std::vector<uint8_t> delta_is_zero_for_each_block_encoded = {0x80, 0x01, 0x04,
  0x81, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
  0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01,
  0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
  0x00, 0x00};

const std::vector<int32_t> data_is_not_aligned_with_block_plain = {-10, -2, -1, -3, 5, 3,
  1, -9, 9, 4, 7, 7, 3, -8, 5, -6, -6, 5, -9, -10, -7, -2, -6, -3, 2, -10, -7, -8, 2, -7,
  -5, -5, 7, -3, -8, -8, -5, 3, 8, 5, -10, 5, 5, 0, 8, 1, 4, 6, 2, -8, 9, 7, 6, -4, -3, 6,
  -2, 7, -7, -3, -3, -2, -5, 8, -2, 7, 1, 8, 8, 8, 8, 2, 7, 1, -4, -10, 7, -8, 5, -10, -3,
  3, 7, -1, -5, 2, -1, -7, -9, -9, -7, -10, -2, 4, 6, -7, 7, -8, -5, 1, 6, -5, 9, 3, -3,
  -3, 1, 7, -2, -7, 0, -5, 8, -9, 7, 7, -4, -9, 0, 1, 1, 4, 0, -5, -4, -6, -6, -8, 2, 0,
  -2, 3, -1, -5, -9, 0, -5, -8, 4, 9, 7, 1, 1, -7, 7, 8, 6, 1, 5, -9, -4, 3, 3, 7, 8, 8,
  6, 8, -10, 5, 5, -5, -2, -1, 5, 5, 3, 0, -9, -4, 8, 2, 8, -9, -3, -1, 9, 1, -8, -9, 7,
  -9, 8, -9, -6, -7, -1, 2, -10, 3, 2, 5, 3, 1, 3, -8, -10, 4, 4, 8, 5, -8, -1, -2, 8, 0,
  -10, 6, 3, -8, 4, 4, -8, -2, -1, -8, 9, 4, 5, 0, 3, -10, 7, 0, 2, 2, 0, -7, 4, -8, -10,
  -4, 2, -5, -5, 3, 3, -2, 7, -10, -4, -8, -5, -4, 9, -7, 2, 7, 0, 5, 4, 6, 4, -2, -1, -2,
  -2, 5, 6, -4, -3, -6, -6, -1, -7, 3, -7, 2, -4, 4, 5, -6, 5, -4, 1, 7, -6, 7, 7, 4, 9,
  -8, -10, 8, -5, 2, 1, -10, -4, 8, 9, 1, -4, 2, -5, -6, 7, 6, -7, -8, 5, -8, -2, -9, 1,
  6, 2, -7, 3, 8, 0, -2, 9, 8, 1, -10, -1, 7, 6, 5, 7, 5, 5, 6, -4, -6, 1, 5, 0, 1, -10,
  4, -3, -10, -7, -3, -3, -4, 2, -6, 8, 9, -7, -2, -9, -9, -10, -4, 4, 3, -8, 6, 8, -3,
  -3, 4, -1, -8, -8, 7, 3, 8, 2, 9, 0, 6, -10, 3, 4, -1, 8, 7, 2, 6, 8, -1, -4, -4, 4, 6,
  -10, -7, 6, -2, -9, 5, -9, -8, -4, 8, 1, 5, -4, -8, -1, -3, -2, 1, -10, -10, 3, -8, -3,
  -1, 7, 5, 7, 7, -1, -2, -2, 9, 9, -10, -8, 1, 7, -9, 1, 1, -9, -7, -9, 7, -8, -9, -6, 8,
  3, -1, 8, 6, -3, 5, -7, 5, -8, -4, -7, 0, 2, -9, -1, 0, 9, -4, -6, -5, 9, 4, 0, 7, -8,
  3, 2, -8, -1, 9, 2, -4, 7, -10, 8, -6, -5, -1, 0, 7, -10, -9, -6, 6, -3, -3, -8, 3, -6,
  7, -3, -5, 8, -4, 3, -10, -6, -9, 6, -3, -4, 0, -3, -9, -10, 5, -6, -1, 4, 2, 2, 5, -1,
  9, 3, -10, 1, 5, 9, -10, 4, -6, 8, 6, -7, 7, -8, 6, -1, 0, 0, -3, -4, -5, -7, 7, 8, 9,
  7, 1, 5, 9, -6, -7, -4, -2, 3, 4, -3, -5, 1, -3, -9, 1, 4, -9, -1, 0, 5, 0, -6, 6, -5,
  -6, -2, -10, 7, 2, -7, 4, -6, -5, -6, -6, 5, 8, -2, -5, 6, 4, 2, -8, -2, 9, -5, 1, 7, 7,
  -10, -3, 3, -10, 3, 6, 4, -7, -9, 0, -1, 3, 5, 6, -5, -2, 2, 0, 7, 3, 3, -1, 1, 5, 8,
  -6, -2, 8, 8, 1, 3, 6, 0, -3, -7, 6, -10, -4, -7, 5, 2, 6, 4, 6, -4, 1, -1, -7, -7, -10,
  -5, 9, 1, 7, 6, -4, -2, 9, 6, -4, -2, -7, 7, -1, 3, -5, 9};

const std::vector<uint8_t> data_is_not_aligned_with_block_encoded = {0x80, 0x01, 0x04,
  0x83, 0x05, 0x13, 0x21, 0x06, 0x06, 0x06, 0x06, 0x99, 0xf4, 0x64, 0xcf, 0x73, 0x8c,
  0x0c, 0x15, 0x35, 0x86, 0x67, 0x44, 0xdc, 0x00, 0x51, 0x56, 0x43, 0x59, 0x05, 0x05,
  0x6d, 0xc8, 0x14, 0x75, 0x07, 0x13, 0x51, 0x99, 0xe5, 0x08, 0x60, 0xc4, 0x64, 0x0a,
  0x35, 0x35, 0x87, 0xf8, 0x40, 0x87, 0xa4, 0x25, 0xda, 0x50, 0x45, 0x92, 0xe3, 0x1d,
  0xda, 0x82, 0x45, 0x51, 0xb4, 0x58, 0x0b, 0xb3, 0x88, 0x82, 0x27, 0x60, 0x57, 0x95,
  0x34, 0x98, 0xb3, 0x3c, 0xd1, 0xe4, 0x64, 0xd7, 0x44, 0x7c, 0x02, 0x75, 0x59, 0xc6,
  0xb7, 0x2c, 0x51, 0x75, 0x21, 0x0c, 0xc6, 0x78, 0x40, 0x18, 0x19, 0x8c, 0x26, 0x45,
  0x54, 0xc3, 0x48, 0x4f, 0xf4, 0x6c, 0x23, 0x06, 0x06, 0x06, 0x05, 0x10, 0x74, 0x39,
  0x8e, 0xb3, 0x35, 0x8f, 0x77, 0x41, 0x8c, 0xa4, 0x80, 0x13, 0xd4, 0x58, 0xc4, 0x95,
  0x49, 0xd6, 0x24, 0x41, 0x14, 0x10, 0x4a, 0x48, 0x35, 0x61, 0x12, 0xf4, 0x24, 0x97,
  0xc7, 0x60, 0x01, 0x46, 0x71, 0x4a, 0x12, 0x89, 0xc2, 0x18, 0x54, 0x11, 0x56, 0x19,
  0x5f, 0x54, 0x41, 0x10, 0x75, 0x40, 0xa0, 0x64, 0x3d, 0x45, 0x16, 0x71, 0x0a, 0x22,
  0x3e, 0x87, 0x27, 0x19, 0xd8, 0xb4, 0x8c, 0xcd, 0xd4, 0x54, 0xc5, 0xb8, 0x50, 0x12,
  0xae, 0x6e, 0x20, 0xc6, 0x4b, 0x6a, 0xd9, 0x76, 0xc0, 0xae, 0xce, 0x2f, 0xf6, 0x5d,
  0x37, 0x52, 0xc8, 0x66, 0x94, 0x21, 0x06, 0x05, 0x05, 0x06, 0x98, 0x74, 0x48, 0x4e,
  0x64, 0x2d, 0xdb, 0xa1, 0x2d, 0x99, 0x64, 0x70, 0x88, 0x75, 0x11, 0x5e, 0xe4, 0x58,
  0xc0, 0x33, 0x12, 0x18, 0x64, 0x5c, 0x5d, 0x26, 0x76, 0x15, 0xf4, 0x90, 0x40, 0x4f,
  0xae, 0xda, 0xb6, 0xa1, 0x6d, 0xd3, 0xe3, 0x50, 0x19, 0x9d, 0x21, 0x9c, 0x2f, 0xca,
  0xf3, 0x70, 0x65, 0xd2, 0x7c, 0xa5, 0x68, 0x8d, 0xf0, 0xa6, 0x2f, 0x83, 0x55, 0x11,
  0xde, 0x0c, 0xcd, 0x9f, 0x46, 0x84, 0x31, 0x4a, 0x04, 0x36, 0xd6, 0x82, 0x21, 0x57,
  0xe0, 0x49, 0x8c, 0x06, 0x31, 0xd5, 0x84, 0x38, 0x51, 0x36, 0x05, 0x94, 0x97, 0x28,
  0x25, 0x06, 0x06, 0x06, 0x06, 0x61, 0x41, 0x5d, 0x1f, 0x73, 0x29, 0x8f, 0x16, 0x51,
  0x16, 0x32, 0x81, 0x08, 0x56, 0x6d, 0x51, 0x35, 0x2d, 0xd2, 0xe4, 0x4d, 0x40, 0xc5,
  0x65, 0x43, 0x37, 0x25, 0x55, 0x34, 0x12, 0x92, 0x15, 0x3a, 0x0f, 0x17, 0x29, 0xdb,
  0xf1, 0x19, 0x17, 0xa4, 0x55, 0xc8, 0x46, 0x71, 0x46, 0x44, 0x85, 0xce, 0xa3, 0x11,
  0x9e, 0x94, 0x68, 0x1d, 0xd3, 0x78, 0x42, 0x59, 0x50, 0x17, 0xa5, 0x09, 0x94, 0xf5,
  0x29, 0x93, 0xe3, 0x29, 0x60, 0x12, 0x81, 0x87, 0x66, 0x5c, 0x90, 0xa8, 0x48, 0x17,
  0xd4, 0x48, 0x22, 0x82, 0x61, 0xd1, 0x64, 0x35, 0x5d, 0x63, 0x78, 0xd7, 0x05, 0x84,
  0x49, 0x18, 0x19, 0x21, 0x05, 0x06, 0x05, 0x05, 0x5f, 0x7c, 0x25, 0xa3, 0x83, 0xf0,
  0x7d, 0x29, 0xdf, 0xaa, 0x55, 0x40, 0x3a, 0xad, 0x54, 0xef, 0xb6, 0xb5, 0x29, 0xc9,
  0x92, 0xc5, 0x2c, 0x9d, 0x01, 0x55, 0x89, 0xc8, 0x20, 0xdc, 0x21, 0x41, 0x11, 0x47,
  0x1d, 0x0e, 0xf7, 0x3c, 0xc7, 0xc5, 0x0d, 0xd7, 0x15, 0x01, 0xf8, 0x12, 0x4f, 0x9f,
  0x79, 0x1a, 0xd6, 0x29, 0x0d, 0xad, 0x0f, 0xb7, 0xd8, 0x66, 0xa5, 0xa3, 0xee, 0xa8,
  0x26, 0x5d, 0xae, 0xf9, 0x70, 0x5d, 0x77, 0xf5, 0xcd, 0x63, 0xdf, 0x8a, 0xce, 0xfe,
  0x74, 0xe1, 0x99, 0xdc, 0x9d, 0xc9, 0x7e, 0xaa, 0x0f, 0x05, 0x00, 0x00, 0x00, 0xc0,
  0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00};

const std::vector<int32_t> max_min_value_plain = {-2147483648, 2147483647, -2147483648,
  2147483647, -2147483648, 2147483647, -2147483648, 2147483647, -2147483648, 2147483647};

const std::vector<uint8_t> max_min_value_encoded = {0x80, 0x01, 0x04, 0x0a, 0xff, 0xff,
  0xff, 0xff, 0x0f, 0x01, 0x02, 0x00, 0x00, 0x00, 0x88, 0x88, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00};

const std::vector<int64_t> block_aligned_64_bits_plain = {5564394361705317565,
  2723284350022887550, 8965616010139000000, -525290723066757833, -256215638214760247,
  2685417292544003998, 4420714740565130201, -7966248783787801342, 1597492495300769866,
  -2961770377674635165, 3638439436798852355, 3670188200530077150, -7775406056454400049,
  -4652152859962012205, -9061770062339801048, 9058335836950191827, 4825919987395976325,
  -4263680224293442400, -3653666216282834286, 3840100219171669977, -1733280462939499801,
  -2075671678347285046, 3306394860239266817, -5286179565387424357, -4682280274939429955,
  -4235648901493336701, -5845610914943065188, -7172379389390334454, 8865062033576882864,
  7118308542749032443, 3136526046193576663, -709247069004872658, -5247498640335065052,
  -3668021642544658016, -1646359613322077637, 1774533014451554068, -1477020000120611798,
  8752771089307757899, -6861564888296028498, 115487847681409176, -3443849941154961469,
  -1995170765479996813, -2461972003540180210, 4977452146124170419, 1267872101171146602,
  7584956963993264869, -8815696705871086780, 4590349657533168299, 3978902005544676138,
  6187833401460793642, -5980181900537739562, -995633791918306088, 1442164110046208645,
  -1396777273742157117, -8422380775602076660, 5058907708441209356, 4291940183440865254,
  -2924178952607862925, -8335742388824276463, -3597460785120635170, -5412445430577590732,
  6993672702338942743, 2024173938507558712, -7954912369173485979, 7283487385335641908,
  8796504034093158353, 8295347694517162966, 4519803267427783613, 6532057550702017214,
  -8822975597452080319, 8418737724759971324, -5423796180070248981, -3314189747582094469,
  -5146420353236934756, -7289309079612308560, -2955048761935010880, -2767253108477691688,
  -3052870016398336764, -5372621181794865391, -3032423883483789152, 9021980008735745759,
  4184806633696302860, -8418381998046290057, -4963393833657354811, 2008653466903433214,
  -2629252080927359626, 1018227968641167688, -1859495675188329772, 2941600877265041330,
  -3442671452921256306, 3743726566489743332, 5600015620387082297, 4715700299786426668,
  761308619457241185, 2599263569351295205, 1861426242331193875, 2674150796295073173,
  -6958180393192730533, 1635845983989368836, -1892474133651507887, 9016981338556810378,
  707145154816397502, 5555068361804567871, -7867210299029578780, -3805792386106328090,
  2436483863710790770, 2452626077273063381, 6999033308347166588, -14623136089808879,
  6755748362404515422, -6762028491100349505, 8807193533846339203, -178956146770039889,
  -4651034594958624271, -239722105908073508, 2698844442922716068, -8035434128046147535,
  1314910580697103634, -4767969827616557937, 3803383836437910678, -3529929826556632386,
  -7400231241931027152, -149302163268565517, -5314823991102441440, 7939721349014423935,
  8980166903831882552, 5408951751631511909, -2975684751936782207, -283925187419752432,
  7601408549359970429, 7600336207467298162, -8823566837612316404, -8806255522560325705,
  7520396725671151600, 6779178524615421808, -7161068744707171724, 3752859805948947904,
  2624816850222429258, 7607694511540288231, -3531645316295787916, -1182432434833177999,
  -2822298332577764647, 4414385455567876494, 2157839447427887091, -1844799327144502552,
  5337975704492557212, 8397620056047772880, -261514986938051955, 2979318440036817183,
  2596877943173423078, 733401753630607440, 4161312721592587183, 3553229893499880699,
  -5721421616806442137, -2789835741288978274, -5254268138401615332, 8206256020759442069,
  650355891233989455, 2348367281573912659, 8578891921070716774, 6067061858960609567,
  4871868099759537007, 5642906287936269491, 5232937214984412694, -5093714855275820827,
  5378107864813017438, 4269810675626950478, 6116209494173032386, -3231764432464187723,
  -7345321370284549331, 232930130415159594, -3264833893781646753, 1704783340429228590,
  1067306117748566169, -63479222854647102, -5075337055125846210, -875632253616129681,
  -3499160870073771267, 7714320673908596556, -7925691325436799071, 5698983928756164680,
  -2807212773054677102, 3711440678194705116, -5566507348934688014, 7231604524824077757,
  -2224452081070488134, 911083723171463212, 9052221365972414118, -3424522036810302967,
  5638898644968340016, 9164339621871693626, 3001218250351810586, -6579063451458687535,
  -8219208949138807705, -2342936252471885667, -1564586168969967400, -765655698652419740,
  6936414893237423841, 2248600469043906531, 4775710155547386401, -8578095384429550454,
  6343461282257277068, -8561321752726062494, -5474103134727408125, -9035076773001873157,
  -380296783751229881, -3966254706930003109, -3206638917656533431, 6373437803673613169,
  -8260238917844823733, 4111593021001875284, 1383741938285565128, 8601141046588395896,
  132484629211317091, -4936484817376656881, 6859959813118624265, -1075011440305416336,
  2228713306849498968, 6724442400532002212, -5313134752015404088, -4244429381652968287,
  -8324561625190216850, -3808849353234928868, -408511836484986417, -6809800573004624088,
  1892071284246799098, 4139819085504872559, 857833863422653679, -1613083322361987942,
  564242440612139557, -682581566465509787, -7121375316306014774, 4212039336549384664,
  -1484705903175896963, -2907994602896961845, -3053237334876426573, -2306886855239827804,
  -2435755571624913620, 8923362597857546680, 8942235912020327282, 3867641573475765766,
  5579054135399926478, -2963527963098065888, -2265146724541771171, -5102786265873522041,
  -6471988754069057109, -3931631116758640028, 2159535115687040626, -7903617578983787994,
  7083228615649847444, -3929841298990179408, 3177684559100098206, -3930547715941903940,
  8223703531679816526, -6432407851205134529, 7405296692383228086, 1459547832900254038,
  4532396312086947928, -4418733970506377473, -3127346264844288809, 6572655119144717339,
  6572655119144717339, -1445593966947920455, -9200450218275468183, -8859565935339498466,
  8480825508232441667, -1112096250991420170, 4774911531107869364, -5185298204123123511,
  -8797765112853289392, -175546849890710190, 6202727909381052078, 7742820613896094143,
  4013575199375053357, 1512510069625143454, -2729430379491759030, -7434963742833246314,
  5905245926336530802, -8260911426973668081, 2694948372504369637, 4740339323970439596,
  4109128916085469324, 3835929223428822729, -4079131443229395061, -5659441767292677864,
  3017862419943638867, -7398743972729757957, 7419434046289446049, -3274393161205250648,
  605060358940115949, -7846121548878794479, -3808722348256002328, -2239319454255130392,
  -7668839813222792427, 125095197862695012, -7975374720437356263, 8447552829515411247,
  -197026706838948981, 7219217227426642112, 9204075690520259724, 5651032002208488232,
  6647630038795348082, 1559515478180510265, -4870531535911284797, 6971244209687191309,
  -230274043259618062, 2995487713339500938, 6473960880637046452, -7014806101877539853,
  -5936939561923709060, -2240949779103950267, 8687994920813552579, 4541675331043753537,
  -3136685060641275594, 6930269378505959980, -5970203002025295719, -1467880518120116375,
  -253409114015568880, 4001171526032939646, 1919244285936821995, -1498149273987692525,
  -2007722153464404658, 5024413252618871796, 305369915660212350, -6805291463091899082,
  8918148781052401948, -6636775902584471489, 4345873711512543363, 9024898643161028471,
  5178947557114574033, 5599948706816694040, -7333629229858755266, 4287928744575100584,
  3270982521980382067, -1774094286088109012, 2120739967981084215, 789244373593591052,
  6955451490121697212, -4653017976897169000, -3809466139891132197, -2566827619600949862,
  -2662108696965796502, 4135270250631289879, -7396920309929112562, -5613327455186286725,
  -3158299296991636231, 8031758261891402166, -7776623117278919641, -3210307312503786659,
  8902461935243185533, -4496689813580683013, 347067035705423499, -4660395297627224948,
  -3741957281861122396, 2583420634707791716, -1105309442044041300, 6367837045738306963,
  -7729229045387224000, 2356732071492961220, 2417074749476734117, -1679587354075763203,
  1253214349404833256, -2980801132601470862, -6220921637988652849, 9184353476005989693,
  -762944759053753838, -1885373774542211384, 4650630651295888764, 7550056253986021642,
  6083486619529420932, 1966154948483357069, 4566692737963309884, -322795716692080136,
  -4517272333985424820, 326496899901956403, -1468093297549861706, 6432525978000548736,
  -2655981049842060080, 3257173739429652809, 7408411190670461763, -4408860971113349770,
  2525815479965616385, 6685095017570842090, -8703307807309230770, -6266405959812100230,
  5542076004726338235, -2184493126685566276, -6767132106156753908, -2999451201562593636,
  -2627433139774265407, -3495621722233849109, -3823538861876924412, 6129082016264556526,
  8439964686134610271, -2378310550688885429, -8642471780655462636, 1117944408697148906,
  6777634429852875416, -8604153005314517970, -2372088077957021672, 2298174613921864064,
  -5620071107734796246, -952764993383727098, 6016183812513968945, 2037321014552461752,
  8401302348672797186, 6705001832222294422, 9076927857858756906, 6751943733025127253,
  -271479213999476919, 1742739327270749034, 8982971909551252637, -8317789836611279551,
  2823072916549174445, 3788769611721083994, 4548290939930043644, 196242524096411887,
  -3835003617203600118, 218822992575524407, -2541086198386953853, -6957595585279927471,
  8821870970693696186, -7837022118204397595, -9048420991534684545, -327019392806705180,
  7226641310744701298, -3772304373287120721, -486791493216779290, 4047072789716644244,
  -9020900142612098415, -4720907651073018249, 6714402434202172454, 7675502635449212911,
  4163008236491446259, 4390134283738207300, 7546472133544954022, 4930610427074226473,
  -6167011621887409352, -5798638284354391984, -889217000960972250, -201340015952665473,
  2023164489243377781, -1346696871644314787, -3871293915369403644, 259857264136807085,
  -3646145382038437548, 4111448051785872641, -5478810845406175931, 7572874438652712837,
  -7846063592351794710, 7042633264701661592, 5044522948034335392, 6221924006117699671,
  1553907059227196143, -7712685892672893960, 8052006613230657093, -824534471744872877,
  -3910354617906417818, 2973280545047653334, 2822730184664772564, -7556338309863103728,
  -1630087101505061022, 2738125142988673797, 16062676226188714, -899436248088437285,
  -5047806008500280056, -896451806300597317, -1359219867622480238, 1563782903825724037,
  -4572216883684723181, -6154237979080040379, -8534026015670213907, 3655102141793033627,
  2270711209004920594, -271049025195580540, -2413678761072906027, -5654773514781748123,
  592641740583434804, -3672939346943895150, 3392274734796885007, -2144580448132110681,
  -4573188197207506347, -3036061507826880565, 7658262541823230255, 3250989688152271064,
  -1883897513992510202, 9086716432456442395, 6089010580687545457, -5381390281369380510,
  -1507826025156333101, -8654757380773078207, -3701039397558967546, -5819699417758754605,
  8386555126854956362, 6872622720250036491, 8691044593027521743, 1391926461937950673,
  -7173207165967617214, 2952914856181720962, -4169915698780150729, -4751969766879231270,
  -8877980575011891936, -5664543455622154189, -8683187004247435472, 8909991169623503572,
  -2242999216027830595, -5657088284774265450, -9009728354873717200, -4682922358324047212,
  831159264216934830, -4357902759909849761, 453291626031788910, -5600876922961503838,
  -2349739797065490213, -1783363344273710440, 2691973575047241511, 5178248030516698027,
  3391837051397254640, -5360193439394332352, -2657169830843505690, 6440158249460793343,
  1956554550279898725, 4441377384265249320, -9166685338805181137, -5332236738067264907,
  2081112303666228845, -6887080699234153512, 4481825581210462663, -7110359965132355921,
  -6723609638021355003, -4101485458205447802, -8922443422816747084, -9187622421381981694,
  -6294693333005590385, 1282044414223475656, 8761939212635812410, 8471343986143597853,
  7127902465402690464, 7388851158400979154, -2962317625512023508, 7547618630925789201,
  3243434367520300335, -449111685068299958, 2282272148126289838, -677138762005876621,
  -7739974695735607513, 8456792712211799318, -3610501688132114331, -6685509579770069374,
  3594711297912237119, 8353855287991593955, 9199949676981774576, -1207689542080521871,
  -3067486917587227243, -1916107699415072235, 2458550225152979405, -1671172039493800176,
  -2425800377722976690, -4435737025753261350, 75996345734337509, 1646302458536598772,
  -6065418457271695365, -72490221055654775, 7719302778996340384, -4558474093379970073,
  7266437101609096522, 2188503932653454806, 774756490492409156, 28303612142889093,
  -5046977805528897462, 938673317915259127, -1834957538071936691, 5861259732941170,
  -5207923426989816818, 1280691769853663519, 1390608800525510789, 4663995164018280258,
  112652620476498786, 4146614714016115495, -5148983112017192399, 3615003910075661395,
  -1722178378864872202, -110410609354086880, 7497182818331107808, 2633807972649330407,
  -8656210068706414381, -2855958786801643121, -2274283920399510974, 5037904037152911582,
  6089053608293594124, -365401499157903387, -8848863532420428373, -6013731457837374792,
  -2067646518980876792, 8729829337800120064, -4926022745523722173, -5626628929295929930,
  3570109879878904201, -4601680787272884598, 5922347387916062411, -2815647500599352479,
  7002247339212877489, 5166215197030938210, -5421524922846668459, 3244018995269884038,
  575339819029269103, -6311536173081082478, -1319128982977709318,
  std::numeric_limits<int64_t>::min(), std::numeric_limits<int64_t>::max(),
  std::numeric_limits<int64_t>::min(), -6687722445408496718, -5212936506468904298,
  4910700210696574724, 1327259068127703339, 6281550587844036510, 8812924971576543152,
  6013411819313117783, 3662977086906118430, -8819538840539939638, 6802224338184360510,
  9186939048786418053, -3771085531489618321, 8980606383633430507, 43956325894508378,
  7021444215162616124, -938023211113508960, 1420566447036232278, -6591209337437782952,
  -2320873369073703768, 6486484775628171670, -4633828031561688156, 4659429086465542268,
  -4516675769355985758, -903759626910133897, -8642829235471962236, 3589671192380332332,
  8164775684969599760, -3098734183592372739, 510889925047057746, -5862322742667827217};

std::vector<uint8_t> block_aligned_64_bits_encoded = {0x80, 0x01, 0x04, 0x81, 0x05, 0xfa,
  0x92, 0x8e, 0xfa, 0x98, 0xcc, 0xd9, 0xb8, 0x9a, 0x01, 0xbd, 0xec, 0xbd, 0xbe, 0xe2,
  0xc1, 0xf3, 0xbc, 0xfc, 0x01, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x9e, 0xc4, 0x5d, 0x86,
  0x40, 0xcf, 0x56, 0x61, 0xef, 0x4c, 0x67, 0x99, 0x1e, 0xde, 0xd4, 0x96, 0x1b, 0xd0,
  0x86, 0xa3, 0x6d, 0x86, 0xfa, 0xb1, 0xa6, 0x6a, 0x46, 0x64, 0xd9, 0xf8, 0x81, 0xf4,
  0x61, 0x5f, 0xbd, 0x97, 0xae, 0x0f, 0xa7, 0x5a, 0xa7, 0xd8, 0xd4, 0xdc, 0xea, 0x51,
  0x96, 0x48, 0xc4, 0x20, 0xc9, 0x54, 0x91, 0x55, 0xd2, 0x67, 0x2e, 0xb8, 0xd3, 0x0c,
  0x23, 0xf6, 0x02, 0x38, 0x4b, 0x94, 0xc6, 0x99, 0x25, 0xf7, 0x3e, 0xbf, 0xef, 0xa4,
  0x6e, 0xe2, 0x8e, 0xd5, 0xd9, 0xfa, 0xa3, 0x29, 0x2c, 0x5c, 0xb2, 0xad, 0x7e, 0x10,
  0x2d, 0x18, 0x39, 0xb1, 0xfb, 0x65, 0xdf, 0x23, 0xf1, 0x6f, 0x1c, 0x43, 0xed, 0x94,
  0xa9, 0x74, 0xad, 0xb9, 0x94, 0x8b, 0xcb, 0x0a, 0x41, 0xca, 0x99, 0x8b, 0xaf, 0x56,
  0x73, 0xb4, 0x79, 0xd1, 0x38, 0x7e, 0x9d, 0x3c, 0x57, 0x80, 0x43, 0x3a, 0x8b, 0xd7,
  0x3b, 0xcb, 0x27, 0x18, 0x00, 0x11, 0xc9, 0x63, 0x33, 0x89, 0x1b, 0xb4, 0x86, 0x66,
  0x88, 0xea, 0xa2, 0xe5, 0x1b, 0x3c, 0xe6, 0x2d, 0x06, 0xa5, 0x88, 0xd3, 0x46, 0xe4,
  0x30, 0x02, 0x56, 0x4f, 0x03, 0x08, 0x7c, 0x7c, 0x79, 0x56, 0x71, 0xdf, 0x71, 0x07,
  0xd8, 0xed, 0xc8, 0xb9, 0xf4, 0x51, 0xea, 0x1c, 0xf2, 0xfd, 0x06, 0x41, 0x19, 0x2b,
  0x93, 0x3b, 0x62, 0x9e, 0x86, 0xe5, 0x54, 0xb4, 0x3b, 0xe5, 0xa7, 0x6f, 0x84, 0x38,
  0x89, 0x5c, 0x3a, 0x1b, 0x2b, 0xe5, 0x67, 0x8d, 0xed, 0x49, 0x31, 0x21, 0x46, 0xd3,
  0x6b, 0xc5, 0x57, 0x00, 0x26, 0x08, 0x57, 0xcd, 0x5c, 0x6a, 0x08, 0xf0, 0x80, 0xfb,
  0x2f, 0xff, 0x65, 0xfb, 0x65, 0xd9, 0x83, 0xf1, 0xc4, 0xfa, 0x46, 0x76, 0x14, 0xc1,
  0x51, 0xc1, 0xf8, 0xdd, 0x48, 0x15, 0x27, 0x69, 0xa9, 0x44, 0xcb, 0x41, 0x3f, 0x9b,
  0x54, 0x1a, 0x1c, 0xfd, 0x54, 0x28, 0x94, 0xba, 0xcb, 0x9c, 0x97, 0xee, 0x49, 0x4b,
  0x9a, 0xf8, 0xe7, 0x1a, 0x79, 0xbf, 0x5a, 0xb6, 0xad, 0x35, 0x10, 0xfb, 0xd1, 0xc9,
  0x10, 0x1d, 0x51, 0x40, 0x4c, 0xbd, 0xdf, 0xd2, 0x6b, 0x34, 0x0c, 0x82, 0xd4, 0x3a,
  0x2f, 0x2f, 0xa3, 0x8b, 0xa5, 0x09, 0xf5, 0x01, 0x67, 0x86, 0x5f, 0x10, 0xdf, 0x4a,
  0x62, 0xb0, 0xb8, 0x2a, 0x98, 0xd7, 0x4c, 0xcf, 0x65, 0x84, 0xff, 0x0f, 0xa5, 0x57,
  0x92, 0xba, 0x8b, 0x9d, 0x5c, 0xc6, 0x7d, 0xc2, 0x77, 0xc4, 0x7c, 0x1c, 0x42, 0xe0,
  0x0b, 0x7b, 0xe5, 0xd6, 0x45, 0x9a, 0x86, 0xa1, 0xd3, 0xc1, 0x4a, 0x9a, 0x2e, 0x6a,
  0xb4, 0x3d, 0xb2, 0xe7, 0xd5, 0x7e, 0x13, 0x6f, 0xc7, 0x68, 0x13, 0xa2, 0x9a, 0x86,
  0x7a, 0x08, 0xb9, 0x2a, 0xbf, 0x48, 0x38, 0x9e, 0x6b, 0x62, 0xb4, 0xa0, 0x9a, 0xc0,
  0x75, 0x1f, 0x71, 0xb1, 0xc6, 0x6d, 0x9a, 0xe4, 0x9c, 0xcb, 0xe0, 0xc3, 0x52, 0x94,
  0x6d, 0x5f, 0xd5, 0x21, 0x01, 0x03, 0x88, 0x1f, 0x93, 0x69, 0xc3, 0xcc, 0x50, 0xb1,
  0xb0, 0x4d, 0xb3, 0x11, 0xa0, 0x5d, 0x33, 0x0b, 0x65, 0xe1, 0xf4, 0xd6, 0x56, 0x68,
  0xe4, 0xcd, 0xae, 0xdd, 0xf1, 0xbc, 0x1c, 0x1f, 0xa1, 0xde, 0x19, 0x7a, 0x0f, 0x54,
  0x39, 0xf9, 0xc8, 0xff, 0x91, 0x20, 0x16, 0x98, 0x73, 0xac, 0x72, 0xd2, 0xf2, 0xe7,
  0x18, 0x18, 0x1a, 0xbd, 0xcd, 0xbb, 0x0c, 0xc1, 0x2a, 0x23, 0x33, 0xec, 0xcf, 0x6d,
  0xae, 0x0a, 0xa9, 0xfe, 0xbf, 0x75, 0xb2, 0xa7, 0xda, 0x1b, 0xc8, 0x0c, 0x65, 0x02,
  0x74, 0xed, 0x35, 0xbd, 0x49, 0x68, 0x2a, 0x40, 0x23, 0xfe, 0x41, 0x3b, 0xb2, 0x45,
  0x39, 0x4c, 0xb2, 0x9c, 0xfb, 0xe6, 0x10, 0xc0, 0xf3, 0xee, 0x4b, 0xd2, 0x2c, 0xde,
  0x92, 0xb6, 0x51, 0xbc, 0x53, 0x59, 0x44, 0xa7, 0x37, 0x3c, 0x93, 0x24, 0xff, 0x8a,
  0xf6, 0xfd, 0x6f, 0x48, 0x77, 0x06, 0x7b, 0x89, 0xf9, 0x5f, 0x79, 0xd7, 0x49, 0x20,
  0x62, 0x39, 0x0c, 0xa3, 0xdd, 0x29, 0x9a, 0xa2, 0x0b, 0xd5, 0xdd, 0xbb, 0xdd, 0x24,
  0xa9, 0xda, 0x85, 0x79, 0x95, 0xf1, 0xc5, 0x83, 0x6d, 0x0e, 0xf3, 0x32, 0x37, 0xc8,
  0x57, 0x22, 0xbe, 0xaf, 0xa0, 0x3d, 0x9c, 0xe6, 0xba, 0x83, 0x9b, 0x40, 0xe7, 0x3f,
  0xd7, 0xff, 0x82, 0xcf, 0x64, 0x33, 0x9f, 0x1c, 0xa3, 0x15, 0xd5, 0x7f, 0x60, 0x2f,
  0xdb, 0x21, 0x9b, 0xd2, 0x49, 0x63, 0xba, 0x37, 0x50, 0x97, 0xf2, 0x2d, 0x16, 0xd8,
  0x80, 0x4b, 0x1f, 0x73, 0x91, 0xf6, 0x2f, 0x46, 0x7a, 0x2c, 0x8d, 0x25, 0x1f, 0xa2,
  0x7d, 0x0b, 0x5e, 0xae, 0xd4, 0x10, 0x52, 0x12, 0xf4, 0xb6, 0x9e, 0x5e, 0x99, 0xbc,
  0x2a, 0x80, 0xbf, 0x86, 0x25, 0x4c, 0xab, 0xb3, 0xb3, 0x78, 0xcf, 0x1b, 0x3b, 0x8a,
  0x83, 0xa5, 0xdb, 0xbd, 0x61, 0x55, 0xcf, 0x6d, 0x91, 0x7c, 0x35, 0x75, 0x7c, 0x2f,
  0xae, 0x58, 0x55, 0x0d, 0xa7, 0x1b, 0x97, 0xfe, 0xde, 0x97, 0x28, 0xa3, 0x2b, 0x81,
  0xc0, 0xdf, 0x3d, 0xf1, 0xee, 0xc2, 0xbb, 0xce, 0x5a, 0xdb, 0xb0, 0xab, 0x8c, 0xb4,
  0xd3, 0x9e, 0x2c, 0x4d, 0x56, 0xfd, 0xaf, 0x3d, 0xa1, 0xe7, 0xd2, 0xdd, 0xc0, 0xfb,
  0xa9, 0xcb, 0x2d, 0x2b, 0x69, 0xa3, 0x25, 0x75, 0xf4, 0x77, 0x7d, 0xf3, 0x1d, 0xf8,
  0xe1, 0x74, 0xeb, 0xe6, 0x4d, 0x16, 0xc4, 0xff, 0x97, 0x12, 0x44, 0x11, 0xca, 0xed,
  0x2e, 0xf7, 0x71, 0x54, 0xc2, 0x1f, 0xe3, 0xbe, 0x14, 0x5c, 0x47, 0xa3, 0xdb, 0x82,
  0x11, 0x51, 0xa1, 0xbe, 0x97, 0x4d, 0x2c, 0xd9, 0x50, 0xe3, 0x93, 0xff, 0x73, 0xa1,
  0xf6, 0x28, 0x17, 0xb4, 0x47, 0x84, 0x89, 0xe5, 0x9d, 0x91, 0x57, 0xd8, 0xfc, 0x8f,
  0xf8, 0xc8, 0x72, 0x2e, 0x0e, 0x7c, 0x04, 0x81, 0xf5, 0x6c, 0x68, 0x0e, 0x12, 0x93,
  0xca, 0x45, 0x4d, 0x58, 0xca, 0xd2, 0x95, 0xfa, 0x12, 0xa3, 0x15, 0x53, 0xc7, 0x5e,
  0xc2, 0x0b, 0x6f, 0xea, 0x0a, 0xa0, 0xbf, 0xce, 0xbe, 0x7f, 0x2f, 0x84, 0xc1, 0xc4,
  0x81, 0x94, 0x2c, 0xb3, 0x63, 0xf7, 0xc3, 0x21, 0x9b, 0x1a, 0xe4, 0x19, 0xf5, 0x99,
  0xb6, 0xab, 0xb3, 0x1c, 0x30, 0x34, 0xec, 0xdd, 0xd4, 0x82, 0x46, 0x2c, 0x0d, 0x49,
  0x40, 0x76, 0x7e, 0xc6, 0x2a, 0x58, 0x00, 0x51, 0xfc, 0x54, 0xbd, 0xb4, 0x2b, 0x9c,
  0xc0, 0xa6, 0x63, 0xe7, 0x1c, 0x6c, 0xa9, 0x16, 0xd9, 0x08, 0x18, 0x32, 0xdc, 0x80,
  0x08, 0xc0, 0x81, 0x99, 0x1c, 0xa4, 0xc2, 0xe3, 0x5d, 0x0a, 0x56, 0xf8, 0xe3, 0x4d,
  0x56, 0x4b, 0xa8, 0x46, 0xf8, 0x80, 0xaf, 0x87, 0x01, 0x61, 0xc9, 0xf1, 0x83, 0x5f,
  0xe3, 0x2c, 0x40, 0x0a, 0xed, 0x01, 0x43, 0x5d, 0x08, 0x75, 0xbb, 0xe7, 0x5a, 0xcf,
  0x0f, 0xbc, 0xc9, 0x04, 0xa7, 0xac, 0xe7, 0x37, 0x3d, 0x8a, 0x15, 0x45, 0xe9, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9c, 0xb2, 0x2f, 0x57, 0x86, 0x2b, 0xd2,
  0x29, 0x26, 0x5b, 0x1d, 0xaa, 0xc5, 0x77, 0x30, 0xf5, 0x47, 0xc9, 0x98, 0xce, 0x1a,
  0xbd, 0x77, 0x18, 0x91, 0xa5, 0xd2, 0x66, 0x65, 0xd4, 0x86, 0x48, 0xe2, 0xb3, 0x21,
  0x8d, 0x9b, 0x60, 0xdd, 0xe2, 0x4c, 0x2d, 0x57, 0x18, 0x4b, 0x48, 0x8d, 0x36, 0x7e,
  0x4c, 0xe1, 0xf4, 0xc9, 0x81, 0x2e, 0x36, 0xd8, 0x1c, 0xef, 0x4a, 0xbf, 0x4e, 0xad,
  0x8c, 0x4c, 0x71, 0x82, 0x90, 0xc4, 0x65, 0xad, 0x4c, 0x3b, 0x9a, 0xf1, 0x25, 0x8d,
  0xb0, 0xe0, 0x09, 0xae, 0xf2, 0xb3, 0x0b, 0x17, 0xf4, 0x97, 0xa3, 0xaf, 0xcb, 0x85,
  0x82, 0xd7, 0xbe, 0xdd, 0x8c, 0xf6, 0x01, 0x40, 0x40, 0x40, 0x40, 0x45, 0x63, 0x49,
  0x14, 0x3c, 0x10, 0x7b, 0xe8, 0xcd, 0x87, 0x89, 0xaa, 0xb0, 0xeb, 0x08, 0x7b, 0x72,
  0x6e, 0xb3, 0x6c, 0x2d, 0x4e, 0x1f, 0x97, 0x83, 0x81, 0x65, 0x58, 0x87, 0x3b, 0x4a,
  0x7b, 0x11, 0x33, 0x4c, 0x63, 0xb2, 0xa6, 0xa0, 0x5d, 0x58, 0x9a, 0x25, 0xe1, 0xf3,
  0x64, 0xc3, 0x70, 0xdc, 0x9d, 0x0b, 0x4f, 0xf2, 0x05, 0x97, 0xb9, 0x24, 0xc6, 0xfa,
  0xf5, 0x2b, 0xcb, 0x82, 0x12, 0x62, 0xb9, 0xaa, 0x44, 0xe3, 0x1d, 0x65, 0x6b, 0x75,
  0x39, 0x35, 0xda, 0xcf, 0x77, 0x33, 0xc0, 0x65, 0xf2, 0x98, 0xb9, 0x6c, 0xd8, 0x75,
  0xe0, 0xd5, 0x7a, 0x7a, 0xfd, 0xa5, 0xcf, 0xa6, 0x9b, 0x40, 0xdb, 0xc8, 0x3d, 0xa2,
  0xc1, 0x4a, 0x64, 0x8d, 0x79, 0x60, 0x08, 0x8b, 0x98, 0x7a, 0xdf, 0x3d, 0x31, 0x9e,
  0x35, 0x57, 0xde, 0xbb, 0x5b, 0xcd, 0x1d, 0x89, 0x85, 0x38, 0x80, 0x80, 0x43, 0x8c,
  0x5b, 0x66, 0x8e, 0xd0, 0x12, 0xbb, 0xde, 0x0c, 0x54, 0x9b, 0x71, 0x4e, 0xc5, 0x82,
  0xa5, 0x95, 0x20, 0xd2, 0x2c, 0x89, 0x4d, 0xe1, 0x02, 0x6a, 0x39, 0x2d, 0x7f, 0xcf,
  0x7b, 0x06, 0xa8, 0x9f, 0x79, 0xdc, 0xac, 0x5f, 0x07, 0xbe, 0x75, 0x42, 0x9b, 0x07,
  0x13, 0x42, 0x55, 0x30, 0x61, 0x37, 0xbe, 0x2a, 0xc7, 0xd7, 0x1d, 0x9f, 0xaa, 0x24,
  0xa4, 0xa5, 0x75, 0xde, 0x62, 0x9c, 0x72, 0x44, 0x79, 0xc7, 0xe8, 0x91, 0x8c, 0x56,
  0xfa, 0x0f, 0x10, 0xe6, 0x0a, 0xcd, 0xd0, 0xbb, 0xa3, 0x56, 0xbc, 0x43, 0x75, 0xc5,
  0x4e, 0xd9, 0x58, 0x51, 0x73, 0x7c, 0x9d, 0x62, 0x1e, 0xda, 0x35, 0x92, 0x63, 0x6f,
  0x2d, 0xd8, 0xc7, 0x30, 0x12, 0xdc, 0xef, 0xe7, 0xa9, 0x57, 0x47, 0x9d, 0x92, 0xeb,
  0x49, 0xa7, 0xd5, 0x20, 0x00, 0x84, 0xd1, 0x91, 0x4c, 0xe8, 0x9b, 0xd8, 0xea, 0x30,
  0x58, 0x28, 0x1d, 0x2f, 0x79, 0x96, 0x8c, 0x76, 0x6a, 0x1c, 0xa4, 0xa9, 0x5a, 0x20,
  0x02, 0xc0, 0x85, 0x3b, 0xa4, 0xef, 0x95, 0x47, 0x3a, 0x5c, 0x75, 0xa7, 0xdd, 0x36,
  0x69, 0xa2, 0x17, 0xbd, 0xeb, 0x51, 0x43, 0xe3, 0xdd, 0x42, 0x1e, 0x60, 0x0c, 0xc8,
  0xd8, 0x20, 0xcd, 0x8e, 0x44, 0xab, 0x6b, 0x4c, 0x93, 0x44, 0x15, 0xec, 0x74, 0xac,
  0x94, 0xcb, 0x81, 0x4d, 0x85, 0x38, 0x0e, 0x52, 0xf9, 0x50, 0x5f, 0x00, 0x2e, 0xbd,
  0x70, 0xf6, 0x41, 0xd5, 0xa0, 0xe7, 0x65, 0x91, 0x16, 0x38, 0xe4, 0x0d, 0x7c, 0xbc,
  0x9d, 0x21, 0x2d, 0x82, 0x4a, 0xa7, 0x1a, 0xed, 0x12, 0x86, 0x5b, 0x04, 0xc0, 0x43,
  0x21, 0xd8, 0xa5, 0xd2, 0xf4, 0x33, 0x72, 0x01, 0x0d, 0x41, 0x08, 0xb4, 0x5f, 0x5b,
  0x6b, 0x54, 0xf7, 0xd7, 0xba, 0xd5, 0x08, 0x7f, 0x35, 0x09, 0xe5, 0x9d, 0xcf, 0x3e,
  0x14, 0x55, 0xb5, 0x66, 0x14, 0x19, 0x4e, 0x9a, 0x15, 0xa4, 0x56, 0x27, 0x77, 0x98,
  0x34, 0x08, 0x05, 0xab, 0x16, 0x2d, 0x6b, 0x4c, 0x28, 0xee, 0x3e, 0x00, 0xa2, 0x7f,
  0xb3, 0x5a, 0xba, 0xef, 0x4c, 0x21, 0x38, 0x22, 0xda, 0x17, 0x97, 0x25, 0xa6, 0x00,
  0x05, 0x22, 0xa6, 0xb8, 0x87, 0xc1, 0xa3, 0x83, 0xd5, 0xee, 0x42, 0x8b, 0xc6, 0x67,
  0xd6, 0x4a, 0xfa, 0xa3, 0x55, 0x3a, 0xd7, 0xeb, 0xc1, 0xa8, 0x2c, 0xd5, 0xd2, 0xcb,
  0xbd, 0x98, 0x11, 0xd2, 0xf7, 0x4a, 0xed, 0x7c, 0x38, 0x2e, 0x64, 0x90, 0xa6, 0x52,
  0xe1, 0xd6, 0x2a, 0x88, 0xdc, 0x07, 0xec, 0x3b, 0xbe, 0x90, 0x8b, 0xf5, 0x6e, 0xe6,
  0xcd, 0xff, 0xf2, 0x78, 0xc3, 0x11, 0x78, 0xd4, 0xf8, 0xe2, 0xe7, 0x0e, 0x89, 0xde,
  0x9c, 0xf9, 0xab, 0xb8, 0x1b, 0x11, 0xdf, 0xbe, 0xec, 0x84, 0x25, 0x8f, 0xc8, 0x89,
  0xb3, 0x87, 0xbb, 0x18, 0xf6, 0x6e, 0xa9, 0xf2, 0xce, 0x56, 0xc3, 0x49, 0x64, 0x0e,
  0x4f, 0x49, 0xf8, 0xdc, 0x71, 0x99, 0xcc, 0x13, 0xbb, 0x27, 0xa6, 0x41, 0xfc, 0xd9,
  0x85, 0x64, 0x13, 0xa4, 0x5b, 0x02, 0x1a, 0x23, 0x86, 0x55, 0x58, 0xb4, 0x18, 0x60,
  0xfb, 0xef, 0xe5, 0xda, 0xeb, 0x1e, 0x01, 0x99, 0x44, 0xfe, 0x39, 0x16, 0xcd, 0x9d,
  0x9c, 0xd9, 0xd3, 0x1e, 0x9e, 0x41, 0xb5, 0x28, 0x9e, 0x97, 0x7b, 0xba, 0xc1, 0xda,
  0x9e, 0xdb, 0xbd, 0xb5, 0xbf, 0x20, 0x4a, 0xae, 0x5c, 0xad, 0x29, 0xc7, 0x4d, 0x34,
  0xac, 0x79, 0xee, 0xe8, 0xa5, 0xf4, 0xbb, 0xe4, 0xa5, 0xd0, 0x75, 0x3f, 0x7c, 0x52,
  0x9c, 0xa1, 0x49, 0x24, 0x48, 0x92, 0x4c, 0x86, 0xaf, 0x28, 0xf3, 0xec, 0x37, 0x8f,
  0x59, 0x3f, 0xd9, 0x48, 0x49, 0xc6, 0x05, 0x8d, 0xdd, 0x83, 0x6d, 0x97, 0x85, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb2, 0x50, 0x1e, 0xa0, 0x88, 0x77, 0xf7,
  0xaf, 0xe1, 0xe0, 0xea, 0x37, 0x94, 0x4e, 0xbe, 0x26, 0x4c, 0x48, 0xa8, 0x56, 0xdf,
  0x74, 0x31, 0x55, 0x88, 0x2f, 0xbd, 0x9c, 0x3a, 0x15, 0x36, 0xdf, 0xc3, 0xe4, 0x88,
  0x2d, 0xd6, 0x04, 0x86, 0x05, 0x84, 0x71, 0xf1, 0xce, 0x1f, 0x22, 0xb4, 0x34, 0xd2,
  0xf2, 0x5c, 0xdd, 0xde, 0x1e, 0xc2, 0x1e, 0x3f, 0x08, 0xda, 0x68, 0xaa, 0x0c, 0xee,
  0x0c, 0xc0, 0xb2, 0xf6, 0x0d, 0x22, 0xeb, 0xe5, 0xa8, 0x24, 0x51, 0x76, 0x4b, 0xc5,
  0xc4, 0x70, 0xb9, 0xfc, 0x70, 0x2e, 0x42, 0x56, 0xaa, 0xfe, 0xd3, 0xb1, 0x0b, 0x22,
  0xfa, 0xd9, 0x88, 0xe1, 0x89, 0xa5, 0xf1, 0x78, 0x43, 0x52, 0x30, 0x6d, 0x42, 0x86,
  0x12, 0xe5, 0x8d, 0x5d, 0xc3, 0xb7, 0xb9, 0x8b, 0x21, 0x71, 0x62, 0xef, 0x27, 0x3d,
  0xaa, 0x31, 0x68, 0x2a, 0xfb, 0xc9, 0xc8, 0x36, 0x22, 0xaa, 0x46, 0x91, 0x3c, 0x54,
  0xfd, 0xcf, 0xf3, 0x4d, 0xd0, 0xc7, 0x0e, 0xb2, 0x55, 0x3e, 0x9a, 0x58, 0xe7, 0xd5,
  0xff, 0xce, 0xc6, 0x80, 0x4d, 0x83, 0xba, 0x06, 0xf0, 0xe3, 0x44, 0xc2, 0x58, 0x63,
  0xc8, 0x81, 0x8e, 0x39, 0x25, 0x44, 0x99, 0x18, 0xd7, 0xd5, 0xa0, 0x26, 0x1f, 0xbf,
  0x69, 0x3d, 0x0a, 0xe1, 0x96, 0x2d, 0x8a, 0xb1, 0x21, 0xe6, 0xe6, 0xc7, 0x23, 0x8f,
  0x1b, 0x55, 0x18, 0x7d, 0xa1, 0x67, 0xd2, 0x5d, 0xd3, 0xfd, 0x2b, 0x26, 0x11, 0x6d,
  0xa8, 0x72, 0x31, 0x4c, 0x67, 0xc0, 0xd2, 0x48, 0xf2, 0x7b, 0xb9, 0x08, 0x79, 0xc9,
  0x42, 0xcf, 0x48, 0xc9, 0x4c, 0x68, 0x85, 0x60, 0x89, 0x55, 0x89, 0x90, 0xe5, 0x42,
  0x79, 0x64, 0xaf, 0xd0, 0x75, 0xc5, 0x6c, 0xb0, 0x18, 0x92, 0xdc, 0x88, 0x9a, 0x28,
  0xc8, 0x4f, 0x7b, 0x6c, 0x85, 0xf8, 0x97, 0x50, 0x26, 0xa0, 0x34, 0xa0, 0x2f, 0xc2,
  0x11, 0xa3, 0xe3, 0xcc, 0x92, 0x2a, 0x3c, 0x8b, 0x42, 0xd2, 0x61, 0x7f, 0x04, 0x15,
  0xb1, 0x12, 0x02, 0x04, 0xe1, 0xbd, 0x84, 0x02, 0x43, 0x72, 0xda, 0xd9, 0x68, 0xab,
  0x53, 0xfc, 0xe1, 0xa6, 0x19, 0x92, 0x58, 0x0c, 0x68, 0x91, 0x3b, 0x77, 0x86, 0xca,
  0xe4, 0x4d, 0x9e, 0xe6, 0x2e, 0xe8, 0x38, 0x65, 0xe6, 0x94, 0xcf, 0x8c, 0x0e, 0xc2,
  0xa1, 0xeb, 0x3a, 0x65, 0xef, 0x46, 0x29, 0xae, 0x8e, 0x30, 0xb4, 0x08, 0x4b, 0xf4,
  0xa5, 0x77, 0xe3, 0x3d, 0x72, 0x36, 0xe2, 0xc6, 0x8d, 0x80, 0x0b, 0x17, 0xbc, 0xaf,
  0xdd, 0xf6, 0x4d, 0x48, 0xe1, 0x62, 0x37, 0x67, 0x18, 0x6a, 0xfc, 0x77, 0xd5, 0x17,
  0x4e, 0xb9, 0x23, 0xc9, 0x0e, 0xb6, 0xfa, 0x53, 0xc3, 0xa7, 0xaf, 0x4f, 0x5c, 0x85,
  0x4b, 0xf2, 0x21, 0x16, 0x3b, 0x78, 0x0b, 0xf6, 0xc2, 0xe4, 0x2f, 0x89, 0x28, 0xdf,
  0xba, 0x99, 0xd3, 0xa4, 0xbf, 0xee, 0xa0, 0xfc, 0x01, 0x40, 0x40, 0x40, 0x40, 0xb2,
  0xdd, 0xb8, 0x67, 0x66, 0xd0, 0xc5, 0xa8, 0x57, 0x9d, 0xfb, 0x04, 0x6c, 0x0f, 0xe8,
  0x01, 0x88, 0xd4, 0x7e, 0xdc, 0x43, 0xcb, 0x0c, 0x90, 0xf4, 0x7d, 0xa9, 0xd2, 0xda,
  0x30, 0xbe, 0x04, 0xb0, 0x2e, 0x33, 0x25, 0xfd, 0xdc, 0x20, 0x7e, 0x4e, 0x38, 0x0c,
  0x74, 0xed, 0x51, 0xda, 0x0e, 0x60, 0xa5, 0x9a, 0xb4, 0x50, 0x14, 0x82, 0x12, 0x65,
  0xb6, 0x1c, 0x27, 0x70, 0xed, 0xdb, 0x82, 0xd5, 0x49, 0x87, 0x29, 0x22, 0x4f, 0xc6,
  0x6e, 0x63, 0xdc, 0x8b, 0x6c, 0x79, 0xf5, 0xff, 0xf8, 0x6e, 0x34, 0x30, 0x3f, 0x60,
  0xb7, 0xd3, 0xcf, 0xc5, 0x7c, 0x5e, 0x90, 0x01, 0x17, 0xe7, 0xf3, 0x37, 0x24, 0x7c,
  0x9b, 0x79, 0xcd, 0xfe, 0x4b, 0xb2, 0x55, 0xf8, 0x08, 0xd1, 0x22, 0xc9, 0xf5, 0x0c,
  0xf8, 0x6e, 0xbf, 0x16, 0x0c, 0xa5, 0xd6, 0xc1, 0x59, 0xfd, 0xd2, 0x24, 0x5f, 0x80,
  0x93, 0x1e, 0x3f, 0x6b, 0xf0, 0x0a, 0xec, 0x5f, 0x4a, 0x21, 0x1d, 0x57, 0x2a, 0x81,
  0x4b, 0x6b, 0x5b, 0x5c, 0xda, 0x39, 0x09, 0xa0, 0x76, 0x42, 0x43, 0xfc, 0x41, 0x7b,
  0x6c, 0x53, 0x73, 0xd3, 0x3c, 0x8c, 0xb8, 0x4b, 0xe5, 0x00, 0xcf, 0x42, 0x37, 0x4d,
  0x92, 0xdb, 0x78, 0xee, 0x8f, 0x88, 0xb9, 0x86, 0x1b, 0x12, 0x0a, 0x6e, 0xe5, 0x2b,
  0x16, 0x77, 0x1e, 0x22, 0xe6, 0x38, 0x8d, 0x83, 0x9a, 0x90, 0xd5, 0xaf, 0x98, 0x77,
  0x5a, 0x5e, 0x75, 0xed, 0x94, 0xa7, 0xdb, 0x50, 0x43, 0x56, 0x7a, 0x72, 0xcf, 0x5e,
  0x91, 0x3f, 0xeb, 0x48, 0x10, 0x3d, 0x88, 0x86, 0x38, 0x1f, 0x79, 0x32, 0x68, 0xeb,
  0x24, 0x6e, 0x3d, 0x2c, 0xd7, 0x8c, 0xf6, 0x58, 0xc6, 0xd6, 0x0c, 0xaa, 0xa4, 0x91,
  0xed, 0x56, 0x54, 0x57, 0x17, 0x63, 0x9b, 0xc5, 0x4b, 0xb7, 0x2f, 0x12, 0x4d, 0xcd,
  0xc1, 0xb8, 0xe9, 0xf5, 0x6c, 0x1a, 0x44, 0x69, 0x73, 0xf7, 0xb3, 0xd4, 0x1b, 0x72,
  0xe9, 0xd1, 0x3b, 0xd8, 0x08, 0x87, 0x7c, 0xc4, 0xb6, 0x2b, 0x96, 0x28, 0xb6, 0xb0,
  0x88, 0x97, 0x1b, 0x9b, 0x80, 0xe8, 0x93, 0xdd, 0x58, 0x2e, 0x3c, 0xfd, 0x54, 0xc7,
  0x32, 0xff, 0x3b, 0x6e, 0x5f, 0x96, 0x7b, 0x4a, 0xea, 0x65, 0x17, 0x30, 0x2b, 0x84,
  0x36, 0xb6, 0x0d, 0xc6, 0x64, 0x1c, 0x48, 0x95, 0xd2, 0x0a, 0x62, 0x0c, 0xf3, 0x2f,
  0x94, 0xb3, 0x25, 0x29, 0x06, 0xe5, 0xf3, 0xbe, 0xda, 0x8d, 0xa7, 0x0c, 0xe5, 0x7c,
  0xa2, 0x52, 0xab, 0x3e, 0x7f, 0xac, 0x99, 0x4c, 0xdd, 0xee, 0x43, 0x94, 0xea, 0xd1,
  0x4c, 0xfa, 0xbb, 0xcf, 0x77, 0x9f, 0x7d, 0xf5, 0x8b, 0x77, 0x80, 0x42, 0x03, 0xc1,
  0x3f, 0x84, 0x37, 0x3a, 0x84, 0x32, 0x21, 0x4c, 0xbf, 0xe4, 0x24, 0xfa, 0xfd, 0x6e,
  0xff, 0x49, 0x4d, 0x77, 0x22, 0x95, 0x4c, 0xd0, 0x06, 0x4b, 0xee, 0x2f, 0x1a, 0x48,
  0x9b, 0x54, 0x35, 0x37, 0x12, 0xe5, 0xaa, 0xda, 0xb3, 0xb2, 0x9e, 0xe7, 0xe1, 0x66,
  0xae, 0xef, 0x7b, 0x66, 0x82, 0xe2, 0x22, 0xef, 0xc2, 0x39, 0x6a, 0x6d, 0x6a, 0xe5,
  0x36, 0x16, 0x8d, 0x79, 0x11, 0xa5, 0x37, 0x1d, 0xa8, 0x6b, 0xb1, 0x2e, 0xb8, 0xab,
  0x15, 0x4a, 0x46, 0xcc, 0x15, 0x2e, 0x49, 0xba, 0x43, 0x4a, 0x2d, 0x96, 0x44, 0xa5,
  0x71, 0x62, 0xf1, 0xe4, 0xd8, 0x91, 0x13, 0xa6, 0xd7, 0x9c, 0xd8, 0xb6, 0xde, 0xd5,
  0x09, 0x1d, 0x85, 0xe5, 0x58, 0xb5, 0x2d, 0x19, 0xcb, 0x80, 0x5d, 0x9f, 0x02, 0x6f,
  0x53, 0x9c, 0xbc, 0x57, 0xab, 0xb8, 0xdc, 0x4b, 0x88, 0xfb, 0x8e, 0x1e, 0x9d, 0x8b,
  0x0b, 0x8a, 0x2b, 0x2c, 0xb9, 0x1d, 0xbb, 0x0f, 0xcb, 0x31, 0x5f, 0x3c, 0x61, 0xd8,
  0x0f, 0x31, 0xf8, 0xa6, 0xd7, 0xb3, 0x4e, 0xeb, 0x4b, 0xd1, 0xa0, 0x27, 0x7f, 0x0e,
  0x77, 0x56, 0xad, 0x3d, 0x01, 0xe2, 0x06, 0xb8, 0xdf, 0x3a, 0xe3, 0x1a, 0x92, 0x13,
  0x74, 0xa3, 0x3c, 0x68, 0x4b, 0x97, 0x86, 0x23, 0xb8, 0x72, 0x1b, 0x96, 0x92, 0x80,
  0xa4, 0x95, 0xbe, 0x55, 0x58, 0xd3, 0x95, 0x11, 0x11, 0x67, 0xab, 0x42, 0xa6, 0xf4,
  0x8e, 0x08, 0xd2, 0x9f, 0x12, 0x8b, 0x16, 0xa4, 0xe5, 0x4d, 0xa7, 0x5f, 0x19, 0x10,
  0xbf, 0x0a, 0x8c, 0xdf, 0x51, 0xda, 0x4c, 0xc1, 0x48, 0xf7, 0xd0, 0x39, 0xcf, 0x4d,
  0x8f, 0xf8, 0x83, 0xd6, 0x84, 0xd4, 0x66, 0x61, 0x90, 0xa3, 0xca, 0x1a, 0x28, 0xf0,
  0x97, 0x64, 0xee, 0x68, 0x1f, 0x7b, 0x33, 0xc4, 0x93, 0xc9, 0xf1, 0x03, 0x70, 0x69,
  0x47, 0xb7, 0xa3, 0x5c, 0x26, 0x1d, 0x38, 0xbb, 0x88, 0x04, 0x72, 0x1b, 0x18, 0x2e,
  0xb4, 0x85, 0x5d, 0xbc, 0xe1, 0xcc, 0x70, 0xa6, 0x6b, 0x60, 0xe9, 0x57, 0xfd, 0xb0,
  0xa1, 0xb3, 0xd3, 0x8c, 0x80, 0x92, 0x4d, 0x83, 0x4b, 0x07, 0xdd, 0xf3, 0xad, 0x92,
  0x4f, 0xea, 0xc2, 0xd5, 0x89, 0x6f, 0x33, 0x50, 0xa7, 0x23, 0x9a, 0x5f, 0x8f, 0x80,
  0x3e, 0x0d, 0x93, 0x5a, 0x5b, 0xce, 0x7c, 0x5d, 0x5d, 0x71, 0xac, 0x20, 0x01, 0x76,
  0xdc, 0xa7, 0x5e, 0xe2, 0x30, 0xc6, 0x4a, 0x16, 0xde, 0x1d, 0x2a, 0x12, 0x88, 0x3d,
  0x75, 0xe1, 0x96, 0x2e, 0x3c, 0xf9, 0x67, 0x17, 0xe0, 0x32, 0xa0, 0x6d, 0x23, 0x99,
  0x1a, 0x0d, 0xef, 0x6b, 0x19, 0x21, 0x69, 0xda, 0x81, 0xe1, 0x35, 0xbe, 0xa2, 0xe6,
  0xdd, 0x12, 0x19, 0x05, 0xad, 0x7f, 0xbd, 0xd0, 0x08, 0x6f, 0x87, 0x71, 0x10, 0x3a,
  0x26, 0x2e, 0xca, 0xff, 0x0c, 0x77, 0x83, 0x2d, 0xc4, 0x40, 0x7c, 0x43, 0x11, 0x2e,
  0x58, 0x59, 0xc1, 0xb1, 0x3c, 0xc2, 0x00, 0x87, 0xc8, 0xa2, 0x38, 0xc8, 0x30, 0x27,
  0x83, 0x7f, 0xcf, 0xdf, 0x8a, 0x70, 0xf7, 0x63, 0xd0, 0xb0, 0x1e, 0xe9, 0xd5, 0xf8,
  0x86, 0x1b, 0x1c, 0x87, 0xdc, 0xef, 0x4a, 0x97, 0xc4, 0xec, 0x03, 0x20, 0xd0, 0xd6,
  0xe5, 0x5d, 0x4d, 0x75, 0x84, 0x11, 0x06, 0x7e, 0xba, 0x34, 0xd6, 0x3e, 0xdb, 0x2f,
  0x65, 0x19, 0x0a, 0x91, 0xef, 0x37, 0xb3, 0x55, 0x3e, 0xf7, 0x7e, 0x08, 0x18, 0x26,
  0xbe, 0x82, 0x98, 0x46, 0x45, 0x9b, 0x46, 0xd2, 0x79, 0x99, 0x44, 0xd4, 0xa6, 0x3a,
  0xb9, 0xb5, 0x3a, 0x57, 0x9e, 0x5e, 0x43, 0x0d, 0x47, 0x23, 0x5c, 0x8e, 0xa4, 0x29,
  0x51, 0x1e, 0x13, 0x04, 0x9a, 0x15, 0x65, 0xeb, 0x53, 0x85, 0x03, 0x98, 0x75, 0xf2,
  0xf5, 0x14, 0xf4, 0x66, 0x43, 0x6b, 0x03, 0xbf, 0x31, 0x8d, 0x6e, 0x64, 0x2d, 0xa9,
  0x89, 0x62, 0x6a, 0xd5, 0xd8, 0x3e, 0xda, 0x37, 0x72, 0x36, 0xb1, 0x5d, 0xa6, 0x2a,
  0x2e, 0x0a, 0x31, 0xad, 0x8f, 0xc6, 0x69, 0xb9, 0xa7, 0x08, 0x0a, 0xa6, 0x29, 0xfd,
  0x44, 0x5f, 0xd0, 0x21, 0x84, 0x54, 0xd4, 0x37, 0xa2, 0x6c, 0x7d, 0x17, 0x1c, 0x25,
  0xe9, 0x45, 0x3a, 0x04, 0x7f, 0xbf, 0x8c, 0xba, 0x16, 0xeb, 0x43, 0x97, 0xf5, 0x05,
  0x95, 0x71, 0x63, 0x59, 0xc1, 0x33, 0xaa, 0xaf, 0x60, 0xb6, 0x32, 0x39, 0x65, 0x7a,
  0x75, 0xfc, 0x55, 0x5c, 0x80, 0xc5, 0xeb, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x29, 0xe3, 0xa3, 0xc1, 0xf0, 0x9b, 0x30, 0xd0, 0xaa, 0xd0, 0xb4, 0xe0, 0x74,
  0x05, 0xbd, 0xb7, 0xe3, 0xa0, 0xf6, 0x7b, 0x91, 0x7a, 0x21, 0xda, 0x3b, 0x3a, 0x07,
  0xc8, 0x7a, 0xc8, 0x5d, 0xde, 0x99, 0xdb, 0xb5, 0x05, 0xb1, 0x97, 0xd9, 0xb7, 0x14,
  0x06, 0x9b, 0x9f, 0x26, 0x46, 0x92, 0xa8, 0xdc, 0x70, 0xc2, 0xd1, 0x4e, 0x7a, 0xf2,
  0x9f, 0xe3, 0xff, 0x83, 0xc2, 0xc7, 0xc5, 0xe7, 0xaf, 0xf6, 0x01, 0x40, 0x40, 0x40,
  0x40, 0x33, 0x23, 0x0c, 0x9d, 0xde, 0xf6, 0x0f, 0x1f, 0xf3, 0x67, 0xd2, 0x21, 0x66,
  0x85, 0xf5, 0x0f, 0x42, 0x61, 0xe3, 0x02, 0x37, 0x01, 0x97, 0x3b, 0x82, 0xfa, 0x7b,
  0x55, 0xe8, 0x4c, 0x79, 0xaf, 0x17, 0xb1, 0x0c, 0x03, 0x8b, 0x7b, 0x59, 0x80, 0x1c,
  0xd3, 0x92, 0xc4, 0x2b, 0x62, 0x23, 0x6f, 0x0b, 0x75, 0xf0, 0x8b, 0x2f, 0xd0, 0xa2,
  0x76, 0xdc, 0xdf, 0xe2, 0x55, 0x0b, 0x9f, 0x4e, 0x05, 0x63, 0x79, 0x56, 0xab, 0xa1,
  0xb6, 0x41, 0x9b, 0xde, 0x8b, 0xdb, 0x9e, 0x13, 0x93, 0x0d, 0xe5, 0xbb, 0xf1, 0x31,
  0x1c, 0xa3, 0x09, 0x41, 0x24, 0xc8, 0x0e, 0x76, 0xee, 0xe5, 0xc5, 0xa3, 0x02, 0xa0,
  0xe4, 0xef, 0xbd, 0x2c, 0x11, 0xbb, 0xc9, 0x88, 0x59, 0x68, 0x28, 0xe9, 0xb8, 0xb8,
  0xa5, 0xdc, 0xeb, 0x70, 0x90, 0x1e, 0x8d, 0xac, 0xd1, 0x5a, 0xd9, 0x53, 0xd1, 0x42,
  0xea, 0xff, 0xbb, 0x9c, 0x6e, 0x86, 0x40, 0x8e, 0x8c, 0x4c, 0x0d, 0xce, 0xf7, 0x07,
  0xf8, 0x44, 0x69, 0xf5, 0xbb, 0x1d, 0xb3, 0x8a, 0x4d, 0x1a, 0x7d, 0xe6, 0xdb, 0x79,
  0x4e, 0x9d, 0xf9, 0x73, 0x0c, 0xf8, 0x43, 0x3c, 0x08, 0x96, 0x75, 0x64, 0x36, 0x81,
  0xd3, 0x86, 0x77, 0xc8, 0x28, 0xc1, 0x56, 0xa5, 0x63, 0x86, 0x3f, 0xb1, 0x7b, 0x3d,
  0x95, 0x1a, 0x9c, 0x1d, 0xfa, 0xfe, 0xf6, 0x57, 0xce, 0xeb, 0x5a, 0xe6, 0xf3, 0x65,
  0xd3, 0x20, 0x99, 0xb7, 0x19, 0x13, 0x54, 0x8b, 0xa8, 0x2d, 0xc0, 0x23, 0x97, 0x25,
  0x61, 0x2a, 0x92, 0x42, 0x48, 0xaa, 0xdf, 0x96, 0x40, 0x71, 0x2c, 0xf1, 0x27, 0x17,
  0x8b, 0x5e, 0xdf, 0xa2, 0xe9, 0xbb, 0x1a, 0xcc, 0x15, 0x9f, 0x07, 0x9e, 0x58, 0x1d,
  0xa7, 0x96, 0x88, 0x94, 0xd8, 0x5a, 0xf3, 0xb5, 0x2b, 0xba, 0x85, 0xe5, 0xd3, 0x0f,
  0x5d, 0x1c, 0x3a, 0xca, 0x3e, 0x0d, 0x8c, 0xc6, 0x6d, 0x14, 0xf2, 0x3d, 0x43, 0x1f,
  0xa9, 0x09, 0x0d, 0xe9, 0xe4, 0x71, 0xb3, 0x3e, 0x03, 0x79, 0xe0, 0x4c, 0xa4, 0xe2,
  0x54, 0xc0, 0x69, 0x31, 0xfb, 0x32, 0x37, 0xe5, 0x3d, 0x5b, 0x5f, 0x17, 0xfa, 0x50,
  0xbc, 0x2b, 0x56, 0x1d, 0x6d, 0x25, 0xc9, 0x15, 0x88, 0xff, 0x93, 0x8c, 0x7e, 0xf7,
  0xd5, 0x3a, 0x0e, 0x60, 0x6a, 0x57, 0x01, 0xd5, 0x07, 0xa2, 0x73, 0x38, 0xf4, 0x80,
  0xc1, 0x22, 0xc9, 0x79, 0xcd, 0x03, 0xe4, 0x2f, 0x27, 0x98, 0x1b, 0x43, 0xb4, 0x8b,
  0xe2, 0x29, 0xb3, 0xb0, 0xd5, 0xa4, 0x4b, 0xc8, 0xa8, 0xa0, 0x6d, 0x34, 0x78, 0xa1,
  0x54, 0x1b, 0xba, 0xef, 0xa0, 0x40, 0x26, 0xf1, 0x0a, 0xd5, 0xc5, 0xd8, 0x23, 0x29,
  0xbf, 0x79, 0x73, 0xdc, 0xb6, 0xa1, 0x85, 0x65, 0xc5, 0x05, 0x31, 0xe2, 0x19, 0xbb,
  0x63, 0x03, 0xb5, 0xa0, 0x52, 0x86, 0x88, 0xf6, 0x37, 0xb3, 0x7e, 0x0a, 0xec, 0x70,
  0x4a, 0x43, 0x3c, 0x82, 0x01, 0x06, 0xb9, 0x56, 0x7e, 0x54, 0x74, 0xfa, 0x64, 0xa3,
  0x5f, 0xfd, 0xa6, 0x75, 0x60, 0x64, 0x68, 0xb9, 0x66, 0xe2, 0x56, 0x01, 0xda, 0x39,
  0x8c, 0xa1, 0x22, 0x2d, 0xe1, 0x0a, 0xed, 0x81, 0x84, 0xaf, 0x88, 0x4c, 0x80, 0xc8,
  0xb9, 0x88, 0x9d, 0xbb, 0x93, 0x51, 0xbf, 0x4b, 0xba, 0xb6, 0x68, 0x8f, 0xa3, 0xbb,
  0x84, 0xe8, 0xdb, 0x97, 0xfd, 0x25, 0xd6, 0x0e, 0x9a, 0xda, 0x02, 0x1a, 0x8e, 0x08,
  0xa8, 0x6b, 0x4c, 0x99, 0x47, 0xe4, 0x93, 0x73, 0xa3, 0x26, 0x58, 0x9b, 0xd7, 0x07,
  0x64, 0x32, 0x9b, 0x84, 0xb4, 0x99, 0x9e, 0x1d, 0x03, 0x60, 0xe6, 0xfa, 0x44, 0x9f,
  0xb3, 0x20, 0xa3, 0x36, 0x51, 0xd8, 0xe6, 0x36, 0x28, 0xf7, 0x64, 0x6e, 0x5d, 0x18,
  0xf6, 0x32, 0x7a, 0xa6, 0xb0, 0xb3, 0xb5, 0x50, 0x30, 0x57, 0xa2, 0x57, 0x04, 0xa4,
  0xbc, 0x34, 0xa5, 0xa0, 0xe7, 0x45, 0xa1, 0xfe, 0x15, 0xcf, 0x49, 0xfa, 0xb0, 0x20,
  0x1d, 0x57, 0x18, 0x75, 0x5f, 0xa9, 0xdd, 0x5a, 0x6d, 0x1c, 0xc7, 0x86, 0x8b, 0x8a,
  0x0a, 0x37, 0xe7, 0x67, 0xae, 0x67, 0x3a, 0xfb, 0xd0, 0xfb, 0xc3, 0xe4, 0x41, 0x96,
  0xfa, 0x3f, 0x52, 0xc5, 0x63, 0x65, 0x3f, 0xf7, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x05, 0x91, 0x6c, 0x67, 0x03, 0xc6, 0x5c, 0x50, 0x62, 0x7c, 0x64,
  0xe7, 0xcb, 0x64, 0xb7, 0xda, 0xf0, 0xc3, 0xb2, 0xec, 0x54, 0xf2, 0x18, 0x79, 0x2e,
  0x33, 0x58, 0x20, 0x48, 0xf3, 0x25, 0xeb, 0x44, 0x9c, 0x09, 0x88, 0x29, 0x15, 0x6e,
  0xcd, 0x95, 0x13, 0xb0, 0xbc, 0xfa, 0xd0, 0xce, 0xb7, 0x97, 0x52, 0x53, 0x08, 0xb2,
  0x19, 0x69, 0x55, 0x23, 0xf8, 0x19, 0x6c, 0xaa, 0x4d, 0x7b, 0x6e, 0x1f, 0x87, 0xce,
  0xcd, 0xc4, 0xd6, 0x9d, 0x41, 0xa5, 0x6e, 0x10, 0x2b, 0xbd, 0x61, 0xcc, 0xb4, 0xc9,
  0xd2, 0xa3, 0xa7, 0xfc, 0xb9, 0xc3, 0x74, 0xe5, 0x8b, 0xc7, 0xe9, 0x9e, 0x66, 0xc0,
  0xa3, 0x80, 0xc3, 0xb1, 0x3f, 0xca, 0x5b, 0x08, 0x26, 0x24, 0x2a, 0xea, 0x51, 0x48,
  0x57, 0x3b, 0x65, 0x9a, 0x02, 0x72, 0xea, 0x79, 0x1a, 0x29, 0x5a, 0xa0, 0x7a, 0x6f,
  0xb7, 0x90, 0x4a, 0x58, 0x24, 0x69, 0x11, 0x05, 0x49, 0xde, 0x76, 0xf9, 0x67, 0x64,
  0x74, 0x64, 0xe4, 0x9e, 0xa9, 0xe9, 0x57, 0x43, 0x4d, 0xf8, 0xb7, 0xb1, 0xa8, 0x73,
  0x5d, 0x82, 0xe3, 0xed, 0x95, 0x94, 0x20, 0x35, 0x4e, 0xc1, 0x65, 0x86, 0x50, 0x29,
  0x16, 0xe3, 0xd1, 0x50, 0x9f, 0x6f, 0x11, 0xb0, 0x6b, 0xfd, 0x3f, 0x6f, 0xb2, 0x5f,
  0x70, 0xdc, 0x7d, 0x3c, 0xdd, 0x8a, 0x1a, 0xdd, 0x2c, 0xa2, 0xf2, 0x58, 0x2e, 0xa0,
  0x1b, 0xba, 0xd5, 0x34, 0xa9, 0x7b, 0x59, 0x68, 0x6d, 0x6e, 0xb3, 0xaa, 0xc7, 0x84,
  0x90, 0x56, 0x51, 0x70, 0x28, 0x26, 0xae, 0x99, 0x0f, 0x9b, 0xe7, 0xd8, 0xf4, 0xc6,
  0x07, 0x06, 0x3e, 0x20, 0x7c, 0x96, 0xde, 0x61, 0x06, 0xed, 0x33, 0x07, 0xbd, 0x29,
  0xaa, 0x58, 0x42, 0x6f, 0x13, 0x48, 0x02, 0x55, 0x48, 0x7f, 0xd1, 0x95, 0x51, 0xe3,
  0xb4, 0xe3, 0x8d, 0x46, 0xc2, 0x00, 0xdc, 0x63, 0x18, 0x5b, 0x3e, 0x41, 0x79, 0xf1,
  0xb0, 0x60, 0x69, 0x7d, 0x8e, 0xe2, 0xce, 0x00, 0x18, 0xb7, 0x87, 0x11, 0xbf, 0x58,
  0xf3, 0xee, 0xbf, 0xbf, 0xf5, 0x4b, 0x0c, 0x06, 0xd1, 0xc8, 0x5d, 0x69, 0x1c, 0x9b,
  0xc4, 0x53, 0x90, 0x56, 0x40, 0xb3, 0x0b, 0x7d, 0x8e, 0x95, 0x3d, 0x2d, 0x66, 0xb6,
  0x5b, 0xd4, 0x4e, 0x25, 0x24, 0x6c, 0x94, 0xf4, 0x82, 0xeb, 0xaa, 0xd1, 0x21, 0xe4,
  0x15, 0x63, 0x13, 0x10, 0xab, 0x6e, 0x57, 0x52, 0x04, 0x32, 0x34, 0x99, 0x5e, 0x6a,
  0x05, 0xb7, 0x07, 0xa7, 0x7c, 0x53, 0x7c, 0x6f, 0x6e, 0x56, 0x18, 0x95, 0x9e, 0x02,
  0xeb, 0xfe, 0xef, 0x1b, 0x73, 0x38, 0x12, 0xf5, 0x0d, 0x1e, 0x46, 0xed, 0x41, 0x05,
  0x23, 0x68, 0x15, 0x1f, 0x3b, 0xc8, 0xa7, 0xef, 0xde, 0xca, 0xd8, 0xc5, 0x6e, 0x4b,
  0x51, 0x96, 0x1b, 0xdc, 0x5f, 0x6c, 0x55, 0x57, 0x6f, 0xdb, 0x43, 0x08, 0x14, 0x6c,
  0x6d, 0x68, 0xe0, 0xcb, 0x96, 0xf3, 0xdd, 0x2a, 0x87, 0xce, 0x4b, 0x8c, 0xc0, 0x1b,
  0xc6, 0xb4, 0xd6, 0xa8, 0x4c, 0x56, 0x04, 0x80, 0x60, 0x37, 0xb6, 0x3b, 0xb7, 0x0c,
  0x17, 0x31, 0x92, 0x1b, 0xc3, 0xb5, 0xc7, 0xa3, 0x9b, 0x94, 0x32, 0xab, 0x8e, 0x2c,
  0x33, 0x01, 0x2e, 0x88, 0xdb, 0xe3, 0x9a, 0xf4, 0xbd, 0x26, 0x26, 0x0a, 0x2f, 0xde,
  0x14, 0x2b, 0x27, 0xb3, 0xaf, 0xd5, 0x8f, 0xfa, 0xaf, 0x89, 0xd8, 0xfe, 0x01, 0x40,
  0x40, 0x40, 0x40, 0x13, 0x73, 0x18, 0x72, 0xbd, 0x6e, 0x76, 0xac, 0x97, 0x6d, 0xc7,
  0xde, 0x21, 0x3f, 0x34, 0x87, 0x69, 0xa8, 0x4b, 0x4f, 0xf7, 0xa9, 0x73, 0xbd, 0x5e,
  0x28, 0x5d, 0x95, 0x31, 0x18, 0xd9, 0xa1, 0x1f, 0xfe, 0xb4, 0xb7, 0x6d, 0x77, 0x8d,
  0x66, 0x2a, 0x7f, 0xf5, 0x5d, 0x63, 0x9d, 0xe2, 0x05, 0x80, 0x4e, 0x25, 0x68, 0x66,
  0x24, 0xdb, 0xa4, 0xf3, 0x07, 0xb4, 0x48, 0x70, 0x46, 0x98, 0xfd, 0x40, 0xb2, 0xac,
  0xa5, 0xf0, 0x1c, 0x1f, 0x41, 0x9d, 0x97, 0x74, 0x9e, 0xf3, 0xef, 0xd3, 0xa1, 0xe1,
  0x9e, 0x21, 0x1f, 0xcd, 0x85, 0x7e, 0xc2, 0x20, 0xd1, 0x74, 0x95, 0x6f, 0xc5, 0x8e,
  0xb4, 0xd2, 0x8f, 0x7c, 0xd5, 0x6c, 0x94, 0x39, 0xe6, 0x45, 0x49, 0xf5, 0x89, 0xba,
  0xa6, 0xe2, 0x02, 0xc9, 0x89, 0x96, 0xbb, 0xc6, 0x8a, 0x1e, 0x1d, 0xc2, 0x48, 0xcc,
  0xb4, 0x69, 0x5b, 0x78, 0xde, 0x30, 0x4f, 0x32, 0x69, 0x1f, 0x16, 0xb6, 0x84, 0x5b,
  0x6b, 0xde, 0x57, 0xcc, 0xba, 0xbb, 0xa3, 0x08, 0x84, 0x3b, 0xe3, 0x05, 0x97, 0x70,
  0x3c, 0x28, 0xc4, 0x9a, 0xe5, 0xda, 0xf7, 0xa9, 0x7b, 0x67, 0x72, 0x0c, 0x82, 0x6d,
  0xd2, 0x7d, 0xa7, 0x13, 0x6f, 0xdc, 0x4c, 0x96, 0x0d, 0x7e, 0xe8, 0x4c, 0x7a, 0xa3,
  0x31, 0x70, 0xff, 0x25, 0xe7, 0xbd, 0x8e, 0x7c, 0x30, 0xee, 0xab, 0x4f, 0x7b, 0x5d,
  0xf2, 0xb4, 0xd4, 0xcd, 0x35, 0xb3, 0x6c, 0x0c, 0xa1, 0xbe, 0xf7, 0x33, 0x26, 0xf7,
  0x82, 0x34, 0x85, 0xbb, 0x49, 0x95, 0x55, 0xb1, 0xef, 0xbf, 0x29, 0xf4, 0x29, 0x17,
  0xde, 0x32, 0x11, 0xf8, 0x60, 0xde, 0x5c, 0xf7, 0x89, 0x9c, 0x43, 0xf5, 0x2f, 0xaa,
  0xab, 0xad, 0x83, 0x19, 0x4c, 0x3e, 0x9a, 0xc9, 0xc3, 0xdf, 0xe5, 0x3f, 0xa5, 0x9f,
  0xcc, 0x22, 0xc6, 0x34, 0x22, 0x46, 0x56, 0x8e, 0x7a, 0x27, 0x58, 0xe3, 0xd6, 0x53,
  0x1d, 0xc9, 0x09, 0x4f, 0x85, 0xe4, 0x8c, 0x1e, 0x60, 0x29, 0x77, 0x6b, 0x6c, 0x6d,
  0x6e, 0xe0, 0xd7, 0xf7, 0x41, 0x60, 0x2a, 0x5d, 0x73, 0xab, 0x54, 0x97, 0x99, 0xc9,
  0xab, 0x2b, 0xc1, 0x02, 0x0e, 0x7e, 0x2b, 0x98, 0x44, 0xfe, 0xf2, 0x63, 0xc1, 0xe7,
  0xa8, 0x30, 0xff, 0x1c, 0x01, 0x16, 0x8b, 0x5b, 0xdc, 0x58, 0x86, 0x05, 0xb6, 0xe8,
  0xee, 0xfe, 0xbf, 0xb2, 0xcf, 0xe3, 0xbe, 0x88, 0x65, 0x5a, 0xc0, 0xe5, 0x45, 0x0c,
  0x98, 0x52, 0x8f, 0x92, 0x9b, 0xb4, 0x28, 0xf4, 0xfa, 0x0d, 0xbc, 0x1d, 0xa9, 0xaf,
  0xd3, 0x3a, 0x5a, 0x08, 0x46, 0x18, 0xc3, 0x62, 0x1b, 0x46, 0x18, 0xdf, 0x74, 0x66,
  0x40, 0x18, 0x5a, 0x04, 0x58, 0x73, 0x63, 0xe5, 0xe4, 0xc5, 0xb1, 0x58, 0xf8, 0xf4,
  0xbd, 0xe9, 0xb4, 0x2e, 0xcb, 0xd6, 0xeb, 0x22, 0x95, 0xe1, 0x96, 0x46, 0x4f, 0x6f,
  0x89, 0x52, 0x14, 0x68, 0x68, 0x0b, 0x6d, 0x3a, 0x3b, 0x83, 0xd2, 0xf1, 0x1d, 0x74,
  0x3c, 0x33, 0x15, 0x7a, 0xeb, 0x21, 0x91, 0x7b, 0xfb, 0xbb, 0xa5, 0xf4, 0xd4, 0x3d,
  0xa5, 0xd7, 0x7e, 0xd4, 0x98, 0x72, 0x23, 0x66, 0x5c, 0x34, 0xa8, 0x6f, 0xa1, 0xdf,
  0x38, 0x48, 0x5b, 0xda, 0xe4, 0xf0, 0x6e, 0xb9, 0x6b, 0x1b, 0x5f, 0x63, 0xb1, 0xcf,
  0x23, 0xfc, 0x74, 0x9f, 0xeb, 0x82, 0x9d, 0x30, 0x0d, 0xe9, 0x38, 0x87, 0x34, 0x49,
  0x93, 0xbb, 0x60, 0x69, 0xd2, 0x30, 0xbc, 0xcc, 0x5b, 0x2e, 0x28, 0xda, 0x58, 0xff,
  0x23, 0x00, 0x5c, 0xb1, 0xf9, 0xe3, 0x98, 0x76, 0x76, 0x7f, 0x55, 0x33, 0xfd, 0xfc,
  0x36, 0xeb, 0x40, 0x09, 0x85, 0xe4, 0x43, 0x64, 0xd9, 0x40, 0xbb, 0xf0, 0xd0, 0xb4,
  0x93, 0xde, 0x80, 0x97, 0xca, 0x92, 0xc1, 0x4d, 0x7a, 0xc5, 0xac, 0xfa, 0x67, 0x0c,
  0xea, 0xd1, 0x74, 0x2e, 0x40, 0x9f, 0x93, 0xcc, 0x87, 0xe6, 0x95, 0x53, 0xb7, 0xe4,
  0xda, 0x5e, 0xd5, 0xc6, 0x79, 0x57, 0xfe, 0xfc, 0x0d, 0xf5, 0x24, 0x83, 0x02, 0xf8,
  0xf8, 0x7d, 0x24, 0x2c, 0x2f, 0xbc, 0x97, 0x46, 0x35, 0x04, 0xdd, 0x17, 0x03, 0xff,
  0x38, 0xb6, 0x95, 0x9a, 0x67, 0xc0, 0x3d, 0xaa, 0xac, 0xeb, 0xe8, 0xe1, 0xf0, 0x8b,
  0x2a, 0x18, 0xe5, 0xd9, 0x3b, 0xc6, 0xf5, 0x37, 0x62, 0x28, 0xdf, 0xa9, 0xe2, 0x96,
  0xcc, 0xea, 0x22, 0x8f, 0xb4, 0xd6, 0xcf, 0x8d, 0xac, 0xa6, 0x5b, 0xf5, 0x98, 0x6a,
  0x87, 0x76, 0xca, 0xc7, 0x7d, 0xf4, 0x2e, 0xd2, 0xe4, 0x08, 0xaf, 0x9f, 0x0a, 0xb7,
  0x81, 0xee, 0x8d, 0xb3, 0x7f, 0xa0, 0xb6, 0x06, 0x3e, 0xc5, 0x25, 0xa0, 0x71, 0xda,
  0x32, 0xfa, 0xc2, 0x9c, 0x09, 0xe7, 0x80, 0xd1, 0x95, 0x59, 0x7c, 0xb0, 0xa6, 0x2a,
  0x2f, 0x0a, 0x0b, 0x18, 0x62, 0x1b, 0xb6, 0xd2, 0x5c, 0x44, 0x63, 0xd5, 0x69, 0x30,
  0x15, 0x1d, 0xfd, 0x2e, 0x9d, 0xa1, 0xbd, 0xd4, 0xc1, 0x4d, 0x71, 0xcb, 0x19, 0x23,
  0x05, 0x9f, 0x75, 0xad, 0xe7, 0xe4, 0x45, 0xf8, 0x73, 0xf9, 0xfe, 0xdb, 0x60, 0x8b,
  0xfa, 0x6e, 0x0a, 0xf0, 0x0d, 0x1b, 0x9c, 0x51, 0xad, 0x9d, 0xee, 0x64, 0x11, 0x70,
  0xc0, 0xd6, 0xad, 0xb1, 0x7a, 0x14, 0x06, 0x2a, 0xb7, 0x20, 0xb2, 0x16, 0x3e, 0x98,
  0x07, 0x8b, 0x73, 0x06, 0x62, 0x3e, 0x2d, 0xdd, 0x65, 0xcd, 0xce, 0x85, 0xdd, 0x34,
  0xdd, 0x68, 0xec, 0x0b, 0xeb, 0xdc, 0xe1, 0x07, 0x45, 0x9a, 0xf7, 0xc3, 0xdd, 0xac,
  0x81, 0x2d, 0x05, 0x4f, 0x5a, 0xfd, 0x66, 0x13, 0xcf, 0x85, 0xf9, 0xc4, 0x1f, 0x42,
  0x61, 0x32, 0x45, 0xa3, 0xaa, 0xa0, 0xc4, 0xe0, 0xb0, 0x49, 0xfc, 0x91, 0x8f, 0xa6,
  0x11, 0xd9, 0xab, 0xfa, 0xd0, 0xbf, 0x12, 0x58, 0x7f, 0xdb, 0xab, 0xfa, 0xd0, 0xbf,
  0x12, 0x58, 0x7f, 0x8c, 0xbf, 0xff, 0x45, 0x9e, 0x82, 0x88, 0xa2, 0xbe, 0xea, 0x36,
  0x3b, 0xbf, 0x90, 0xcf, 0x93, 0x48, 0xb0, 0xba, 0xca, 0xb4, 0x74, 0xd6, 0x0b, 0x01,
  0x46, 0xa4, 0x18, 0x04, 0x22, 0x9d, 0x4d, 0x4d, 0xc2, 0x66, 0x33, 0xa3, 0x40, 0x19,
  0xc4, 0xec, 0xeb, 0xe0, 0xe8, 0x56, 0x52, 0x79, 0xa2, 0x81, 0x9e, 0x48, 0xa8, 0x5e,
  0x34, 0x7e, 0x58, 0xa1, 0xa2, 0x1e, 0x34, 0xd0, 0xa6, 0xb9, 0x5e, 0x86, 0xb7, 0x66,
  0x02, 0xa6, 0x44, 0x1d, 0xd2, 0x4e, 0x41, 0x1e, 0xba, 0x97, 0xb9, 0x23, 0x58, 0x21,
  0x33, 0x10, 0xd1, 0x24, 0x48, 0x70, 0xa0, 0xc4, 0x88, 0xd8, 0x4e, 0x0c, 0xec, 0x83,
  0xcb, 0x56, 0xed, 0x1b, 0x17, 0xfc, 0x2e, 0x4f, 0x30, 0x49, 0x03, 0x7a, 0xc3, 0xde,
  0xb6, 0x52, 0x03, 0xbc, 0x15, 0x65, 0x29, 0x04, 0x17, 0x2d, 0xe0, 0x3e, 0x82, 0x7f,
  0xd0, 0x4b, 0x5d, 0xe2, 0x10, 0x90, 0x8e, 0xe7, 0x09, 0x8c, 0x77, 0x13, 0xa0, 0xdc,
  0x35, 0x54, 0x1b, 0x1f, 0x87, 0x28, 0x10, 0x2a, 0xd4, 0xe4, 0xf2, 0xaf, 0x5a, 0x9b,
  0xba, 0xc8, 0xf0, 0x93, 0x89, 0x51, 0x18, 0x92, 0xf9, 0xe8, 0x5d, 0x33, 0x49, 0x2b,
  0xc9, 0x04, 0xe5, 0xb2, 0xd8, 0x5c, 0x3f, 0xdc, 0x5a, 0x50, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xaf, 0x20, 0xbd, 0xba, 0xd6, 0xba, 0x7b, 0xb1, 0xe7,
  0x49, 0xa1, 0x92, 0xf1, 0x5f, 0xf1, 0x13, 0x82, 0xd5, 0xed, 0xed, 0x13, 0xa5, 0x1a,
  0x29, 0xbe, 0x65, 0x0e, 0x9c, 0x0b, 0x1c, 0xd6, 0xbe, 0xc7, 0x6e, 0x3c, 0x51, 0x33,
  0x0c, 0x08, 0xe3, 0x2f, 0x53, 0x1e, 0x61, 0xc0, 0x08, 0x70, 0xb1, 0x77, 0x56, 0xd0,
  0x3f, 0x98, 0xdf, 0xe5, 0x26};
} /// namespace impala
