/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hive.common.util;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.TemporalField;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Objects;
import org.apache.hadoop.hive.common.type.Timestamp;
import org.apache.hadoop.hive.common.type.TimestampTZ;
import org.apache.hadoop.hive.common.type.TimestampTZUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimestampParser {
    private static final Logger LOG = LoggerFactory.getLogger(TimestampParser.class);
    public static final String MILLIS_FORMAT_STR = "millis";
    public static final String ISO_8601_FORMAT_STR = "iso8601";
    public static final String RFC_1123_FORMAT_STR = "rfc1123";
    private final Collection<DateTimeFormatter> dtFormatters;
    private final boolean supportMillisEpoch;

    public TimestampParser() {
        this(Collections.emptyList());
    }

    public TimestampParser(TimestampParser tsParser) {
        this.dtFormatters = tsParser.dtFormatters;
        this.supportMillisEpoch = tsParser.supportMillisEpoch;
    }

    public TimestampParser(String[] formatStrings) {
        this(formatStrings == null ? Collections.emptyList() : Arrays.asList(formatStrings));
    }

    public TimestampParser(Collection<String> patterns) {
        HashSet<String> patternSet = new HashSet<String>(patterns);
        this.supportMillisEpoch = patternSet.remove(MILLIS_FORMAT_STR);
        if (patternSet.isEmpty()) {
            this.dtFormatters = Collections.emptyList();
            return;
        }
        this.dtFormatters = new ArrayList<DateTimeFormatter>();
        Iterator iterator = patternSet.iterator();
        while (iterator.hasNext()) {
            String patternText;
            this.dtFormatters.add(switch (patternText = (String)iterator.next()) {
                case ISO_8601_FORMAT_STR -> DateTimeFormatter.ISO_INSTANT;
                case RFC_1123_FORMAT_STR -> DateTimeFormatter.RFC_1123_DATE_TIME;
                default -> DateTimeFormatter.ofPattern(patternText);
            });
        }
    }

    public Timestamp parseTimestamp(String text) {
        if (this.supportMillisEpoch) {
            try {
                long millis = new BigDecimal(text).setScale(0, RoundingMode.DOWN).longValueExact();
                return Timestamp.ofEpochMilli(millis);
            }
            catch (NumberFormatException e) {
                LOG.debug("Could not format millis: {}", (Object)text);
            }
        }
        for (DateTimeFormatter formatter : this.dtFormatters) {
            try {
                TemporalAccessor parsed = formatter.parse(text);
                Instant inst = Instant.from(this.wrap(parsed));
                return Timestamp.ofEpochMilli(inst.toEpochMilli());
            }
            catch (DateTimeParseException dtpe) {
                LOG.debug("Could not parse timestamp text: {}", (Object)text);
            }
        }
        try {
            return Timestamp.valueOf(text);
        }
        catch (IllegalArgumentException e) {
            return null;
        }
    }

    public TimestampTZ parseTimestamp(String text, ZoneId defaultTimeZone) {
        Objects.requireNonNull(text);
        for (DateTimeFormatter f : this.dtFormatters) {
            try {
                return TimestampTZUtil.parse(text, defaultTimeZone, f);
            }
            catch (DateTimeException dateTimeException) {
            }
        }
        return TimestampTZUtil.parse(text, defaultTimeZone);
    }

    private TemporalAccessor wrap(TemporalAccessor in) {
        if (in.isSupported(ChronoField.INSTANT_SECONDS) && in.isSupported(ChronoField.NANO_OF_SECOND)) {
            return in;
        }
        return new DefaultingTemporalAccessor(in);
    }

    private static class DefaultingTemporalAccessor
    implements TemporalAccessor {
        private static final EnumSet<ChronoField> FIELDS = EnumSet.of(ChronoField.YEAR, new ChronoField[]{ChronoField.MONTH_OF_YEAR, ChronoField.DAY_OF_MONTH, ChronoField.HOUR_OF_DAY, ChronoField.MINUTE_OF_HOUR, ChronoField.SECOND_OF_MINUTE, ChronoField.MILLI_OF_SECOND, ChronoField.NANO_OF_SECOND});
        private final TemporalAccessor wrapped;

        DefaultingTemporalAccessor(TemporalAccessor in) {
            ZonedDateTime dateTime = ZonedDateTime.ofInstant(Instant.EPOCH, ZoneOffset.UTC);
            for (ChronoField field : FIELDS) {
                if (!in.isSupported(field)) continue;
                dateTime = dateTime.with(field, in.getLong(field));
            }
            this.wrapped = dateTime.toInstant();
        }

        @Override
        public long getLong(TemporalField field) {
            return this.wrapped.getLong(field);
        }

        @Override
        public boolean isSupported(TemporalField field) {
            return this.wrapped.isSupported(field);
        }
    }
}

