/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.mr.hive;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.commons.lang3.SerializationUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.common.type.TimestampTZ;
import org.apache.hadoop.hive.common.type.TimestampTZUtil;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.metastore.IMetaStoreClient;
import org.apache.hadoop.hive.metastore.Warehouse;
import org.apache.hadoop.hive.metastore.api.ColumnStatisticsObj;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.hive.metastore.utils.TableFetcher;
import org.apache.hadoop.hive.ql.Context;
import org.apache.hadoop.hive.ql.io.sarg.SearchArgument;
import org.apache.hadoop.hive.ql.metadata.DummyPartition;
import org.apache.hadoop.hive.ql.metadata.HiveException;
import org.apache.hadoop.hive.ql.metadata.HiveUtils;
import org.apache.hadoop.hive.ql.metadata.Partition;
import org.apache.hadoop.hive.ql.parse.AlterTableExecuteSpec;
import org.apache.hadoop.hive.ql.parse.SemanticException;
import org.apache.hadoop.hive.ql.parse.TransformSpec;
import org.apache.hadoop.hive.ql.plan.PlanUtils;
import org.apache.hadoop.hive.ql.session.SessionState;
import org.apache.hadoop.hive.ql.session.SessionStateUtil;
import org.apache.hadoop.util.Sets;
import org.apache.iceberg.ContentFile;
import org.apache.iceberg.DeleteFiles;
import org.apache.iceberg.ManageSnapshots;
import org.apache.iceberg.ManifestFile;
import org.apache.iceberg.MetadataTableType;
import org.apache.iceberg.MetadataTableUtils;
import org.apache.iceberg.PartitionData;
import org.apache.iceberg.PartitionField;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.PartitionStatisticsFile;
import org.apache.iceberg.Partitioning;
import org.apache.iceberg.PartitionsTable;
import org.apache.iceberg.RowLevelOperationMode;
import org.apache.iceberg.Schema;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.SnapshotRef;
import org.apache.iceberg.StatisticsFile;
import org.apache.iceberg.StructLike;
import org.apache.iceberg.Table;
import org.apache.iceberg.UpdatePartitionSpec;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.expressions.Expression;
import org.apache.iceberg.expressions.Expressions;
import org.apache.iceberg.expressions.ResidualEvaluator;
import org.apache.iceberg.expressions.UnboundPredicate;
import org.apache.iceberg.io.CloseableIterable;
import org.apache.iceberg.mr.Catalogs;
import org.apache.iceberg.mr.hive.HiveIcebergFilterFactory;
import org.apache.iceberg.puffin.BlobMetadata;
import org.apache.iceberg.puffin.Puffin;
import org.apache.iceberg.puffin.PuffinReader;
import org.apache.iceberg.relocated.com.google.common.collect.FluentIterable;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableList;
import org.apache.iceberg.relocated.com.google.common.collect.Iterables;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.collect.Maps;
import org.apache.iceberg.types.Conversions;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;
import org.apache.iceberg.util.ByteBuffers;
import org.apache.iceberg.util.Pair;
import org.apache.iceberg.util.PropertyUtil;
import org.apache.iceberg.util.SnapshotUtil;
import org.apache.iceberg.util.StructProjection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IcebergTableUtil {
    private static final Logger LOG = LoggerFactory.getLogger(IcebergTableUtil.class);
    public static final String PARTITION_TRANSFORM_SPEC_NOT_FOUND = "Iceberg partition transform spec is not found in QueryState.";
    public static final int SPEC_IDX = 1;
    public static final int PART_IDX = 0;

    private IcebergTableUtil() {
    }

    static Table getTable(Configuration configuration, org.apache.hadoop.hive.metastore.api.Table hmsTable, boolean skipCache) {
        Properties properties = new Properties();
        properties.setProperty("name", TableIdentifier.of(hmsTable.getDbName(), hmsTable.getTableName()).toString());
        properties.setProperty("location", hmsTable.getSd().getLocation());
        hmsTable.getParameters().computeIfPresent("iceberg.catalog", (k, v) -> {
            properties.setProperty((String)k, (String)v);
            return v;
        });
        return IcebergTableUtil.getTable(configuration, properties, skipCache);
    }

    public static Table getTable(Configuration configuration, org.apache.hadoop.hive.metastore.api.Table hmsTable) {
        return IcebergTableUtil.getTable(configuration, hmsTable, false);
    }

    static Table getTable(Configuration configuration, Properties properties, boolean skipCache) {
        String metaTable = properties.getProperty("metaTable");
        Properties props = new Properties(properties);
        if (metaTable != null) {
            props.put("name", String.valueOf(properties.get("name")) + "." + metaTable);
            props.put("location", String.valueOf(properties.get("location")) + "#" + metaTable);
        }
        String tableIdentifier = props.getProperty("name");
        Function<Void, Table> tableLoadFunc = unused -> {
            Table tab = Catalogs.loadTable(configuration, props);
            SessionStateUtil.addResource((Configuration)configuration, (String)tableIdentifier, (Object)tab);
            return tab;
        };
        if (skipCache) {
            return tableLoadFunc.apply(null);
        }
        return SessionStateUtil.getResource((Configuration)configuration, (String)tableIdentifier).filter(o -> o instanceof Table).map(o -> (Table)o).orElseGet(() -> {
            LOG.debug("Iceberg table {} is not found in QueryState. Loading table from configured catalog", (Object)tableIdentifier);
            return (Table)tableLoadFunc.apply(null);
        });
    }

    static Table getTable(Configuration configuration, Properties properties) {
        return IcebergTableUtil.getTable(configuration, properties, false);
    }

    static Snapshot getTableSnapshot(Table table, org.apache.hadoop.hive.ql.metadata.Table hmsTable) {
        long snapshotId = -1L;
        if (hmsTable.getAsOfTimestamp() != null) {
            ZoneId timeZone = SessionState.get() == null ? new HiveConf().getLocalTimeZone() : SessionState.get().getConf().getLocalTimeZone();
            TimestampTZ time = TimestampTZUtil.parse((String)hmsTable.getAsOfTimestamp(), (ZoneId)timeZone);
            snapshotId = SnapshotUtil.snapshotIdAsOfTime(table, time.toEpochMilli());
        } else if (hmsTable.getAsOfVersion() != null) {
            try {
                snapshotId = Long.parseLong(hmsTable.getAsOfVersion());
            }
            catch (NumberFormatException e) {
                SnapshotRef ref = table.refs().get(hmsTable.getAsOfVersion());
                if (ref == null) {
                    throw new RuntimeException("Cannot find matching snapshot ID or reference name for version " + hmsTable.getAsOfVersion());
                }
                snapshotId = ref.snapshotId();
            }
        }
        if (snapshotId > 0L) {
            return table.snapshot(snapshotId);
        }
        return IcebergTableUtil.getTableSnapshot(table, hmsTable.getSnapshotRef());
    }

    static Snapshot getTableSnapshot(Table table, String snapshotRef) {
        if (snapshotRef != null) {
            String ref = HiveUtils.getTableSnapshotRef((String)snapshotRef);
            return table.snapshot(ref);
        }
        return table.currentSnapshot();
    }

    static String getColStatsPath(Table table) {
        return IcebergTableUtil.getColStatsPath(table, table.currentSnapshot().snapshotId());
    }

    static String getColStatsPath(Table table, long snapshotId) {
        return table.statisticsFiles().stream().filter(stats -> stats.snapshotId() == snapshotId).filter(stats -> stats.blobMetadata().stream().anyMatch(metadata -> ColumnStatisticsObj.class.getSimpleName().equals(metadata.type()))).map(StatisticsFile::path).findAny().orElse(null);
    }

    static PartitionStatisticsFile getPartitionStatsFile(Table table, long snapshotId) {
        return table.partitionStatisticsFiles().stream().filter(stats -> stats.snapshotId() == snapshotId).findAny().orElse(null);
    }

    public static PartitionSpec spec(Configuration configuration, Schema schema) {
        List partitionTransformSpecList = SessionStateUtil.getResource((Configuration)configuration, (String)"partition_transform_spec").map(o -> (List)o).orElse(null);
        if (partitionTransformSpecList == null) {
            LOG.warn(PARTITION_TRANSFORM_SPEC_NOT_FOUND);
            return null;
        }
        PartitionSpec.Builder builder = PartitionSpec.builderFor(schema);
        partitionTransformSpecList.forEach(spec -> {
            switch (spec.getTransformType()) {
                case IDENTITY: {
                    builder.identity(spec.getColumnName().toLowerCase());
                    break;
                }
                case YEAR: {
                    builder.year(spec.getColumnName());
                    break;
                }
                case MONTH: {
                    builder.month(spec.getColumnName());
                    break;
                }
                case DAY: {
                    builder.day(spec.getColumnName());
                    break;
                }
                case HOUR: {
                    builder.hour(spec.getColumnName());
                    break;
                }
                case TRUNCATE: {
                    builder.truncate(spec.getColumnName(), spec.getTransformParam());
                    break;
                }
                case BUCKET: {
                    builder.bucket(spec.getColumnName(), spec.getTransformParam());
                }
            }
        });
        return builder.build();
    }

    public static void updateSpec(Configuration configuration, Table table) {
        PartitionSpec newPartitionSpec = IcebergTableUtil.spec(configuration, table.schema());
        if (newPartitionSpec == null) {
            LOG.warn("Iceberg partition spec is not updated due to empty partition spec definition.");
            return;
        }
        UpdatePartitionSpec updatePartitionSpec = table.updateSpec().caseSensitive(false);
        table.spec().fields().forEach(field -> updatePartitionSpec.removeField(field.name()));
        List partitionTransformSpecList = SessionStateUtil.getResource((Configuration)configuration, (String)"partition_transform_spec").map(o -> (List)o).orElse(null);
        if (partitionTransformSpecList == null) {
            LOG.warn(PARTITION_TRANSFORM_SPEC_NOT_FOUND);
            return;
        }
        partitionTransformSpecList.forEach(spec -> {
            switch (spec.getTransformType()) {
                case IDENTITY: {
                    updatePartitionSpec.addField(spec.getColumnName());
                    break;
                }
                case YEAR: {
                    updatePartitionSpec.addField(Expressions.year(spec.getColumnName()));
                    break;
                }
                case MONTH: {
                    updatePartitionSpec.addField(Expressions.month(spec.getColumnName()));
                    break;
                }
                case DAY: {
                    updatePartitionSpec.addField(Expressions.day(spec.getColumnName()));
                    break;
                }
                case HOUR: {
                    updatePartitionSpec.addField(Expressions.hour(spec.getColumnName()));
                    break;
                }
                case TRUNCATE: {
                    updatePartitionSpec.addField(Expressions.truncate(spec.getColumnName(), spec.getTransformParam()));
                    break;
                }
                case BUCKET: {
                    updatePartitionSpec.addField(Expressions.bucket(spec.getColumnName(), spec.getTransformParam()));
                }
            }
        });
        updatePartitionSpec.commit();
    }

    public static boolean isBucketed(Table table) {
        return table.spec().fields().stream().anyMatch(f -> f.transform().toString().startsWith("bucket["));
    }

    public static boolean isBucket(TransformSpec spec) {
        return spec.getTransformType() == TransformSpec.TransformType.BUCKET && spec.getTransformParam() != null;
    }

    public static void rollback(Table table, AlterTableExecuteSpec.RollbackSpec.RollbackType type, Long value) {
        ManageSnapshots manageSnapshots = table.manageSnapshots();
        if (type == AlterTableExecuteSpec.RollbackSpec.RollbackType.TIME) {
            LOG.debug("Trying to rollback iceberg table to snapshot before timestamp {}", (Object)value);
            manageSnapshots.rollbackToTime(value);
        } else {
            LOG.debug("Trying to rollback iceberg table to snapshot ID {}", (Object)value);
            manageSnapshots.rollbackTo(value);
        }
        manageSnapshots.commit();
    }

    public static void setCurrentSnapshot(Table table, String value) {
        long snapshotId;
        ManageSnapshots manageSnapshots = table.manageSnapshots();
        try {
            snapshotId = Long.parseLong(value);
            LOG.debug("Rolling the iceberg table {} from snapshot id {} to snapshot ID {}", new Object[]{table.name(), table.currentSnapshot().snapshotId(), snapshotId});
        }
        catch (NumberFormatException e) {
            String refName = PlanUtils.stripQuotes((String)value);
            snapshotId = Optional.ofNullable(table.refs().get(refName)).map(SnapshotRef::snapshotId).orElseThrow(() -> new IllegalArgumentException(String.format("SnapshotRef %s does not exist", refName)));
            LOG.debug("Rolling the iceberg table {} from snapshot id {} to the snapshot ID {} of SnapshotRef {}", new Object[]{table.name(), table.currentSnapshot().snapshotId(), snapshotId, refName});
        }
        manageSnapshots.setCurrentSnapshot(snapshotId);
        manageSnapshots.commit();
    }

    public static void fastForwardBranch(Table table, String sourceBranch, String targetBranch) {
        LOG.debug("Fast Forwarding the iceberg table {} branch {} to {}", new Object[]{table.name(), sourceBranch, targetBranch});
        table.manageSnapshots().fastForwardBranch(sourceBranch, targetBranch).commit();
    }

    public static void cherryPick(Table table, long snapshotId) {
        LOG.debug("Cherry-Picking {} to {}", (Object)snapshotId, (Object)table.name());
        table.manageSnapshots().cherrypick(snapshotId).commit();
    }

    public static boolean isV2TableOrAbove(Map<String, String> props) {
        return IcebergTableUtil.formatVersion(props) >= 2;
    }

    public static boolean isV2TableOrAbove(BinaryOperator<String> props) {
        return IcebergTableUtil.formatVersion(props) >= 2;
    }

    public static Integer formatVersion(Map<String, String> props) {
        if (props == null) {
            return 2;
        }
        return IcebergTableUtil.formatVersion(props::getOrDefault);
    }

    private static Integer formatVersion(BinaryOperator<String> props) {
        String version = (String)props.apply("format-version", null);
        if (version == null) {
            return 2;
        }
        try {
            return Integer.parseInt(version);
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("Invalid format version: " + version, e);
        }
    }

    private static String getWriteModeDefault(BinaryOperator<String> props) {
        return (IcebergTableUtil.isV2TableOrAbove(props) ? RowLevelOperationMode.MERGE_ON_READ : RowLevelOperationMode.COPY_ON_WRITE).modeName();
    }

    public static boolean isCopyOnWriteMode(Context.Operation operation, BinaryOperator<String> props) {
        String mode = switch (operation) {
            case Context.Operation.DELETE -> (String)props.apply("write.delete.mode", IcebergTableUtil.getWriteModeDefault(props));
            case Context.Operation.UPDATE -> (String)props.apply("write.update.mode", IcebergTableUtil.getWriteModeDefault(props));
            case Context.Operation.MERGE -> (String)props.apply("write.merge.mode", IcebergTableUtil.getWriteModeDefault(props));
            default -> null;
        };
        return RowLevelOperationMode.COPY_ON_WRITE.modeName().equalsIgnoreCase(mode);
    }

    public static boolean isFanoutEnabled(Map<String, String> props) {
        return PropertyUtil.propertyAsBoolean(props, "write.fanout.enabled", true);
    }

    public static void performMetadataDelete(Table icebergTable, String branchName, SearchArgument sarg) {
        Expression exp = HiveIcebergFilterFactory.generateFilterExpression(sarg);
        DeleteFiles deleteFiles = icebergTable.newDelete();
        if (StringUtils.isNotEmpty((CharSequence)branchName)) {
            deleteFiles = (DeleteFiles)deleteFiles.toBranch(HiveUtils.getTableSnapshotRef((String)branchName));
        }
        deleteFiles.deleteFromRowFilter(exp).commit();
    }

    public static PartitionData toPartitionData(StructLike key, Types.StructType keyType) {
        PartitionData keyTemplate = new PartitionData(keyType);
        return keyTemplate.copyFor(key);
    }

    public static PartitionData toPartitionData(StructLike sourceKey, Types.StructType sourceKeyType, Types.StructType targetKeyType) {
        StructProjection projection = StructProjection.create(sourceKeyType, targetKeyType).wrap(sourceKey);
        return IcebergTableUtil.toPartitionData(projection, targetKeyType);
    }

    public static Expression generateExpressionFromPartitionSpec(Table table, Map<String, String> partitionSpec, boolean latestSpecOnly) throws SemanticException {
        Map partitionFieldMap = IcebergTableUtil.getPartitionFields(table, latestSpecOnly).stream().collect(Collectors.toMap(PartitionField::name, Function.identity()));
        Expression finalExp = Expressions.alwaysTrue();
        for (Map.Entry<String, String> entry : partitionSpec.entrySet()) {
            String partColName = entry.getKey();
            if (partitionFieldMap.containsKey(partColName)) {
                PartitionField partitionField = (PartitionField)partitionFieldMap.get(partColName);
                if (partitionField.transform().isIdentity()) {
                    Type partKeyType = table.schema().findField(partitionField.sourceId()).type();
                    Object partKeyVal = Conversions.fromPartitionString(partKeyType, entry.getValue());
                    UnboundPredicate<Object> boundPredicate = Expressions.equal(partColName, partKeyVal);
                    finalExp = Expressions.and(finalExp, boundPredicate);
                    continue;
                }
                throw new SemanticException(String.format("Partition transforms are not supported here: %s", partColName));
            }
            throw new SemanticException(String.format("No partition column by the name: %s", partColName));
        }
        return finalExp;
    }

    public static List<PartitionField> getPartitionFields(Table table, boolean latestSpecOnly) {
        return latestSpecOnly ? table.spec().fields() : table.specs().values().stream().flatMap(spec -> spec.fields().stream().filter(f -> !f.transform().isVoid())).distinct().collect(Collectors.toList());
    }

    public static List<String> getPartitionNames(Table table, Map<String, String> partSpecMap, boolean latestSpecOnly) throws SemanticException {
        ImmutableList<String> immutableList;
        block8: {
            Expression expression = IcebergTableUtil.generateExpressionFromPartitionSpec(table, partSpecMap, latestSpecOnly);
            PartitionsTable partitionsTable = (PartitionsTable)MetadataTableUtils.createMetadataTableInstance(table, MetadataTableType.PARTITIONS);
            CloseableIterable fileScanTasks = partitionsTable.newScan().planFiles();
            try {
                immutableList = FluentIterable.from(fileScanTasks).transformAndConcat(task -> task.asDataTask().rows()).transform(row -> {
                    StructLike data = row.get(0, StructProjection.class);
                    PartitionSpec spec = table.specs().get(row.get(1, Integer.class));
                    return Maps.immutableEntry(IcebergTableUtil.toPartitionData(data, Partitioning.partitionType(table), spec.partitionType()), spec);
                }).filter(e -> {
                    ResidualEvaluator resEval = ResidualEvaluator.of((PartitionSpec)e.getValue(), expression, false);
                    return ((PartitionSpec)e.getValue()).isPartitioned() && resEval.residualFor((StructLike)e.getKey()).isEquivalentTo(Expressions.alwaysTrue()) && (((PartitionSpec)e.getValue()).specId() == table.spec().specId() || !latestSpecOnly);
                }).transform(e -> ((PartitionSpec)e.getValue()).partitionToPath((StructLike)e.getKey())).toSortedList(Comparator.naturalOrder());
                if (fileScanTasks == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (fileScanTasks != null) {
                        try {
                            fileScanTasks.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e2) {
                    throw new SemanticException(String.format("Error while fetching the partitions due to: %s", e2));
                }
            }
            fileScanTasks.close();
        }
        return immutableList;
    }

    public static PartitionSpec getPartitionSpec(Table icebergTable, String partitionPath) throws MetaException, HiveException {
        if (icebergTable == null || partitionPath == null || partitionPath.isEmpty()) {
            throw new HiveException("Table and partitionPath must not be null or empty.");
        }
        ArrayList fieldNames = Lists.newArrayList(Warehouse.makeSpecFromName((String)partitionPath).keySet());
        return icebergTable.specs().values().stream().filter(spec -> {
            List<String> specFieldNames = spec.fields().stream().map(PartitionField::name).toList();
            return specFieldNames.equals(fieldNames);
        }).findFirst().orElseThrow(() -> new HiveException("No matching partition spec found for partition path: " + partitionPath));
    }

    public static TransformSpec getTransformSpec(Table table, String transformName, int sourceId) {
        TransformSpec spec = TransformSpec.fromString((String)transformName.toUpperCase(), (String)table.schema().findColumnName(sourceId));
        return spec;
    }

    public static <T> List<T> readColStats(Table table, Long snapshotId, Predicate<BlobMetadata> filter) {
        ArrayList<Object> colStats = Lists.newArrayList();
        String statsPath = IcebergTableUtil.getColStatsPath(table, snapshotId);
        if (statsPath == null) {
            return colStats;
        }
        try (PuffinReader reader = Puffin.read(table.io().newInputFile(statsPath)).build();){
            List<BlobMetadata> blobMetadata = reader.fileMetadata().blobs();
            if (filter != null) {
                blobMetadata = blobMetadata.stream().filter(filter).toList();
            }
            Iterator it = Iterables.transform(reader.readAll(blobMetadata), Pair::second).iterator();
            LOG.info("Using column stats from: {}", (Object)statsPath);
            while (it.hasNext()) {
                byte[] byteBuffer = ByteBuffers.toByteArray((ByteBuffer)it.next());
                colStats.add(SerializationUtils.deserialize((byte[])byteBuffer));
            }
        }
        catch (Exception e) {
            LOG.warn("Unable to read column stats: {}", (Object)e.getMessage());
        }
        return colStats;
    }

    public static ExecutorService newDeleteThreadPool(String completeName, int numThreads) {
        AtomicInteger deleteThreadsIndex = new AtomicInteger(0);
        return Executors.newFixedThreadPool(numThreads, runnable -> {
            Thread thread = new Thread(runnable);
            thread.setName("remove-snapshot-" + completeName + "-" + deleteThreadsIndex.getAndIncrement());
            return thread;
        });
    }

    public static boolean hasUndergonePartitionEvolution(Table table) {
        return table.currentSnapshot() != null && table.currentSnapshot().allManifests(table.io()).parallelStream().map(ManifestFile::partitionSpecId).anyMatch(id -> id.intValue() != table.spec().specId());
    }

    public static <T extends ContentFile<?>> Set<String> getPartitionNames(Table icebergTable, Iterable<T> files, Boolean latestSpecOnly) {
        HashSet partitions = Sets.newHashSet();
        int tableSpecId = icebergTable.spec().specId();
        for (ContentFile file : files) {
            if (latestSpecOnly != null && (!Boolean.TRUE.equals(latestSpecOnly) || file.specId() != tableSpecId) && (!Boolean.FALSE.equals(latestSpecOnly) || file.specId() == tableSpecId)) continue;
            String partName = icebergTable.specs().get(file.specId()).partitionToPath(file.partition());
            partitions.add(partName);
        }
        return partitions;
    }

    public static List<Partition> convertNameToMetastorePartition(org.apache.hadoop.hive.ql.metadata.Table hmsTable, Collection<String> partNames) {
        ArrayList<Partition> partitions = Lists.newArrayList();
        for (String partName : partNames) {
            LinkedHashMap partSpecMap = Maps.newLinkedHashMap();
            Warehouse.makeSpecFromName(partSpecMap, (Path)new Path(partName), null);
            partitions.add((Partition)new DummyPartition(hmsTable, partName, partSpecMap));
        }
        return partitions;
    }

    public static TableFetcher getTableFetcher(IMetaStoreClient msc, String catalogName, String dbPattern, String tablePattern) {
        return new TableFetcher.Builder(msc, catalogName, dbPattern, tablePattern).tableTypes("EXTERNAL_TABLE").tableCondition("hive_filter_field_params__table_type like \"ICEBERG\" ").build();
    }

    public static String defaultWarehouseLocation(TableIdentifier tableIdentifier, Configuration conf, Properties catalogProperties) {
        StringBuilder sb = new StringBuilder();
        String warehouseLocation = conf.get(String.format("iceberg.catalog.%s.warehouse", catalogProperties.getProperty("iceberg.catalog")));
        sb.append(warehouseLocation).append('/');
        for (String level : tableIdentifier.namespace().levels()) {
            sb.append(level).append('/');
        }
        sb.append(tableIdentifier.name());
        return sb.toString();
    }
}

