/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.task;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeoutException;
import org.apache.helix.AccessOption;
import org.apache.helix.BaseDataAccessor;
import org.apache.helix.ConfigAccessor;
import org.apache.helix.HelixAdmin;
import org.apache.helix.HelixDataAccessor;
import org.apache.helix.HelixException;
import org.apache.helix.HelixManager;
import org.apache.helix.HelixProperty;
import org.apache.helix.PropertyKey;
import org.apache.helix.PropertyPathBuilder;
import org.apache.helix.manager.zk.ZKHelixAdmin;
import org.apache.helix.manager.zk.ZKHelixDataAccessor;
import org.apache.helix.manager.zk.ZkBaseDataAccessor;
import org.apache.helix.model.ClusterConfig;
import org.apache.helix.model.InstanceConfig;
import org.apache.helix.model.LiveInstance;
import org.apache.helix.store.HelixPropertyStore;
import org.apache.helix.store.zk.ZkHelixPropertyStore;
import org.apache.helix.task.JobConfig;
import org.apache.helix.task.JobContext;
import org.apache.helix.task.JobDag;
import org.apache.helix.task.JobQueue;
import org.apache.helix.task.TargetState;
import org.apache.helix.task.TaskConfig;
import org.apache.helix.task.TaskExecutionInfo;
import org.apache.helix.task.TaskPartitionState;
import org.apache.helix.task.TaskState;
import org.apache.helix.task.TaskUtil;
import org.apache.helix.task.UserContentStore;
import org.apache.helix.task.Workflow;
import org.apache.helix.task.WorkflowConfig;
import org.apache.helix.task.WorkflowContext;
import org.apache.helix.util.HelixUtil;
import org.apache.helix.zookeeper.api.client.RealmAwareZkClient;
import org.apache.helix.zookeeper.datamodel.ZNRecord;
import org.apache.helix.zookeeper.zkclient.DataUpdater;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TaskDriver {
    private static final Logger LOG = LoggerFactory.getLogger(TaskDriver.class);
    private static final int DEFAULT_TIMEOUT = 300000;
    private static final long DEFAULT_SLEEP = 1000L;
    private static final Set<TaskState> ILLEGAL_JOB_STATES_FOR_TASK_MODIFICATION = new HashSet<TaskState>(Arrays.asList(TaskState.TIMING_OUT, TaskState.TIMED_OUT, TaskState.FAILING, TaskState.FAILED, TaskState.ABORTED, TaskState.COMPLETED, TaskState.STOPPING, TaskState.STOPPED));
    private static final long DEFAULT_CONFIGS_LIMITATION = HelixUtil.getSystemPropertyAsLong("helixTask.configsLimitation", 100000L);
    private static final String TASK_START_TIME_KEY = "START_TIME";
    protected long _configsLimitation = DEFAULT_CONFIGS_LIMITATION;
    private final HelixDataAccessor _accessor;
    private final HelixPropertyStore<ZNRecord> _propertyStore;
    private final HelixAdmin _admin;
    private final String _clusterName;

    public TaskDriver(HelixManager manager) {
        this(manager.getClusterManagmentTool(), manager.getHelixDataAccessor(), manager.getHelixPropertyStore(), manager.getClusterName());
    }

    @Deprecated
    public TaskDriver(RealmAwareZkClient client, String clusterName) {
        this(client, new ZkBaseDataAccessor<ZNRecord>(client), clusterName);
    }

    @Deprecated
    public TaskDriver(RealmAwareZkClient client, ZkBaseDataAccessor<ZNRecord> baseAccessor, String clusterName) {
        this(new ZKHelixAdmin(client), new ZKHelixDataAccessor(clusterName, baseAccessor), new ZkHelixPropertyStore<ZNRecord>(baseAccessor, PropertyPathBuilder.propertyStore(clusterName), null), clusterName);
    }

    @Deprecated
    public TaskDriver(HelixAdmin admin, HelixDataAccessor accessor, ConfigAccessor cfgAccessor, HelixPropertyStore<ZNRecord> propertyStore, String clusterName) {
        this(admin, accessor, propertyStore, clusterName);
    }

    public TaskDriver(HelixAdmin admin, HelixDataAccessor accessor, HelixPropertyStore<ZNRecord> propertyStore, String clusterName) {
        this._admin = admin;
        this._accessor = accessor;
        this._propertyStore = propertyStore;
        this._clusterName = clusterName;
    }

    public void start(Workflow flow) {
        LOG.info("Starting workflow {}", (Object)flow.getName());
        flow.validate();
        this.validateZKNodeLimitation(flow.getJobConfigs().keySet().size() + 1);
        WorkflowConfig newWorkflowConfig = new WorkflowConfig.Builder(flow.getWorkflowConfig()).setWorkflowId(flow.getName()).build();
        HashMap<String, String> jobTypes = new HashMap<String, String>();
        for (String job : flow.getJobConfigs().keySet()) {
            JobConfig jobCfg;
            JobConfig.Builder jobCfgBuilder = JobConfig.Builder.fromMap(flow.getJobConfigs().get(job));
            if (flow.getTaskConfigs() != null && flow.getTaskConfigs().containsKey(job)) {
                jobCfgBuilder.addTaskConfigs(flow.getTaskConfigs().get(job));
            }
            if ((jobCfg = jobCfgBuilder.build()).getJobType() != null) {
                jobTypes.put(job, jobCfg.getJobType());
            }
            this.addJobConfig(job, jobCfg);
        }
        newWorkflowConfig.setJobTypes(jobTypes);
        if (!TaskUtil.createWorkflowConfig(this._accessor, flow.getName(), newWorkflowConfig)) {
            HashSet<String> failedJobRemoval = new HashSet<String>();
            for (String job : flow.getJobConfigs().keySet()) {
                if (TaskUtil.removeJobConfig(this._accessor, job)) continue;
                failedJobRemoval.add(job);
            }
            throw new HelixException(String.format("Failed to add workflow configuration for workflow %s. It's possible that a workflow of the same name already exists or there was a connection issue. JobConfig deletion attempted but failed for the following jobs: %s", flow.getName(), failedJobRemoval));
        }
    }

    public void updateWorkflow(String workflow, WorkflowConfig newWorkflowConfig) {
        if (newWorkflowConfig.getWorkflowId() == null || newWorkflowConfig.getWorkflowId().isEmpty()) {
            newWorkflowConfig.getRecord().setSimpleField(WorkflowConfig.WorkflowConfigProperty.WorkflowID.name(), workflow);
        }
        if (workflow == null || !workflow.equals(newWorkflowConfig.getWorkflowId())) {
            throw new HelixException(String.format("Workflow name {%s} does not match the workflow Id from WorkflowConfig {%s}", workflow, newWorkflowConfig.getWorkflowId()));
        }
        WorkflowConfig currentConfig = TaskUtil.getWorkflowConfig(this._accessor, workflow);
        if (currentConfig == null) {
            throw new HelixException("Workflow " + workflow + " does not exist!");
        }
        if (currentConfig.isTerminable()) {
            throw new HelixException("Workflow " + workflow + " is terminable, not allow to change its configuration!");
        }
        newWorkflowConfig.setJobDag(currentConfig.getJobDag());
        if (!TaskUtil.setWorkflowConfig(this._accessor, workflow, newWorkflowConfig)) {
            LOG.error("Failed to update workflow configuration for workflow {}", (Object)workflow);
        }
    }

    public void createQueue(JobQueue queue) {
        this.start(queue);
    }

    public void flushQueue(String queue) {
        this.cleanupQueue(queue);
    }

    public void deleteJob(String queue, String job) {
        this.deleteNamespacedJob(queue, TaskUtil.getNamespacedJobName(queue, job), false);
    }

    public void deleteJob(String queue, String job, boolean forceDelete) {
        this.deleteNamespacedJob(queue, TaskUtil.getNamespacedJobName(queue, job), forceDelete);
    }

    public void deleteNamespacedJob(String queue, String job, boolean forceDelete) {
        WorkflowConfig jobQueueConfig = TaskUtil.getWorkflowConfig(this._accessor, queue);
        if (forceDelete) {
            LOG.info("Forcefully removing job: {} from queue: {}", (Object)job, (Object)queue);
            if (!TaskUtil.removeJob(this._accessor, this._propertyStore, job)) {
                LOG.info("Failed to delete job: {} from queue: {}", (Object)job, (Object)queue);
                throw new HelixException("Failed to delete job: " + job + " from queue: " + queue);
            }
            if (jobQueueConfig != null) {
                boolean isRecurringWorkflow;
                boolean bl = isRecurringWorkflow = jobQueueConfig.getScheduleConfig() != null && jobQueueConfig.getScheduleConfig().isRecurring();
                if (isRecurringWorkflow) {
                    this.deleteJobFromLastScheduledQueue(queue, TaskUtil.getDenamespacedJobName(queue, job));
                }
            }
            return;
        }
        if (jobQueueConfig == null) {
            throw new IllegalArgumentException(String.format("JobQueue %s's config is not found!", queue));
        }
        if (!jobQueueConfig.isJobQueue()) {
            throw new IllegalArgumentException(String.format("%s is not a queue!", queue));
        }
        boolean isRecurringWorkflow = jobQueueConfig.getScheduleConfig() != null && jobQueueConfig.getScheduleConfig().isRecurring();
        String denamespacedJob = TaskUtil.getDenamespacedJobName(queue, job);
        if (isRecurringWorkflow) {
            this.deleteJobFromLastScheduledQueue(queue, denamespacedJob);
        }
        this.deleteJobFromQueue(queue, denamespacedJob);
    }

    private void deleteJobFromLastScheduledQueue(String queue, String denamespacedJob) {
        WorkflowConfig lastWorkflowCfg;
        WorkflowContext wCtx = TaskUtil.getWorkflowContext(this._propertyStore, queue);
        String lastScheduledQueue = null;
        if (wCtx != null) {
            lastScheduledQueue = wCtx.getLastScheduledSingleWorkflow();
        }
        if (lastScheduledQueue != null && (lastWorkflowCfg = TaskUtil.getWorkflowConfig(this._accessor, lastScheduledQueue)) != null) {
            this.deleteJobFromQueue(lastScheduledQueue, denamespacedJob);
        }
    }

    private void deleteJobFromQueue(String queue, String job) {
        String workflowState;
        WorkflowContext workflowCtx = TaskUtil.getWorkflowContext(this._propertyStore, queue);
        String string = workflowState = workflowCtx != null ? workflowCtx.getWorkflowState().name() : TaskState.NOT_STARTED.name();
        if (workflowState.equals(TaskState.IN_PROGRESS.name())) {
            throw new IllegalStateException("Queue " + queue + " is still running!");
        }
        if (workflowState.equals(TaskState.COMPLETED.name()) || workflowState.equals(TaskState.FAILED.name()) || workflowState.equals(TaskState.ABORTED.name())) {
            LOG.warn("Queue {} has already reached its final state, skip deleting job from it.", (Object)queue);
            return;
        }
        if (!TaskUtil.removeJobsFromWorkflow(this._accessor, this._propertyStore, queue, Collections.singleton(TaskUtil.getNamespacedJobName(queue, job)), true)) {
            LOG.error("Failed to delete job {} from queue {}.", (Object)job, (Object)queue);
            throw new HelixException("Failed to delete job " + job + " from queue " + queue);
        }
    }

    public void enqueueJob(String queue, String job, JobConfig.Builder jobBuilder) {
        this.enqueueJobs(queue, Collections.singletonList(job), Collections.singletonList(jobBuilder));
    }

    public void enqueueJobs(String queue, List<String> jobs, List<JobConfig.Builder> jobBuilders) {
        WorkflowConfig workflowConfig = TaskUtil.getWorkflowConfig(this._accessor, queue);
        if (workflowConfig == null) {
            throw new IllegalArgumentException("Queue " + queue + " config does not yet exist!");
        }
        if (workflowConfig.isTerminable()) {
            throw new IllegalArgumentException(queue + " is not a queue!");
        }
        int capacity = workflowConfig.getCapacity();
        int queueSize = workflowConfig.getJobDag().size();
        if (capacity > 0 && queueSize >= capacity) {
            Set<String> expiredJobs;
            WorkflowContext workflowContext = TaskUtil.getWorkflowContext(this._propertyStore, queue);
            if (workflowContext != null && !TaskUtil.removeJobsFromWorkflow(this._accessor, this._propertyStore, queue, expiredJobs = TaskUtil.getExpiredJobs(this._accessor, this._propertyStore, workflowConfig, workflowContext), true)) {
                LOG.warn("Failed to clean up expired and completed jobs from queue {}", (Object)queue);
            }
            if ((workflowConfig = TaskUtil.getWorkflowConfig(this._accessor, queue)).getJobDag().size() >= capacity) {
                throw new HelixException(String.format("Failed to enqueue job, queue %s is full.", queue));
            }
        }
        if ((workflowConfig = TaskUtil.getWorkflowConfig(this._accessor, queue)).getJobDag().size() + jobs.size() >= capacity) {
            throw new IllegalStateException(String.format("Queue %s already reaches its max capacity %d, failed to add %s", queue, capacity, jobs.toString()));
        }
        this.validateZKNodeLimitation(1);
        ArrayList<Object> jobConfigs = new ArrayList<Object>();
        ArrayList<String> namespacedJobNames = new ArrayList<String>();
        ArrayList<String> jobTypeList = new ArrayList<String>();
        try {
            for (int i = 0; i < jobBuilders.size(); ++i) {
                JobConfig jobConfig = jobBuilders.get(i).setWorkflow(queue).build();
                String namespacedJobName = TaskUtil.getNamespacedJobName(queue, jobs.get(i));
                this.addJobConfig(namespacedJobName, jobConfig);
                jobConfigs.add(jobConfig);
                namespacedJobNames.add(namespacedJobName);
                jobTypeList.add(jobConfig.getJobType());
            }
        }
        catch (HelixException e) {
            LOG.error("Failed to add job configs {}. Remove them all!", (Object)jobs.toString());
            for (String job : jobs) {
                String namespacedJobName = TaskUtil.getNamespacedJobName(queue, job);
                TaskUtil.removeJobConfig(this._accessor, namespacedJobName);
            }
        }
        DataUpdater<ZNRecord> updater = currentData -> {
            if (currentData == null) {
                throw new HelixException(String.format("enqueueJobs DataUpdater: JobQueue %s config is not found!", queue));
            }
            JobDag jobDag = JobDag.fromJson(currentData.getSimpleField(WorkflowConfig.WorkflowConfigProperty.Dag.name()));
            Set<String> allNodes = jobDag.getAllNodes();
            if (capacity > 0 && allNodes.size() + jobConfigs.size() >= capacity) {
                for (String job : jobs) {
                    String namespacedJobName = TaskUtil.getNamespacedJobName(queue, job);
                    TaskUtil.removeJobConfig(this._accessor, namespacedJobName);
                }
                throw new IllegalStateException(String.format("Queue %s already reaches its max capacity %d, failed to add %s", queue, capacity, jobs.toString()));
            }
            String lastNodeName = null;
            for (int i = 0; i < namespacedJobNames.size(); ++i) {
                String namespacedJobName = (String)namespacedJobNames.get(i);
                if (allNodes.contains(namespacedJobName)) {
                    throw new IllegalStateException(String.format("Could not add to queue %s, job %s already exists", queue, jobs.get(i)));
                }
                jobDag.addNode(namespacedJobName);
                String candidate = null;
                if (lastNodeName == null) {
                    for (String node : allNodes) {
                        if (node.equals(namespacedJobName) || !jobDag.getDirectChildren(node).isEmpty()) continue;
                        candidate = node;
                        break;
                    }
                } else {
                    candidate = lastNodeName;
                }
                if (candidate == null) continue;
                jobDag.addParentToChild(candidate, namespacedJobName);
                lastNodeName = namespacedJobName;
            }
            Map<String, String> jobTypes = currentData.getMapField(WorkflowConfig.WorkflowConfigProperty.JobTypes.name());
            for (String jobType : jobTypeList) {
                if (jobType == null) continue;
                if (jobTypes == null) {
                    jobTypes = new HashMap<String, String>();
                }
                jobTypes.put(queue, jobType);
            }
            if (jobTypes != null) {
                currentData.setMapField(WorkflowConfig.WorkflowConfigProperty.JobTypes.name(), jobTypes);
            }
            try {
                currentData.setSimpleField(WorkflowConfig.WorkflowConfigProperty.Dag.name(), jobDag.toJson());
            }
            catch (Exception e) {
                throw new IllegalStateException(String.format("Could not add jobs %s to queue %s", jobs.toString(), queue), e);
            }
            return currentData;
        };
        String path = this._accessor.keyBuilder().resourceConfig(queue).getPath();
        boolean status = this._accessor.getBaseDataAccessor().update(path, updater, AccessOption.PERSISTENT);
        if (!status) {
            LOG.error("Failed to update WorkflowConfig, remove all jobs {}", (Object)jobs.toString());
            for (String job : jobs) {
                TaskUtil.removeJobConfig(this._accessor, job);
            }
            throw new HelixException("Failed to enqueue job");
        }
    }

    public void addTask(String workflowName, String jobName, TaskConfig taskConfig) throws TimeoutException, InterruptedException {
        this.addTask(workflowName, jobName, taskConfig, 300000L);
    }

    public void addTask(String workflowName, String jobName, TaskConfig taskConfig, long timeoutMs) throws TimeoutException, InterruptedException {
        TaskState jobState;
        if (timeoutMs < 1000L) {
            throw new IllegalArgumentException(String.format("Timeout is less than the minimum acceptable timeout value which is %s ms", 1000L));
        }
        long endTime = System.currentTimeMillis() + timeoutMs;
        this.validateConfigsForTaskModifications(workflowName, jobName, taskConfig);
        String nameSpaceJobName = TaskUtil.getNamespacedJobName(workflowName, jobName);
        JobConfig jobConfig = TaskUtil.getJobConfig(this._accessor, nameSpaceJobName);
        for (String taskEntry : jobConfig.getMapConfigs().keySet()) {
            if (!taskEntry.equals(taskConfig.getId())) continue;
            throw new HelixException("Task cannot be added because another task with the same ID already exists!");
        }
        WorkflowContext workflowContext = this.getWorkflowContext(workflowName);
        JobContext jobContext = this.getJobContext(nameSpaceJobName);
        if (workflowContext != null && jobContext != null && (jobState = workflowContext.getJobState(nameSpaceJobName)) != null && ILLEGAL_JOB_STATES_FOR_TASK_MODIFICATION.contains((Object)jobState)) {
            throw new HelixException("Job " + nameSpaceJobName + " is in illegal state for task addition. Job State is " + jobState);
        }
        DataUpdater<ZNRecord> updater = currentData -> {
            if (currentData != null) {
                currentData.setMapField(taskConfig.getId(), taskConfig.getConfigMap());
            } else {
                LOG.error("JobConfig DataUpdater: Fails to update JobConfig. CurrentData is null.");
            }
            return currentData;
        };
        this.updateTaskInJobConfig(workflowName, jobName, updater);
        workflowContext = (WorkflowContext)this._accessor.getProperty(this._accessor.keyBuilder().workflowContextZNode(workflowName));
        jobContext = (JobContext)this._accessor.getProperty(this._accessor.keyBuilder().jobContextZNode(workflowName, jobName));
        if (workflowContext == null || jobContext == null) {
            return;
        }
        String taskID = taskConfig.getId();
        while (System.currentTimeMillis() <= endTime) {
            jobContext = (JobContext)this._accessor.getProperty(this._accessor.keyBuilder().jobContextZNode(workflowName, jobName));
            workflowContext = (WorkflowContext)this._accessor.getProperty(this._accessor.keyBuilder().workflowContextZNode(workflowName));
            if (jobContext.getTaskIdPartitionMap().containsKey(taskID) && workflowContext.getJobState(nameSpaceJobName) == TaskState.IN_PROGRESS) {
                return;
            }
            Thread.sleep(1000L);
        }
        throw new TimeoutException("An unexpected issue happened while task being added to the job!");
    }

    public void deleteTask(String workflowName, String jobName, String taskID) throws TimeoutException, InterruptedException {
        this.deleteTask(workflowName, jobName, taskID, 300000L);
    }

    public void deleteTask(String workflowName, final String jobName, final String taskID, long timeoutMs) throws TimeoutException, InterruptedException {
        TaskState jobState;
        long endTime = System.currentTimeMillis() + timeoutMs;
        String nameSpaceJobName = TaskUtil.getNamespacedJobName(workflowName, jobName);
        JobConfig jobConfig = this.getJobConfig(nameSpaceJobName);
        if (jobConfig == null) {
            throw new IllegalArgumentException("Job " + nameSpaceJobName + " config does not exist!");
        }
        TaskConfig taskConfig = null;
        Map<String, TaskConfig> allTaskConfigs = jobConfig.getTaskConfigMap();
        for (Map.Entry<String, TaskConfig> entry : allTaskConfigs.entrySet()) {
            if (!entry.getKey().equals(taskID)) continue;
            taskConfig = entry.getValue();
        }
        this.validateConfigsForTaskModifications(workflowName, jobName, taskConfig);
        WorkflowContext workflowContext = this.getWorkflowContext(workflowName);
        JobContext jobContext = this.getJobContext(nameSpaceJobName);
        if (workflowContext != null && jobContext != null && (jobState = workflowContext.getJobState(nameSpaceJobName)) != null && ILLEGAL_JOB_STATES_FOR_TASK_MODIFICATION.contains((Object)jobState)) {
            throw new HelixException("Job " + nameSpaceJobName + " is in illegal state for task deletion. Job State is " + jobState);
        }
        DataUpdater<ZNRecord> taskRemover = new DataUpdater<ZNRecord>(){

            @Override
            public ZNRecord update(ZNRecord currentData) {
                if (currentData != null) {
                    Map<String, Map<String, String>> taskMap = currentData.getMapFields();
                    if (taskMap == null) {
                        LOG.warn("Could not update the jobConfig: {} Znode MapField is null.", (Object)jobName);
                        return null;
                    }
                    HashMap<String, Map<String, String>> newTaskMap = new HashMap<String, Map<String, String>>();
                    for (Map.Entry<String, Map<String, String>> entry : taskMap.entrySet()) {
                        if (entry.getKey().equals(taskID)) continue;
                        newTaskMap.put(entry.getKey(), entry.getValue());
                    }
                    currentData.setMapFields(newTaskMap);
                }
                return currentData;
            }
        };
        this.updateTaskInJobConfig(workflowName, jobName, taskRemover);
        workflowContext = (WorkflowContext)this._accessor.getProperty(this._accessor.keyBuilder().workflowContextZNode(workflowName));
        jobContext = (JobContext)this._accessor.getProperty(this._accessor.keyBuilder().jobContextZNode(workflowName, jobName));
        if (workflowContext == null || jobContext == null) {
            return;
        }
        while (System.currentTimeMillis() <= endTime) {
            jobContext = (JobContext)this._accessor.getProperty(this._accessor.keyBuilder().jobContextZNode(workflowName, jobName));
            workflowContext = (WorkflowContext)this._accessor.getProperty(this._accessor.keyBuilder().workflowContextZNode(workflowName));
            if (!jobContext.getTaskIdPartitionMap().containsKey(taskID)) {
                return;
            }
            Thread.sleep(1000L);
        }
        throw new TimeoutException("An unexpected issue happened while task being deleted from the job!");
    }

    private void validateConfigsForTaskModifications(String workflowName, String jobName, TaskConfig taskConfig) {
        WorkflowConfig workflowConfig = TaskUtil.getWorkflowConfig(this._accessor, workflowName);
        String nameSpaceJobName = TaskUtil.getNamespacedJobName(workflowName, jobName);
        JobConfig jobConfig = TaskUtil.getJobConfig(this._accessor, nameSpaceJobName);
        if (workflowConfig == null) {
            throw new IllegalArgumentException(String.format("Workflow config for workflow %s does not exist!", workflowName));
        }
        if (jobConfig == null) {
            throw new IllegalArgumentException(String.format("Job config for job %s does not exist!", nameSpaceJobName));
        }
        if (taskConfig == null) {
            throw new IllegalArgumentException("TaskConfig is null!");
        }
        if (taskConfig.getId() == null) {
            throw new HelixException("Task cannot be added or deleted because taskID is null!");
        }
        if (jobConfig.getTargetResource() != null) {
            throw new HelixException(String.format("Job %s is a targeted job. New task cannot be added/deleted to/from this job!", nameSpaceJobName));
        }
        if (taskConfig.getCommand() == null == (jobConfig.getCommand() == null)) {
            throw new HelixException(String.format("Command must exist in either jobconfig (%s) or taskconfig (%s), not both!", jobName, taskConfig.getId()));
        }
    }

    private void updateTaskInJobConfig(String workflowName, String jobName, DataUpdater<ZNRecord> updater) {
        String nameSpaceJobName = TaskUtil.getNamespacedJobName(workflowName, jobName);
        String path = this._accessor.keyBuilder().resourceConfig(nameSpaceJobName).getPath();
        boolean status = this._accessor.getBaseDataAccessor().update(path, updater, AccessOption.PERSISTENT);
        if (!status) {
            LOG.error("Failed to update task in the job {}", (Object)nameSpaceJobName);
            throw new HelixException("Failed to update task in the job");
        }
    }

    @Deprecated
    public void cleanupJobQueue(String queue) {
        this.cleanupQueue(queue);
    }

    public void cleanupQueue(String queue) {
        WorkflowConfig workflowConfig = TaskUtil.getWorkflowConfig(this._accessor, queue);
        if (workflowConfig == null) {
            throw new IllegalArgumentException("Queue " + queue + " does not yet exist!");
        }
        if (!workflowConfig.isJobQueue() || workflowConfig.isTerminable()) {
            throw new IllegalArgumentException(queue + " is not a queue!");
        }
        WorkflowContext wCtx = TaskUtil.getWorkflowContext(this._propertyStore, queue);
        if (wCtx == null || wCtx.getWorkflowState() == null) {
            throw new IllegalStateException("Queue " + queue + " does not have a valid work state!");
        }
        HashSet<String> jobs = new HashSet<String>();
        for (String jobNode : workflowConfig.getJobDag().getAllNodes()) {
            TaskState curState = wCtx.getJobState(jobNode);
            if ((curState == null || curState != TaskState.ABORTED) && curState != TaskState.COMPLETED && curState != TaskState.FAILED) continue;
            jobs.add(jobNode);
        }
        TaskUtil.removeJobsFromWorkflow(this._accessor, this._propertyStore, queue, jobs, true);
    }

    private void addJobConfig(String job, JobConfig jobConfig) {
        LOG.info("Add job configuration " + job);
        JobConfig newJobCfg = new JobConfig(job, jobConfig);
        if (!TaskUtil.createJobConfig(this._accessor, job, newJobCfg)) {
            throw new HelixException("Failed to add job configuration for job " + job + ". It's possible that a job of the same name already exists or there was a connection issue");
        }
    }

    public void resume(String workflow) {
        this.setWorkflowTargetState(workflow, TargetState.START);
    }

    public void stop(String workflow) {
        this.setWorkflowTargetState(workflow, TargetState.STOP);
    }

    public void waitToStop(String workflow, long timeout) throws InterruptedException {
        this.setWorkflowTargetState(workflow, TargetState.STOP);
        long endTime = System.currentTimeMillis() + timeout;
        while (System.currentTimeMillis() <= endTime) {
            WorkflowContext workflowContext = this.getWorkflowContext(workflow);
            if (workflowContext == null || !TaskState.STOPPED.equals((Object)workflowContext.getWorkflowState())) {
                Thread.sleep(1000L);
                continue;
            }
            return;
        }
        throw new HelixException(String.format("Fail to stop the workflow/queue %s with in %d milliseconds.", workflow, timeout));
    }

    public void delete(String workflow) {
        this.delete(workflow, false);
    }

    public void delete(String workflow, boolean forceDelete) {
        WorkflowContext wCtx = TaskUtil.getWorkflowContext(this._propertyStore, workflow);
        if (forceDelete) {
            LOG.info("Forcefully removing workflow: " + workflow);
            this.removeWorkflowFromZK(workflow);
        } else {
            this.setWorkflowTargetState(workflow, TargetState.DELETE);
        }
        if (wCtx != null && wCtx.getScheduledWorkflows() != null) {
            for (String scheduledWorkflow : wCtx.getScheduledWorkflows()) {
                if (forceDelete) {
                    WorkflowContext scheduledWorkflowCtx = TaskUtil.getWorkflowContext(this._propertyStore, scheduledWorkflow);
                    if (scheduledWorkflowCtx == null || scheduledWorkflowCtx.getFinishTime() == -1L) continue;
                    this.removeWorkflowFromZK(scheduledWorkflow);
                    continue;
                }
                this.setWorkflowTargetState(scheduledWorkflow, TargetState.DELETE);
            }
        }
    }

    private void removeWorkflowFromZK(String workflow) {
        boolean success;
        HashSet<String> jobSet = new HashSet<String>();
        WorkflowConfig wCfg = TaskUtil.getWorkflowConfig(this._accessor, workflow);
        if (wCfg != null) {
            jobSet.addAll(wCfg.getJobDag().getAllNodes());
        }
        if (!(success = TaskUtil.removeWorkflow(this._accessor, this._propertyStore, workflow, jobSet))) {
            LOG.info("Failed to delete the workflow " + workflow);
            throw new HelixException("Failed to delete the workflow " + workflow);
        }
    }

    public void deleteAndWaitForCompletion(String workflow, long timeout) throws InterruptedException {
        this.delete(workflow);
        long endTime = System.currentTimeMillis() + timeout;
        BaseDataAccessor<ZNRecord> baseDataAccessor = this._accessor.getBaseDataAccessor();
        PropertyKey.Builder keyBuilder = this._accessor.keyBuilder();
        String idealStatePath = keyBuilder.idealStates(workflow).getPath();
        String workflowConfigPath = keyBuilder.resourceConfig(workflow).getPath();
        String workflowContextPath = keyBuilder.workflowContext(workflow).getPath();
        while (System.currentTimeMillis() <= endTime) {
            if (baseDataAccessor.exists(idealStatePath, AccessOption.PERSISTENT) || baseDataAccessor.exists(workflowConfigPath, AccessOption.PERSISTENT) || baseDataAccessor.exists(workflowContextPath, AccessOption.PERSISTENT)) {
                Thread.sleep(1000L);
                continue;
            }
            return;
        }
        StringBuilder failed = new StringBuilder();
        if (baseDataAccessor.exists(idealStatePath, AccessOption.PERSISTENT)) {
            failed.append("IdealState ");
        }
        if (baseDataAccessor.exists(workflowConfigPath, AccessOption.PERSISTENT)) {
            failed.append("WorkflowConfig ");
        }
        if (baseDataAccessor.exists(workflowContextPath, AccessOption.PERSISTENT)) {
            failed.append("WorkflowContext ");
        }
        throw new HelixException(String.format("Failed to delete the workflow/queue %s within %d milliseconds. The following components still remain: %s", workflow, timeout, failed.toString()));
    }

    private void setWorkflowTargetState(String workflow, TargetState state) {
        String lastScheduledWorkflow;
        this.setSingleWorkflowTargetState(workflow, state);
        WorkflowContext wCtx = TaskUtil.getWorkflowContext(this._propertyStore, workflow);
        if (wCtx != null && (lastScheduledWorkflow = wCtx.getLastScheduledSingleWorkflow()) != null) {
            this.setSingleWorkflowTargetState(lastScheduledWorkflow, state);
        }
    }

    private void setSingleWorkflowTargetState(String workflow, TargetState state) {
        LOG.info("Set " + workflow + " to target state " + state);
        WorkflowConfig workflowConfig = TaskUtil.getWorkflowConfig(this._accessor, workflow);
        if (workflowConfig == null) {
            LOG.warn("WorkflowConfig for {} not found!", (Object)workflow);
            return;
        }
        WorkflowContext workflowContext = TaskUtil.getWorkflowContext(this._propertyStore, workflow);
        if (state != TargetState.DELETE && workflowContext != null && workflowContext.getFinishTime() != -1L) {
            LOG.info("Workflow {} is already completed, skip to update its target state {}", (Object)workflow, (Object)state);
            return;
        }
        DataUpdater<ZNRecord> updater = currentData -> {
            if (currentData != null) {
                currentData.setSimpleField(WorkflowConfig.WorkflowConfigProperty.TargetState.name(), state.name());
            } else {
                LOG.warn("TargetState DataUpdater: Fails to update target state for {}. CurrentData is null.", (Object)workflow);
            }
            return currentData;
        };
        PropertyKey workflowConfigKey = TaskUtil.getWorkflowConfigKey(this._accessor, workflow);
        this._accessor.getBaseDataAccessor().update(workflowConfigKey.getPath(), updater, AccessOption.PERSISTENT);
    }

    public WorkflowConfig getWorkflowConfig(String workflow) {
        return TaskUtil.getWorkflowConfig(this._accessor, workflow);
    }

    public WorkflowContext getWorkflowContext(String workflow) {
        return TaskUtil.getWorkflowContext(this._propertyStore, workflow);
    }

    public JobConfig getJobConfig(String job) {
        return TaskUtil.getJobConfig(this._accessor, job);
    }

    public JobContext getJobContext(String job) {
        return TaskUtil.getJobContext(this._propertyStore, job);
    }

    public static JobContext getJobContext(HelixManager manager, String job) {
        return TaskUtil.getJobContext(manager, job);
    }

    public static WorkflowConfig getWorkflowConfig(HelixManager manager, String workflow) {
        return TaskUtil.getWorkflowConfig(manager, workflow);
    }

    public static WorkflowContext getWorkflowContext(HelixManager manager, String workflow) {
        return TaskUtil.getWorkflowContext(manager, workflow);
    }

    public static JobConfig getJobConfig(HelixManager manager, String job) {
        return TaskUtil.getJobConfig(manager, job);
    }

    public Map<String, WorkflowConfig> getWorkflows() {
        HashMap<String, WorkflowConfig> workflowConfigMap = new HashMap<String, WorkflowConfig>();
        Map resourceConfigMap = this._accessor.getChildValuesMap(this._accessor.keyBuilder().resourceConfigs(), true);
        for (Map.Entry resource : resourceConfigMap.entrySet()) {
            try {
                WorkflowConfig config = WorkflowConfig.fromHelixProperty((HelixProperty)resource.getValue());
                workflowConfigMap.put(resource.getKey(), config);
            }
            catch (IllegalArgumentException illegalArgumentException) {}
        }
        return workflowConfigMap;
    }

    public TaskState pollForWorkflowState(String workflowName, long timeout, TaskState ... targetStates) throws InterruptedException {
        WorkflowContext ctx;
        long st = System.currentTimeMillis();
        HashSet<TaskState> allowedStates = new HashSet<TaskState>(Arrays.asList(targetStates));
        long timeToSleep = timeout > 100L ? 100L : timeout;
        do {
            Thread.sleep(timeToSleep);
        } while (((ctx = this.getWorkflowContext(workflowName)) == null || ctx.getWorkflowState() == null || !allowedStates.contains((Object)ctx.getWorkflowState())) && System.currentTimeMillis() < st + timeout);
        if (ctx == null || !allowedStates.contains((Object)ctx.getWorkflowState())) {
            throw new HelixException(String.format("Workflow %s context is empty or not in states: %s, current state: %s.", workflowName, Arrays.asList(targetStates), ctx == null ? "null" : ctx.getWorkflowState().toString()));
        }
        return ctx.getWorkflowState();
    }

    public TaskState pollForWorkflowState(String workflowName, TaskState ... targetStates) throws InterruptedException {
        return this.pollForWorkflowState(workflowName, 300000L, targetStates);
    }

    public TaskState pollForJobState(String workflowName, String jobName, long timeout, TaskState ... states) throws InterruptedException {
        WorkflowContext ctx;
        long timeToSleep;
        WorkflowConfig workflowConfig = this.getWorkflowConfig(workflowName);
        if (workflowConfig == null) {
            throw new HelixException(String.format("Workflow %s does not exists!", workflowName));
        }
        long l = timeToSleep = timeout > 50L ? 50L : timeout;
        if (workflowConfig.isRecurring()) {
            do {
                Thread.sleep(timeToSleep);
            } while ((ctx = this.getWorkflowContext(workflowName)) == null || ctx.getLastScheduledSingleWorkflow() == null);
            jobName = jobName.substring(workflowName.length() + 1);
            workflowName = ctx.getLastScheduledSingleWorkflow();
        }
        HashSet<TaskState> allowedStates = new HashSet<TaskState>(Arrays.asList(states));
        long st = System.currentTimeMillis();
        do {
            Thread.sleep(timeToSleep);
        } while (((ctx = this.getWorkflowContext(workflowName)) == null || ctx.getJobState(jobName) == null || !allowedStates.contains((Object)ctx.getJobState(jobName))) && System.currentTimeMillis() < st + timeout);
        if (ctx == null || !allowedStates.contains((Object)ctx.getJobState(jobName))) {
            WorkflowConfig wfcfg = this.getWorkflowConfig(workflowName);
            JobConfig jobConfig = this.getJobConfig(jobName);
            JobContext jbCtx = this.getJobContext(jobName);
            throw new HelixException(String.format("Workflow %s context is null or job %s is not in states: %s; ctx is %s, jobState is %s, wf cfg %s, jobcfg %s, jbctx %s", workflowName, jobName, allowedStates, ctx == null ? "null" : ctx, ctx != null ? ctx.getJobState(jobName) : "null", wfcfg, jobConfig, jbCtx));
        }
        return ctx.getJobState(jobName);
    }

    public TaskState pollForJobState(String workflowName, String jobName, TaskState ... states) throws InterruptedException {
        return this.pollForJobState(workflowName, jobName, 300000L, states);
    }

    public long getLastScheduledTaskTimestamp(String workflowName) {
        return this.getLastScheduledTaskExecutionInfo(workflowName).getStartTimeStamp();
    }

    public TaskExecutionInfo getLastScheduledTaskExecutionInfo(String workflowName) {
        long lastScheduledTaskTimestamp = -1L;
        String jobName = null;
        Integer taskPartitionIndex = null;
        TaskPartitionState state = null;
        WorkflowContext workflowContext = this.getWorkflowContext(workflowName);
        if (workflowContext != null) {
            Map<String, TaskState> allJobStates = workflowContext.getJobStates();
            for (Map.Entry<String, TaskState> jobState : allJobStates.entrySet()) {
                JobContext jobContext;
                if (jobState.getValue().equals((Object)TaskState.NOT_STARTED) || (jobContext = this.getJobContext(jobState.getKey())) == null) continue;
                Set<Integer> allPartitions = jobContext.getPartitionSet();
                for (Integer partition : allPartitions) {
                    long startTimeLong;
                    String startTime = jobContext.getMapField(partition).get(TASK_START_TIME_KEY);
                    if (startTime == null || (startTimeLong = Long.parseLong(startTime)) <= lastScheduledTaskTimestamp) continue;
                    lastScheduledTaskTimestamp = startTimeLong;
                    jobName = jobState.getKey();
                    taskPartitionIndex = partition;
                    state = jobContext.getPartitionState(partition);
                }
            }
        }
        return new TaskExecutionInfo(jobName, taskPartitionIndex, state, lastScheduledTaskTimestamp);
    }

    @Deprecated
    public String getUserContent(String key, UserContentStore.Scope scope, String workflowName, String jobName, String taskName) {
        return TaskUtil.getUserContent(this._propertyStore, key, scope, workflowName, jobName, taskName);
    }

    public Map<String, String> getWorkflowUserContentMap(String workflowName) {
        return TaskUtil.getWorkflowJobUserContentMap(this._propertyStore, workflowName);
    }

    public Map<String, String> getJobUserContentMap(String workflowName, String jobName) {
        String namespacedJobName = TaskUtil.getNamespacedJobName(workflowName, jobName);
        return TaskUtil.getWorkflowJobUserContentMap(this._propertyStore, namespacedJobName);
    }

    public Map<String, String> getTaskUserContentMap(String workflowName, String jobName, String taskPartitionId) {
        String namespacedJobName = TaskUtil.getNamespacedJobName(workflowName, jobName);
        String namespacedTaskName = TaskUtil.getNamespacedTaskName(namespacedJobName, taskPartitionId);
        return TaskUtil.getTaskUserContentMap(this._propertyStore, namespacedJobName, namespacedTaskName);
    }

    public void addOrUpdateWorkflowUserContentMap(String workflowName, Map<String, String> contentToAddOrUpdate) {
        TaskUtil.addOrUpdateWorkflowJobUserContentMap(this._propertyStore, workflowName, contentToAddOrUpdate);
    }

    public void addOrUpdateJobUserContentMap(String workflowName, String jobName, Map<String, String> contentToAddOrUpdate) {
        String namespacedJobName = TaskUtil.getNamespacedJobName(workflowName, jobName);
        TaskUtil.addOrUpdateWorkflowJobUserContentMap(this._propertyStore, namespacedJobName, contentToAddOrUpdate);
    }

    public void addOrUpdateTaskUserContentMap(String workflowName, String jobName, String taskPartitionId, Map<String, String> contentToAddOrUpdate) {
        String namespacedJobName = TaskUtil.getNamespacedJobName(workflowName, jobName);
        String namespacedTaskName = TaskUtil.getNamespacedTaskName(namespacedJobName, taskPartitionId);
        TaskUtil.addOrUpdateTaskUserContentMap(this._propertyStore, namespacedJobName, namespacedTaskName, contentToAddOrUpdate);
    }

    private void validateZKNodeLimitation(int newConfigNodeCount) {
        List<String> resourceConfigs = this._accessor.getChildNames(this._accessor.keyBuilder().resourceConfigs());
        if ((long)(resourceConfigs.size() + newConfigNodeCount) > this._configsLimitation) {
            throw new HelixException("Cannot create more workflows or jobs because there are already too many items created in the path CONFIGS.");
        }
    }

    public int getTargetTaskThreadPoolSize(String instanceName) {
        InstanceConfig instanceConfig = this.getInstanceConfig(instanceName);
        return instanceConfig.getTargetTaskThreadPoolSize();
    }

    public void setTargetTaskThreadPoolSize(String instanceName, int targetTaskThreadPoolSize) {
        InstanceConfig instanceConfig = this.getInstanceConfig(instanceName);
        instanceConfig.setTargetTaskThreadPoolSize(targetTaskThreadPoolSize);
        this._accessor.setProperty(this._accessor.keyBuilder().instanceConfig(instanceName), instanceConfig);
    }

    private InstanceConfig getInstanceConfig(String instanceName) {
        InstanceConfig instanceConfig = (InstanceConfig)this._accessor.getProperty(this._accessor.keyBuilder().instanceConfig(instanceName));
        if (instanceConfig == null) {
            throw new IllegalArgumentException("Failed to find InstanceConfig with provided instance name " + instanceName + "!");
        }
        return instanceConfig;
    }

    public int getGlobalTargetTaskThreadPoolSize() {
        ClusterConfig clusterConfig = this.getClusterConfig();
        return clusterConfig.getGlobalTargetTaskThreadPoolSize();
    }

    public void setGlobalTargetTaskThreadPoolSize(int globalTargetTaskThreadPoolSize) {
        ClusterConfig clusterConfig = this.getClusterConfig();
        clusterConfig.setGlobalTargetTaskThreadPoolSize(globalTargetTaskThreadPoolSize);
        this._accessor.setProperty(this._accessor.keyBuilder().clusterConfig(), clusterConfig);
    }

    private ClusterConfig getClusterConfig() {
        ClusterConfig clusterConfig = (ClusterConfig)this._accessor.getProperty(this._accessor.keyBuilder().clusterConfig());
        if (clusterConfig == null) {
            throw new IllegalStateException("Failed to find ClusterConfig for cluster " + this._clusterName + "!");
        }
        return clusterConfig;
    }

    public int getCurrentTaskThreadPoolSize(String instanceName) {
        LiveInstance liveInstance = (LiveInstance)this._accessor.getProperty(this._accessor.keyBuilder().liveInstance(instanceName));
        if (liveInstance == null) {
            throw new IllegalArgumentException("Failed to find LiveInstance with provided instance name " + instanceName + "!");
        }
        return liveInstance.getCurrentTaskThreadPoolSize();
    }

    public static enum DriverCommand {
        start,
        stop,
        delete,
        resume,
        list,
        flush,
        clean;

    }
}

