/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.rest;

import java.io.IOException;
import javax.servlet.ServletContext;
import org.apache.hadoop.hbase.rest.model.NamespacesModel;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.apache.hbase.thirdparty.javax.ws.rs.GET;
import org.apache.hbase.thirdparty.javax.ws.rs.Path;
import org.apache.hbase.thirdparty.javax.ws.rs.PathParam;
import org.apache.hbase.thirdparty.javax.ws.rs.Produces;
import org.apache.hbase.thirdparty.javax.ws.rs.core.Context;
import org.apache.hbase.thirdparty.javax.ws.rs.core.Response;
import org.apache.hbase.thirdparty.javax.ws.rs.core.UriInfo;

/**
 * Implements REST GET list of all namespaces.
 * <p>
 * <tt>/namespaces</tt>
 * <p>
 */
@InterfaceAudience.Private
public class NamespacesResource extends ResourceBase {

  private static final Logger LOG = LoggerFactory.getLogger(NamespacesResource.class);

  /**
   * Constructor
   */
  public NamespacesResource() throws IOException {
    super();
  }

  /**
   * Build a response for a list of all namespaces request.
   * @param context servlet context
   * @param uriInfo (JAX-RS context variable) request URL
   * @return a response for a version request
   */
  @GET
  @Produces({ MIMETYPE_TEXT, MIMETYPE_XML, MIMETYPE_JSON, MIMETYPE_PROTOBUF,
    MIMETYPE_PROTOBUF_IETF })
  public Response get(final @Context ServletContext context, final @Context UriInfo uriInfo) {
    if (LOG.isTraceEnabled()) {
      LOG.trace("GET " + uriInfo.getAbsolutePath());
    }
    servlet.getMetrics().incrementRequests(1);
    try {
      NamespacesModel rowModel = null;
      rowModel = new NamespacesModel(servlet.getAdmin());
      servlet.getMetrics().incrementSucessfulGetRequests(1);
      return Response.ok(rowModel).build();
    } catch (IOException e) {
      servlet.getMetrics().incrementFailedGetRequests(1);
      throw new RuntimeException("Cannot retrieve list of namespaces.");
    }
  }

  /**
   * Dispatch to NamespaceInstanceResource
   */
  @Path("{namespace}")
  public NamespacesInstanceResource getNamespaceInstanceResource(
    final @PathParam("namespace") String namespace) throws IOException {
    return new NamespacesInstanceResource(namespace);
  }
}
