/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.tools.fedbalance.procedure;

import java.io.IOException;
import java.util.Collection;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.lang3.builder.CompareToBuilder;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.tools.fedbalance.procedure.BalanceJob;
import org.apache.hadoop.tools.fedbalance.procedure.BalanceJournal;
import org.apache.hadoop.tools.fedbalance.procedure.BalanceJournalInfoHDFS;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BalanceProcedureScheduler {
    public static final Logger LOG = LoggerFactory.getLogger(BalanceProcedureScheduler.class);
    private ConcurrentHashMap<BalanceJob, BalanceJob> jobSet;
    private LinkedBlockingQueue<BalanceJob> runningQueue;
    private DelayQueue<DelayWrapper> delayQueue;
    private LinkedBlockingQueue<BalanceJob> recoverQueue;
    private Configuration conf;
    private BalanceJournal journal;
    private Thread readerThread;
    private ThreadPoolExecutor workersPool;
    private Thread roosterThread;
    private Thread recoverThread;
    private AtomicBoolean running = new AtomicBoolean(true);

    public BalanceProcedureScheduler(Configuration conf) {
        this.conf = conf;
    }

    public synchronized void init(boolean recoverJobs) throws IOException {
        this.runningQueue = new LinkedBlockingQueue();
        this.delayQueue = new DelayQueue();
        this.recoverQueue = new LinkedBlockingQueue();
        this.jobSet = new ConcurrentHashMap();
        this.roosterThread = new Rooster();
        this.roosterThread.setDaemon(true);
        this.roosterThread.start();
        this.recoverThread = new Recover();
        this.recoverThread.setDaemon(true);
        this.recoverThread.start();
        int workerNum = this.conf.getInt("hdfs.fedbalance.procedure.work.thread.num", 10);
        this.workersPool = new ThreadPoolExecutor(workerNum, workerNum * 2, 1L, TimeUnit.MILLISECONDS, new LinkedBlockingDeque<Runnable>());
        this.readerThread = new Reader();
        this.readerThread.start();
        this.journal = new BalanceJournalInfoHDFS();
        this.journal.setConf(this.conf);
        if (recoverJobs) {
            this.recoverAllJobs();
        }
    }

    public synchronized void submit(BalanceJob job) throws IOException {
        if (!this.running.get()) {
            throw new IOException("Scheduler is shutdown.");
        }
        String jobId = BalanceProcedureScheduler.allocateJobId();
        job.setId(jobId);
        job.setScheduler(this);
        this.journal.saveJob(job);
        this.jobSet.put(job, job);
        this.runningQueue.add(job);
        LOG.info("Add new job={}", (Object)job);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public BalanceJob remove(BalanceJob job) {
        BalanceJob inner = this.findJob(job);
        if (inner == null) {
            return null;
        }
        if (job.isJobDone()) {
            BalanceProcedureScheduler balanceProcedureScheduler = this;
            synchronized (balanceProcedureScheduler) {
                return this.jobSet.remove(inner);
            }
        }
        return null;
    }

    public BalanceJob findJob(BalanceJob job) {
        BalanceJob found = null;
        for (BalanceJob j : this.jobSet.keySet()) {
            if (!j.getId().equals(job.getId())) continue;
            found = j;
            break;
        }
        return found;
    }

    public Collection<BalanceJob> getAllJobs() {
        return this.jobSet.values();
    }

    public void waitUntilDone(BalanceJob job) {
        BalanceJob found = this.findJob(job);
        if (found == null || found.isJobDone()) {
            return;
        }
        while (!found.isJobDone()) {
            try {
                found.waitJobDone();
            }
            catch (InterruptedException interruptedException) {}
        }
    }

    void delay(BalanceJob job, long delayInMilliseconds) {
        this.delayQueue.add(new DelayWrapper(job, delayInMilliseconds));
        LOG.info("Need delay {}ms. Add to delayQueue. job={}", (Object)delayInMilliseconds, (Object)job);
    }

    boolean jobDone(BalanceJob job) {
        try {
            this.journal.clear(job);
            if (job.shouldRemoveAfterDone()) {
                this.jobSet.remove(job);
            }
            return true;
        }
        catch (IOException e) {
            LOG.warn("Clear journal failed, add to recoverQueue. job=" + job, (Throwable)e);
            this.recoverQueue.add(job);
            return false;
        }
    }

    boolean writeJournal(BalanceJob job) {
        try {
            this.journal.saveJob(job);
            return true;
        }
        catch (Exception e) {
            LOG.warn("Save procedure failed, add to recoverQueue. job=" + job, (Throwable)e);
            this.recoverQueue.add(job);
            return false;
        }
    }

    public boolean isRunning() {
        return this.running.get();
    }

    public synchronized void shutDown() {
        if (!this.running.get()) {
            return;
        }
        this.running.set(false);
        this.readerThread.interrupt();
        this.roosterThread.interrupt();
        this.recoverThread.interrupt();
        this.workersPool.shutdownNow();
    }

    public synchronized void shutDownAndWait(int timeout) {
        this.shutDown();
        while (this.readerThread.isAlive()) {
            try {
                this.readerThread.join();
            }
            catch (InterruptedException interruptedException) {}
        }
        while (this.roosterThread.isAlive()) {
            try {
                this.roosterThread.join();
            }
            catch (InterruptedException interruptedException) {}
        }
        while (this.recoverThread.isAlive()) {
            try {
                this.recoverThread.join();
            }
            catch (InterruptedException interruptedException) {}
        }
        while (!this.workersPool.isTerminated()) {
            try {
                this.workersPool.awaitTermination(timeout, TimeUnit.SECONDS);
            }
            catch (InterruptedException interruptedException) {}
        }
    }

    private void recoverAllJobs() throws IOException {
        BalanceJob[] jobs;
        for (BalanceJob job : jobs = this.journal.listAllJobs()) {
            this.recoverQueue.add(job);
            this.jobSet.put(job, job);
            LOG.info("Recover federation balance job {}.", (Object)job);
        }
    }

    @VisibleForTesting
    static String allocateJobId() {
        return "job-" + UUID.randomUUID();
    }

    @VisibleForTesting
    public void setJournal(BalanceJournal journal) {
        this.journal = journal;
    }

    private static class DelayWrapper
    implements Delayed {
        private BalanceJob job;
        private long time;

        DelayWrapper(BalanceJob job, long delayInMilliseconds) {
            this.job = job;
            this.time = Time.monotonicNow() + delayInMilliseconds;
        }

        BalanceJob getJob() {
            return this.job;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            long delay = this.time - Time.monotonicNow();
            if (delay < 0L) {
                delay = 0L;
            }
            return unit.convert(delay, TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed o) {
            DelayWrapper dw = (DelayWrapper)o;
            return new CompareToBuilder().append(this.time, dw.time).append((Object)this.job, (Object)dw.job).toComparison();
        }

        public int hashCode() {
            return new HashCodeBuilder(17, 37).append(this.time).append((Object)this.job).toHashCode();
        }

        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (obj == this) {
                return true;
            }
            if (obj.getClass() != this.getClass()) {
                return false;
            }
            DelayWrapper dw = (DelayWrapper)obj;
            return new EqualsBuilder().appendSuper(super.equals(obj)).append(this.time, dw.time).append((Object)this.job, (Object)dw.job).build();
        }
    }

    class Recover
    extends Thread {
        Recover() {
        }

        @Override
        public void run() {
            while (BalanceProcedureScheduler.this.running.get()) {
                BalanceJob job = null;
                try {
                    job = (BalanceJob)BalanceProcedureScheduler.this.recoverQueue.poll(500L, TimeUnit.MILLISECONDS);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                if (job == null) continue;
                try {
                    BalanceProcedureScheduler.this.journal.recoverJob(job);
                    job.setScheduler(BalanceProcedureScheduler.this);
                    BalanceProcedureScheduler.this.runningQueue.add(job);
                    LOG.info("Recover success, add to runningQueue. job={}", (Object)job);
                }
                catch (IOException e) {
                    LOG.warn("Recover failed, re-add to recoverQueue. job=" + job, (Throwable)e);
                    BalanceProcedureScheduler.this.recoverQueue.add(job);
                }
            }
        }
    }

    class Reader
    extends Thread {
        Reader() {
        }

        @Override
        public void run() {
            while (BalanceProcedureScheduler.this.running.get()) {
                try {
                    BalanceJob job = (BalanceJob)BalanceProcedureScheduler.this.runningQueue.poll(500L, TimeUnit.MILLISECONDS);
                    if (job == null) continue;
                    BalanceProcedureScheduler.this.workersPool.submit(() -> {
                        LOG.info("Start job. job_msg={}", (Object)job.getDetailMessage());
                        job.execute();
                        if (!BalanceProcedureScheduler.this.running.get()) {
                            return;
                        }
                        if (job.isJobDone()) {
                            if (job.getError() == null) {
                                LOG.info("Job done. job={}", (Object)job);
                            } else {
                                LOG.warn("Job failed. job=" + job, (Throwable)job.getError());
                            }
                        }
                    });
                }
                catch (InterruptedException interruptedException) {}
            }
        }
    }

    class Rooster
    extends Thread {
        Rooster() {
        }

        @Override
        public void run() {
            while (BalanceProcedureScheduler.this.running.get()) {
                try {
                    DelayWrapper dJob = (DelayWrapper)BalanceProcedureScheduler.this.delayQueue.take();
                    BalanceProcedureScheduler.this.runningQueue.add(dJob.getJob());
                    LOG.info("Wake up job={}", (Object)dJob.getJob());
                }
                catch (InterruptedException interruptedException) {}
            }
        }
    }
}

