/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.util.functional;

import java.io.Closeable;
import java.io.IOException;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.statistics.IOStatisticAssertions;
import org.apache.hadoop.fs.statistics.IOStatistics;
import org.apache.hadoop.fs.statistics.IOStatisticsSnapshot;
import org.apache.hadoop.fs.statistics.IOStatisticsSource;
import org.apache.hadoop.test.AbstractHadoopTestBase;
import org.apache.hadoop.test.LambdaTestUtils;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.functional.ConsumerRaisingIOE;
import org.apache.hadoop.util.functional.RemoteIterators;
import org.assertj.core.api.AbstractIntegerAssert;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.AbstractStringAssert;
import org.assertj.core.api.Assertions;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestRemoteIterators
extends AbstractHadoopTestBase {
    private static final Logger LOG = LoggerFactory.getLogger(TestRemoteIterators.class);
    private static final String[] DATA = new String[]{"a", "b", "c"};
    private int counter;

    @Test
    public void testIterateArray() throws Throwable {
        this.verifyInvoked(RemoteIterators.remoteIteratorFromArray((Object[])DATA), DATA.length, s -> LOG.info(s));
    }

    @Test
    public void testIterateArrayMapped() throws Throwable {
        this.verifyInvoked(RemoteIterators.mappingRemoteIterator((RemoteIterator)RemoteIterators.remoteIteratorFromArray((Object[])DATA), d -> {
            this.counter += d.length();
            return d;
        }), DATA.length, this::log);
        this.assertCounterValue(3);
    }

    public void log(Object o) {
        LOG.info("{}", o);
    }

    @Test
    public void testSingleton() throws Throwable {
        StringBuffer result = new StringBuffer();
        String name = "singleton";
        RemoteIterator it = RemoteIterators.remoteIteratorFromSingleton((Object)name);
        this.assertStringValueContains(it, "SingletonIterator");
        this.assertStringValueContains(it, name);
        this.verifyInvoked(it, 1, s -> result.append((String)s));
        Assertions.assertThat((String)result.toString()).isEqualTo((Object)name);
    }

    @Test
    public void testSingletonNotClosed() throws Throwable {
        CloseCounter closeCounter = new CloseCounter();
        RemoteIterator it = RemoteIterators.remoteIteratorFromSingleton((Object)closeCounter);
        this.verifyInvoked(it, 1, this::log);
        this.close(it);
        closeCounter.assertCloseCount(0);
    }

    @Test
    public void testNullSingleton() throws Throwable {
        this.verifyInvoked(RemoteIterators.remoteIteratorFromSingleton(null), 0, this::log);
    }

    @Test
    public void testSingletonStats() throws Throwable {
        IOStatsInstance singleton = new IOStatsInstance();
        RemoteIterator it = RemoteIterators.remoteIteratorFromSingleton((Object)singleton);
        IOStatisticAssertions.extractStatistics(it);
    }

    @Test
    public void testMappedSingletonStats() throws Throwable {
        IOStatsInstance singleton = new IOStatsInstance();
        RemoteIterator it = RemoteIterators.mappingRemoteIterator((RemoteIterator)RemoteIterators.remoteIteratorFromSingleton((Object)singleton), Object::toString);
        this.verifyInvoked(it, 1, this::log);
        IOStatisticAssertions.extractStatistics(it);
    }

    @Test
    public void testClosePassthrough() throws Throwable {
        CountdownRemoteIterator countdown = new CountdownRemoteIterator(0);
        RemoteIterator it = RemoteIterators.mappingRemoteIterator((RemoteIterator)countdown, i -> i);
        this.verifyInvoked(it, 0, this::log);
        countdown.assertCloseCount(1);
        IOStatisticAssertions.extractStatistics((Object)countdown);
        ((Closeable)it).close();
        countdown.assertCloseCount(1);
    }

    @Test
    public void testMapping() throws Throwable {
        CountdownRemoteIterator countdown = new CountdownRemoteIterator(100);
        RemoteIterator it = RemoteIterators.mappingRemoteIterator((RemoteIterator)countdown, i -> i);
        this.verifyInvoked(it, 100, c -> ++this.counter);
        this.assertCounterValue(100);
        IOStatisticAssertions.extractStatistics(it);
        this.assertStringValueContains(it, "CountdownRemoteIterator");
        this.close(it);
        countdown.assertCloseCount(1);
    }

    @Test
    public void testFiltering() throws Throwable {
        CountdownRemoteIterator countdown = new CountdownRemoteIterator(100);
        RemoteIterator it = RemoteIterators.filteringRemoteIterator((RemoteIterator)countdown, i -> i % 2 == 0);
        this.verifyInvoked(it, 50, c -> ++this.counter);
        this.assertCounterValue(50);
        IOStatisticAssertions.extractStatistics(it);
        this.close(it);
        countdown.assertCloseCount(1);
    }

    @Test
    public void testFilterNoneAccepted() throws Throwable {
        RemoteIterator it = RemoteIterators.filteringRemoteIterator((RemoteIterator)new CountdownRemoteIterator(100), i -> false);
        this.verifyInvoked(it, 0, c -> ++this.counter);
        this.assertCounterValue(0);
        IOStatisticAssertions.extractStatistics(it);
    }

    @Test
    public void testFilterAllAccepted() throws Throwable {
        RemoteIterator it = RemoteIterators.filteringRemoteIterator((RemoteIterator)new CountdownRemoteIterator(100), i -> true);
        this.verifyInvoked(it, 100, c -> ++this.counter);
        this.assertStringValueContains(it, "CountdownRemoteIterator");
    }

    @Test
    public void testJavaIteratorSupport() throws Throwable {
        CountdownIterator countdownIterator = new CountdownIterator(100);
        RemoteIterator it = RemoteIterators.remoteIteratorFromIterator((Iterator)countdownIterator);
        this.verifyInvoked(it, 100, c -> ++this.counter);
        this.assertStringValueContains(it, "CountdownIterator");
        IOStatisticAssertions.extractStatistics(it);
        this.close(it);
        countdownIterator.assertCloseCount(1);
    }

    @Test
    public void testJavaIterableSupport() throws Throwable {
        CountdownIterable countdown = new CountdownIterable(100);
        RemoteIterator it = RemoteIterators.remoteIteratorFromIterable((Iterable)countdown);
        this.verifyInvoked(it, 100, c -> ++this.counter);
        this.assertStringValueContains(it, "CountdownIterator");
        IOStatisticAssertions.extractStatistics(it);
        this.close(it);
        countdown.assertCloseCount(0);
        this.verifyInvoked(RemoteIterators.remoteIteratorFromIterable((Iterable)countdown), 100, c -> ++this.counter);
    }

    @Test
    public void testJavaIterableClose() throws Throwable {
        CountdownIterable countdown = new CountdownIterable(100);
        RemoteIterator it = RemoteIterators.closingRemoteIterator((RemoteIterator)RemoteIterators.remoteIteratorFromIterable((Iterable)countdown), (Closeable)countdown);
        this.verifyInvoked(it, 100, c -> ++this.counter);
        this.assertStringValueContains(it, "CountdownIterator");
        IOStatisticAssertions.extractStatistics(it);
        countdown.assertCloseCount(1);
        this.close(it);
        countdown.assertCloseCount(1);
        LambdaTestUtils.intercept(IllegalStateException.class, () -> RemoteIterators.remoteIteratorFromIterable((Iterable)countdown));
    }

    @Test
    public void testJavaIterableCloseInNextLoop() throws Throwable {
        CountdownIterable countdown = new CountdownIterable(100);
        RemoteIterator it = RemoteIterators.closingRemoteIterator((RemoteIterator)RemoteIterators.remoteIteratorFromIterable((Iterable)countdown), (Closeable)countdown);
        try {
            while (true) {
                it.next();
            }
        }
        catch (NoSuchElementException noSuchElementException) {
            countdown.assertCloseCount(1);
            return;
        }
    }

    @Test
    public void testHaltableIterator() throws Throwable {
        int limit = 4;
        AtomicInteger count = new AtomicInteger(4);
        RemoteIterator it = RemoteIterators.haltableRemoteIterator((RemoteIterator)RemoteIterators.rangeExcludingIterator((long)0L, (long)10L), () -> count.get() > 0);
        this.verifyInvoked(it, 4, v -> count.decrementAndGet());
    }

    @Test
    public void testHaltableIteratorNoHalt() throws Throwable {
        int finish = 10;
        RemoteIterator it = RemoteIterators.haltableRemoteIterator((RemoteIterator)RemoteIterators.rangeExcludingIterator((long)0L, (long)10L), () -> true);
        this.verifyInvoked(it, 10);
    }

    @Test
    public void testRangeExcludingIterator() throws Throwable {
        this.verifyInvoked(RemoteIterators.rangeExcludingIterator((long)0L, (long)0L), 0);
        this.verifyInvoked(RemoteIterators.rangeExcludingIterator((long)0L, (long)-1L), 0);
        this.verifyInvoked(RemoteIterators.rangeExcludingIterator((long)0L, (long)100L), 100);
        LambdaTestUtils.intercept(NoSuchElementException.class, () -> (Long)RemoteIterators.rangeExcludingIterator((long)0L, (long)0L).next());
    }

    protected void assertStringValueContains(Object o, String expected) {
        ((AbstractStringAssert)Assertions.assertThat((String)o.toString()).describedAs("Object string value", new Object[0])).contains(new CharSequence[]{expected});
    }

    protected void assertCounterValue(int expected) {
        ((AbstractIntegerAssert)Assertions.assertThat((int)this.counter).describedAs("Counter value", new Object[0])).isEqualTo(expected);
    }

    protected <T> void verifyInvoked(RemoteIterator<T> it, int length, ConsumerRaisingIOE<T> consumer) throws IOException {
        ((AbstractLongAssert)Assertions.assertThat((long)RemoteIterators.foreach(it, consumer)).describedAs("Scan through iterator %s", new Object[]{it})).isEqualTo((long)length);
    }

    protected <T> void verifyInvoked(RemoteIterator<T> it, int length) throws IOException {
        this.verifyInvoked(it, length, t -> {});
    }

    private <T> void close(RemoteIterator<T> it) throws IOException {
        if (it instanceof Closeable) {
            ((Closeable)it).close();
        }
    }

    private static final class CountdownIterable
    extends CloseCounter
    implements Iterable<Integer> {
        private int limit;

        private CountdownIterable(int limit) {
            this.limit = limit;
        }

        @Override
        public Iterator<Integer> iterator() {
            Preconditions.checkState((this.getCloseCount() == 0 ? 1 : 0) != 0);
            return new CountdownIterator(this.limit);
        }
    }

    private static final class CountdownIterator
    extends CloseCounter
    implements Iterator<Integer> {
        private int limit;

        private CountdownIterator(int limit) {
            this.limit = limit;
        }

        @Override
        public boolean hasNext() {
            return this.limit > 0;
        }

        @Override
        public Integer next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException("limit reached");
            }
            return this.limit--;
        }

        public String toString() {
            return "CountdownIterator{limit=" + this.limit + '}';
        }
    }

    private static final class CountdownRemoteIterator
    extends CloseCounter
    implements RemoteIterator<Integer> {
        private int limit;

        private CountdownRemoteIterator(int limit) {
            this.limit = limit;
        }

        public boolean hasNext() throws IOException {
            return this.limit > 0;
        }

        public Integer next() throws IOException {
            return this.limit--;
        }

        public String toString() {
            return "CountdownRemoteIterator{limit=" + this.limit + '}';
        }
    }

    private static class IOStatsInstance
    implements IOStatisticsSource {
        private IOStatisticsSnapshot stats = new IOStatisticsSnapshot();

        private IOStatsInstance() {
        }

        public IOStatistics getIOStatistics() {
            return this.stats;
        }
    }

    private static class CloseCounter
    extends IOStatsInstance
    implements Closeable {
        private int closeCount;

        private CloseCounter() {
        }

        @Override
        public void close() throws IOException {
            ++this.closeCount;
            LOG.info("close ${}", (Object)this.closeCount);
        }

        public int getCloseCount() {
            return this.closeCount;
        }

        public void reset() {
            this.closeCount = 0;
        }

        public void assertCloseCount(int expected) {
            ((AbstractIntegerAssert)Assertions.assertThat((int)this.closeCount).describedAs("Close count", new Object[0])).isEqualTo(expected);
        }
    }
}

