/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azurebfs.services;

import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.fs.azurebfs.AbfsConfiguration;
import org.apache.hadoop.fs.azurebfs.services.AbfsOperationMetrics;
import org.apache.hadoop.fs.azurebfs.services.TimerFunctionality;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class AbfsClientThrottlingAnalyzer {
    private static final Logger LOG = LoggerFactory.getLogger(AbfsClientThrottlingAnalyzer.class);
    private static final int MIN_ANALYSIS_PERIOD_MS = 1000;
    private static final int MAX_ANALYSIS_PERIOD_MS = 30000;
    private static final double MIN_ACCEPTABLE_ERROR_PERCENTAGE = 0.1;
    private static final double MAX_EQUILIBRIUM_ERROR_PERCENTAGE = 1.0;
    private static final double RAPID_SLEEP_DECREASE_FACTOR = 0.75;
    private static final double RAPID_SLEEP_DECREASE_TRANSITION_PERIOD_MS = 150000.0;
    private static final double SLEEP_DECREASE_FACTOR = 0.975;
    private static final double SLEEP_INCREASE_FACTOR = 1.05;
    private int analysisPeriodMs;
    private volatile int sleepDuration = 0;
    private long consecutiveNoErrorCount = 0L;
    private String name = null;
    private Timer timer = null;
    private AtomicReference<AbfsOperationMetrics> blobMetrics = null;
    private AtomicLong lastExecutionTime = null;
    private final AtomicBoolean isOperationOnAccountIdle = new AtomicBoolean(false);
    private AbfsConfiguration abfsConfiguration = null;
    private boolean accountLevelThrottlingEnabled = true;

    private AbfsClientThrottlingAnalyzer() {
    }

    AbfsClientThrottlingAnalyzer(String name, AbfsConfiguration abfsConfiguration) throws IllegalArgumentException {
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((CharSequence)name), (Object)"The argument 'name' cannot be null or empty.");
        int period = abfsConfiguration.getAnalysisPeriod();
        Preconditions.checkArgument((period >= 1000 && period <= 30000 ? 1 : 0) != 0, (Object)"The argument 'period' must be between 1000 and 30000.");
        this.name = name;
        this.abfsConfiguration = abfsConfiguration;
        this.accountLevelThrottlingEnabled = abfsConfiguration.accountThrottlingEnabled();
        this.analysisPeriodMs = abfsConfiguration.getAnalysisPeriod();
        this.lastExecutionTime = new AtomicLong(Time.now());
        this.blobMetrics = new AtomicReference<AbfsOperationMetrics>(new AbfsOperationMetrics(System.currentTimeMillis()));
        this.timer = new Timer(String.format("abfs-timer-client-throttling-analyzer-%s", name), true);
        this.timer.schedule((TimerTask)new TimerTaskImpl(), this.analysisPeriodMs, (long)this.analysisPeriodMs);
    }

    private void resumeTimer() {
        this.blobMetrics = new AtomicReference<AbfsOperationMetrics>(new AbfsOperationMetrics(System.currentTimeMillis()));
        this.timer.schedule((TimerTask)new TimerTaskImpl(), this.analysisPeriodMs, (long)this.analysisPeriodMs);
        this.isOperationOnAccountIdle.set(false);
    }

    private synchronized boolean timerOrchestrator(TimerFunctionality timerFunctionality, TimerTask timerTask) {
        switch (timerFunctionality) {
            case RESUME: {
                if (!this.isOperationOnAccountIdle.get()) break;
                this.resumeTimer();
                break;
            }
            case SUSPEND: {
                if (!this.accountLevelThrottlingEnabled || System.currentTimeMillis() - this.lastExecutionTime.get() < (long)this.getOperationIdleTimeout()) break;
                this.isOperationOnAccountIdle.set(true);
                timerTask.cancel();
                this.timer.purge();
                return true;
            }
        }
        return false;
    }

    public void addBytesTransferred(long count, boolean isFailedOperation) {
        AbfsOperationMetrics metrics = this.blobMetrics.get();
        if (isFailedOperation) {
            metrics.addBytesFailed(count);
            metrics.incrementOperationsFailed();
        } else {
            metrics.addBytesSuccessful(count);
            metrics.incrementOperationsSuccessful();
        }
        this.blobMetrics.set(metrics);
    }

    public boolean suspendIfNecessary() {
        this.lastExecutionTime.set(Time.now());
        this.timerOrchestrator(TimerFunctionality.RESUME, null);
        int duration = this.sleepDuration;
        if (duration > 0) {
            try {
                Thread.sleep(duration);
                return true;
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
            }
        }
        return false;
    }

    @VisibleForTesting
    int getSleepDuration() {
        return this.sleepDuration;
    }

    int getOperationIdleTimeout() {
        return this.abfsConfiguration.getAccountOperationIdleTimeout();
    }

    AtomicBoolean getIsOperationOnAccountIdle() {
        return this.isOperationOnAccountIdle;
    }

    private int analyzeMetricsAndUpdateSleepDuration(AbfsOperationMetrics metrics, int sleepDuration) {
        double newSleepDuration;
        double percentageConversionFactor = 100.0;
        double bytesFailed = metrics.getBytesFailed().get();
        double bytesSuccessful = metrics.getBytesSuccessful().get();
        double operationsFailed = metrics.getOperationsFailed().get();
        double operationsSuccessful = metrics.getOperationsSuccessful().get();
        double errorPercentage = bytesFailed <= 0.0 ? 0.0 : 100.0 * bytesFailed / (bytesFailed + bytesSuccessful);
        long periodMs = metrics.getEndTime() - metrics.getStartTime();
        if (errorPercentage < 0.1) {
            ++this.consecutiveNoErrorCount;
            double reductionFactor = (double)(this.consecutiveNoErrorCount * (long)this.analysisPeriodMs) >= 150000.0 ? 0.75 : 0.975;
            newSleepDuration = (double)sleepDuration * reductionFactor;
        } else if (errorPercentage < 1.0) {
            newSleepDuration = sleepDuration;
        } else {
            this.consecutiveNoErrorCount = 0L;
            double additionalDelayNeeded = 5 * this.analysisPeriodMs;
            if (bytesSuccessful > 0.0) {
                additionalDelayNeeded = (bytesSuccessful + bytesFailed) * (double)periodMs / bytesSuccessful - (double)periodMs;
            }
            newSleepDuration = additionalDelayNeeded / (operationsFailed + operationsSuccessful);
            double maxSleepDuration = this.analysisPeriodMs;
            double minSleepDuration = (double)sleepDuration * 1.05;
            newSleepDuration = Math.max(newSleepDuration, minSleepDuration) + 1.0;
            newSleepDuration = Math.min(newSleepDuration, maxSleepDuration);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug(String.format("%5.5s, %10d, %10d, %10d, %10d, %6.2f, %5d, %5d, %5d", this.name, (int)bytesFailed, (int)bytesSuccessful, (int)operationsFailed, (int)operationsSuccessful, errorPercentage, periodMs, sleepDuration, (int)newSleepDuration));
        }
        return (int)newSleepDuration;
    }

    class TimerTaskImpl
    extends TimerTask {
        private AtomicInteger doingWork = new AtomicInteger(0);

        TimerTaskImpl() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            boolean doWork = false;
            try {
                doWork = this.doingWork.compareAndSet(0, 1);
                if (!doWork) {
                    return;
                }
                long now = System.currentTimeMillis();
                if (AbfsClientThrottlingAnalyzer.this.timerOrchestrator(TimerFunctionality.SUSPEND, this)) {
                    return;
                }
                if (now - ((AbfsOperationMetrics)AbfsClientThrottlingAnalyzer.this.blobMetrics.get()).getStartTime() >= (long)AbfsClientThrottlingAnalyzer.this.analysisPeriodMs) {
                    AbfsOperationMetrics oldMetrics = AbfsClientThrottlingAnalyzer.this.blobMetrics.getAndSet(new AbfsOperationMetrics(now));
                    oldMetrics.setEndTime(now);
                    AbfsClientThrottlingAnalyzer.this.sleepDuration = AbfsClientThrottlingAnalyzer.this.analyzeMetricsAndUpdateSleepDuration(oldMetrics, AbfsClientThrottlingAnalyzer.this.sleepDuration);
                }
            }
            finally {
                if (doWork) {
                    this.doingWork.set(0);
                }
            }
        }
    }
}

