/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.tserver.log;

import com.google.common.annotations.VisibleForTesting;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ThreadPoolExecutor;
import org.apache.accumulo.core.conf.AccumuloConfiguration;
import org.apache.accumulo.core.conf.ConfigurationCopy;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.crypto.CryptoEnvironmentImpl;
import org.apache.accumulo.core.data.Key;
import org.apache.accumulo.core.data.Mutation;
import org.apache.accumulo.core.file.FileOperations;
import org.apache.accumulo.core.file.FileSKVWriter;
import org.apache.accumulo.core.manager.thrift.RecoveryStatus;
import org.apache.accumulo.core.metadata.TabletFile;
import org.apache.accumulo.core.metadata.UnreferencedTabletFile;
import org.apache.accumulo.core.spi.crypto.CryptoEnvironment;
import org.apache.accumulo.core.spi.crypto.CryptoService;
import org.apache.accumulo.core.util.Pair;
import org.apache.accumulo.core.util.threads.ThreadPools;
import org.apache.accumulo.server.ServerContext;
import org.apache.accumulo.server.fs.VolumeManager;
import org.apache.accumulo.server.log.SortedLogState;
import org.apache.accumulo.server.zookeeper.DistributedWorkQueue;
import org.apache.accumulo.tserver.log.DfsLogger;
import org.apache.accumulo.tserver.logger.LogFileKey;
import org.apache.accumulo.tserver.logger.LogFileValue;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LogSorter {
    private static final Logger log = LoggerFactory.getLogger(LogSorter.class);
    AccumuloConfiguration sortedLogConf;
    private final Map<String, LogProcessor> currentWork = Collections.synchronizedMap(new HashMap());
    ThreadPoolExecutor threadPool;
    private final ServerContext context;
    private final double walBlockSize;
    private final CryptoService cryptoService;

    public LogSorter(ServerContext context, AccumuloConfiguration conf) {
        this.context = context;
        this.sortedLogConf = this.extractSortedLogConfig(conf);
        int threadPoolSize = conf.getCount(Property.TSERV_WAL_SORT_MAX_CONCURRENT);
        this.threadPool = ThreadPools.getServerThreadPools().createFixedThreadPool(threadPoolSize, this.getClass().getName(), true);
        this.walBlockSize = DfsLogger.getWalBlockSize(conf);
        CryptoEnvironmentImpl env = new CryptoEnvironmentImpl(CryptoEnvironment.Scope.RECOVERY);
        this.cryptoService = context.getCryptoFactory().getService((CryptoEnvironment)env, conf.getAllCryptoProperties());
    }

    private AccumuloConfiguration extractSortedLogConfig(AccumuloConfiguration conf) {
        String tablePrefix = "table.file.";
        Map props = conf.getAllPropertiesWithPrefixStripped(Property.TSERV_WAL_SORT_FILE_PREFIX);
        ConfigurationCopy copy = new ConfigurationCopy((Iterable)conf);
        props.forEach((prop, val) -> {
            String tableProp = "table.file." + prop;
            if (!Property.isValidProperty((String)tableProp, (String)val) || !Property.isValidTablePropertyKey((String)tableProp)) {
                throw new IllegalArgumentException("Invalid sort file property " + prop + "=" + val);
            }
            log.debug("Using property for writing sorted files: {}={}", (Object)tableProp, val);
            copy.set(tableProp, val);
        });
        return copy;
    }

    @VisibleForTesting
    void writeBuffer(String destPath, List<Pair<LogFileKey, LogFileValue>> buffer, int part) throws IOException {
        String filename = String.format("part-r-%05d.rf", part);
        Path path = new Path(destPath, filename);
        FileSystem fs = this.context.getVolumeManager().getFileSystemByPath(path);
        Path fullPath = fs.makeQualified(path);
        TreeMap<Key, List<Mutation>> keyListMap = new TreeMap<Key, List<Mutation>>();
        for (Pair<LogFileKey, LogFileValue> pair : buffer) {
            LogFileKey logFileKey = (LogFileKey)pair.getFirst();
            LogFileValue logFileValue = (LogFileValue)pair.getSecond();
            Key k = logFileKey.toKey();
            List<Mutation> list = keyListMap.putIfAbsent(k, logFileValue.mutations);
            if (list == null) continue;
            ArrayList<Mutation> muts = new ArrayList<Mutation>(list);
            muts.addAll(logFileValue.mutations);
            keyListMap.put(logFileKey.toKey(), muts);
        }
        try (FileSKVWriter writer = FileOperations.getInstance().newWriterBuilder().forFile((TabletFile)UnreferencedTabletFile.of((FileSystem)fs, (Path)fullPath), fs, fs.getConf(), this.cryptoService).withTableConfiguration(this.sortedLogConf).build();){
            writer.startDefaultLocalityGroup();
            for (Map.Entry entry : keyListMap.entrySet()) {
                LogFileValue val = new LogFileValue();
                val.mutations = (List)entry.getValue();
                writer.append((Key)entry.getKey(), val.toValue());
            }
        }
    }

    public void startWatchingForRecoveryLogs(ThreadPoolExecutor distWorkQThreadPool) throws KeeperException, InterruptedException {
        this.threadPool = distWorkQThreadPool;
        new DistributedWorkQueue(this.context.getZooKeeperRoot() + "/recovery", this.sortedLogConf, this.context).startProcessing((DistributedWorkQueue.Processor)new LogProcessor(), this.threadPool);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<RecoveryStatus> getLogSorts() {
        ArrayList<RecoveryStatus> result = new ArrayList<RecoveryStatus>();
        Map<String, LogProcessor> map = this.currentWork;
        synchronized (map) {
            for (Map.Entry<String, LogProcessor> entries : this.currentWork.entrySet()) {
                RecoveryStatus status = new RecoveryStatus();
                status.name = entries.getKey();
                try {
                    double progress = (double)entries.getValue().getBytesCopied() / this.walBlockSize;
                    status.progress = Math.min(progress, 99.9);
                }
                catch (IOException ex) {
                    log.warn("Error getting bytes read");
                }
                status.runtime = (int)entries.getValue().getSortTime();
                result.add(status);
            }
            return result;
        }
    }

    class LogProcessor
    implements DistributedWorkQueue.Processor {
        private FSDataInputStream input;
        private DataInputStream decryptingInput;
        private long bytesCopied = -1L;
        private long sortStart = 0L;
        private long sortStop = -1L;

        LogProcessor() {
        }

        public DistributedWorkQueue.Processor newProcessor() {
            return new LogProcessor();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void process(String child, byte[] data) {
            String work = new String(data);
            String[] parts = work.split("\\|");
            String src = parts[0];
            String dest = parts[1];
            String sortId = new Path(src).getName();
            log.debug("Sorting {} to {} using sortId {}", new Object[]{src, dest, sortId});
            Object object = LogSorter.this.currentWork;
            synchronized (object) {
                if (LogSorter.this.currentWork.containsKey(sortId)) {
                    return;
                }
                LogSorter.this.currentWork.put(sortId, this);
            }
            object = this;
            synchronized (object) {
                this.sortStart = System.currentTimeMillis();
            }
            VolumeManager fs = LogSorter.this.context.getVolumeManager();
            String formerThreadName = Thread.currentThread().getName();
            try {
                this.sort(fs, sortId, new Path(src), dest);
            }
            catch (Exception t) {
                try {
                    fs.mkdirs(new Path(dest));
                    fs.create(SortedLogState.getFailedMarkerPath((String)dest)).close();
                }
                catch (IOException e) {
                    log.error("Error creating failed flag file " + sortId, (Throwable)e);
                }
                log.error("Caught exception", (Throwable)t);
            }
            finally {
                Thread.currentThread().setName(formerThreadName);
                try {
                    this.close();
                }
                catch (Exception e) {
                    log.error("Error during cleanup sort/copy " + sortId, (Throwable)e);
                }
                LogProcessor logProcessor = this;
                synchronized (logProcessor) {
                    this.sortStop = System.currentTimeMillis();
                }
                LogSorter.this.currentWork.remove(sortId);
            }
        }

        public void sort(VolumeManager fs, String name, Path srcPath, String destPath) throws IOException {
            int part = 0;
            if (fs.exists(SortedLogState.getFinishedMarkerPath((String)destPath))) {
                log.debug("Sorting already finished at {}", (Object)destPath);
                return;
            }
            log.info("Copying {} to {}", (Object)srcPath, (Object)destPath);
            fs.deleteRecursively(new Path(destPath));
            this.input = fs.open(srcPath);
            try {
                this.input.setDropBehind(Boolean.TRUE);
            }
            catch (UnsupportedOperationException e) {
                log.debug("setDropBehind reads not enabled for wal file: {}", (Object)this.input);
            }
            catch (IOException e) {
                log.debug("IOException setting drop behind for file: {}, msg: {}", (Object)this.input, (Object)e.getMessage());
            }
            try {
                this.decryptingInput = DfsLogger.getDecryptingStream(this.input, LogSorter.this.cryptoService);
            }
            catch (DfsLogger.LogHeaderIncompleteException e) {
                log.warn("Could not read header from write-ahead log {}. Not sorting.", (Object)srcPath);
                fs.mkdirs(new Path(destPath));
                LogSorter.this.writeBuffer(destPath, Collections.emptyList(), part++);
                fs.create(SortedLogState.getFinishedMarkerPath((String)destPath)).close();
                return;
            }
            long bufferSize = LogSorter.this.sortedLogConf.getAsBytes(Property.TSERV_WAL_SORT_BUFFER_SIZE);
            Thread.currentThread().setName("Sorting " + name + " for recovery");
            while (true) {
                ArrayList<Pair<LogFileKey, LogFileValue>> buffer = new ArrayList<Pair<LogFileKey, LogFileValue>>();
                try {
                    long start = this.input.getPos();
                    while (this.input.getPos() - start < bufferSize) {
                        LogFileKey key = new LogFileKey();
                        LogFileValue value = new LogFileValue();
                        key.readFields(this.decryptingInput);
                        value.readFields(this.decryptingInput);
                        buffer.add((Pair<LogFileKey, LogFileValue>)new Pair((Object)key, (Object)value));
                    }
                    LogSorter.this.writeBuffer(destPath, buffer, part++);
                    buffer.clear();
                }
                catch (EOFException ex) {
                    LogSorter.this.writeBuffer(destPath, buffer, part++);
                    fs.create(new Path(destPath, "finished")).close();
                    log.info("Finished log sort {} {} bytes {} parts in {}ms", new Object[]{name, this.getBytesCopied(), part, this.getSortTime()});
                    return;
                }
            }
        }

        synchronized void close() throws IOException {
            if (this.input != null) {
                this.bytesCopied = this.input.getPos();
                this.input.close();
                if (this.decryptingInput != null) {
                    this.decryptingInput.close();
                }
                this.input = null;
            }
        }

        public synchronized long getSortTime() {
            if (this.sortStart > 0L) {
                if (this.sortStop > 0L) {
                    return this.sortStop - this.sortStart;
                }
                return System.currentTimeMillis() - this.sortStart;
            }
            return 0L;
        }

        synchronized long getBytesCopied() throws IOException {
            return this.input == null ? this.bytesCopied : this.input.getPos();
        }
    }
}

