/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.server.fs;

import com.github.benmanes.caffeine.cache.Cache;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.accumulo.core.client.SampleNotPresentException;
import org.apache.accumulo.core.conf.AccumuloConfiguration;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.data.Key;
import org.apache.accumulo.core.data.Value;
import org.apache.accumulo.core.dataImpl.KeyExtent;
import org.apache.accumulo.core.file.FileOperations;
import org.apache.accumulo.core.file.FileSKVIterator;
import org.apache.accumulo.core.file.blockfile.impl.CacheProvider;
import org.apache.accumulo.core.iterators.IteratorEnvironment;
import org.apache.accumulo.core.iterators.SortedKeyValueIterator;
import org.apache.accumulo.core.iteratorsImpl.system.InterruptibleIterator;
import org.apache.accumulo.core.iteratorsImpl.system.SourceSwitchingIterator;
import org.apache.accumulo.core.metadata.StoredTabletFile;
import org.apache.accumulo.core.metadata.TabletFile;
import org.apache.accumulo.core.metadata.schema.DataFileValue;
import org.apache.accumulo.core.sample.impl.SamplerConfigurationImpl;
import org.apache.accumulo.core.util.threads.ThreadPools;
import org.apache.accumulo.server.ServerContext;
import org.apache.accumulo.server.conf.TableConfiguration;
import org.apache.accumulo.server.fs.TooManyFilesException;
import org.apache.accumulo.server.problems.ProblemReport;
import org.apache.accumulo.server.problems.ProblemReportingIterator;
import org.apache.accumulo.server.problems.ProblemReports;
import org.apache.accumulo.server.problems.ProblemType;
import org.apache.hadoop.fs.FileSystem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileManager {
    private static final Logger log = LoggerFactory.getLogger(FileManager.class);
    private int maxOpen;
    private Map<StoredTabletFile, List<OpenReader>> openFiles;
    private HashMap<FileSKVIterator, StoredTabletFile> reservedReaders;
    private Semaphore filePermits;
    private Cache<String, Long> fileLenCache;
    private long maxIdleTime;
    private long slowFilePermitMillis;
    private final ServerContext context;

    public FileManager(ServerContext context, int maxOpen, Cache<String, Long> fileLenCache) {
        if (maxOpen <= 0) {
            throw new IllegalArgumentException("maxOpen <= 0");
        }
        this.context = context;
        this.fileLenCache = fileLenCache;
        this.filePermits = new Semaphore(maxOpen, true);
        this.maxOpen = maxOpen;
        this.openFiles = new HashMap<StoredTabletFile, List<OpenReader>>();
        this.reservedReaders = new HashMap();
        this.maxIdleTime = this.context.getConfiguration().getTimeInMillis(Property.TSERV_MAX_IDLE);
        ThreadPools.watchCriticalScheduledTask(this.context.getScheduledExecutor().scheduleWithFixedDelay(new IdleFileCloser(), this.maxIdleTime, this.maxIdleTime / 2L, TimeUnit.MILLISECONDS));
        this.slowFilePermitMillis = this.context.getConfiguration().getTimeInMillis(Property.TSERV_SLOW_FILEPERMIT_MILLIS);
    }

    private static int countReaders(Map<StoredTabletFile, List<OpenReader>> files) {
        int count = 0;
        for (List<OpenReader> list : files.values()) {
            count += list.size();
        }
        return count;
    }

    private List<FileSKVIterator> takeLRUOpenFiles(int numToTake) {
        ArrayList openReaders = new ArrayList();
        for (Map.Entry<StoredTabletFile, List<OpenReader>> entry : this.openFiles.entrySet()) {
            openReaders.addAll(entry.getValue());
        }
        Collections.sort(openReaders);
        ArrayList<FileSKVIterator> ret = new ArrayList<FileSKVIterator>();
        for (int i = 0; i < numToTake && i < openReaders.size(); ++i) {
            OpenReader or = (OpenReader)openReaders.get(i);
            List<OpenReader> ofl = this.openFiles.get(or.file);
            if (!ofl.remove(or)) {
                throw new IllegalStateException("Failed to remove open reader that should have been there");
            }
            if (ofl.isEmpty()) {
                this.openFiles.remove(or.file);
            }
            ret.add(or.reader);
        }
        return ret;
    }

    private void closeReaders(Collection<FileSKVIterator> filesToClose) {
        for (FileSKVIterator reader : filesToClose) {
            try {
                reader.close();
            }
            catch (Exception e) {
                log.error("Failed to close file {}", (Object)e.getMessage(), (Object)e);
            }
        }
    }

    private List<StoredTabletFile> takeOpenFiles(Collection<StoredTabletFile> files, Map<FileSKVIterator, StoredTabletFile> readersReserved) {
        List<StoredTabletFile> filesToOpen = Collections.emptyList();
        for (StoredTabletFile file : files) {
            List<OpenReader> ofl = this.openFiles.get(file);
            if (ofl != null && !ofl.isEmpty()) {
                OpenReader openReader = ofl.remove(ofl.size() - 1);
                readersReserved.put(openReader.reader, file);
                if (!ofl.isEmpty()) continue;
                this.openFiles.remove(file);
                continue;
            }
            if (filesToOpen.isEmpty()) {
                filesToOpen = new ArrayList<StoredTabletFile>(files.size());
            }
            filesToOpen.add(file);
        }
        return filesToOpen;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Map<FileSKVIterator, StoredTabletFile> reserveReaders(KeyExtent tablet, Collection<StoredTabletFile> files, boolean continueOnFailure, CacheProvider cacheProvider) throws IOException {
        if (!tablet.isMeta() && files.size() >= this.maxOpen) {
            throw new IllegalArgumentException("requested files exceeds max open");
        }
        if (files.isEmpty()) {
            return Collections.emptyMap();
        }
        List<StoredTabletFile> filesToOpen = null;
        List<FileSKVIterator> filesToClose = Collections.emptyList();
        HashMap<FileSKVIterator, StoredTabletFile> readersReserved = new HashMap<FileSKVIterator, StoredTabletFile>();
        if (!tablet.isMeta()) {
            long start = System.currentTimeMillis();
            this.filePermits.acquireUninterruptibly(files.size());
            long waitTime = System.currentTimeMillis() - start;
            if (waitTime >= this.slowFilePermitMillis) {
                log.warn("Slow file permits request: {} ms, files requested: {}, max open files: {}, tablet: {}", new Object[]{waitTime, files.size(), this.maxOpen, tablet});
            }
        }
        Object object = this;
        synchronized (object) {
            filesToOpen = this.takeOpenFiles(files, readersReserved);
            if (!filesToOpen.isEmpty()) {
                int numOpen = FileManager.countReaders(this.openFiles);
                if (filesToOpen.size() + numOpen + this.reservedReaders.size() > this.maxOpen) {
                    filesToClose = this.takeLRUOpenFiles(filesToOpen.size() + numOpen + this.reservedReaders.size() - this.maxOpen);
                }
            }
        }
        readersReserved.forEach((k, v) -> k.setCacheProvider(cacheProvider));
        this.closeReaders(filesToClose);
        for (StoredTabletFile file : filesToOpen) {
            try {
                FileSystem ns = this.context.getVolumeManager().getFileSystemByPath(file.getPath());
                TableConfiguration tableConf = this.context.getTableConfiguration(tablet.tableId());
                FileSKVIterator reader = FileOperations.getInstance().newReaderBuilder().forFile((TabletFile)file, ns, ns.getConf(), tableConf.getCryptoService()).withTableConfiguration((AccumuloConfiguration)tableConf).withCacheProvider(cacheProvider).withFileLenCache(this.fileLenCache).build();
                readersReserved.put(reader, file);
            }
            catch (Exception e) {
                ProblemReports.getInstance(this.context).report(new ProblemReport(tablet.tableId(), ProblemType.FILE_READ, file.toString(), e));
                if (continueOnFailure) {
                    if (!tablet.isMeta()) {
                        this.filePermits.release(1);
                    }
                    log.warn("Failed to open file {} {} continuing...", new Object[]{file, e.getMessage(), e});
                    continue;
                }
                this.closeReaders(readersReserved.keySet());
                if (!tablet.isMeta()) {
                    this.filePermits.release(files.size());
                }
                log.error("Failed to open file {} {}", (Object)file, (Object)e.getMessage());
                throw new IOException("Failed to open " + file, e);
            }
        }
        object = this;
        synchronized (object) {
            this.reservedReaders.putAll(readersReserved);
        }
        return readersReserved;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void releaseReaders(KeyExtent tablet, List<FileSKVIterator> readers, boolean sawIOException) {
        FileManager fileManager = this;
        synchronized (fileManager) {
            if (!this.reservedReaders.keySet().containsAll(readers)) {
                throw new IllegalArgumentException("Asked to release readers that were never reserved ");
            }
            for (FileSKVIterator reader : readers) {
                try {
                    reader.closeDeepCopies();
                }
                catch (IOException e) {
                    log.warn("{}", (Object)e.getMessage(), (Object)e);
                    sawIOException = true;
                }
            }
            for (FileSKVIterator reader : readers) {
                StoredTabletFile file = this.reservedReaders.remove(reader);
                if (sawIOException) continue;
                this.openFiles.computeIfAbsent(file, k -> new ArrayList()).add(new OpenReader(file, reader));
            }
        }
        if (sawIOException) {
            this.closeReaders(readers);
        }
        if (!tablet.isMeta()) {
            this.filePermits.release(readers.size());
        }
    }

    public ScanFileManager newScanFileManager(KeyExtent tablet, CacheProvider cacheProvider) {
        return new ScanFileManager(tablet, cacheProvider);
    }

    public class ScanFileManager {
        private ArrayList<FileDataSource> dataSources;
        private ArrayList<FileSKVIterator> tabletReservedReaders = new ArrayList();
        private KeyExtent tablet;
        private boolean continueOnFailure;
        private CacheProvider cacheProvider;

        ScanFileManager(KeyExtent tablet, CacheProvider cacheProvider) {
            this.dataSources = new ArrayList();
            this.tablet = tablet;
            this.cacheProvider = cacheProvider;
            this.continueOnFailure = FileManager.this.context.getTableConfiguration(tablet.tableId()).getBoolean(Property.TABLE_FAILURES_IGNORE);
            if (tablet.isMeta()) {
                this.continueOnFailure = false;
            }
        }

        private Map<FileSKVIterator, StoredTabletFile> openFiles(List<StoredTabletFile> files) throws TooManyFilesException, IOException {
            if (this.tabletReservedReaders.size() + files.size() >= FileManager.this.maxOpen) {
                throw new TooManyFilesException("Request to open files would exceed max open files reservedReaders.size()=" + this.tabletReservedReaders.size() + " files.size()=" + files.size() + " maxOpen=" + FileManager.this.maxOpen + " tablet = " + this.tablet);
            }
            Map<FileSKVIterator, StoredTabletFile> newlyReservedReaders = FileManager.this.reserveReaders(this.tablet, files, this.continueOnFailure, this.cacheProvider);
            this.tabletReservedReaders.addAll(newlyReservedReaders.keySet());
            return newlyReservedReaders;
        }

        public synchronized List<InterruptibleIterator> openFiles(Map<StoredTabletFile, DataFileValue> files, boolean detachable, SamplerConfigurationImpl samplerConfig) throws IOException {
            Map<FileSKVIterator, StoredTabletFile> newlyReservedReaders = this.openFiles(new ArrayList<StoredTabletFile>(files.keySet()));
            ArrayList<InterruptibleIterator> iters = new ArrayList<InterruptibleIterator>();
            boolean someIteratorsWillWrap = files.values().stream().anyMatch(DataFileValue::willWrapIterator);
            for (Map.Entry<FileSKVIterator, StoredTabletFile> entry : newlyReservedReaders.entrySet()) {
                FileSKVIterator source = entry.getKey();
                StoredTabletFile file = entry.getValue();
                if (samplerConfig != null && (source = source.getSample(samplerConfig)) == null) {
                    throw new SampleNotPresentException();
                }
                ProblemReportingIterator iter = new ProblemReportingIterator(FileManager.this.context, this.tablet.tableId(), file.toString(), this.continueOnFailure, (SortedKeyValueIterator<Key, Value>)(detachable ? this.getSsi(file, source) : source));
                if (someIteratorsWillWrap) {
                    DataFileValue value = files.get(file);
                    iter = value.wrapFileIterator((InterruptibleIterator)iter);
                }
                iters.add(iter);
            }
            return iters;
        }

        private SourceSwitchingIterator getSsi(StoredTabletFile file, FileSKVIterator source) {
            FileDataSource fds = new FileDataSource(file, (SortedKeyValueIterator<Key, Value>)source);
            this.dataSources.add(fds);
            return new SourceSwitchingIterator((SourceSwitchingIterator.DataSource)fds);
        }

        public synchronized void detach() {
            FileManager.this.releaseReaders(this.tablet, this.tabletReservedReaders, false);
            this.tabletReservedReaders.clear();
            for (FileDataSource fds : this.dataSources) {
                fds.unsetIterator();
            }
        }

        public synchronized void reattach(SamplerConfigurationImpl samplerConfig) throws IOException {
            if (!this.tabletReservedReaders.isEmpty()) {
                throw new IllegalStateException();
            }
            List<StoredTabletFile> files = this.dataSources.stream().map(x -> x.file).collect(Collectors.toList());
            Map<FileSKVIterator, StoredTabletFile> newlyReservedReaders = this.openFiles(files);
            HashMap map = new HashMap();
            newlyReservedReaders.forEach((reader, file) -> map.computeIfAbsent(file, k -> new LinkedList()).add(reader));
            for (FileDataSource fds : this.dataSources) {
                FileSKVIterator source = (FileSKVIterator)((List)map.get(fds.file)).remove(0);
                if (samplerConfig != null && (source = source.getSample(samplerConfig)) == null) {
                    throw new SampleNotPresentException();
                }
                fds.setIterator((SortedKeyValueIterator<Key, Value>)source);
            }
        }

        public synchronized void releaseOpenFiles(boolean sawIOException) {
            FileManager.this.releaseReaders(this.tablet, this.tabletReservedReaders, sawIOException);
            this.tabletReservedReaders.clear();
            this.dataSources.clear();
        }

        public synchronized int getNumOpenFiles() {
            return this.tabletReservedReaders.size();
        }
    }

    static class FileDataSource
    implements SourceSwitchingIterator.DataSource {
        private SortedKeyValueIterator<Key, Value> iter;
        private ArrayList<FileDataSource> deepCopies;
        private boolean current = true;
        private IteratorEnvironment env;
        private StoredTabletFile file;
        private AtomicBoolean iflag;

        FileDataSource(StoredTabletFile file, SortedKeyValueIterator<Key, Value> iter) {
            this.file = file;
            this.iter = iter;
            this.deepCopies = new ArrayList();
        }

        public FileDataSource(IteratorEnvironment env, SortedKeyValueIterator<Key, Value> deepCopy, ArrayList<FileDataSource> deepCopies) {
            this.iter = deepCopy;
            this.env = env;
            this.deepCopies = deepCopies;
            deepCopies.add(this);
        }

        public boolean isCurrent() {
            return this.current;
        }

        public SourceSwitchingIterator.DataSource getNewDataSource() {
            this.current = true;
            return this;
        }

        public SourceSwitchingIterator.DataSource getDeepCopyDataSource(IteratorEnvironment env) {
            return new FileDataSource(env, (SortedKeyValueIterator<Key, Value>)this.iter.deepCopy(env), this.deepCopies);
        }

        public SortedKeyValueIterator<Key, Value> iterator() {
            return this.iter;
        }

        void unsetIterator() {
            this.current = false;
            this.iter = null;
            for (FileDataSource fds : this.deepCopies) {
                fds.current = false;
                fds.iter = null;
            }
        }

        void setIterator(SortedKeyValueIterator<Key, Value> iter) {
            this.current = false;
            this.iter = iter;
            if (this.iflag != null) {
                ((InterruptibleIterator)this.iter).setInterruptFlag(this.iflag);
            }
            for (FileDataSource fds : this.deepCopies) {
                fds.current = false;
                fds.iter = iter.deepCopy(fds.env);
            }
        }

        public void setInterruptFlag(AtomicBoolean flag) {
            this.iflag = flag;
            ((InterruptibleIterator)this.iter).setInterruptFlag(this.iflag);
        }
    }

    private class IdleFileCloser
    implements Runnable {
        private IdleFileCloser() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            long curTime = System.currentTimeMillis();
            ArrayList<FileSKVIterator> filesToClose = new ArrayList<FileSKVIterator>();
            FileManager fileManager = FileManager.this;
            synchronized (fileManager) {
                Iterator<Map.Entry<StoredTabletFile, List<OpenReader>>> iter = FileManager.this.openFiles.entrySet().iterator();
                while (iter.hasNext()) {
                    Map.Entry<StoredTabletFile, List<OpenReader>> entry = iter.next();
                    List<OpenReader> ofl = entry.getValue();
                    Iterator<OpenReader> oflIter = ofl.iterator();
                    while (oflIter.hasNext()) {
                        OpenReader openReader = oflIter.next();
                        if (curTime - openReader.releaseTime <= FileManager.this.maxIdleTime) continue;
                        filesToClose.add(openReader.reader);
                        oflIter.remove();
                    }
                    if (!ofl.isEmpty()) continue;
                    iter.remove();
                }
            }
            FileManager.this.closeReaders(filesToClose);
        }
    }

    private static class OpenReader
    implements Comparable<OpenReader> {
        long releaseTime;
        FileSKVIterator reader;
        StoredTabletFile file;

        public OpenReader(StoredTabletFile file, FileSKVIterator reader) {
            this.file = file;
            this.reader = reader;
            this.releaseTime = System.currentTimeMillis();
        }

        @Override
        public int compareTo(OpenReader o) {
            return Long.compare(this.releaseTime, o.releaseTime);
        }

        public boolean equals(Object obj) {
            if (obj instanceof OpenReader) {
                return this.compareTo((OpenReader)obj) == 0;
            }
            return false;
        }

        public int hashCode() {
            return this.file.hashCode();
        }
    }
}

